#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <float.h>
#include <math.h>
#include "common.h"

/* March 22, 2007: line buffer size increased to 2MB */
#define LBUFFERSIZE 2000000
#define APOOLSIZE 16

extern long nchromo, nmarkers, npermu;
extern long na, nc, nmuta;
extern char *mdata;
extern char *status, *mutation;
extern double obs_pval1[], obs_score1[];
extern double *obs_pval2;
extern double correct_locus;
extern long closest_mk;
extern char *maxallele;
extern double *location;
extern int muta_info;

static long nid;
static int dir;

char map_allele(long, int);
int original_allele(long, char);
void rec_write_tree(FILE *, struct Node *, long, long, long);

struct AllelePool {
	struct AllelePool *next;
	int cnt;
	int key[APOOLSIZE];
};

static struct AllelePool **first_apool;

static char li[LBUFFERSIZE];

/* input file format:
* [Idn]* Status [Mn]* 
*/
int read_data(FILE *f)
{
	long i, j;
	int error, m;
	char *r, *s, *ptr;

	/* fprintf(stderr, "Reading data..\n");*/
	
	/* read data: pass 1 */
	nmarkers = 0; nid = 0; error = 0;
	fgets(li, LBUFFERSIZE, f);
	
	s = li; m = 0;
	muta_info = 0;
	while (error == 0 && (r = strtok(s, " \n")))
	{
		if (m == 1)
			if (*r == 'M') nmarkers++;
			else if (*r == 'X') muta_info = 1;
			else error = 1;
		if (m == 0)
		{
			if (strcmp(r, "Status") == 0)  m = 1;
			else if (strncmp(r, "Id", 2) == 0) nid++;
			else error = 1;
		}
		s = NULL;
	}
	if (error)
	{
		fprintf(stderr, "Error on title line: %s\n", r);
		fclose(f);
		return -2;
	}

	while (fgets(li, LBUFFERSIZE, f)) nchromo++;

	fprintf(stderr, "%ld chromosomes, %ld markers\n", nchromo, nmarkers);


	/* memory allocation */
	location = (double *)malloc(nmarkers * sizeof(double));
	mdata = (char *)malloc(nmarkers * nchromo * sizeof(char));
	status = (char *)malloc(nchromo * sizeof(char));
	mutation = (char *)malloc(nchromo * sizeof(char));
	maxallele = (char *)calloc(nmarkers, sizeof(char));
	first_apool = (struct AllelePool **)calloc(nmarkers, sizeof(struct AllelePool));
	for (j = 0; j < nmarkers; j++) first_apool[j] = NULL;
	for (j = 0; j < nmarkers; j++) location[j] = j + 1.0;

	/*fprintf(stderr, "Memory allocated \n");*/
	
	/* read data: pass 2 */
	rewind(f);
	ptr = mdata;
	fgets(li, LBUFFERSIZE, f);
	na = nc = nmuta = 0;
	for (i = 0; i < nchromo; i++)
	{
		fgets(li, LBUFFERSIZE, f);
		s = li;
		for (j = 0; j < nid; j++)
		{
			if (strtok(s, " \n") == NULL)
			{
				fprintf(stderr, "Too few fields on line %ld\n", i+1);
				fclose(f);
				return -3;
			}
			s = NULL;
		}
		r = strtok(s, " \n");
		if (r == NULL)
		{
			fprintf(stderr, "Too few fields on line %ld\n", i+1);
			fclose(f);
			return -3;
		}
		if (*r != 'a' && *r != 'c')
		{
			fprintf(stderr, "Unknown status on line %ld\n", i+1);
			fclose(f);
			return -4;
		}
		else
		{
			status[i] = *r;
			if (*r == 'a') na++; else nc++;
		}
		for (j = 0; j < nmarkers; j++)
		{
			int a;
			r = strtok(s, " \n");
			if (r == NULL)
			{
				fprintf(stderr, "Too few fields on line %ld\n", i+1);
				fclose(f);
				return -3;
			}
			a = atoi(r);
			*ptr++ = map_allele(j, a);
			(void)original_allele(j, *(ptr-1));		
		}
		if (muta_info)
		{
			r = strtok(s, " \n");
			if (r == NULL)
			{
				fprintf(stderr, "Too few fields on line %ld\n", i+1);
				fclose(f);
				return -3;
			}
			mutation[i] = (char)atoi(r);
			if (mutation[i]) nmuta++;
		}
	}

	if (muta_info) fprintf(stderr, "%ld mutations\n", nmuta);
	fprintf(stderr, "done\n");
	return 0;
}

void dispose_data(void)
{
	struct AllelePool *ap, *ap2;
	long m;

	for (m = 0; m < nmarkers; m++)
	{
		ap = first_apool[m];
		while (ap)
		{
			ap2 = ap->next;
			free(ap);
			ap = ap2;
		}
	}
	free(mdata);
	free(status);
	free(mutation);
	free(maxallele);
	free(location);
}

void write_power(FILE *f)
{
	long nmin = 0, pos, m, miss = 0;
	double min = 2.0;

	for (m = 0; m < nmarkers + 1; m++)
		if (obs_pval2[m] < min) { nmin = 1; pos = m; min = obs_pval2[m]; }
		else if (obs_pval2[m] == min)
			if (rand() / (RAND_MAX + 1.0) < 1.0 / ++nmin) pos = m;

	if (closest_mk > -1)
		for (m = 0; m < nmarkers + 1; m++)
			if (obs_pval2[m] <= obs_pval2[closest_mk]) miss++;

	fprintf(f, "#Location p-value\n");
	for (m = 0; m < nmarkers + 1; m++) fprintf(f, "%9.1f %1.9f\n", m + 0.5, obs_pval2[m]);
	fprintf(f, "PREDICTED_LOC %f\n", pos + 0.5);
	if (closest_mk > -1) {
		fprintf(f, "CORRECT_LOC   %f\n", correct_locus);
		fprintf(f, "ERROR_ABS     %f\n", fabs(pos + 0.5 - correct_locus));
		fprintf(f, "FALSE_POS     %f\n", miss / (nmarkers + 1.0));
		fprintf(f, "CORR_LOC_P    %f\n", obs_pval2[closest_mk]);
	}
	fprintf(f, "MIN_PVAL      %f\n", obs_pval2[pos]);
	fprintf(f, "OVERALL_P     %f\n", total_p(obs_pval2[pos]));
}

/* obsolete
void write_pvalues(FILE *f)
{
	long m;

	fprintf(f, "#Location p-value\n");

	for (m = 0; m < nmarkers + 1; m++) fprintf(f, "%9.1f %7.5f\n", m + 0.5, obs_pval2[m]);
}*/

void debug_dump_chromo(long i)
{
	long j = 0;
	i--;
	printf("%c ", status[i]);
	for (j = 0; j < nmarkers; j++) printf(" %d", original_allele(j, mdata[i*nmarkers+j]));
	printf("\n");
}


void rec_write_tree(FILE *f, struct Node *n, long level, long pos, long shared_length)
{
	long i;
	struct ListNode *ln = n->first_child;

	for (i = 0; i < shared_length; i++) fprintf(f, "  "); 
	if (ln == NULL)
	{
		long max = n->shared_length;
		if (max > shared_length + 10) max = shared_length + 10;

		for (i = shared_length; i < max; i++)
			fprintf(f, "%2d", (int)original_allele(pos + i * dir, mdata[n->chromo * nmarkers + pos + i * dir]));
		
		if (n->shared_length > max) fprintf(f, " ...");

		fprintf(f, " :status: %c", status[n->chromo]);
		if (muta_info && mutation[n->chromo]) fprintf(f, "X");
		fprintf(f, "  chromo: %ld\n", n->chromo);
	}
	else
	{
		for (i = shared_length; i < n->shared_length; i++)
			fprintf(f, "%2d", (int)original_allele(pos + i * dir, mdata[n->chromo * nmarkers + pos + i * dir]));
		fprintf(f, " :depth: %ld  a/c: %ld/%ld  shared: %ld  mean_shared: %1.2f root_location: %1.2f chromo: %ld\n",
			n->depth, n->na, n->nc, n->shared_length, n->mean_shared, n->loc, n->chromo);

		while (ln)
		{
			rec_write_tree(f, ln->ptr, level+1, pos, n->shared_length);
			ln = ln->next;
		}
	}
}


void write_left_tree(FILE *f, struct Node *root, long pos)
{
	fprintf(f, "Left-side haplotype tree at marker %d\n", pos);

	dir = -1;
	rec_write_tree(f, root, 0, pos, 0);
}

void write_right_tree(FILE *f, struct Node *root, long pos)
{
	fprintf(f, "Right-side haplotype tree at marker %d\n", pos);
	
	dir = 1;
	rec_write_tree(f, root, 0, pos, 0);
}

void write_subscores(FILE *f, long m)
{
	int i;

	fprintf(f, "%5ld", m+1);
	for (i = 0; i < MAX_SUBTREES; i++)
		if (obs_score1[i] > -FLT_MAX) fprintf(f, " %f", obs_score1[i]);
		else fprintf(f, " -");
	fprintf(f, "\n");
}

void write_sub_pvalues(FILE *f, long m)
{
	int i;

	fprintf(f, "%5ld", m+1);
	for (i = 0; i < MAX_SUBTREES; i++)
		fprintf(f, " %8.5f", obs_pval1[i]);

	fprintf(f, "\n");
}

	
char map_allele(long m, int a)
{
	register int found = 0, ix = 0, i;
	register struct AllelePool *ap = first_apool[m], *ap2 = NULL;
		
	/* printf("Mk = %ld, orig: %d; ", m, a); */

	if (a == 0) return 0;
	a--;
	while (found == 0 && ap)
	{	
		/* printf("cnt = %ld; ", ap->cnt); */
		for (i = 0; i < ap->cnt; i++, ix++) 
			if (ap->key[i] == a) { found = 1; break; }
		if (i == APOOLSIZE) { ap2 = ap; ap = ap->next; }
		else break;
	}
	/* if (found) printf("i = %d; ", i); */
	if (found == 0)
	{
		if (ap == NULL)
		{
			/* printf("Allocate pool; "); */
			ap = (struct AllelePool *)malloc(sizeof(struct AllelePool));
			if (first_apool[m] == NULL) { first_apool[m] = ap; /* printf("head; "); */ }
			else ap2->next = ap;
			ap->next = NULL; ap->cnt = 0;
		}
		/* printf("cnt = %d; ", ap->cnt); */
		ap->key[ap->cnt++] = a;
		maxallele[m]++;
		/* printf("(not found); mapped: %d -> %d\n", ap->key[ap->cnt-1] + 1, ix+1); */
	}
	/* else printf("(found); mapped: %d -> %d\n", ap->key[i] + 1, ix+1); */
	return ix + 1;
}

int original_allele(long m, char c)
{
	int ix = 0;
	struct AllelePool *ap = first_apool[m];

	/* printf("Mk = %ld, mapped: %d; ", m, (int)c); */
	
	if (c == 0) return 0;
	c--;
	while (c >= ix + APOOLSIZE)
	{
		ap = ap->next;
		ix += APOOLSIZE;
	}

	/* printf("orig: %d\n", ap->key[c-ix] + 1); */
	return ap->key[c - ix] + 1;
}
