#include <stdio.h>
#include <stdlib.h>
#include "common.h"
#define POOLSIZE 1024

extern long nmarkers, nchromo;
extern long na, nc;
extern char *maxallele;
extern char *mdata;
extern struct Node *root;
extern double *location;

long *sortlist;

static long pos, sortlistbase;
static struct Node *root_a[256], *last_node[256];
static struct Node *first_free_node = NULL;
static struct ListNode *first_free_listnode = NULL;

struct NodePool {
	struct NodePool *next;
	struct Node node[POOLSIZE];
};

struct ListNodePool {
	struct ListNodePool *next;
	struct ListNode listnode[POOLSIZE];
};

static struct NodePool *first_npool = NULL;
static struct ListNodePool *first_lnpool = NULL;

int rec_update(struct Node *);
void update(void);
void treelink(struct Node *, struct Node *);
struct Node *allocate_node(void);
struct ListNode *allocate_listnode(void);
void free_node(struct Node *);
void free_listnode(struct ListNode *);
void free_subtree(struct Node *, int);
long finalize_subtree(struct Node *, long, long);
void init_sort(void);

void allocate_sortlist(void)
{
	sortlist = (long *)malloc(nchromo * nmarkers * sizeof(long));
}

void free_sortlist(void)
{
	free(sortlist);
}

void init_left_sort(void)
{
	init_sort();
	pos = 0;
	root->loc = 0;
}

void init_right_sort(void)
{
	init_sort();
	pos = nmarkers - 1;
	root->loc = 0;
}

void init_sort(void)
{
	long i;
	
	/* printf("Initializing sort..\n"); */
	root = allocate_node();
	root->depth = 0; root->min = 0; root->max = nchromo - 1; root->nchr = nchromo;
	root->shared_length = 0; root->chromo = 0;
	for (i = 0; i < nchromo; i++)
	{
		struct Node *n = allocate_node();
		treelink(root, n);
		n->min = n->max = n->chromo = i; n->nchr = 1;
		n->depth = 1; n->shared_length = 0;
		n->mean_shared = 0; n->loc = 0;
	}
	/* printf("done\n"); */
}

void finalize_sort(void)
{
	/* printf("Finalizing sort..\n"); */
	free_subtree(root, 1);
	while (first_npool)
	{
		struct NodePool *np = first_npool->next;
		free(first_npool);
		first_npool = np;
	}
	while (first_lnpool)
	{
		struct ListNodePool *lnp = first_lnpool->next;
		free(first_lnpool);
		first_lnpool = lnp;
	}
	first_free_node = NULL; first_free_listnode = NULL;
	/* printf("done\n"); */
} 

void left_update(void)
{
	update();
	root->loc = location[pos];
	pos++;
}

void right_update(void)
{
	update();
	root->loc = location[pos];
	pos--;
}

int rec_update(struct Node *n)
{
	struct ListNode *ln = n->first_child;

	if (ln == NULL) /* leaf */
	{
		char c = mdata[pos+nmarkers*n->chromo];
		struct Node *n2 = last_node[c], *n1 = n;	/* n1: current source node, n2: most recent new node for allele c */
		struct Node *nn1 = allocate_node(), *nn2;   /* nn1: new leaf node, nn2: possible new internal node */
		long last_chr, next_chr = n->min;			/* in source order */
		
		/* printf("chr: %ld, allele: %d, ", n->chromo, (int)c); */

		if (n2) last_chr = n2->min; else last_chr = next_chr;
		nn1->min = nn1->max = next_chr; nn1->depth = n->depth; nn1->chromo = n->chromo;
		nn1->shared_length = n->shared_length + 1; nn1->loc = n->loc;

		/* find the lowest common ancestor of last_chr & next_chr in the source tree */
		while (n1->min > last_chr) n1 = n1->father;
		
		while (n2 && n2->depth > n1->depth) n2 = n2->father;
		if (root_a[c] == NULL) root_a[c] = nn1;  /* first chromosome */
		else if (n2 && n2->depth == n1->depth) treelink(n2, nn1); /* new leaf only */
		else /* new internal node & leaf */
		{
			nn2 = allocate_node();
			nn2->min = n1->min; nn2->max = n1->max; nn2->depth = n1->depth; nn2->chromo = n->chromo;
			nn2->shared_length = n1->shared_length + 1; nn2->loc = n1->loc;
			if (n2 == NULL) /* new internal node = root */
			{
				/* printf("new root "); */
				treelink(nn2, root_a[c]);
				root_a[c] = nn2;
			}
			else
			{
				struct ListNode *l = n2->last_child;
				treelink(nn2, l->ptr);
				n2->last_child = l->prev;		/* unlink */
				n2->last_child->next = NULL;
				free_listnode(l);
				treelink(n2, nn2);
			}
			/* printf("I&L\n"); */
			treelink(nn2, nn1);
		}

		last_node[c] = nn1;
		return c;
	}

	else /* internal node */
	{
		int c, c_ret = 0, c_cmp = rec_update(ln->ptr);

		if (c_cmp == -1) c_ret = -1;
		ln = ln->next;
		
		/* debug */ if (ln == NULL) { fprintf(stderr, "Malformed tree\n"); exit(-1); }

		while (ln)
		{
			c = rec_update(ln->ptr);
			if (c == -1 || c != c_cmp) c_ret = -1;
			ln = ln->next;
		}
		if (c_ret == -1) add_pattern(n); /* n is closed */
		return c_ret;
	}
}


void update(void)
{
	long k;
	int n = 0;

	/* printf("update %ld, maxa = %d\n", pos, maxallele[pos]); */

	for (k = 0; k <= maxallele[pos]; k++) root_a[k] = last_node[k] = NULL;

	/* printf("Recursively updating..\n"); */
	(void)rec_update(root);
	/* printf("done\n"); */

	/* printf("Freeing old tree..\n"); */
	free_subtree(root, 0);
	/* printf("done\n"); */

	root = allocate_node();

	for (k = 0; k <= maxallele[pos]; k++) if (root_a[k]) { treelink(root, root_a[k]); n++; }

	if (n == 1)
	{
		struct Node *n = root;
		root = root->first_child->ptr;
		free_node(n);
		fprintf(stderr, "Marker %ld is monomorphic\n", pos + 1);
	}

	sortlistbase = pos * nchromo;

	/* printf("Finalizing new tree..\n"); */
	(void)finalize_subtree(root, 0, 0);
	/* printf("done\n"); */
	
}

long finalize_subtree(struct Node *n, long min, long depth)
{
	struct ListNode *ln = n->first_child;

	n->min = min; n->depth = depth;
	if (ln) /* internal node */
	{
		while (ln)
		{
			min = finalize_subtree(ln->ptr, min, depth+1) + 1;
			ln = ln->next;
		}
		n->max = min - 1; n->nchr = n->max - n->min + 1;
	}
	else
	{
		sortlist[min + sortlistbase] = n->chromo;
		n->max = min; n->nchr = 1;
	}
	return n->max;
}

void free_subtree(struct Node *n, int last)
{
	struct ListNode *ln = n->first_child;

	if (last && n->shared_length > 0) add_pattern(n);

	while (ln)
	{
		struct ListNode *ln2 = ln->next;
		free_subtree(ln->ptr, last);
		free_listnode(ln);
		ln = ln2;
	}
	free_node(n);
}


void treelink(struct Node *n1, struct Node *n2)
{
	register struct ListNode *l1 = allocate_listnode();

	n2->father = n1;
	l1->ptr = n2; l1->prev = n1->last_child;
	if (n1->last_child) n1->last_child->next = l1; else n1->first_child = l1;
	n1->last_child = l1;
}

struct Node *allocate_node(void)
{
	struct Node *n;

	if (first_free_node == NULL)
	{
		struct NodePool *np = (struct NodePool *)malloc(sizeof(struct NodePool));
		long i;

		/*printf("ALLOCATING NPOOL\n");*/
		np->next = first_npool; first_npool = np;

		for (i = 0; i < POOLSIZE - 1; i++) np->node[i].next = &(np->node[i+1]);
		np->node[POOLSIZE-1].next = NULL;
		first_free_node = &(np->node[0]);
	}
	n = first_free_node; first_free_node = first_free_node->next;
	n->first_child = n->last_child = NULL; n->father = n->next = NULL;
	return n;
}

struct ListNode *allocate_listnode(void)
{
	struct ListNode *ln;

	if (first_free_listnode == NULL)
	{
		struct ListNodePool *lnp = (struct ListNodePool *)malloc(sizeof(struct ListNodePool));
		long i;

		/*printf("ALLOCATING LNPOOL\n");*/
		lnp->next = first_lnpool; first_lnpool = lnp;

		for (i = 0; i < POOLSIZE - 1; i++) lnp->listnode[i].next = &(lnp->listnode[i+1]);
		lnp->listnode[POOLSIZE-1].next = NULL;
		first_free_listnode = &(lnp->listnode[0]);
	}
	ln = first_free_listnode; first_free_listnode = first_free_listnode->next;
	ln->next = ln->prev = NULL; ln->ptr = NULL;
	return ln;
}

void free_node(struct Node *n)
{
	n->next = first_free_node; first_free_node = n;
}

void free_listnode(struct ListNode *ln)
{
	ln->next = first_free_listnode; first_free_listnode = ln;
}
