#!/usr/bin/env python
# -*- coding: utf-8 -*-

from plasTeX.Renderers import Renderer as BaseRenderer
import textwrap, re, string, sys, os.path, codecs
from plasTeX.ConfigManager import *
from plasTeX.DOM import Node

class ParempiRenderer(BaseRenderer):
    """Renderer for plain text documents."""
    
    # default values (Use plasTeXrc configuration file to customize these values.)
    outputType = unicode
    fileExtension = '.txt'

    conf = { # configurable variables and their default values. will also be set under self while processing document-node
        "wrapText":True, # Wrap long lines to multiple lines.
        "lineWidth":76,
        "itemMark":" - ", # Appears before an item in itemize. Also sets the indentation length, if item content is wrapped.
        "enumMark":" %s. ", # Appears before an item in enumerate. %s is replaced with item count. Also sets the indentation length, if item content is wrapped.
        "emphasizeFormat":'{em %s}', #if changed: default value also in code
        "boldfaceFormat":'{bf %s}', #if changed: default value also in code
        "quotePre":"\\begin{quote}", # line before quote
        "quotePost":"\\end{quote}", # line after quote
        "quoteIndent":3, # Amount of whitespace characters before each line of quoted content.
        "refFormat":"{ref%s}", # %s is the count number
        "labelFormat":"{label%s}", # %s is the count number
        "pagerefFormat":"{pageref%s}", # %s is the count number
        "cellspacingHorizontal":1, # empty spaces between cells or vertical lines: [spacing] cell1 [spacing] cell2 [spacing] etc.
        "dollarsInText":False,
        "symbolProfile":'hlub', # default profile for rendering math symbols
        "mathIndent":1,
        "renderUndefinedCommands":True,
        "formulaNumbering":25  #Amount of whitespace charechters before numbering in equations and alings
    }

    fetch_func = { # variable types : fetching function names (from configmanager) -mapping. just a helper.
        type(True):"getboolean",
        type(42):"getint",
        type("string"):"get_opt",
    }

    aliases = { # our names for bad python characters
        'superscript': '^',
        'dollar': '$',
        'percent': '%',
        'opencurly': '{',
        'closecurly': '}',
        'subscript': '_',
        'underscore': '_',
        'ampersand': '&',
        'hashmark': '#',
        'backslash': '\\',
        'star':'*',
    }

    ignoredMath = set() # Contains those math environment commands that will be ignored
    symbols = dict() # mapping from math commands to math command output
    unknownListFile = None # the file that holds the list of the unknown nodes met.
    unknownsSoFar = set() # a way to avoid duplicates in the list of the unknown nodes met.
    lrdict = dict() # label/ref dictionary that maps from internal labels to the count of custom labels.
    footnotes = [] # Stack containing expanded footnotes temporarily, until it is their time.
    globalIndent = [] # Stack containing the current indentation. In pieces, as nested structures may have multiple indentations in effect.
    figureIterator = -1 # used in do_figure, do_caption
    slideIterator = 0 # used in do_slide
    
    def __init__(self, *args, **kwargs):
        """
        Constructor, which has following responsibilities
            - fills the dictionary of functions in self
            - loads general configuration file C{~/.plasTeXrc} with C{ConfigManager}
            - possibly sets some functions to point at the ignore function, depending on the configuration
            - loads math configuration file C{~/.ParempiMathrc} with C{ConfigManager}
        """

        BaseRenderer.__init__(self, *args, **kwargs)
        
        # Load dictionary with methods
        for key in vars(type(self)):
            if key.startswith('do___'): # typical name that ends with *
                self[key[5:]+self.aliases["star"]]=getattr(self, key)
            elif key.startswith('do__'): # bad python characters
                self[self.aliases[key[4:]]] = getattr(self, key)
            elif key.startswith('do_'): # typical case
                self[key[3:]] = getattr(self, key)

        self['default-layout'] = self['document-layout'] = self.textDefault # unknown, this line removes warnings for now

        # Load rendering settings from the configuration file
        c = ConfigManager() # Use ConfigManager for parsing the INI-file
        c.read("~/.plasTeXrc") # Search from home directory

        for key in self.conf: # for each relevant variable name
            if c.has_option("parempi",key):
                self.conf[key]=getattr(c,self.fetch_func[type(self.conf[key])])("parempi",key) # fetched function depends on variable type

        if(c.has_option("parempi", "ignore")):
            # have all ignored commands (tags) point to ignore-function
            for key in re.split(",", str(c.getraw("parempi","ignore")) ):
                if key != '':
                    self[key.strip()] = getattr(self, 'ignore')

        if(c.has_option("parempi", "mathignore")):
            # collect those math commands, that we want to ignore
            for key in re.split(",", str(c.getraw("parempi","mathignore")) ):
                if key != '':
                    self.ignoredMath.add(key.strip())

        # Load settings for rendering math symbols
        m = ConfigManager() # Use ConfigManager for parsing the math INI-file
        m.read("~/.ParempiMathrc")    # Load math symbols

        if m.has_section(self.conf["symbolProfile"]): # perhaps configmanager should be a bit better here
            for var in m.options(self.conf["symbolProfile"]):
                self.symbols[var]=m.get_opt(self.conf["symbolProfile"],var) # overrides or creates new

    def logUnknowns(self, node, inmath=False):
        """
        Handles the logging of unknown node names into a text file.
        """
        if node.nodeName not in self.unknownsSoFar: # log each new unhandled command type to filename+_unknowns.txt, if there are any
           self.unknownsSoFar.add(node.nodeName)
           if self.unknownListFile is None:
               try:
                   self.unknownListFile=open(node.ownerDocument.userdata['jobname']+"_unknowns.txt",'w',0)
               except:
                   print "Error while trying to open the file for writing the list of the unknown commands"
           try:
               if inmath:
                   self.unknownListFile.write("Under math environment: "+node.nodeName+"\n")
               else:
                   self.unknownListFile.write(node.nodeName+"\n")
           except:
               print "Error while writing a log entry of an unknown command to the file of the unknown commands"

    def default(self, node):
        """
        Default function for all undefined macros and commands. Logs all
        unrecognised commands to I{<filename>_unknowns.txt}. If option
        C{renderUndefinedCommands} is set in general configuration file, this
        function will render unrecognised commands the same way they are in the
        input file.

        """

        self.logUnknowns(node)

        if self.renderUndefinedCommands == False:
            return unicode(node)
        else: 
            # First \begin{command} ... \end{command}
            # Check if undefined command starts with '\begin'
            if node.source[0:7].find('\\begin')==0:
                nodestr = u'\n\\begin{' + node.nodeName + u'}'
                return unicode(nodestr) + unicode(node)
                
            # Check if an undefined command starts with '\end'
            elif node.source[0:4].find('\\end')==0:    
                return u'\end{' + node.nodeName + '}'
            
            # Undefined command is a single command in form of '\undefined'
            else:
                nodestr = unicode(node)
                # These commands plastex somehow is able to recognize
                if len(nodestr)>0:
                    return unicode('\\'+node.nodeName + '{' + unicode(node) + '}')
                # These commands plastex doesn't recognize at all
                else:
                    if node.nextSibling is not None:
                        # Assume that command is in form \command{}
                        # Thus, bgroup is implicitly included in the command
                        if node.nextSibling.nodeName == 'bgroup':
                                # Parser works wrong for really unknown nodes
                                # Thus, fix the problem by  putting the next sibling 
                                # as a child for this node
                                node.append(node.nextSibling)
                                node.parentNode.removeChild(node.nextSibling)
                                return unicode('\\'+node.nodeName+'{'+unicode(node)+'}')
                        else:
                            return unicode('\\'+node.nodeName+' ')
                    else:
                        return unicode('\\'+node.nodeName+' ')
    
    def textDefault(self, node):
        """
        Renders node as text, if it has a valid text form, or ignores it.
        Continues rendering document in any case.
        """
        return unicode(node)
    
    def ignore(self, node):
        """
        Continues the recursion in the node-tree.
        Meant for hiding unwanted nodes, while allowing them to act as capsules around desirable content.
        """
        return unicode(node)

    def do_document(self, node): # should be the first node met. atm used for some initialization, solely because of jobname and for overriding.
        """
        Should be the first node rendered. The root node of the node-tree that represents the .tex document.
        Currently used for initialization, where jobname (name of the input file without its extension) is required.
        """
        override=node.ownerDocument.userdata['jobname']+".ini"
        o=ConfigManager()
        o.read(override)
        
        for key in self.conf: # for each relevant variable name
            if o.has_option("parempi",key):
                self.conf[key]=getattr(o,self.fetch_func[type(self.conf[key])])("parempi",key) # fetched function depends on variable type
        
        if(o.has_option("parempi", "ignore")):
            # have all ignored commands (tags) point to ignore-function (in addition to the ones in main config)
            for key in re.split(",", str(o.getraw("parempi","ignore")) ):
                if key != '':
                    self[key.strip()] = getattr(self, 'ignore')

        if(o.has_option("parempi", "mathignore")):
            # collect those math commands, that we want to ignore (in addition to the ones in main config)
            for key in re.split(",", str(o.getraw("parempi","mathignore")) ):
                if key != '':
                    self.ignoredMath.add(key.strip())
        
        if o.has_section(self.conf["symbolProfile"]): # perhaps configmanager should be a bit better here
            for var in o.options(self.conf["symbolProfile"]):
                self.symbols[var]=o.get_opt(self.conf["symbolProfile"],var) # overrides or creates new
        # NOTE: if the symbolProfile is named as 'parempi', ordinary config will get mixed in with math symbols!

        for var in self.conf: # configuration is now known, time to set the variables (not necessary if code always refers to the dict)
            setattr(self,var,self.conf[var])
        
        # also delete filename _unknowns.txt for better usability (for unknown command logging)
        try:
            os.remove(node.ownerDocument.userdata['jobname']+"_unknowns.txt")
        except:
            pass
        return unicode(node)

    #ignore artificial nodes automatically.. and others?
    do_bgroup=do_newblock=ignore
    do_bibliography=ignore
    do_hline=do_multicolumn=ignore # These nodes will be rendered in a tabular, so there is no need to handle them.

    def nothing(self, node):
        """
        This function returns an empty string. These nodes will be processed
        elsewhere, and their content should not appear in their place.
        """
        return u''
    
    do_bibliographystyle=do_author=do_date=do_thanks=do_title=nothing

    def fill(self, s, **kwargs): 
        """
        Returns the string in a wrapped form (with newlines).
        In case globalIndent is not empty, uses its last item as the indentation mark for the first line, while ensuring the desired amount of indentation for all lines.

        @type s:    string
        @param s:   unwrapped text
        @return:    wrapped text
        """
        gi_length=len("".join(self.globalIndent)) # globalIndent-code ensures indent+bullet for the first line, and indent+bullet worth of whitespace as the indent for the following lines
        if len(self.globalIndent)==0:
            gi_last=""
        else:
            gi_last=self.globalIndent[len(self.globalIndent)-1]

        if self.wrapText == True:
            return textwrap.fill(unicode(s).strip(), self.lineWidth, initial_indent=u" "*(gi_length-len(gi_last))+unicode(gi_last),subsequent_indent=" "*gi_length,break_long_words=False, **kwargs)
        else: # Keep text (e.g. paragraphs) in single lines.
            return u" "*(gi_length-len(gi_last))+unicode(gi_last)+unicode(s).strip()
    
    def do_tabular(self, node, render=unicode):
        """
        For rendering tabular-environments
        features:
            - hlines
                - single hline:        -> '---------------'
                - two or more hlines:  -> '==============='
            - vlines                   -> '|' chars
            - alignments
                - table contents get left justified, right justified or centered
                  according to the parameters in 'colspec' attributes
            - horizontal cellspacing: 
                - This adds empty space between cells or between cells and
                  vertical lines '|'. There is no spacing before the first 
                  cell if there are no vlines before it. Similarly there is
                  no spacing after the last cell if there are no vlines after
                  it. 
                - e.g. C{cell1 [spacing] '|' [spacing] cell2 [spacing] '||'}
           - multicolumns
                - Multicolumns are implemented in a natural way, but they do not
                  have own borders. Tabular's 'colspec' is used for determining
                  their borders. 
         
        I{Notes:}  'hline' after the last row won't be rendered because
        the 'tabular'-node misses it for some unknown reason.
        """
        class Cell:
            """ Class for cell elements, used in do_tabular """
            def __init__( self, cellSpan = 1, content = '',  cStart=0, cEnd=0, leftVlines = 0, rightVlines = 0):
                self.span = cellSpan    # If span is one, the cell is a normal cell, otherwise it is multicolumned
                self.content = content  # Actual content of the cell
                self.colStart = cStart  # Starting column of the cell. 
                self.colEnd = cEnd      # Ending column of the cell. For normal cells, colStart and colEnd are equal.
                self.spaceReserved = 0  # Minimun length of the cell. This is needed when counting widths of the multicolumned cells.
                self.closed = False     # This is used for rendering cells from left to right. Cells remain open until colEnd is reached.
                self.align = 'l'        # Default alignment of the cell
            def size(self):
                return len(self.content)
            def close(self):            # State of the cell; when a cell is closed, its final width is known.
                self.closed=True
        
        # The 'cellspacing' variable indicates how much empty space gets added
        # between cells or between cells and vertical lines '|'.
        cellspacing = self.cellspacingHorizontal
        
        # the lists to store hlines and 'colspec' attributes in more usable format
        hlines = []     # the numbers of hlines in each row
        colaligns = []  # the alignments to be used in each column
        vlines = []     # the numbers of 'vlines' in each column
                        
        # convert 'colspec' to string and remove whitespacies
        argstr = re.sub(r'\s', '', str(node.attributes['colspec']))

        # Replace p{...}, m{...} and b{...} by a left-justified column 'l' because
        # these arguments are not implemented.
        argstr = re.sub(r'[pmb]{\w*}', 'l', argstr)
        
        # Initialize vlines and colaligns with zeros and enpty chars
        for i in range(0, len(re.sub(r'[^lcr]', '', argstr))): # range(0, number of columns)
            vlines.append(0)
            colaligns.append('')
        vlines.append(0)    # one more place for the rightmost vlines
          
        # Parse vertical lines and column alignments from the 'argstr' string.
        colIndex = 0 # current column being prosessed
        for attr in argstr:
            
            if attr == '|':             # vline
                vlines[colIndex] += 1   # increase the number of vlines at the current column
            
            elif attr in 'lcr':         # chars indicating columns
                colaligns[colIndex] = attr  
                colIndex += 1           # iterate to the next column
        
        # Parse hlines and cells. Cells can be single or multicolumned cells.
        rows = []
        for r, row in enumerate(node.childNodes):   # Iterate all rows
            cells = []
            curColumn = 0       # Starting from the left, curColumn indicates the first column.
            hlines.append(0)    # Reserve space for hlines in each row. 
            
            for c, cell in enumerate(row.childNodes):   # Iterate all cells
                try:
                    align = colaligns[c]
                except BaseException, (ErrorMessage):
                        print 'WARNING: error in %s: %s, its content will not be rendered' % (node.nodeName, ErrorMessage) 
                        return u''  # In case of error, abort processing and return empty string.
                
                colspan = 1
                for cellPar in cell:
                    cell.align = argstr[c]
                    for parChild in cellPar:
                        if parChild.nodeName == 'hline':
                            hlines[r] += 1
                        if parChild.nodeName == 'multicolumn':
                            colspan = parChild.attributes['colspan']
                            multicolumnArgstr = parChild.attributes['colspec'] 
                            for attr in multicolumnArgstr:
                                if attr in 'lcr':       # chars indicating columns
                                    align = attr
                           
                contentStr = render(cell).strip()   
                contentStr = re.sub(r'\n', '', contentStr) # remove newlines
                curCell = Cell(colspan, contentStr, curColumn, curColumn+colspan-1)
                curCell.align = align
                cells.append(curCell);
                curColumn += colspan
            rows.append(cells)  
        
        # Count total amount of columns
        colSum = 0  
        for cell in rows[0]:
            colSum += cell.span
        
        activeCells = []    # This is needed for processing multicolumned cells.
        curCol = 0          # current column
        colWidth = 0        # This is used for determining correct width of the column.
        
        # Calculate widths of the cells and expand cells to correct form from the left to the right in tabular
        for curCol in range(0, colSum): # Start from the leftmost column
            colWidth=0
            for r in rows:
                for cell in r:
                    # Set all cells starting from current column active
                    if cell.colStart == curCol:
                        activeCells.append(cell)   # Put all cells starting from the current column to list being processed.
            
            # Count common length for all cells in the current column.
            for cell in activeCells:
                
                # Calculate the widest widht of the column.
                if curCol == cell.colEnd:  # When cell ends, it width can be determined.
                    if cell.span == 1:
                        colWidth = max(colWidth , cell.size() ) 
                    
                    else:
                        colWidth = max(colWidth, cell.size() - cell.spaceReserved)
                    
            # Process open cells
            for cell in activeCells:
                
                # When processing has reached cell's end, it will be rendered to correct width and closed.
                # Normal cell starts and ends from the same column, but multicolumned cells are wider.
                if (curCol == cell.colEnd) and (not cell.closed):
                    
                    if cell.span == 1:  # normal cell
                        cell.spaceReserved = colWidth
                   
                    else: # Multicolumned cell
                        inc = colWidth 
                        if vlines[curCol] > 0:
                            inc += vlines[curCol]
                            inc += 1
                        inc += cellspacing
                        cell.spaceReserved += inc
                    
                    leftborder = vlines[cell.colStart] * '|' # Add vlines to left of a cell
                    
                    # Add empty space after vline
                    if vlines[cell.colStart] > 0:
                        leftborder += cellspacing * ' '
                    if cell.colStart > 0:
                        leftborder = cellspacing * u' ' + leftborder # empty space after vline
                    rightborder = u''
                    
                    # Add rightborder if cell being processed is in the rightmost column
                    if curCol == colSum - 1:
                        # Add vlines to the right of the cell if it ends to the last column
                        if cell.span == 1 and vlines[colSum] > 0:
                            rightborder = cellspacing * u' ' + vlines[curCol + 1] * '|'
                        # Or if the case is a multicolmned cell 
                        elif vlines[colSum] > 0:
                            rightborder =   cellspacing * u' ' + vlines[cell.colEnd + 1] * '|'
                    
                    extra = 0
                    if cell.span > 1 and cell.colStart != 0: # if cell is multicolumned, don't count space reserved for leftborder
                            extra = len(leftborder)
                    if cell.align == 'l':
                        cell.content = leftborder + cell.content.ljust(cell.spaceReserved - extra, ' ') + rightborder
                    if cell.align == 'c':
                        cell.content = leftborder + cell.content.center(cell.spaceReserved - extra, ' ') + rightborder
                    if cell.align == 'r':
                        cell.content = leftborder + cell.content.rjust(cell.spaceReserved - extra, ' ') + rightborder
                    
                    cell.close()   # Current cell has been processed.
                
                # Process cell that has not yet reached end (multicolumn) and thus stays active in the processing list.
                # Length of the column will be added to cell's spaceReserved variable.
                elif cell.colStart <= curCol and (not cell.closed):
                    
                    if curCol == 0:                     # if current column is the leftmost
                        cell.spaceReserved += colWidth  # add space corresponding the width of the widest cell in this column
                    
                    else:
                        inc = colWidth              # add space corresponding the width of the widest cell in this column
                        inc += cellspacing       # add space reserved for cellspacing (whitespacies) that is between each cell
                        if vlines[curCol] > 0:      # add space reserved for vlines and cellspacing of the vlines
                            inc += vlines[curCol]   # in the beginning of the first line, there is no cellspacing
                            inc += 1
                        cell.spaceReserved += inc
        
        # Calculate width of the array from the first row
        tableWidth = 0
        for cell in rows[0]:
            tableWidth = tableWidth + cell.size()
        
        # Render rows to output
        output = []
        for rIndex, row in enumerate(rows):
            if hlines[rIndex] == 1: output.append(tableWidth*'-' + '\n')
            elif hlines[rIndex] >1: output.append(tableWidth*'=' + '\n')
            for c in row:
                output.append(c.content)
            output.append('\n')   
        
        return u''.join(output)

    def do_array(self, node):
        """
        Uses do_tabular function with mathArrayCell as the render function.
        """
        
        return self.do_tabular(node, render = self.mathArrayCell)

    def mathArrayCell(self, cellNode):
        """
        This is a small renderer for do_array's cell, that is needed to be able to use do_tabular function in do_array.
        It calls do_math for all childs of the given cell and combines the return values into a single string.
        """
        
        cellOutput = []
        for child in cellNode:
            cellS = self.do_math(child)
            cellOutput.append(cellS)
        return u''.join(cellOutput).strip()

    def do_table(self, node):
        return unicode(node)

    def do_thebibliography(self, node):
        """
        Responsible for rendering the bibliography list.
        """
        term = 'Bibliography'   # default
        
        # Get a language specific term, that corresponds to 'ref' 
        # in dictionary 'terms' (see file 'i18n.xml').
        try :
            term = node.ownerDocument.context.terms['ref']
        except KeyError:    # catch the error, handle it and continue
            print "WARNING (parempi): Didn't find a key 'ref' term from the dictionary."
                 
        output = unicode(term + '\n\n')
        for item in node:

            itemstr = u'%s' % item.bibcite  # Convert to the Unicode string
            
            if re.search(r"\w{3}\$\^\{\+\}\$\d\d", itemstr):
                # Handle a special case where bibstyle is 'alpha' or similar
                # and the current bibitem in the .aux file contains more than four authors.
                # Thus bibitem is in the form of '\bibcite{ABC}{ABC{$^{+}$}94}' giving 
                # a citation in a form '[ABC{$^{+}$}94]' which we'll convert to the form '[ABC+94]'.
                itemstr = re.sub(r'[\$\{\}\^]', '', itemstr) # Remove characters '${}^'.

            bullet = u'[%s] ' % itemstr
            oneitem=unicode(item) #fill is called here
            self.globalIndent.append(bullet)
            output=output+self.fill(oneitem)+u'\n\n'
            self.globalIndent.pop()
        return output
  
    def do_cite(self, node):
        """
        Responsible for rendering a citation.
        """
        output = []
        for item in node.citation():
            itemstr = u'%s' % item # Convert to the Unicode string
            if re.search(r"\w{3}\$\^\{\+\}\$\d\d", itemstr):
                # Handle a special case where bibstyle is 'alpha' or similar
                # and the current bibitem in the .aux file contains more than four authors
                # in the same way as in the do_thebibliography.
                itemstr = re.sub(r'[\$\{\}\^]', '', itemstr) # Remove characters '${}^'.
            output.append(unicode(itemstr))
        return u''.join(output)

    def do_ref(self, node):
        """
        Renders a number surrounded by the user configurable ref string. The number is unique for each referenced label in the source .tex.
        """
        if node.attributes['label'] not in self.lrdict: # if the internal label is not in our mapping
            self.lrdict[node.attributes['label']]=len(self.lrdict) # let's add it with the count of mappings as the value
        return unicode(self.refFormat.replace("%s",str(self.lrdict[node.attributes['label']]),1))

    def do_pageref(self, node):
        """
        Renders a number surrounded by the user configurable pageref string. The number is unique for each referenced label in the source .tex.
        """
        if node.attributes['label'] not in self.lrdict:
            self.lrdict[node.attributes['label']]=len(self.lrdict)
        return unicode(self.pagerefFormat.replace("%s",str(self.lrdict[node.attributes['label']]),1))

    def do_label(self, node):
        """
        Renders a number surrounded by the user configurable label string. The number is unique for each label.
        """
        if node.attributes['label'] not in self.lrdict:
            self.lrdict[node.attributes['label']]=len(self.lrdict)
        return unicode(self.labelFormat.replace("%s",str(self.lrdict[node.attributes['label']]),1))

    # Assumes that unicode(node) returns a string that is once separeted by \
    # This is quaranteed by do_includegraphics
    def do_figure(self, node):
        """
        Used as a parent environment for caption and includegraphics. Takes care of
        correct output format
        """
        # First take the text from do_includegraphics, do_caption to a variable
        helper = unicode(node)
        if len(helper)>0:
            # Now change the text a little
            helper2 = helper.split(' ')
            helper = []
            # Get rid of blanks where not needed
            for x in helper2:
                if x != '':
                    helper.append(x)
            helper = ' '.join(helper)
            # Get rid of row changes
            helper = ''.join(helper.split('\n'))
            # Use the separator '\' to change the direction
            helper = helper.split('\\')
            if(len(helper)==2):
                stack = []
                stack.append('{')
                stack.append(helper.pop().strip()+' ')
                stack.append(helper.pop())
                # Find page numbers in pdf
                helper = ""
                fileName= node.ownerDocument.userdata['jobname']
                workingDir = node.ownerDocument.userdata['working-dir']
                wholePath= ''.join([workingDir,'/',fileName,'.aux'])
                if os.path.isfile(wholePath):
                    openfile = codecs.open(wholePath, 'r',"utf-8")
                    filecontent = openfile.read().split('\n')
                    if filecontent:
                        for content in filecontent:
                            if  re.match(r'\\\@writefile\{lof\}\{\\contentsline',content):
                                content = re.sub('^((.)+\\\\numberline {)()','',content)
                                content = content.replace('}',' ',1)
                                if str(self.figureIterator) == content[0]:
                                    content = re.sub('.+}}{',"",content)
                                    helper = content.split('}')[0]

                else:
                    print('WARNING (parempi): couldn\'t find page numbers for figures, .aux file is missing')

                # Find name for page from i18.xml
                term = 'Page' # default
                # Get a language specific term, that corresponds to 'page'
                # in dictionary 'terms' (see file 'i18n.xml').
                try :
                    term = node.ownerDocument.context.terms['page']
                except KeyError:    # catch the error, handle it and continue
                    print "WARNING (parempi): Didn't find a key 'page' term from the dictionary."

                stack.append('('+term+' '+helper+')}')
            # Something wrong with either includegraphics or caption
            else:
                stack = []
                stack.append('{')
                stack.append(u''.join(helper))
                stack.append('}')
                print 'WARNING (parempi): something wrong inside figure environment. Check figure with output: '+u''.join(stack)
            # Ensure that in .tex-file \caption{} is used only once inside \begin{figure}
            self.figureIterator = -1
            return self.fill(u''.join(stack))
        else:
            print 'WARNING (parempi): something wrong inside figure environment, check output for missing includegraphics, caption statements'
            return u'{Figure ? PAREMPI RENDERER: missing includegraphics, caption statements inside figure environment!}'

    # Needs \usepackage{graphics} or \usepackage{graphicsx}
    def do_includegraphics(self, node):
        """
        Child environment for figure, outputs filename
        """
        s = []
        if node.hasAttributes():
            # keys available are loc,ll, file, *modifier*, ur                 
            for key, value in node.attributes.items():
                if key == 'file':
                    s.append('('+value+')')
        return u''.join(s)+'\\'

    # Caption-class in Base.LaTeX.Floats produces output "Figure X:"
    # Assumes that do_caption is called after do_figure only once
    # This way the correct page number can be found in do_figure using self.figureIterator
    def do_caption(self, node):
        """
        Child environment for figure, outputs 'Figure X:' by using Base.LaTeX.Floats.Caption
        """
        if(self.figureIterator != -1):
            print "WARNING (parempi): \caption used twice inside \begin{figure} in .tex-file"
        # Search for a parentNode named figure, to check wheter dealing 
        # with a figure caption. Thus it is assumed, that any other 
        #caption-using environment is used inside figure-environment 
        parent = node.parentNode
        while(parent.nodeName != 'figure' and parent.nodeName != 'document'):
            parent = parent.parentNode
        if(parent.nodeName == 'figure'):
            self.figureIterator = node.ref
        return u'%s %s: %s' % (node.title, node.ref, unicode(node))

    def do_footnote(self, node):
        """
        This function returns the footnote mark and adds the expanded footnote in the footnote list. L{do_par} handles the rest
        """
        self.footnotes.append( unicode("[fn"+str(node.attributes["num"])+"]")+unicode(node) )
        return unicode("[fn"+str(node.attributes["num"])+"]")

    def do_itemize(self, node):
        """Handles the C{itemize} enviroment. Appends global indent and item mark to each item in list."""
        r=""
        self.globalIndent.append(self.itemMark)
        for item in node:
            r=r+unicode(item).rstrip()+u'\n'
        self.globalIndent.pop()
        return r
        
    def do_enumerate(self, node):
        """Handles the C{enumerate} enviroment. Appends global indent and a running number to each item in list."""
        r=""
        for i, item in enumerate(node):
            bullet=re.sub("%s",str(i+1),self.enumMark,1) #if user has %s, show enumeration
            self.globalIndent.append(bullet)
            r=r+unicode(item).rstrip()+u'\n'
            self.globalIndent.pop()
        return r

    def do_tableofcontents (self,node):
        """Handles the C{tableofcontents} command. Searches for a C{.toc} file corresponding the input file (without the suffix). Renders table of contents."""
        fileName= node.ownerDocument.userdata['jobname']
        workingDir = node.ownerDocument.userdata['working-dir']
        wholePath= "".join([workingDir,'/',fileName,'.toc'])
        term = 'Table of contents' # default
        # Get a language specific term, that corresponds to 'ref' 
        # in dictionary 'terms' (see file 'i18n.xml').
        try :
            term = node.ownerDocument.context.terms['contents']
        except KeyError:    # catch the error, handle it and continue
            print "WARNING (parempi): Didn't find a key 'ref' term from the dictionary."
        output = '\n\n' + term + '\n' 
        if os.path.isfile(wholePath):
           try:
                 openfile = codecs.open(wholePath, 'r','utf8')
                 filecontent = openfile.read().split('\n')
           except UnicodeDecodeError:
                 openfile = codecs.open(wholePath,'r','iso-8859-15')
                 filecontent = openfile.read().split('\n')

           if filecontent:
                for i in filecontent:
                    if  re.match(r'^\\contentsline',i):
                        i = re.sub(r'^\\contentsline {','',i)
                        if re.match(r'^section',i):
                            i = re.sub(r'^section}{\\numberline {','',i)
                        if re.match(r'^subsection',i):
                            i = re.sub(r'^subsection}{\\numberline {',' ',i)
                        if re.match(r'^subsubsection',i):
                            i = re.sub(r'^subsubsection}{\\numberline {','  ',i)
                        i=i.replace('}',' ',1)
                        i = re.sub('}{.+}$','',i) #deleting pagenumber
                        output = output+'\n'+i
        else:
            print('WARNING: could\'t generate table of content, .toc file is missing')
        return unicode(output)
    
    def do_quote(self, node):
        """
        Inserts the quoted content between user configurable lines. The indentation for the content is also configurable.
        """
        self.globalIndent.append(" "*self.quoteIndent)
        r=self.quotePre+u'\n'+unicode(node).rstrip()+u'\n'+self.quotePost #rstrip eats the newlines of the last par
        self.globalIndent.pop()
        return unicode(r)

    do_quotation=do_quote

    def do_maketitle(self, node):
        """
        Extracts title, author, date and thanks from node userdata and returns
        it as string.
        """
        output = []
        metadata = node.ownerDocument.userdata
        if 'title' in metadata:
            output.append(self.fill(metadata['title']).strip())
        if 'author' in metadata:
            output.append(self.fill(metadata['author']).strip())
        if 'date' in metadata:
            output.append(self.fill(metadata['date']).strip())
        if 'thanks' in metadata:
            output.append(self.fill(metadata['thanks']).strip())
        return u'\n%s\n\n' % u'\n'.join(output)

    do_titlepage = do_maketitle

    def do_section(self, node):
        """Extracts full title from node."""
        return self.fill(node.fullTitle)+u'\n\n'+unicode(node)

    do_part=do_chapter=do_subsection=do_subsubsection=do_paragraph=do_subparagraph=do_subsubparagraph=do_section
    
    def do_newpage(self, node):
        """
        There are no pages in .txt, so this function causes an empty line instead.
        """
        return u'\n\n'

    def do_par(self, node):
        """
        For handling paragraphs that are created during the parsing process by PlasTeX parser.
        Also inserts expanded footnotes after the paragraph, for the footnotes met in the paragraph.
        """
        directText = False
        for child in node.childNodes: #check if we have any direct text
            if isinstance(child, basestring):
                if not unicode(child).strip()=="":
                    directText = True
                    break
        if directText: #handle the text.. basically call fill for wrap and add 2 newlines
            output=self.fill(node).rstrip()+u'\n\n'
            if node.parentNode.nodeName=="item": #for better looking itemize/enumerate when the next par does not contain text directly. hacky
                for threat in node.parentNode.childNodes:
                    if threat.nodeName=="par":
                        if not isinstance(threat.childNodes[0],basestring):
                            output = output.rstrip()+u'\n' #after all that, just remove one newline at the end
        else: #if no direct text, continue recursion. maintaining a safe amount of newlines.
            return unicode(node).rstrip()+u'\n\n'
            
        """Footnotes in the footnote list will be removed and inserted in output, after the actual par output"""
        while len(self.footnotes)>0:
            output=output+self.fill(self.footnotes.pop(0))+u'\n'
            if len(self.footnotes)==0:
                output=output+u'\n'
        return output

    def do_em(self, node):
        """Handles the C{em} command according to the form specified in general configuration file, option C{emphasizeFormat}. Defaults to C{{em%s}}."""
        pieces = re.split("%s",self.emphasizeFormat,1)
        if len(pieces)==1:
            print('WARNING: emphasizeFormat in configuration lacks the %s, which represents the content. Using the default value for emphasizeFormat')
            pieces = re.split("%s",'{em %s}',1)
        return unicode(pieces[0] + unicode(node) + pieces[1])

    do_emph=do_em
    
    def do_bf(self, node):
        """Handles the C{bf} command according to the form specified in general configuration file, option C{boldfaceFormat}. Defaults to C{{bf%s}}."""
        pieces = re.split("%s",self.boldfaceFormat,1)
        if len(pieces)==1:
            print('WARNING: boldfaceFormat in configuration lacks the %s, which represents the content. Using the default value for boldfaceFormat')
            pieces = re.split("%s",'{bf %s}',1)
        return unicode(pieces[0] + unicode(node) + pieces[1])

    do_textbf=do_bf

    def do_verbatim(self, node):
        """
        Keeps the content as it is.
        """
        return re.sub(r'^\s*\n', r'', unicode(node)).rstrip()

    # Math
     
    def __mathProcessNonText(self, node):
        """
        Process all non-text elements (symbols, commands and undefined commands)
        in math environments. Has a list of "allowed" commands and enviroments,
        and doesn't work with others.
        """
        
        mathCommandList = {'multicolumn':self.do_math,'hline':self.nothing, 'array':self.do_array,'cases':self.do_cases,
    'bgroup':self.do_math,'left':self.do_left,'right':self.do_right,
    'overline':self.do_overline,'frac':self.do_frac}    # list of methods what you can call in mathEnviroment in runtime
        
        if node.nodeName in self.ignoredMath: # if we want to skip this node
            return self.do_math(node)
        elif node.nodeName == 'active::^':
            return self.do__superscript(node)
        elif node.nodeName == 'active::_':
            return self.do__subscript(node)
        elif node.nodeName == 'text' or node.nodeName=='#text': #latter just for bgroup-madness, is not the perfect way
            return unicode(node)
        elif node.nodeName == 'nonumber':
            return u''
        elif mathCommandList.has_key(node.nodeName):
            s = mathCommandList[node.nodeName](node)
            if(node.nodeName == 'cases'):
                s = self.__mathIndent(s)
            return s
        else:
            s=self.__processSymbol(node) 
            return s

    def __processSymbol(self,node):
        """
        Returns a math symbol corresponding nodeName.
        If none is found, the tex source code of the unhandled symbol is returned. 

        @return:    math symbol or raw source code as string
        """
        if (node.nodeName in self.symbols):
            symbol = self.symbols[node.nodeName]
            return symbol
        else:
            self.logUnknowns(node,True)
            return node.source

    def do_math(self, node):
        """
        Process math environment. Works similar to unicode(), but for the math
        enviroment. Calls C{__mathProcessText} for text nodes and
        C{__mathProcessNonText} for other nodes. Surrounds math enviroment
        with C{$}, if in middle of text and if C{dollarsInText} is toggled in
        general configuration file.

        @return:    a string containing math, rendered
        """
        output = []
        for c, child_iter in enumerate(node.childNodes):
            if child_iter.nodeName=='#text' and node.nodeName!="bgroup":
                output.append(self.__processMathText(child_iter))
            else:
                s = self.__mathProcessNonText(child_iter)
                output.append(s)

        if node.nodeName!="bgroup":
            weAreInText=False
            for any in node.parentNode.childNodes:
                if any.nodeName=='#text':
                    weAreInText=True
            if self.dollarsInText and weAreInText:
                output.insert(0,'$')
                output.append('$')
        output = filter(None,output)
        return u''.join(output)

    do_ensuremath = do_math
   

    def __processMathText(self,node):
        s = re.sub(r'(\s){2,}','',node.source)
        s = re.sub(r'\((\s)*','(',s)
        s = re.sub(r'\[(\s)*','[',s)
        s = re.sub(r'\{(\s)*','{',s)
        s = re.sub(r'(\s)*\)',')',s)
        s = re.sub(r'(\s)*\]',']',s)
        s = re.sub(r'(\s)*\}','}',s)
        return u''+re.sub(r'\s*(_|\^)\s*', r'\1', s.strip())

    def do_equation(self, node, numbering=True):
        """ 
        Processes 'equation' environment. This is much like math environment, 
        but the content of 'equation' will be renderer on its own paragraph.
        Also, equations gets numbered according to the global equation counter'. 
        All childs of 'equation' node are handled as math elements. 
        
        @return:   indented and math-specific rendered equation as a string
        
        I{Notes:} Function 'do___equation' that corresponds the LaTeX 'equation*'
        environment, calls this function as a parameter 'numbering = false'.
        """
    
        output = []
        for c, child_iter in enumerate(node.childNodes):
            if child_iter.nodeName=='#text':
                rendered_text = self.__processMathText(child_iter) 
                output.append(rendered_text)
            else:
                rendered_non_text = self.__mathProcessNonText(child_iter)
                if child_iter.nodeName == 'cases':
                    output.append('\n') # add newline berore content of 'cases'
                    output.append(rendered_non_text)
                elif child_iter.nodeName =='left': 
                    output.append(rendered_non_text)
                    output.append('\n') # add newline after content of 'left'
                else: output.append(rendered_non_text)
                
        origStr = u''.join(output) 
        
        if numbering:   # process equation number
            numberedStr = ''
            row = 0
            for char in origStr:
                if char=='\n':     # add equation number before newline
                    if row == 0:
                        numberedStr = self.__equationCounter(node, numberedStr)
                    numberedStr += '\n'
                    row += 1
                else:
                    numberedStr += char
            if row == 0 and numbering:    # if there is only one row
                numberedStr = self.__equationCounter(node, numberedStr)
        
        else: numberedStr = origStr # no numbering
         
        output = self.__mathIndent(numberedStr)
        return output

    def __mathIndent(self,output):
        """
        Processes indentations in the math enviroment. More specifically,
        adds spaces after each newline according to C{mathIndent}.

        @type output:   list of strings
        @param output:  unindented math elements
        @return:        indented math elements
        """
        output = self.mathIndent * u' ' + output
        indent = u'\n'+self.mathIndent * u' '
        return re.sub(r'\n',indent,output)

    def __equationCounter(self,node,line):
        """
        @type line:     string
        @param line:    equation
        @return:        line given as parameter + equation number
        """
        howClose = self.lineWidth - len(line) - self.formulaNumbering

        # nudge eq counter one step forward and get value
        node.ownerDocument.context.counters['${eq}'].stepcounter()
        equationNumber = node.ownerDocument.context.counters['${eq}'].value

        # append equation number to line
        line += '   (%s)'.rjust(howClose) % str(equationNumber)

        return line     

    def do___equation(self,node):
        """
        Handles mathematical equations in environment 'equation*'. This is very similar as do_equation. 
        The only difference is in numbering; content of 'do___equation' will not get an equation number.  
        @return:   indented and math-specific rendered equation as a string
        """
        return self.do_equation(node, False)

    do_displaymath = do___equation

    def do_left(self,node):
        return u'{'

    def do_right(self,node):
        return u'}'

    def do_cases(self,node):
        output = self.alignFromBeginning(node,len(node[0].childNodes[0].textContent.strip())+1)
        output=[u'{'+cell + u'}' for cell in output]
        return u'\n'.join(output)

    def do_align(self, node):
        """
        Start rendering array from the first cell
            -get the firstCellLength attribute from the first cell on the first row
            -i.e. firstCellLength is behind first ArrayRow and then first ArrayCell
        
        Parser feeds do_align in the following way:
            -f(x) &= 2 is given inside an ArrayRow, where two
            -cells are parsed separately according to &-mark 
        """
        output = self.alignFromBeginning(node,len(node[0].childNodes[0].textContent.strip())+1)
        return self.__mathIndent(u'\n'.join(output))

    def alignFromBeginning(self, node, firstCellLength):
        """
        For doing the rendering job both with align- and eqnarray-environment. Renders nodes
        by going from left to right and up to down. Starts rendering always from beginning if
        the length of the first cell on yet unknown row is larger than firstCellLength.
        """
        skipLineNumbering=False
        output = []
        # for ArrayRows ...
        for row, cnode in enumerate(node.childNodes):
            firstCell = True
            # for ArrayCells ...
            for ccnode in cnode.childNodes:
                # Check for nonumber
                for parnode in ccnode.childNodes:
                    if parnode.nodeName=="nonumber":
                        skipLineNumbering=True
                    else:
                        for nonode in parnode.childNodes:
                            if nonode.nodeName=="nonumber":
                                skipLineNumbering=True
                # Detect the first cell on every row
                if firstCell:
                    firstCell = False
                    # Detect if first cell empty
                    if len(ccnode.textContent)<=0:
                        output.append(firstCellLength*' ')
                    else:
                        # Render the insides of a cell
                        # Assume that all the content of the cell
                        # is inside a single par (see it yourself with debugger)
                        cccnode = ccnode.childNodes[0]
                        cellOut = self.do_math(cccnode).strip()
                        # Need to change also the first cell length
                        # --> render whole align from the beginning
                        if len(cellOut)>=firstCellLength:
                            return self.alignFromBeginning(node,len(cellOut)+1)
                        # This works fine as long as the first cell on the
                        # left side of the =-mark has largest amount of chars
                        # of all cells on the left side
                        else:
                            # In this case cell has to have content 
                            output.append((firstCellLength-len(cellOut)-1)*' '+cellOut+' ')
                else:
                    # Check if cell empty
                    if len(ccnode.textContent)<=0:
                        output[row] += '  '
                    else:
                        # Render the insides of a cell
                        cccnode = ccnode.childNodes[0]
                        cellOut = self.do_math(cccnode).strip()
                        output[row] += cellOut+' '
            if skipLineNumbering==False and node.nodeName!="cases" and node.nodeName!="align*":
                output[row] = self.__equationCounter(node,output[row]) # insert line numbering at the end
            skipLineNumbering=False
        return output

    do_eqnarray = do_align

    def do___align(self, node):
        """
        Start rendering array from the first cell
            -get the firstCellLength attribute from the first cell on the first row
            -i.e. firstCellLength is behind first ArrayRow and then first ArrayCell

        Parser feeds do_align in the following way:
            -f(x) &= 2 is given inside an ArrayRow, where two
            -cells are parsed separately according to &-mark
        """
        output = self.alignFromBeginning(node,len(node[0].childNodes[0].textContent.strip())+1)
        #adds equation numbers in align
        return self.__mathIndent(u'\n'.join(output))

    do___eqnarray = do___align

    def do__superscript(self,node):
        parameter=u'%s' % node.attributes['self']
        if len(parameter) > 1:
            return u'^' + u'{' + parameter + u'}' # return in form a^{bc}
        else: return u'^' + parameter # return in a form a^b

    def do__subscript(self,node):
        parameter=u'%s' % node.attributes['self']
        if len(parameter) > 1:
            return u'_' + u'{' + parameter + u'}' # return in form a_{bc}
        else: return u'_' + parameter # return in a form a_b

    def do_frac(self,node):
        """
        Handles fractions (in math enviroment).
        """

        ops_re = '\+|\-|\/|\*'
        n = self.do_math(node.attributes['numer'])
        d = self.do_math(node.attributes['denom'])

        if re.search(ops_re, n):
            n = '(' + n + ')'
        if re.search(ops_re, d):
            d = '(' + d + ')'

        # if frac not a direct child of a math node (e.g. inside another frac),
        # render brackets
        if node.parentNode.nodeName != 'math':
            return u''.join('(' + n + '/' + d + ')')
        else:
            return u''.join(n + '/' + d)

    def do_overline(self,node):
        if node.attributes.has_key('self'):
            return str(node.attributes['self']) + '^c'
        else:
            return ''

    # Misc
    
    def do_slide(self, node):
        """
        For rendering both \slide-environment and beamer-environment. 
        Beamer demands command \documentclass{beamer} after which 
        \frame-commands are parsed to \frameenv-commands by using 
        Beamer-package. Needs to have slides.py in plasTeX/Packages 
        or otherwise slides will not be parsed correctly be PlasTeX
        parser.
        """
        # Find name for Slide from i18.xml
        term = 'Slide' # default
        # Get a language specific term, that corresponds to 'slide'
        # in dictionary 'terms' (see file 'i18n.xml').
        try :
            term = node.ownerDocument.context.terms['slide']
        except KeyError:    # catch the error, handle it and continue
            print "WARNING (parempi): Didn't find a key 'slide' term from the dictionary."
        # Update slide number
        self.slideIterator += 1

        # If slide has title, print it
        if 'title' in node.attributes:
            return u'\n'+term+' '+str(self.slideIterator)+ u': ' + u''.join(node.attributes['title']) + '\n\n' + unicode(node)
        else:
            return u'\n'+term+' '+str(self.slideIterator)+ u':\n\n' + unicode(node)        # This is reached only if in the end of the slide

    do_frameenv = do_frame  = do_slide

    # the backslash in latex2e, as in $\backslash$

    def do_backslash(self,node):
        return u'\\'

    # special characters (python function naming limitation? not checked for all)

    def do__backslash(self,node):
        return u'\n'

    def do__dollar(self,node):
        return u'$'

    def do__percent(self,node):
        return u'%'

    def do__opencurly(self,node):
        return u'{'

    def do__closecurly(self,node):
        return u'}'

    def do__underscore(self,node):
        return u'_'

    def do__ampersand(self,node):
        return u'&'

    def do__hashmark(self,node):
        return u'#'

    def do_to(self,node):
        return u'-> '

Renderer = ParempiRenderer
