/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "app_context.h"
#include <coemain.h>
#include <flogger.h>
#include <bautils.h>
#include <context_uids.h>
#ifdef __WINS__
#include <apgcli.h>
#include <apgtask.h>
#endif
#include "raii.h"

//#define DEBUG_CALLSTACK 1

class CApp_contextImpl : public CApp_context, public MDiskSpace {
public:
	virtual ~CApp_contextImpl();
	virtual RFs&	Fs();
	virtual RSystemAgent&	SysAgent();
	virtual RTelServer&	TelServer();
	virtual RPhoneType&	Phone();
	virtual CCnvCharacterSetConverter* CC();
	virtual bool NoSpaceLeft();
	virtual void SetFileLog(MPausable* log);
	virtual const TDesC& DataDir();
	virtual const TDesC& AppDir();
	virtual void SetDataDir(const TDesC& Dir, bool UseMMC);
	virtual void SetAppDir(const TDesC& Dir);
	virtual MSettings& Settings(); 
	virtual void SetActiveErrorReporter(MActiveErrorReporter* Reporter);
	virtual void SetSettings(MSettings* Settings);  // takes ownership
	virtual RWsSession& Ws();
	virtual void ReportActiveError(const TDesC& Source,
		const TDesC& Reason, TInt Code);
	virtual const char* CallStack();
	virtual TInt PushCallStack(const TDesC& Name);
	virtual TPtr PopCallStack();
	virtual void ResetCallStack();
	virtual HBufC* GetFormattedCallStack(const TDesC& Prefix);
	virtual void IterateStack(MCallStackVisitor& aVisitor, const char* aStack=0);
	virtual void ShowGlobalNote(TAknGlobalNoteType notetype, const TDesC& msg);

	virtual void RaiseErrorWithInfo(TInt aErrorCode, const TDesC& aSource,
		const TDesC& aUserMessage, const TDesC& aTechMessage,
		CErrorInfo::TErrorType aErrorType, CErrorInfo::TErrorLevel aErrorLevel,
		CErrorInfo* aInnerError); // on success, takes ownership
	virtual CErrorInfo* GetLastErrorInfo();
	virtual void ResetLastErrorInfo();

private:
	CApp_contextImpl();
	void ConstructL(bool aFsOnly, const TDesC& Name);
	virtual void DiskSpaceThreshold(TInt aDrive);

	RFs	iFs;
	RSystemAgent	iSysAgent; bool hasSysAgent;
	RTelServer	iTelServer; bool hasTelServer;
	RPhoneType	iPhone; bool hasPhone;
	CCnvCharacterSetConverter* cc; 
	TInt64		iDiskSpaceLimit;
	TInt		iDrive[2];
	bool		iSpaceLeft;
	CAknGlobalNote *globalNote;
	MPausable	*iFileLog;
	MSettings	*iSettings;
	MActiveErrorReporter	*iReporter;
	TFileName	iDataDir;
	TFileName	iAppDir;
	bool		iFsOnly;
	bool		iUseMMC;

	RChunk		iCallStackChunk; int iCallStackSize; bool has_chunk;
	char		*iCallStackZero, *iCallStackCurrent;

	CDiskSpaceNotifier *iDiskSpaceNotifier[2];

	CErrorInfo	*iLastErrorInfo;

	friend CApp_context;
};

class CDiskSpaceNotifier: public CCheckedActive
{
public:
	static CDiskSpaceNotifier* NewL(RFs& fs, TInt aDrive, MDiskSpace* aNotifier, TInt64 aThreshold);
	~CDiskSpaceNotifier();
private:
	CDiskSpaceNotifier(RFs& fs, TInt aDrive, MDiskSpace* aNotifier, TInt64 aThreshold);
	void ConstructL();
	void CheckedRunL();
	void DoCancel();

	MDiskSpace* iNotifier;
	RFs& iFs;
	TInt iDrive;
	TInt64 iThreshold;
};

EXPORT_C void print_stacktrace(RFs& fs, MApp_context* context, const TDesC& cat, TInt error)
{
	if (!context) return;
	RFile f;
	if (f.Open(fs, _L("C:\\crash.txt"), 
		EFileShareAny | EFileStreamText | EFileWrite)!=KErrNone) {
		if(f.Create(fs, _L("C:\\crash.txt"), 
			EFileShareAny | EFileStreamText | EFileWrite)!=KErrNone) {
			return;
		}
	}
	CleanupClosePushL(f);

	TInt pos=0;
	f.Seek(ESeekEnd, pos);

	TBuf<80> msg;
	msg.Format(_L("context_log crashed with error %d\ncategory "), error);
	TPtrC8 msg2( (TText8*)msg.Ptr(), msg.Length()*2);
	f.Write(msg2);
	TPtrC8 msg3( (TText8*)cat.Ptr(), cat.Length()*2);
	f.Write(msg3);
	f.Write(_L8("\ntrace:\n"));

	TPtr n=context->PopCallStack();
	while (n.Ptr()) {
		TPtrC8 msg2( (TText8*)n.Ptr(), n.Length()*2);
		f.Write(msg2);
		n=context->PopCallStack();
	}
	CleanupStack::PopAndDestroy();
}

EXPORT_C void exceptionhandler(TExcType t)
{
	MApp_context* c;
	c=MApp_context::Static();
	if (c) {
		print_stacktrace(c->Fs(), c, _L("Exception"), t);
	}
	User::Panic(_L("EXCEPTION"), t);
}

MApp_context::~MApp_context()
{
}

CApp_contextImpl::CApp_contextImpl()
{
}

void CApp_contextImpl::ConstructL(bool aFsOnly, const TDesC& Name)
{
	RFileLogger iLog;
	TInt log_err=iLog.Connect();
	if (log_err==KErrNone) {
		iLog.CreateLog(_L("ContextCommon"),_L("CApp_contextImpl"),EFileLoggingModeAppend);
		iLog.Write(_L("constructing"));
	}

	iFsOnly=aFsOnly;

	iDiskSpaceLimit=500*1024; // 500k
	iDrive[0]=2; // C:
	iFileLog=0;

	User::LeaveIfError(iFs.Connect());
	iSpaceLeft=true;

	if (Name.Length() && (iCallStackChunk.OpenGlobal(Name, EFalse)==KErrNone || 
		iCallStackChunk.CreateGlobal(Name, 4096, 4096)==KErrNone)) {
		has_chunk=true;
		iCallStackZero=iCallStackCurrent=(char*)iCallStackChunk.Base() + 4;
		iCallStackSize=iCallStackChunk.Size();
	} else {
		has_chunk=false;
		iCallStackZero=new char[4096];
		if (iCallStackZero) iCallStackZero += 4;
		iCallStackCurrent=iCallStackZero;
		iCallStackSize=4096;
	}

	if (!aFsOnly) {
		globalNote=CAknGlobalNote::NewL();
	}
	

	// only the first app context in a thread gets put
	// in the Tls, so that we can use an instance to look
	// at another thread's call stack
	if (Dll::Tls()==0) Dll::SetTls(this);

	if (log_err==KErrNone) {
		iLog.Write(_L("constructed"));

		iLog.CloseLog();
		iLog.Close();
	}
}

CApp_contextImpl::~CApp_contextImpl()
{
	Dll::SetTls(0);

	delete iDiskSpaceNotifier[0];
	delete iDiskSpaceNotifier[1];

	delete iSettings;
	delete globalNote;

	if (!has_chunk) {
		iCallStackZero-=4;
		delete iCallStackZero;
	} else {
		iCallStackZero-=4;
		*(TInt*)iCallStackZero=0;
		iCallStackChunk.Close();
	}
	iCallStackZero=iCallStackCurrent=0;

	if (!iFsOnly) {
	}

#ifndef __WINS__
	if (hasPhone) iPhone.Close();
	_LIT(KGsmModuleName, "phonetsy.tsy");
	if (hasTelServer) {
		iTelServer.UnloadPhoneModule( KGsmModuleName );
		iTelServer.Close();
	}
#endif
	if (hasSysAgent) iSysAgent.Close();

	delete cc;
	iFs.Close();

	delete iLastErrorInfo;
	iLastErrorInfo=0;

	RFileLogger iLog;
	iLog.Connect();
	iLog.CreateLog(_L("ContextCommon"),_L("CApp_contextImpl"),EFileLoggingModeAppend);
	iLog.Write(_L("deleted"));

	iLog.CloseLog();
	iLog.Close();
}

EXPORT_C CApp_context* CApp_context::NewL(bool aFsOnly, const TDesC& Name)
{
	CApp_contextImpl* ret;
	ret=new (ELeave) CApp_contextImpl();
	CleanupStack::PushL(ret);
	ret->ConstructL(aFsOnly, Name);
	CleanupStack::Pop();
	return ret;
}

CApp_context::~CApp_context()
{
}

RFs& CApp_contextImpl::Fs()
{
	return iFs;
}

RSystemAgent& CApp_contextImpl::SysAgent()
{
	if (!hasSysAgent) {
		User::LeaveIfError(iSysAgent.Connect());
		hasSysAgent=true;
	}

	return iSysAgent;
}

RTelServer& CApp_contextImpl::TelServer()
{
#ifndef __WINS__
	if (!hasTelServer) {
		User::LeaveIfError(iTelServer.Connect());
		_LIT(KGsmModuleName, "phonetsy.tsy");
		TInt err;
		err=iTelServer.LoadPhoneModule( KGsmModuleName );
		if (err==KErrNone) {
			hasTelServer=true;
		} else {
			iTelServer.Close();
			User::Leave(err);
		}
	}
#endif
	return iTelServer;

}

RPhoneType& CApp_contextImpl::Phone()
{
#ifndef __WINS__
	if (!hasPhone) {
		RTelServer::TPhoneInfo info;
		User::LeaveIfError( TelServer().GetPhoneInfo( 0, info ) );
		User::LeaveIfError( iPhone.Open( iTelServer, info.iName ) );
		hasPhone=true;
	}
#endif

	return iPhone;
}

CCnvCharacterSetConverter* CApp_contextImpl::CC()
{
	if (!cc) {
		cc=CCnvCharacterSetConverter::NewL();
		cc->PrepareToConvertToOrFromL(KCharacterSetIdentifierIso88591, iFs);
	}
	return cc;
}

void CApp_contextImpl::SetFileLog(MPausable* log)
{
	iFileLog=log;
	if (iFileLog && !iSpaceLeft) iFileLog->pause();
}

void CApp_contextImpl::DiskSpaceThreshold(TInt /*aDrive*/)
{
	int v;
	iSpaceLeft=true;
	TBuf<100> msg;
	TAknGlobalNoteType notetype;
	for(v=0; v<2; v++) {
		if (iDrive[v] < 0) continue;

		TVolumeInfo info;
		if (iFs.Volume(info, iDrive[v])==KErrNone) {
			if (info.iFree<=iDiskSpaceLimit) {
				iSpaceLeft=false;
			}
		}
	}
	if (!iSpaceLeft) {
		if (iFileLog) iFileLog->pause();
		msg.Append(_L("WARNING! No space left for logs. Please delete some pictures and other media."));
		notetype=EAknGlobalErrorNote;
	} else {
		if (iFileLog) iFileLog->unpause();
		msg.Append(_L("Logging continues."));	
		notetype=EAknGlobalInformationNote;
	}
	if (globalNote) {
		TRAPD(err, globalNote->ShowNoteL(notetype, msg));
	}
}

void CApp_contextImpl::ShowGlobalNote(TAknGlobalNoteType notetype, const TDesC& msg)
{
	if (globalNote) {
		TRAPD(err, globalNote->ShowNoteL(notetype, msg));
	}
}
void CDiskSpaceNotifier::ConstructL()
{
	CActiveScheduler::Add(this);
	iStatus=KRequestPending;
	iFs.NotifyDiskSpace(iThreshold, iDrive, iStatus);
	SetActive();
}

CDiskSpaceNotifier* CDiskSpaceNotifier::NewL(RFs& fs, TInt aDrive, MDiskSpace* aNotifier, TInt64 aThreshold)
{
	auto_ptr<CDiskSpaceNotifier> ret(new (ELeave) CDiskSpaceNotifier(fs, aDrive, aNotifier, aThreshold));
	ret->ConstructL();
	return ret.release();
}

CDiskSpaceNotifier::~CDiskSpaceNotifier()
{
	Cancel();
}

CDiskSpaceNotifier::CDiskSpaceNotifier(RFs& fs, TInt aDrive, MDiskSpace* aNotifier, TInt64 aThreshold) : 
	CCheckedActive(EPriorityNormal, _L("CDiskSpaceNotifier")), iNotifier(aNotifier), iFs(fs), iDrive(aDrive),
	iThreshold(aThreshold)
{
}

void CDiskSpaceNotifier::CheckedRunL()
{
	if (iStatus==KErrNone) {
		iNotifier->DiskSpaceThreshold(iDrive);
	}
	iStatus=KRequestPending;
	iFs.NotifyDiskSpace(iThreshold, iDrive, iStatus);
	SetActive();
}

void CDiskSpaceNotifier::DoCancel()
{
	iFs.NotifyDiskSpaceCancel();
}

bool CApp_contextImpl::NoSpaceLeft()
{
	if (iFsOnly) return false;

	if (!iDiskSpaceNotifier[0]) {
		iDrive[0]=2;
		iDiskSpaceNotifier[0]=CDiskSpaceNotifier::NewL(iFs, 2, this, iDiskSpaceLimit);

#ifndef __WINS__
		{
		iDrive[1]=-1;
		TDriveInfo i;
		if (iUseMMC && iFs.Drive(i, EDriveE)==KErrNone) {
			if (i.iType!=EMediaNotPresent &&
				i.iType!=EMediaUnknown &&
				i.iType!=EMediaCdRom &&
				i.iType!=EMediaRom) {
				// memory card
				iDrive[1]=4;
				iDiskSpaceNotifier[1]=CDiskSpaceNotifier::NewL(iFs, 4, this, iDiskSpaceLimit);
			}
		}
		}
#endif

		int v;
		for(v=0; v<2; v++) {
			if (iDrive[v] < 0) continue;
			TVolumeInfo volinfo;
			if (iFs.Volume(volinfo, iDrive[v])==KErrNone) {
				if (volinfo.iFree<=iDiskSpaceLimit) {
					iSpaceLeft=false;
				}
			}
		}
	}

	return !iSpaceLeft;
}

const TDesC& CApp_contextImpl::DataDir()
{
	return iDataDir;
}

void CApp_contextImpl::SetDataDir(const TDesC& Dir, bool UseMMC)
{
	iUseMMC=UseMMC;
	BaflUtils::EnsurePathExistsL(iFs, Dir);
	TDriveInfo i;
	if (UseMMC && iFs.Drive(i, EDriveE)==KErrNone) {
		if (i.iType!=EMediaNotPresent &&
			i.iType!=EMediaUnknown &&
			i.iType!=EMediaCdRom &&
			i.iType!=EMediaRom) {

			// memory card

			TFileName f=Dir;
			f.Replace(0, 1, _L("E"));
			TRAPD(err, BaflUtils::EnsurePathExistsL(iFs, f));
		}
	}
	iDataDir=Dir;
}

const TDesC& CApp_contextImpl::AppDir()
{
	return iAppDir;
}

void CApp_contextImpl::SetAppDir(const TDesC& Dir)
{
	iAppDir=Dir;
}

void CApp_contextImpl::SetSettings(MSettings* Settings)
{
	iSettings=Settings;
}

MSettings& CApp_contextImpl::Settings()
{
	return *iSettings;
}

RWsSession& CApp_contextImpl::Ws()
{
	return CCoeEnv::Static()->WsSession();
}

const char* CApp_contextImpl::CallStack()
{
	return iCallStackCurrent;
}

void CApp_contextImpl::SetActiveErrorReporter(MActiveErrorReporter* Reporter)
{
	iReporter=Reporter;
}

void CApp_contextImpl::ReportActiveError(const TDesC& Source,
	const TDesC& Reason, TInt Code)
{
	if (iReporter) {
		iReporter->ReportError(Source, Reason, Code);
	} else if (globalNote) {
		TBuf<256> msg;
		msg.Format(_L("Unhandled error %d %S %S"), Code, &(Source.Left(100)), &(Reason.Left(120)));
		globalNote->ShowNoteL(EAknGlobalErrorNote, msg);
	}
}

EXPORT_C MApp_context* MApp_context::Static()
{
	return (MApp_context*)Dll::Tls();
}

EXPORT_C const TDesC& MContextBase::DataDir()
{
	return iContext.DataDir();
}

EXPORT_C const TDesC& MContextBase::AppDir()
{
	return iContext.AppDir();
}

EXPORT_C MSettings& MContextBase::Settings()
{
	return iContext.Settings();
}


EXPORT_C bool MContextBase::NoSpaceLeft()
{
	return iContext.NoSpaceLeft();
}

EXPORT_C MContextBase::MContextBase() : iContext(*(MApp_context::Static()))
{
}

EXPORT_C MContextBase::MContextBase(MApp_context& Context) : iContext(Context)
{
}


EXPORT_C MContextBase::~MContextBase()
{
}

EXPORT_C RFs& MContextBase::Fs()
{
	return iContext.Fs();
}

EXPORT_C RSystemAgent& MContextBase::SysAgent()
{
	return iContext.SysAgent();
}

EXPORT_C RTelServer& MContextBase::TelServer()
{
	return iContext.TelServer();
}

EXPORT_C RPhoneType& MContextBase::Phone()
{
	return iContext.Phone();
}

EXPORT_C MApp_context& MContextBase::AppContext()
{
	return iContext;
}

EXPORT_C CCnvCharacterSetConverter* MContextBase::CC()
{
	return iContext.CC();
}

EXPORT_C RWsSession& MContextBase::Ws()
{
	return iContext.Ws();
}

EXPORT_C void MContextBase::ReportActiveError(const TDesC& Source,
	const TDesC& Reason, TInt Code)
{
	iContext.ReportActiveError(Source, Reason, Code);
}

EXPORT_C MApp_context* GetContext()
{
	return MApp_context::Static();
}

EXPORT_C MApp_context* MContextBase::GetContext() const
{
	return &iContext;
}

TInt CApp_contextImpl::PushCallStack(const TDesC& Name)
{
	if (!iCallStackZero) return KErrNotFound;
	if (iCallStackCurrent+Name.Length()*2+8 > iCallStackZero+iCallStackSize) return KErrNoMemory;

	TInt len, added;
	len=Name.Length()*2;
	// align on 4 byte boundaries
	added=(4-(len % 4));
	if (added==4) added=0;
	len+=added;

	TPtr pt( (TUint16*)iCallStackCurrent, len/2);
	pt=Name;

	if(added>=2) pt.Append(_L(" "));

	TInt* pp=(TInt*) (iCallStackCurrent+len);
#ifdef DEBUG_CALLSTACK
	TBuf<200> msg;
	msg.Format(_L("PushCallStack name at %d length %d link at %d next at %d: %S"),
		(TInt)iCallStackCurrent, TInt(len), 
		(TInt)pp, (TInt)(iCallStackCurrent+len + 4), &Name);
	RDebug::Print(msg);
#endif

	*pp=iCallStackCurrent-iCallStackZero;
	iCallStackCurrent+=len + 4;

	*((TInt*) (iCallStackZero-4)) = (iCallStackCurrent - iCallStackZero);

	return KErrNone;
}

TPtr CApp_contextImpl::PopCallStack()
{
	if (iCallStackCurrent==iCallStackZero) {
		return TPtr(0, 0);
	}
	TInt* pd=(TInt*)(iCallStackCurrent - 4);
	char* p= iCallStackZero + *pd;
	TInt len=(iCallStackCurrent - p - 4)/2;
	TPtr ret( (TUint16*)(p), len, len);

#ifdef DEBUG_CALLSTACK
	TBuf<200> msg;
	msg.Format(_L("PopCallStack name at %d length %d link at %d next at %d: %S"),
		(TInt)p, TInt(len*2), 
		(TInt)pd, (TInt)(iCallStackCurrent), &ret);
	RDebug::Print(msg);
#endif

	iCallStackCurrent=p;
	*((TInt*) (iCallStackZero-4)) = (iCallStackCurrent - iCallStackZero);
	return ret;
}

void CApp_contextImpl::ResetCallStack()
{
	iCallStackCurrent=iCallStackZero;
	*((TInt*) (iCallStackZero-4)) = 0;
}

class TFormattedStack : public MCallStackVisitor {
public:
	TFormattedStack(HBufC* Into, const TDesC& Prefix) : iInto(Into), iPrefix(Prefix) { }
private:
	virtual void BeginStack()
	{
		iInto->Des().Zero();
		iInto->Des().Append(iPrefix);
		iInto->Des().Append(_L(":"));
	}
	virtual void VisitItem(const TDesC& Name)
	{
		iInto->Des().Append(Name);
		iInto->Des().Append(_L("|"));
	}
	virtual void EndStack()
	{
	}

	HBufC* iInto;
	const TDesC& iPrefix;
};

HBufC* CApp_contextImpl::GetFormattedCallStack(const TDesC& prefix)
{
	const char* aStack=iCallStackZero-4;
	const char* current=aStack+4+*(TInt*)aStack;
	aStack+=4;

	if (current==aStack) return 0;
	HBufC* ret=0;
	TInt len=((current-aStack)/2) + prefix.Length() + 1;
	TRAPD(err, ret=HBufC::NewL( len ));
	if (err!=KErrNone) return 0;

	TFormattedStack f(ret, prefix);
	IterateStack(f);

#ifdef __WINS__
	RDebug::Print(*ret);
#endif
	return ret;
}


void CApp_contextImpl::IterateStack(MCallStackVisitor& aVisitor, const char* aStack)
{
	if (aStack==0) aStack=iCallStackZero-4;

	aVisitor.BeginStack();
	const char* current=aStack+4+*(TInt*)aStack;
	aStack+=4;
	while (current!=aStack) {
		const char *prev=current -4;
		TInt* p=(TInt*)(prev);
		current=aStack + *p;
		TInt len=(prev-current)/2;
		TPtrC n( (TUint16*)current, len);
		aVisitor.VisitItem(n);
	}
	aVisitor.EndStack();
}

void CApp_contextImpl::RaiseErrorWithInfo(TInt aErrorCode, const TDesC& aSource,
	const TDesC& aUserMessage, const TDesC& aTechMessage,
	CErrorInfo::TErrorType aErrorType, CErrorInfo::TErrorLevel aErrorLevel,
	CErrorInfo* aInnerError)
{
	CErrorInfo* inner=0;
	if (aInnerError) inner=aInnerError;
	else inner=iLastErrorInfo;
	TRAPD(err, {
		auto_ptr<CErrorInfo> i(CErrorInfo::NewL(aErrorCode, aSource,
			aUserMessage, aTechMessage, aErrorType,
			aErrorLevel, aInnerError));
		if (inner!=iLastErrorInfo) delete iLastErrorInfo;
		iLastErrorInfo=i.release();
	});
	if (err!=KErrNone) {
		//FIXME: can we do something reasonable?
	}
	User::Leave(aErrorCode);
}
		
CErrorInfo* CApp_contextImpl::GetLastErrorInfo()
{
	return iLastErrorInfo;
}

void CApp_contextImpl::ResetLastErrorInfo()
{
	delete iLastErrorInfo;
	iLastErrorInfo=0;
}

EXPORT_C TCallStackItem::TCallStackItem(const TDesC& Name, MApp_context* Context)
{
	iAppContext=Context;

	if (!iAppContext) return;
	TInt err=iAppContext->PushCallStack(Name.Left(100));
	if (err!=KErrNone) {
		iAppContext=0;
	}
}

EXPORT_C TCallStackItem::~TCallStackItem()
{
	if (!iAppContext) return;
	iAppContext->PopCallStack();
}

EXPORT_C void MoveIntoDataDirL(MApp_context_access& Context, const TDesC& FileName)
{
	TFileName appfile, datafile;

	appfile.Format(_L("%S%S"), &Context.AppDir(), &FileName);
	datafile.Format(_L("%S%S"), &Context.DataDir(), &FileName);

	// if appdir and datadir are the same
	if (! appfile.CompareF(datafile)) return;

	if (BaflUtils::FileExists(Context.Fs(), appfile)) {
		User::LeaveIfError(BaflUtils::CopyFile(Context.Fs(), appfile, datafile));
		User::LeaveIfError(BaflUtils::DeleteFile(Context.Fs(), appfile));
	}
}

#ifdef __WINS__
EXPORT_C void StartStarterL(const TDesC& StackName, TUid AppUid, bool /*CheckForRunning*/, RWsSession& Ws)
#else
EXPORT_C void StartStarterL(const TDesC& StackName, TUid AppUid, bool CheckForRunning)
#endif
{
	TBuf<100> cmdline=StackName;;
	cmdline.Append(_L(" ")); cmdline.AppendNum((TInt)AppUid.iUid);
#ifdef __WINS__
	const TUid starter_uid= KUidstarter;
	RApaLsSession ls;
	User::LeaveIfError(ls.Connect());
	CleanupClosePushL(ls);

	TApaTaskList tl(Ws);
	TApaTask t=tl.FindApp(KUidstarter);
	if (! t.Exists()) {
		TThreadId dummy;
		User::LeaveIfError( ls.StartDocument(cmdline, starter_uid, dummy) );
	}
	CleanupStack::PopAndDestroy();
#else
	TFileName fnAppPath = _L("c:\\system\\apps\\starter\\starter.exe");

	{
		TFindProcess f(_L("*"));
		TFullName t;
		RProcess r;
		while (f.Next(t)==KErrNone) {
			r.Open(t);
			CleanupClosePushL(r);
			//LogAppEvent(t); LogAppEvent(r.FileName());
			if (r.FileName().Length()>0 && (! r.FileName().Mid(1).CompareF( fnAppPath.Mid(1) )) ) {
				// already running
				bool running=false;
				if (CheckForRunning && r.CommandLineLength()==0) running=true;
				else {
					HBufC* c=HBufC::NewLC(r.CommandLineLength());
					TPtr16 d=c->Des();
					r.CommandLine(d);
					if (! (d.Compare(cmdline)) ) {
						running=true;
					}
					CleanupStack::PopAndDestroy();
				}
				if (running) {
					CleanupStack::PopAndDestroy();
					return;
				}
			}
			CleanupStack::PopAndDestroy();
		}
	}

	{
		RProcess server;

		if (server.Create(fnAppPath, cmdline) != KErrNone) {
			fnAppPath.Replace(0, 1, _L("e"));
			User::LeaveIfError(server.Create(fnAppPath, cmdline));
		}
		CleanupClosePushL(server);
		server.Resume();

		CleanupStack::PopAndDestroy(); // server
	}
#endif
}
