/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "db.h"

#include <bautils.h>
#include <e32std.h>
#include "symbian_auto_ptr.h"

const int COMPACT_BETWEEN_UPDATES=20;
const int COMPACT_BETWEEN_DELETES=20;

EXPORT_C CDb* CDb::NewL(MApp_context& Context, const TDesC& dbname, TInt aFileMode, bool shared)
{
	auto_ptr<CDb> ret(new (ELeave) CDb(Context, shared));
	ret->ConstructL(dbname, aFileMode);
	return ret.release();
}

EXPORT_C CDb::~CDb()
{
	TInt err;
	if (iDbOpen) { 
		TRAP(err, iDb.Compact());
		TRAP(err, iDb.Close());
	}
	delete iStore;
}

EXPORT_C RDbDatabase& CDb::Db()
{
	return iDb;
}

CDb::CDb(MApp_context& Context, bool shared) : MContextBase(Context), iShared(shared)
{
}

void CDb::ConstructL(const TDesC& dbname, TInt aFileMode)
{
	TFileName nameonly;
	nameonly.Format(_L("%S.db"), &dbname);
	MoveIntoDataDirL(AppContext(), nameonly);

	TFileName database_file_install;
	TFileName database_file;

	database_file_install.Format(_L("%S%S_inst.db"), &AppDir(), &dbname);
	database_file.Format(_L("%S%S.db"), &DataDir(), &dbname);

	if (BaflUtils::FileExists(Fs(), database_file_install)) {
		User::LeaveIfError(BaflUtils::CopyFile(Fs(), database_file_install, database_file));
		User::LeaveIfError(BaflUtils::DeleteFile(Fs(), database_file_install));
	}

	bool retry=true;
	while(retry) {
		retry=false;
		TInt store_exists;
		delete iStore; iStore=0;
		if(iDbOpen) iDb.Close(); iDbOpen=false;
		TRAP(store_exists, iStore = CPermanentFileStore::OpenL(Fs(), database_file, aFileMode));
	
		//TRAP (store_exists, iStore = ::OpenL(Fs(), database_file, EFileRead|EFileWrite));

		if (store_exists==KErrNone) { 
			TRAPD(err, iDb.OpenL(iStore, iStore->Root()));
			if (err==KErrNotFound) {
				// store not constructed properly
				Fs().Delete(database_file);
				retry=true;
			} else if(err!=KErrNone) {
				User::Leave(err);
			} else {
				iDbOpen=true;
				User::LeaveIfError(iDb.Recover());
			}
		} else {
			// construct database
			TRAPD(err, {
				iStore = CPermanentFileStore::ReplaceL(Fs(), database_file,aFileMode);
				iStore->SetTypeL(iStore->Layout());
				TStreamId id=iDb.CreateL(iStore);
				iDbOpen=true;
				iStore->SetRootL(id);
				iStore->CommitL();
			});
			if (err!=KErrNone) {
				// roll back
				Fs().Delete(database_file);
				User::Leave(err);
			}
		}
	}
	/*
	 * named databases not supported?
	 *
	if (BaflUtils::FileExists(Fs(), database_file)) {
		TInt err=iDb.Open(Fs(), database_file);
		User::LeaveIfError(err);
	} else {
		TInt err=iDb.Create(Fs(), database_file);
		User::LeaveIfError(err);
	}
	*/
}

EXPORT_C void CDb::BeginL()
{
	User::LeaveIfError(iDb.Begin());
}

EXPORT_C void CDb::CommitL()
{
	TInt err=iDb.Commit();
	if (err!=KErrNone) {
		iDb.Rollback();
		if (iDb.IsDamaged()) iDb.Recover();
	}
	++iTransactionCount;
	if (iTransactionCount>4) iDb.Compact();
	User::LeaveIfError(err);
}

EXPORT_C void CDb::RollBack()
{
	iDb.Rollback();
	if (iDb.IsDamaged()) iDb.Recover();
	++iTransactionCount;
	if (iTransactionCount>4) iDb.Compact();
}

EXPORT_C MDBStore::MDBStore(RDbDatabase& Db) : iDb(Db)
{
}

EXPORT_C MDBStore::~MDBStore()
{
	TInt err;
	if (iTableOpen) { TRAP(err, iTable.Close()); }
}

void MDBStore::CreateIndices(int* columns, int* idx_columns, bool unique_idx, const TDesC& name)
{
	TBuf<10> colname;
	if (idx_columns[0]!=-1) {
		CDbKey* idx_key=CDbKey::NewLC();
		TInt idx_count=0;

		for (int* idx_i=idx_columns; ; idx_i++) {
			if (*idx_i<0) {
				if (unique_idx && idx_count==0) 
					idx_key->MakeUnique();
				TBuf<30> idx_name;
				if (idx_count>0) idx_name.Format(_L("IDX%d"), idx_count);
				else idx_name=_L("IDX");
				TInt err=iDb.CreateIndex(idx_name, name, *idx_key);
				if (err)
					User::Leave(err);
				idx_key->Clear();
				++idx_count;

			} else {
				colname.Format(_L("C%d"), *idx_i);

				/*
				 * we must limit the key size, e.g. 255 characters
				 * is too long, so truncate the last text column in the idx
				 * (but we can't say 50 if the column is actually _shorter_
				 */
				if (*(idx_i+1)>=0) {
					idx_key->AddL(TDbKeyCol(colname));
				} else {
					if ( columns[*idx_i -1] == EDbColText)  {
						if (iTextLen!=0 && iTextLen<50) {
							idx_key->AddL(TDbKeyCol(colname));
						} else {
							idx_key->AddL(TDbKeyCol(colname, 50));
						}

					} else {
						idx_key->AddL(TDbKeyCol(colname));
					}
				}
			}
			if (*idx_i==-1) break;
		}

		CleanupStack::PopAndDestroy(); // idx_key
	}
}

void MDBStore::DeleteIndices(const TDesC& name)
{
	CDbIndexNames* idxs=iDb.IndexNamesL(name);
	CleanupStack::PushL(idxs);
	for (int i=0; i< idxs->Count(); i++) {
		iDb.DropIndex( (*idxs)[i], name);
	}
	CleanupStack::PopAndDestroy();
}

EXPORT_C void MDBStore::ConstructL(int* columns, int* idx_columns, bool unique_idx,
			  const TDesC& name, TBool Alter, int* col_flags)
{
	if (name.Length()>KMaxTableNameLen) User::Leave(KErrTooBig);
	iTableName=name;

	iCurrentIdx=-1;

	TInt table_exists=iTable.Open(iDb, name);
	iTable.Close();

	CDbColSet* cols=CDbColSet::NewLC();

	TBuf<10> colname; int colcount=1;
	for (int* col_i=columns; *col_i!=-1; col_i++) {
		colname.Format(_L("C%d"), colcount);
		colcount++;
		TDbCol n(colname, (TDbColType)*col_i);
		if (col_flags) {
			n.iAttributes=col_flags[colcount-2];
		}
		if (n.iType==EDbColText && iTextLen!=0) n.iMaxLength=iTextLen;
		cols->AddL(n);
	}

	if (table_exists!=KErrNone) {
		TInt err_create=iDb.CreateTable(name, *cols);
		if (err_create==KErrAlreadyExists) 
			User::Leave(table_exists);
		User::LeaveIfError(err_create);

		CleanupStack::PopAndDestroy(); // cols

		TRAPD(err, CreateIndices(columns, idx_columns, unique_idx, name));
		if (err!=KErrNone) {
			DeleteIndices(name);
			iDb.DropTable(name);
			User::Leave(err);
		}

		User::LeaveIfError(iTable.Open(iDb, name));
		iTableOpen=true;
	} else {
		if (Alter) {
			// FIXME:
			// this will fuck up the database if an error occurs:
			// the old indices (or some of them) may be deleted
			// while not all of the new ones are created
			// :-(
			//
			TInt err=iDb.AlterTable(name, *cols);
			if (err!=KErrNone) {
				if (err==KErrArgument) {
					// have to recreate indexes
					DeleteIndices(name);
					TInt err=iDb.AlterTable(name, *cols);
					if (err!=KErrNone) 
						User::Leave(err);
					CreateIndices(columns, idx_columns, unique_idx, name);
				} else {
					User::Leave(err);
				}
			}
		}
		CleanupStack::PopAndDestroy(); // cols
		User::LeaveIfError(iTable.Open(iDb, name));
		iTableOpen=true;
	}

	if (idx_columns[0]!=-1) {
		SwitchIndexL(0);
	}
}

EXPORT_C void MDBStore::SwitchIndexL(TInt Idx)
{
	if (iCurrentIdx==Idx) return;

	TBuf<30> idx_name;
	if (Idx==-1) {
		RESET_IF_NECESSARY( User::LeaveIfError(iTable.SetNoIndex()) );
		iCurrentIdx=Idx;
		return;
	}
	if (Idx>0) idx_name.Format(_L("IDX%d"), Idx);
	else idx_name=_L("IDX");
	RESET_IF_NECESSARY( User::LeaveIfError(iTable.SetIndex(idx_name)) );
	iCurrentIdx=Idx;
}

EXPORT_C TInt MDBStore::GetCurrentIndex()
{
	return iCurrentIdx;
}

EXPORT_C void MDBStore::Reset(TInt aError)
{
	// FIXME!
	// there should be a better way...
	//
	if (iInReset) return;
	iTable.Cancel();
	iInReset=true;
	if (aError==KErrNotReady) {
		iTable.Reset();
	} else if (aError==KErrDisconnected) {
		if (iTableOpen) iTable.Close();
		iTableOpen=false;
		TInt err=iTable.Open(iDb, iTableName);
		if (err!=KErrNone) {
			iInReset=false;
			User::Leave(err);
		}
		iTableOpen=true;
	} else if (aError==KErrCorrupt) {
		if (iTableOpen) iTable.Close();
		iTableOpen=false;
		iDb.Recover();
		TInt err=iTable.Open(iDb, iTableName);
		if (err!=KErrNone) {
			iInReset=false;
			User::Leave(err);
		}
		iTableOpen=true;
	} else {
		iInReset=false;
		User::Leave(aError);
	}
	TInt prev_idx=iCurrentIdx;
	iCurrentIdx=-2;
	TRAPD(err2, SwitchIndexL(prev_idx));
	iInReset=false;
	User::LeaveIfError(err2);
}

EXPORT_C void MDBStore::PutL()
{
	TRAPD(err, iTable.PutL());
	if (err!=KErrNone) {
		iTable.Cancel();
		User::Leave(err);
	}
	if (!iDb.InTransaction() && iUpdateCount>=COMPACT_BETWEEN_UPDATES) {
		iUpdateCount=0;
		iDb.Compact();
	}
	++iUpdateCount;
}

EXPORT_C void MDBStore::SetTextLen(TInt Len)
{
	if (Len>255) 
		iTextLen=255;
	else
		iTextLen=Len;
}

CSingleColDbBase::CSingleColDbBase(MApp_context& Context, RDbDatabase& Db) : MDBStore(Db), MContextBase(Context)
{
}

CSingleColDbBase::~CSingleColDbBase()
{
}

bool CSingleColDbBase::SeekInnerL(TInt Idx, bool Updatable, bool Add)
{
	TDbSeekKey rk(Idx);
	if (iTable.SeekL(rk) ) {
		iTable.GetL();
		if (Updatable) iTable.UpdateL();
		return true;
	} else if (Add) {
		if (!Updatable) User::Leave(-1001);
		iTable.InsertL();
		iTable.SetColL(1, Idx);
		return true;
	}
	return false;
}

bool CSingleColDbBase::SeekL(TInt Idx, bool Updatable, bool Add)
{
	bool ret=false;
	RESET_IF_NECESSARY(ret=SeekInnerL(Idx, Updatable, Add));
	return ret;
}

void CSingleColDbBase::ConstructL(const TDesC& TableName,
	int ColType)
{
	int columns[] = { EDbColUint32, 0, -1 };
	columns[1]=ColType;
	int id_cols[]= { 1, -1 };

	MDBStore::ConstructL(columns, id_cols, true, TableName);
}
