/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


//CContextServer.cpp
#include "ver.h"
#include "ContextServer.h"
#include "ContextServerSession.h"
#include <ContextCommon.h>
#include <E32SVR.H>
#include <basched.h>

#include <app_context.h>
#include <flogger.h>

void Log(const TDesC& msg);

bool ClientAlive(TInt ThreadId);

#pragma warning(disable:4706)

CContextServer::CContextServer(TInt aPriority, MApp_context& Context) : CServer(aPriority), MContextBase(Context)
{
	CALLSTACKITEM(_L("CContextServer::CContextServer"));

	
}

CContextServer::~CContextServer()
{
	CALLSTACKITEM(_L("CContextServer::~CContextServer"));

	delete iPresenceInfo;
	delete iMessageHolder;
	delete iJabberClient;

	/*
	iSessionIter.SetToFirst();
	for (;;)
        {
		CContextServerSession* session;
		session = reinterpret_cast<CContextServerSession*>(iSessionIter++);
		if (!session)
		{
			break;
		}

		session->~CContextServerSession();
        }
	*/
}

CContextServer* CContextServer::NewL(MApp_context& Context)
{
	CALLSTACKITEM2(_L("CContextServer::NewL"), &Context);

	CContextServer* contextServer = CContextServer::NewLC(Context);
    	CleanupStack::Pop(contextServer);
    	return contextServer;
}

CContextServer* CContextServer::NewLC(MApp_context& Context)
{
	CALLSTACKITEM2(_L("CContextServer::NewLC"), &Context);

	CContextServer* contextServer = new (ELeave) CContextServer(EPriorityNormal, Context);
    	CleanupStack::PushL(contextServer);
    	contextServer->ConstructL();
    	return contextServer;
}

void CContextServer::ConstructL()
{
	CALLSTACKITEM(_L("CContextServer::ConstructL"));

	iPresenceInfo = CPresenceInfo::NewL();
	iMessageHolder = CMessageHolder::NewL();

	iJabberClient = CSocketsEngine::NewL(*this, AppContext());

	StartL(KContextServerName);
}

CSharableSession* CContextServer::NewSessionL(const TVersion& aVersion) const
{
	CALLSTACKITEM(_L("CContextServer::NewSessionL"));

	// check version
	if (!User::QueryVersionSupported(TVersion(KContextServMajorVersionNumber,
                                                KContextServMinorVersionNumber,
                                                KContextServBuildVersionNumber),
                                                aVersion))
	{
		User::Leave(KErrNotSupported);
	}
	
	// create new session
	RThread client = Message().Client();
	return CContextServerSession::NewL(client, *const_cast<CContextServer*> (this));
}


void CContextServer::IncrementSessions()
{
	CALLSTACKITEM(_L("CContextServer::IncrementSessions"));

    	iSessionCount++;
}

void CContextServer::DecrementSessions()
{
	CALLSTACKITEM(_L("CContextServer::DecrementSessions"));

    	iSessionCount--;
    	if (iSessionCount <= 0)
	{
      		iSessionCount =0;
			iJabberClient->Disconnect(EFalse);
		CActiveScheduler::Stop();
	}
}

TInt CContextServer::CheckedRunError(TInt aError)
{
	CALLSTACKITEM(_L("CContextServer::CheckedRunError"));

	if (aError == KErrBadDescriptor)
	{
        	// A bad descriptor error implies a badly programmed client, so panic it;
        	// otherwise report the error to the client
        	PanicClient(Message(), EBadDescriptor);
	}
	else
	{
		Message().Complete(aError);
	}

	//
	// The leave will result in an early return from CServer::RunL(), skipping
	// the call to request another message. So do that now in order to keep the
	// server running.
	ReStart();

	return KErrNone;	// handled the error fully
}

void CContextServer::PanicClient(const RMessage& aMessage, TContextServPanic aPanic)
{
	CALLSTACKITEMSTATIC(_L("CContextServer::PanicClient"));

	aMessage.Panic(KContextServer, aPanic);
}

void CContextServer::PanicServer(TContextServPanic aPanic)
{
	CALLSTACKITEMSTATIC(_L("CContextServer::PanicServer"));

    	User::Panic(KContextServer, aPanic);
}

void CContextServer::RunL()
{
	AppContext().ResetCallStack();

	CALLSTACKITEM(_L("CContextServer::RunL"));

	TRAPD(err, CServer::RunL());
	if (err!=KErrNone) {
		TBuf<30> msg;
		msg.Format(_L("Error in RunL: %d"), err);
		Log(msg);
		CheckedRunError(err);
	}

	AppContext().ResetCallStack();
}

void CContextServer::ThreadFunctionL()
{
	#if defined (__WINS__)
	UserSvr::ServerStarted();
	#endif

	__UHEAP_MARK;

	CApp_context* c=CApp_context::NewL(false, _L("ContextServer"));
	CleanupStack::PushL(c);
	{

	// make sure the call stack item is not on the stack
	// after app context is deleted
	CALLSTACKITEM2(_L("CContextServer::ThreadFunctionL"), c);

    	// Construct active scheduler
    	CActiveScheduler* activeScheduler = new (ELeave) CBaActiveScheduler;

    	CleanupStack::PushL(activeScheduler) ;

    	// Install active scheduler
    	// We don't need to check whether an active scheduler is already installed
    	// as this is a new thread, so there won't be one
    	CActiveScheduler::Install(activeScheduler);

    	// Construct our server
	CContextServer::NewLC(*c);    

	RSemaphore semaphore;
	if (semaphore.CreateGlobal(KContextServerSemaphoreName, 0)!=KErrNone) {
		User::LeaveIfError(semaphore.OpenGlobal(KContextServerSemaphoreName));
	}
	
	// Semaphore opened ok
	semaphore.Signal();
	semaphore.Close();

	// Start handling requests
	CActiveScheduler::Start();
	CleanupStack::PopAndDestroy(2); // CServer, activeScheduler, 
	}
		 
	CleanupStack::PopAndDestroy(1); // appcontext

	__UHEAP_MARKEND;

}

TInt CContextServer::ThreadFunction(TAny* /*aNone*/)
{
    	CTrapCleanup* cleanupStack = CTrapCleanup::New();
	if (cleanupStack == NULL)
	{
      	PanicServer(ECreateTrapCleanup);
	}

	TRAPD(err, ThreadFunctionL());
    	if (err != KErrNone)
	{
        	PanicServer(ESrvCreateServer);
	}
	
    	delete cleanupStack;
    	cleanupStack = NULL;

	return KErrNone;
}


//---------------------------------------------------------------------------

void CContextServer::TerminateContextServer()
{
	CALLSTACKITEM(_L("CContextServer::TerminateContextServer"));

	iJabberClient->Disconnect(EFalse);

	NotifySessions(ETerminated);
	//Log();
	CActiveScheduler::Stop();
}

void CContextServer::ConnectToPresenceServer(TDesC & username, TDesC & password, TDesC & server, TUint32 accessPoint)
{
	CALLSTACKITEM(_L("CContextServer::ConnectToPresenceServer"));

	iAccessPoint = accessPoint;
	iUsername.Copy(username);
	iPassword.Copy(password);
	iServer.Copy(server);

	iJabberClient->ConnectL(username,password,server,accessPoint);
}

void CContextServer::ConnectToPresenceServer(const RMessage &aMessage)
{
	CALLSTACKITEM(_L("CContextServer::ConnectToPresenceServer"));

	const TAny* clientsDescriptor0 = aMessage.Ptr0();
	TBuf<30> username;
	aMessage.ReadL(clientsDescriptor0, username);

	const TAny* clientsDescriptor1 = aMessage.Ptr1();
	TBuf<30> password;
	aMessage.ReadL(clientsDescriptor1, password);

	const TAny* clientsDescriptor2 = aMessage.Ptr2();
	TBuf<30> server;
	aMessage.ReadL(clientsDescriptor2, server);

	const TAny* clientsDescriptor3 = aMessage.Ptr3();
	TUint32 accessPointId;
	TPckg<TUint32> apPackage(accessPointId);
	aMessage.ReadL(clientsDescriptor3, apPackage,0);

	ConnectToPresenceServer(username,password,server,accessPointId);
}

void CContextServer::NotifySessions(TEvent aEvent)
{
	CALLSTACKITEM(_L("CContextServer::NotifySessions"));

	CContextServerSession* session=0;

	iSessionIter.SetToFirst();
	while( (session = reinterpret_cast<CContextServerSession*>(iSessionIter++)) ) {
		session->NotifyEvent(aEvent);
	}
}

void CContextServer::SuspendConnection()
{
	CALLSTACKITEM(_L("CContextServer::SuspendConnection"));

	NotifySessions(EDisconnected);
	iJabberClient->Disconnect(ETrue);
}

void CContextServer::ResumeConnection()
{
	CALLSTACKITEM(_L("CContextServer::ResumeConnection"));

	iJabberClient->ConnectL(iUsername,iPassword,iServer,iAccessPoint);
}

CPresenceInfo *  CContextServer::GetPresenceInfo()
{
	CALLSTACKITEM(_L("CContextServer::GetPresenceInfo"));

	return iPresenceInfo;
}

CSocketsEngine * CContextServer::GetJabberClient()
{
	CALLSTACKITEM(_L("CContextServer::GetJabberClient"));

	return iJabberClient;
}


// from MConnectionOpenerObserver --------------------------------------------------------------------
/*void CContextServer::success(CBase* source)
{
	CALLSTACKITEM(_L("CContextServer::success"));

	iJabberClient->ConnectL(iUsername,iPassword,iServer);
}

void CContextServer::error(CBase* source, TInt code, const TDesC& reason)
{
	CALLSTACKITEM(_L("CContextServer::error"));

	Log(reason);
}

void CContextServer::info(CBase* source, const TDesC& msg)
{
	CALLSTACKITEM(_L("CContextServer::info"));

	Log(msg);
}*/

// from MContextServerNotifier --------------------------------------------------------------------
void CContextServer::NotifyConnected()
{
	CALLSTACKITEM(_L("CContextServer::NotifyConnected"));

	NotifySessions(EConnected);	
}

void CContextServer::NotifyNewPresenceInfo(const TDesC & from, const TDesC & info, 
					   const TTime& send_timestamp)
{
	CALLSTACKITEM(_L("CContextServer::NotifyNewPresenceInfo"));

	RDebug::Print(from.Left(128));
	RDebug::Print(info.Left(128));

	//DEBUG
	TBuf<10> buf;
	send_timestamp.FormatL(buf, _L("%H%T%S"));
	RDebug::Print(buf);
	
	iPresenceInfo->Update(from,info,send_timestamp);

	NotifySessions(ENewPresenceInfo);
}

void CContextServer::ReportError(TContextServRqstComplete aErrorType, TDesC & aErrorCode, TDesC & aErrorValue)
{
	CALLSTACKITEM(_L("CContextServer::ReportError"));

	CContextServerSession* session=0;

	iSessionIter.SetToFirst();
	while( (session = reinterpret_cast<CContextServerSession*>(iSessionIter++)) ) {
		session->ReportError(aErrorType, aErrorCode, aErrorValue);
	}
}

void CContextServer::CancelRequest(const RMessage &aMessage)
{
	CALLSTACKITEM(_L("CContextServer::CancelRequest"));

	switch(aMessage.Function()) {
	case EConnectToPresenceServer:
	case EResumeConnection:
		iJabberClient->Disconnect(ETrue);
		break;
	}
}

CMessageHolder  * CContextServer::GetMessageHolder()
{
	CALLSTACKITEM(_L("CContextServer::GetMessageHolder"));

	return iMessageHolder;
}

void CContextServer::NotifyNewMessage(const TDesC & from, const TDesC& subject, const TDesC & message)
{
	CALLSTACKITEM(_L("CContextServer::NotifyNewMessage"));

	RDebug::Print(_L("NewMessage"));
	RDebug::Print(from.Left(128));
	RDebug::Print(subject.Left(128));
	RDebug::Print(message.Left(128));
	iMessageHolder->AppendMessageL(from, subject, message);
	NotifySessions(ENewMessage);
}
