/* 
Copyright (C) 2004  Mika Raento - Renaud Petit

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
   
    
     email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


/* Copyright (c) 2001, Nokia. All rights reserved */

#include <eikgted.h>
#include "SocketsEngine.h"
#include <TimeOut.h>
#include "SocketsReader.h"
#include "SocketsWriter.h"
#include "Sockets.pan"
#include "presence_data.h"

#include <flogger.h>

#include <string.h>
#include <es_sock.h>
#include <app_context.h>

#include "sha1.h"

const TInt CSocketsEngine::KTimeOut = 120; // 60 seconds time-out
const TInt CSocketsEngine::KMaxIdentificationRetry = 1;
const TInt CSocketsEngine::KTimeBeforeRetry = 120; //120 seconds before retry

const TInt CSocketsEngine::KDefaultPortNumber = 5222;

void LogL(const TDesC& msg)
{
	RFileLogger iLog;
	User::LeaveIfError(iLog.Connect());
	CleanupClosePushL(iLog);
	iLog.CreateLog(_L("Context"),_L("Engine"),EFileLoggingModeAppend);
	TInt i=0;
#ifdef __WINS__
	while (i<msg.Length()) {
#else
	{
#endif
		RDebug::Print(msg.Mid(i));
		iLog.Write(msg.Mid(i));
		i+=128;
	}
	
	iLog.CloseLog();
	CleanupStack::PopAndDestroy();
}

void Log(const TDesC& msg)
{
	TRAPD(err, LogL(msg));
	// ignore error, not much else we can do
	// and not critical
}
void LogL(const TDesC8& msg)
{
	RFileLogger iLog;
	User::LeaveIfError(iLog.Connect());
	CleanupClosePushL(iLog);
	iLog.CreateLog(_L("Context"),_L("Engine"),EFileLoggingModeAppend);
	TInt i=0;
#ifdef __WINS__
	while (i<msg.Length()) {
#else
	{
#endif
		iLog.Write(msg.Mid(i));
		i+=128;
	}
	//RDebug::Print(msg);
	
	// Close the log file and the connection to the server.
	iLog.CloseLog();
	CleanupStack::PopAndDestroy();
}

void Log(const TDesC8& msg)
{
	TRAPD(err, LogL(msg));
	// ignore error, not much else we can do
	// and not critical
}

void LogL(const TDesC& msg, TInt i)
{
	RFileLogger iLog;
	User::LeaveIfError(iLog.Connect());
	CleanupClosePushL(iLog);
	iLog.CreateLog(_L("Context"),_L("Engine"),EFileLoggingModeAppend);
	iLog.Write(msg);
	RDebug::Print(msg.Left(128));
	TBuf<10> iBuf;
	iBuf.Format(_L("%d"),i);
	iLog.Write(iBuf);
	RDebug::Print(iBuf);
	// Close the log file and the connection to the server.
	iLog.CloseLog();
	CleanupStack::PopAndDestroy();
}

void Log(const TDesC& msg, TInt /*i*/)
{
	TRAPD(err, LogL(msg));
	// ignore error, not much else we can do
	// and not critical
}


CSocketsEngine* CSocketsEngine::NewL(MContextServerNotifier& aNotifier, MApp_context& Context)
{
	CALLSTACKITEM2(_L("CSocketsEngine::NewL"), &Context);

	CSocketsEngine* self = CSocketsEngine::NewLC(aNotifier, Context);
	CleanupStack::Pop(self);
	return self;
}

CSocketsEngine* CSocketsEngine::NewLC(MContextServerNotifier& aNotifier, MApp_context& Context)
{
	CALLSTACKITEM2(_L("CSocketsEngine::NewLC"), &Context);

	CSocketsEngine* self = new (ELeave) CSocketsEngine(aNotifier, Context);
	CleanupStack::PushL(self);
	self->ConstructL();
	return self;
}

CSocketsEngine::CSocketsEngine(MContextServerNotifier& aNotifier, MApp_context& Context)
: CActive(EPriorityStandard), MContextBase(Context),
iContextNotifier(aNotifier),
iPort(KDefaultPortNumber),
iServerName(KDefaultServerName),
iRessource(KDefaultJabberRessource),
iIdAttempt(0),
iConnectionRetry(0),
iTimeBeforeRetry(KTimeBeforeRetry)

{
	CALLSTACKITEM(_L("CSocketsEngine::CSocketsEngine"));

#ifdef __WINS__
	iPresenceInterval=60;
	iFreshnessInterval=4*60;
#else
	iPresenceInterval=60;
	iFreshnessInterval=4*60;
#endif
	
	iSocketOpen= EFalse;
	iResolverOpen = EFalse;
	
	Log(_L("Socket engine created."));
}


CSocketsEngine::~CSocketsEngine()
{
	CALLSTACKITEM(_L("CSocketsEngine::~CSocketsEngine"));

	Cancel();
	
	delete iSocketsReader;
	iSocketsReader = NULL;
	
	delete iSocketsWriter;
	iSocketsWriter = NULL;
	
	delete iConnectionOpener;
	iConnectionOpener = NULL;
	
#ifdef __S60V2__
	iConnection.Close();
#endif
	iSocketServ.Close();
	
	delete iTimer;
	delete iSendTimer;
	delete iIdentTimer;
	iTimer = NULL;
	
	XML_ParserFree(iParser);
	
	delete iStack;
	iStack = NULL;
	
	delete iXmlBuf;
	
	delete iFrom;
	delete iPresenceInfo;
	delete iMessage; delete iSubject;
	delete iUserPresenceInfo;
	delete iErrorValue;
}


void CSocketsEngine::ConstructL()
{
	CALLSTACKITEM(_L("CSocketsEngine::ConstructL"));

	iPresenceInfo=HBufC::NewL(512);
	iMessage=HBufC::NewL(512);
	iSubject=HBufC::NewL(128);
	iUserPresenceInfo=HBufC::NewL(512);
	
	iSocketServ.Connect();
	
#ifdef __S60V2__
	iConnection.Open(iSocketServ);
#endif
	
#ifndef __S60V2__
	iConnectionOpener = CConnectionOpener::NewL(*this, iSocketServ);
#else
	iConnectionOpener = CConnectionOpener::NewL(*this, iSocketServ, iConnection);
#endif
	
	ChangeStatus(ENotConnected);
	
	// Start a timer
	iTimer = CTimeOut::NewL(*this);
	iSendTimer = CTimeOut::NewL(*this);
	iIdentTimer = CTimeOut::NewL(*this);
	CActiveScheduler::Add(this); 
	
	// Create socket read and write active objects
	iSocketsReader = CSocketsReader::NewL(*this, iSocket, AppContext());
	iSocketsWriter = CSocketsWriter::NewL(*this, iSocket, AppContext());
	
	iStack = CList<TInt>::NewL();
	
	iJabberSessionId.Zero();
	
	iParser = XML_ParserCreate(NULL);
	
	iXmlBuf=CXmlBuf16::NewL(256);
	
	iFrom = HBufC::NewL(64);
	iErrorValue = HBufC::NewL(64);
	
}

void CSocketsEngine::ConnectL(TDesC16 &u, TDesC16 &p, TDesC16 &s, TUint32 iAP)
{
	CALLSTACKITEM(_L("CSocketsEngine::ConnectL"));

	iUsername.Copy(u);
	iFullNick.Zero();
	iFullNick.Append(u);
	iFullNick.Append(_L16("@"));
	iFullNick.Append(s);
	iFullNick.Append(_L("/"));
	iFullNick.Append(iRessource);
	
	iPassword.Copy(p);
	iServerName.Copy(s);
	iStack->reset();
	iAccessPoint = iAP;
	
	iConnectionOpener->MakeConnectionL(iAccessPoint);
}

void CSocketsEngine::ConnectL()
{
	CALLSTACKITEM(_L("CSocketsEngine::ConnectL"));

	// Initiate connection process
	if ( (iEngineStatus == ENotConnected ) )
	{
		TInetAddr addr;
		if (addr.Input(iServerName) == KErrNone)
		{
			// server name is already a valid ip address
			ConnectL(addr.Address());
		}
		else // need to look up name using dns
		{
			// Initiate DNS
#ifndef __S60V2__
			User::LeaveIfError(iResolver.Open(iSocketServ, KAfInet, KProtocolInetUdp));
#else
			User::LeaveIfError(iResolver.Open(iSocketServ, KAfInet, KProtocolInetUdp, iConnection));
#endif
			iResolverOpen = ETrue;
			
			// DNS request for name resolution
			iResolver.GetByName(iServerName, iNameEntry, iStatus);
			
			ChangeStatus(ELookingUp);
			// Request time out
			iTimer->Wait(KTimeOut);
			SetActive();
		}
        }
	
}


void CSocketsEngine::ConnectL(TUint32 aAddr) // <a name="ConnectL32">
{
	CALLSTACKITEM(_L("CSocketsEngine::ConnectL"));

	iConnectionRetry++;
	Log(_L("Connection attempt :"),iConnectionRetry);
	
	
	// Initiate attempt to connect to a socket by IP address	
	if (iEngineStatus == ENotConnected)
        {
		XML_ParserReset(iParser, NULL);
		XML_SetUserData(iParser,this);
		XML_SetElementHandler(iParser, CSocketsEngine::startElement, CSocketsEngine::endElement);
		XML_SetCharacterDataHandler(iParser, CSocketsEngine::charData);
		iJabberSessionId.Zero();
		iXmlBuf->Zero();
		iStack->reset();
		
		
		// Open a TCP socket
#ifndef __S60V2__
		User::LeaveIfError(iSocket.Open(iSocketServ, KAfInet, KSockStream, KProtocolInetTcp));
#else
		User::LeaveIfError(iSocket.Open(iSocketServ, KAfInet, KSockStream, KProtocolInetTcp, iConnection));
#endif
		iSocketOpen = ETrue;
		
		// Set up address information
		iAddress.SetAddress(aAddr);
		iAddress.SetPort(iPort);	
		
		// Initiate socket connection
		iSocket.Connect(iAddress, iStatus);
		ChangeStatus(EConnecting);
		
		// Start a timeout
		iTimer->Wait(KTimeOut);
		SetActive();
        }
}


void CSocketsEngine::Disconnect(TBool closeConnection) 
{
	CALLSTACKITEM(_L("CSocketsEngine::Disconnect"));

	SendDisconnectionL();
	DisconnectSocket();
	if (closeConnection)
	{
		iConnectionOpener->CloseConnection();
	}
}

void CSocketsEngine::DisconnectSocket()
{
	CALLSTACKITEM(_L("CSocketsEngine::DisconnectSocket"));

	// cancel all outstanding operations
	// since we are connected, the only possibilities are read and write
	iSocketsReader->Cancel();
	iSocketsReader->SetIssueRead(EFalse);
	iSocketsWriter->Cancel();
	
	if (iResolverOpen)
	{
		iResolver.Close();
		iResolverOpen=EFalse;
	}
	
	if (iSocketOpen)
	{
		iSocket.Close();
		iSocketOpen=EFalse;
	}
	
	iTimer->Reset();
	iSendTimer->Reset();
	iIdentTimer->Reset();
	
	iFrom->Des().Zero();		
	iPresenceInfo->Des().Zero();
	iMessage->Des().Zero();
	iSubject->Des().Zero();
	
	ChangeStatus(ENotConnected);
}

// from CActive
void CSocketsEngine::DoCancel()
{
	CALLSTACKITEM(_L("CSocketsEngine::DoCancel"));

	iTimer->Reset();
	
	// Cancel appropriate request to socket
	switch (iEngineStatus)
        {
        case EConnecting:
		iSocket.CancelConnect();
		iSocket.Close();
		iSocketOpen = EFalse;
		break;
        case ELookingUp:
		// Cancel look up attempt
		iResolver.Cancel();
		iResolver.Close();
		iResolverOpen=EFalse;
		break;
		
	case ENotConnected:
		// do nothing
		break;
		
	default:
		//User::Panic(KPanicSocketsEngine, ESocketsBadStatus);
		
		break;
	}
	
	ChangeStatus(ENotConnected);
}

void CSocketsEngine::WriteL(const TDesC16& aData)
{
	CALLSTACKITEM(_L("CSocketsEngine::WriteL"));

	// Write data to socket
	if ( (iEngineStatus == EConnected) )
        {
		iSocketsWriter->IssueWriteL(aData);
        }
}

void CSocketsEngine::SendXMLStreamHeaderL()
{
	CALLSTACKITEM(_L("CSocketsEngine::SendXMLStreamHeaderL"));

	if ( (iEngineStatus == EConnected) )
	{
		TBuf16<200> header;
		
		header.Append(_L16("<?xml version='1.0'?><stream:stream"));
		header.Append(_L16(" xmlns='jabber:client' "));
		header.Append(_L16("xmlns:stream='http://etherx.jabber.org/streams'  to='"));
		header.Append(iServerName);
		header.Append(_L16("'>"));
		
		iSocketsWriter->IssueWriteL(header);
	}
}

void CSocketsEngine::SendDisconnectionL()
{
	CALLSTACKITEM(_L("CSocketsEngine::SendDisconnectionL"));

	if ( (iEngineStatus == EConnected) )
	{
		TBuf16<200> msg;
		msg.Append(_L16("</stream:stream>"));
		iSocketsWriter->IssueWriteL(msg);
	}
}

void CSocketsEngine::SendIdentificationL()
{
	CALLSTACKITEM(_L("CSocketsEngine::SendIdentificationL"));

	if (iEngineStatus == EConnected)
	{
		TBuf8<100> iDigest;
		iDigest.Copy(iJabberSessionId);
		iDigest.Append(iPassword);
		
		RDebug::Print(iJabberSessionId);
		
		SHA1 * sha1 = new SHA1;
		sha1->Input((char*)(iDigest.Ptr()),iDigest.Size());
		unsigned int message_digest_array[5];
		
		sha1->Result(message_digest_array);
		
		TBuf<40> digest;
		for (int i=0;i<5;i++)
		{
			TBuf<8> h;
			h.Format(_L("%08x"),message_digest_array[i]);
			digest.Append(h);
		}
		
		RDebug::Print(digest);
		
		delete sha1;
		
		//TPtrC8 digest = TPtrC8( (TUint8 *)(message_digest_array) );//
		
		
		/*CMD5 * md5 = new CMD5;
		Log(_L("md5 created"));
		TBuf16<40> buf;
		buf.Copy(iPassword);
		
		  md5->setPlainText();
		  Log(_L("md5 setPlaintext ok"));
		  const char * digest1 = md5->getMD5Digest();
		  
		    TPtrC8 digest = TPtrC8( (TUint8 *)(digest1) );
		*/
		Log(_L("Identification Attempt:"), iIdAttempt);
		
		TBuf16<400> ident;
		
		ident.Append(_L16("<iq type='set'><query xmlns='jabber:iq:auth'><username>"));
		ident.Append(iUsername);
		
		ident.Append(_L16("</username><resource>"));
		ident.Append(iRessource);
		ident.Append(_L16("</resource><digest>"));
		ident.Append(digest);	
		ident.Append(_L16("</digest></query></iq>"));
		
		//ident.Append(_L16("</resource><password>"));
		//ident.Append(iPassword);
		//ident.Append(_L16("</password></query></iq>"));
		
		
		iSocketsWriter->IssueWriteL(ident);
		iIdentTimer->Wait(KTimeOut);
	}
}

void CSocketsEngine::DoSendPresenceInfoL()
{
	CALLSTACKITEM(_L("CSocketsEngine::DoSendPresenceInfoL"));

	if (iEngineStatus == EConnected && iUserPresenceInfo && iUserPresenceInfo->Des().Length()>0)
	{
		Log(_L("Sending presence"));
		iXmlBuf->Zero();
		auto_ptr<CPtrCArray> attr(new (ELeave) CPtrC16Array(1));
		attr->AppendL(_L16("from"));
		attr->AppendL(iFullNick.Mid(0));
		attr->AppendL(_L16("type"));
		attr->AppendL(_L16("available"));
		iXmlBuf->BeginElement(_L16("presence"), attr.get());
		
		iXmlBuf->Leaf(_L16("show"), _L16("xa"));
		iXmlBuf->BeginElement(_L16("status"));
		
		TBuf<15> timestamp;
		
		TTime time; time.HomeTime();
		
		TDateTime dt;
		dt=time.DateTime();
		_LIT(KFormatTxt,"%04d%02d%02dT%02d%02d%02d");
		timestamp.Format(KFormatTxt, dt.Year(), (TInt)dt.Month()+1, (TInt)dt.Day()+1,
			dt.Hour(), dt.Minute(), dt.Second());
		
		iXmlBuf->Characters(timestamp);
		iXmlBuf->Characters(*iUserPresenceInfo);
		
		//iXmlBuf->Leaf(_L16("status"), _L16("test"));
		iXmlBuf->EndElement(_L16("status"));
		iXmlBuf->EndElement(_L16("presence"));
		
		iSocketsWriter->IssueWriteL(iXmlBuf->Buf());
		
		iLastUpdateSent.HomeTime();
		
		iSendTimer->Wait(iFreshnessInterval);
	}
}

void CSocketsEngine::SendPresenceInfoL(TDesC &presenceInfo)
{
	CALLSTACKITEM(_L("CSocketsEngine::SendPresenceInfoL"));

	//iUserPresenceInfo->Des().Zero();
	//local copy of presenceInfo, in case of disconnection...
	while ( presenceInfo.Length()>iUserPresenceInfo->Des().MaxLength() ) {
		iUserPresenceInfo=iUserPresenceInfo->ReAllocL(iUserPresenceInfo->Des().MaxLength()*2);
	}
	*iUserPresenceInfo=presenceInfo;
	
	TTime now; now.HomeTime();
	
	if (iEngineStatus == EConnected && now-iPresenceInterval > iLastUpdateSent)
	{
		iSendTimer->Reset();
		DoSendPresenceInfoL();
		
	} else if (iEngineStatus == EConnected) {
		TBuf<40> msg;
		TTime next_send=iLastUpdateSent+iPresenceInterval;
		TInt w= ((next_send.Int64() - now.Int64())/(1000*1000)).Low();
		if (w<1) w=1;
		msg.Format(_L("Queuing presence sending after %d secs"), w);
		//Log(msg);
		iSendTimer->Wait(w);
	}
}

void CSocketsEngine::Read()
{
	CALLSTACKITEM(_L("CSocketsEngine::Read"));

	// Initiate read of data from socket
	if ((iEngineStatus == EConnected) )
	{
		if(!iSocketsReader->IsActive()) 
		{
			//Log(_L("Issue Read()"));
			iSocketsReader->SetIssueRead(ETrue);
			iSocketsReader->Start();
		}
	}    
}

TInt CSocketsEngine::CheckedRunError(TInt aError)
{
	CALLSTACKITEM(_L("CSocketsEngine::CheckedRunError"));

	Log(_L("Error in CSocketsEngine::RunL %d"), aError);
	return aError;
}

// from CActive
void CSocketsEngine::RunL()
{
	AppContext().ResetCallStack();

	CALLSTACKITEM(_L("CSocketsEngine::RunL"));

	// Active object request complete handler.
	// iEngineStatus flags what request was made, so its
	// completion can be handled appropriately
	iTimer->Reset(); // Cancel TimeOut timer before completion
	
	switch(iEngineStatus)
	{
	case EConnecting:
		// IP connection request
		if (iStatus == KErrNone)
			// Connection completed successfully
		{
			ChangeStatus(EConnected);
			iConnectionRetry = 0;
			SendXMLStreamHeaderL();
			Read();
		}
		else
		{
			iSocket.Close();
			iSocketOpen = EFalse;
			Log(_L("CSocketsEngine: Conn. failed"), iStatus.Int());
			ChangeStatus(ENotConnected);
		}
		break;
		
	case ELookingUp:
		iResolver.Close();
		iResolverOpen = EFalse;
		if (iStatus == KErrNone)
		{
			// DNS look up successful
			iNameRecord = iNameEntry();
			// Extract domain name and IP address from name record
			//Print(_L("Domain name = "));
			//Print(iNameRecord.iName);
			TBuf<15> ipAddr;
			TInetAddr::Cast(iNameRecord.iAddr).Output(ipAddr);
			//Print(_L("\r\nIP address = "));
			//Print(ipAddr);
			//	Print(_L("\r\n"));
			// And connect to the IP address
			ChangeStatus(ENotConnected);
			ConnectL(TInetAddr::Cast(iNameRecord.iAddr).Address());
		}
		else
		{	
			// DNS lookup failed
			// iConsole.ErrorNotify(_L("CSocketsEngine\nDNS lookup failed"), iStatus.Int());
			ChangeStatus(ENotConnected);
			ReportError(EDNSLookupFailed, 0);
		}
		
		break;
		
	default:
		User::Panic(KPanicSocketsEngine, ESocketsBadStatus);
		break;
		
	};

	AppContext().ResetCallStack();
}

void CSocketsEngine::expired(CBase* Source)
{
	CALLSTACKITEM(_L("CSocketsEngine::expired"));

	//User::Panic(_L("TEST"), 13);

	if (Source==iTimer) 
	{
		if (iEngineStatus == EWaitingForRetry)
		{
			ChangeStatus(ENotConnected);
			ConnectL(iUsername,iPassword,iServerName,iAccessPoint);
		}
		else
		{
			Cancel();
			//Log(_L("CSocketsEngine: Timed out"), KErrTimedOut);
			ReportError(ESocketEngineTimeOut, KErrTimedOut);
		}
	}
	else if (Source==iIdentTimer)
	{
		ReportError(EIdentificationTimeOut,0);
	}
	else if (Source==iSendTimer) 
	{
		DoSendPresenceInfoL();
	}
}

void CSocketsEngine::ReportError(MEngineNotifier::TErrorType aErrorType, TInt aErrorCode)
{
	CALLSTACKITEM(_L("CSocketsEngine::ReportError"));

	DisconnectSocket();
	
	TBool retry = ETrue;
	
	switch (aErrorType)
        {
	case MEngineNotifier::EDNSLookupFailed:
		Log(_L("CSocketsEngine: DNS Lookup Failed"), aErrorCode);
		if (iConnectionRetry >= 3 )
		{
			iConnectionOpener->CloseConnection();
		}
		break;
		
	case MEngineNotifier::EIdentificationFailed:
		
		Log(_L("CSocketsEngine: Identification Failed"), aErrorCode);
		Log(_L("ErrorDetail:"));
		Log(iErrorCode);
		Log(*iErrorValue);
		iContextNotifier.ReportError(EIdentificationError, iErrorCode, *iErrorValue);
		iIdAttempt=0;
		retry = EFalse;
		break;
		
	case MEngineNotifier::ESocketEngineTimeOut:
		Log(_L("CSocketsEngine: Timed out"), aErrorCode);			
		break;
		
		
	case MEngineNotifier::ETimeOutOnRead:
		Log(_L("CSocketsEngine: Time Out on Read"), aErrorCode);
		break;
		
	case MEngineNotifier::EStreamError:
		Log(_L("CSocketsEngine: StreamError"), aErrorCode);
		// report to context_log ?
		break;
		
	case MEngineNotifier::EDisconnected:
		Log(_L("CSocketsEngine: Disconnected"), aErrorCode);
		break;
		
	case MEngineNotifier::EGeneralReadError:
		Log(_L("CSocketsEngine: Read Error"), aErrorCode);
		break;
		
	case MEngineNotifier::ETimeOutOnWrite:
		Log(_L("CSocketsEngine: Time Out on write"), aErrorCode);
		break;
		
	case MEngineNotifier::EGeneralWriteError:
		Log(_L("CSocketsEngine: Write Error"), aErrorCode);
		break;
		
	case MEngineNotifier::EXmlParseError:
		Log(_L("CSocketsEngine: XML Parse Error"), aErrorCode);
		break;
	case MEngineNotifier::ENetworkConnectError:
		Log(_L("CSocketsEngine: Connect to net error"), aErrorCode);
		break;
	case MEngineNotifier::EUnknownError:
		Log(_L("CSocketsEngine: Unknown error"), aErrorCode);
		break;
	default:
		User::Panic(KPanicSocketsEngine, ESocketsBadStatus);
		break;
        }
	
	if ( iConnectionRetry == 5 && retry)
	{
		iContextNotifier.ReportError(EServerUnreachable,iErrorCode, *iErrorValue);
		Log(_L("CSocketsEngine: Server Unreachable"), aErrorCode);
		retry=EFalse;
	}
	
	if (retry)
	{
		//wait and retry
		ChangeStatus(EWaitingForRetry);
		//iTimeBeforeRetry = iTimeBeforeRetry;
		iTimer->Wait(iTimeBeforeRetry);
	}
	
}

void CSocketsEngine::ResponseReceived(const TDesC8& aBuffer)
{
	CALLSTACKITEM(_L("CSocketsEngine::ResponseReceived"));

	Log(_L("Receive:"));
	//Log(aBuffer);
	if (!XML_Parse(iParser,(char*)(aBuffer.Ptr()),aBuffer.Size(), EFalse)) {
		XML_Error code=XML_GetErrorCode(iParser);
		TPtrC descr((TUint16*)XML_ErrorString(code));
		Log(_L("XML parse error"));
		Log(descr);
		ReportError(MEngineNotifier::EXmlParseError, code);
	}
	
}

void CSocketsEngine::ChangeStatus(TSocketsEngineState aNewStatus)
{
	CALLSTACKITEM(_L("CSocketsEngine::ChangeStatus"));

	// Update the status (and the status display)
	switch (aNewStatus)
        {
        case ENotConnected:
		Log(_L("Not connected"));
		break;
		
        case EConnecting:
		Log(_L("Connecting"));
		break;
		
	case EConnected:
		Log(_L("Connected"));
		break;
		
        case ELookingUp:
		Log(_L("Looking up"));
		break;
		
	case EDisconnecting:
		Log(_L("Disconnecting"));
		break;
		
	case EWaitingForRetry:
		Log(_L("Waiting For Retry"));
		break;
		
        default:
		User::Panic(KPanicSocketsEngine, ESocketsBadStatus);
		break;
        }
	
	iEngineStatus = aNewStatus;
}


void CSocketsEngine::SetServerName(const TDesC& aName)
{
	CALLSTACKITEM(_L("CSocketsEngine::SetServerName"));

	iServerName.Copy(aName);
}

const TDesC& CSocketsEngine::ServerName() const
{
	CALLSTACKITEM(_L("CSocketsEngine::ServerName"));

	return iServerName;
}

void CSocketsEngine::SetPort(TInt aPort)
{
	CALLSTACKITEM(_L("CSocketsEngine::SetPort"));

	iPort = aPort;
}

TInt CSocketsEngine::Port() const
{
	CALLSTACKITEM(_L("CSocketsEngine::Port"));

	return iPort;
}

TBool CSocketsEngine::Connected() const
{
	CALLSTACKITEM(_L("CSocketsEngine::Connected"));

	return (iEngineStatus == EConnected);
}


//------------------------------------------------------------------------
// expat handlers

void  CSocketsEngine::startElement(void *userData, const XML_Char *el, const XML_Char **atts)
{

	CSocketsEngine* eng=(CSocketsEngine*)userData;
	eng->startElement(el, atts);
}

void  CSocketsEngine::startElement(const XML_Char *el, const XML_Char **atts)
{
	CALLSTACKITEM(_L("CSocketsEngine::startElement"));

	TInt currentState = EStackUndefined;
	
	if ( iStack->iCurrent != NULL )
	{
		currentState = iStack->iCurrent->Item;
	}
	
	// if we're already ignoring parent tag
	if ( currentState == EStackIgnoreElement || currentState == EStackError)
	{
		// then let's ignore all children
		iStack->AppendL(EStackIgnoreElement);
		return;
	}
	
	//--------------- XML TAGS AND ATTR _LIT-------------
	
	_LIT(KStream,"stream:stream");
	_LIT(KStreamError, "stream:error");
	_LIT(KIq, "iq");
	_LIT(KPresence, "presence");
	_LIT(KMessage, "message");
	_LIT(KStatus, "status");
	_LIT(KBody, "body");
	_LIT(KSubject, "subject");
	
	_LIT(KId, "id");
	_LIT(KType, "type");
	_LIT(KResult, "result");
	_LIT(KFrom, "from");
	_LIT(KError, "error");
	_LIT(KCode, "code");
	_LIT(KUnavailable, "unavailable");
	
	//---------------------------------------------------
	
	TPtrC element = TPtrC( (TUint16*)el );
	
	if (element.Compare(KStream) == 0) {
		// Case: <stream:stream>
		iStack->AppendL(EStackSession);
		
		for (int i = 0; atts[i]; i += 2)
		{
			TPtrC attribute = TPtrC((TUint16*)atts[i]);
			if ( attribute.Compare(KId) == 0)
			{
				(iJabberSessionId).Copy(TPtrC((TUint16*)atts[i+1]));
			}
		}
		SendIdentificationL();
	} else if ( element.Compare(KStreamError) == 0) {
		// Case: <stream:error>
		iStack->AppendL(EStackStreamError);
	} else if ( (element.Compare(KIq) == 0) && ( currentState == EStackSession ) ) {
		// Case: <iq>
		TBool found = EFalse;
		for (int i = 0; atts[i]; i += 2)
		{
			TPtrC attribute = TPtrC((TUint16*)atts[i]);
			TPtrC value = TPtrC((TUint16*)atts[i+1]);
			
			if ( attribute.Compare(KType) == 0 )
			{
				if (value.Compare(KResult) == 0 )
				{
					iStack->AppendL(EStackConnected);
					iIdentTimer->Reset();
					iContextNotifier.NotifyConnected();
					iIdAttempt=0;
				}
				else if (value.Compare(KError) == 0)
				{
					iStack->AppendL(EStackIdentFailure);
					iIdAttempt++;
				}
				found = ETrue;
			}
		}
		if (!found)
		{
			iStack->AppendL(EStackIgnoreElement);				
		}
	} else if ( element.Compare(KError) == 0 ) {
		// Case: <error>
		iErrorCode.Zero();
		
		delete iErrorValue;
		iErrorValue = iErrorValue = HBufC::NewL(0);
		
		for (int i=0; atts[i]; i+=2)
		{
			TPtrC attribute = TPtrC((TUint16*)atts[i]);
			TPtrC value = TPtrC((TUint16*)atts[i+1]);
			if ( attribute.Compare(KCode) == 0 )
			{
				iErrorCode.Copy(value);			
			}
		}
		iStack->AppendL(EStackError);	
	} else if (element.Compare(KPresence) == 0) {
		// Case: <presence>
		TBool contextBuddyFound = EFalse;
		TBool offlinePresenceInfo = EFalse;
		
		for (int i = 0; atts[i]; i += 2)
		{
			TPtrC attribute = TPtrC((TUint16*)atts[i]);
			TPtrC value = TPtrC((TUint16*)atts[i+1]);
			
			if ( (attribute.Compare(KFrom) == 0) && ( value.Find(KDefaultJabberRessource) != KErrNotFound) )
			{
				contextBuddyFound = ETrue;
				
				TInt len=value.Length() - KDefaultJabberRessource().Length() -1;
				if (iFrom->Des().MaxLength() < len) {
					iFrom = iFrom->ReAllocL(len);
				}
				*(iFrom) = value.Left(value.Length() - KDefaultJabberRessource().Length()-1);
			}
			else if ( (attribute.Compare(KType) == 0) && (value.Compare(KUnavailable)==0) )
			{
				offlinePresenceInfo = ETrue;
			}
		}
		if (contextBuddyFound)
		{
			if (!offlinePresenceInfo) 
			{ 
				iStack->AppendL(EStackPresenceInfo); 
			}
			else
			{
				iStack->AppendL(EStackOfflinePresenceInfo);	
			}
		}
		else
		{
			iStack->AppendL(EStackIgnoreElement);	
		}
	} else if (element.Compare(KMessage) == 0) {
		// Case: <message>
		for (int i = 0; atts[i]; i += 2)
		{
			TPtrC attribute = TPtrC((TUint16*)atts[i]);
			TPtrC value = TPtrC((TUint16*)atts[i+1]);
			
			if (attribute.Compare(KFrom) == 0)
			{
				TInt len=value.Length();
				if (iFrom->Des().MaxLength() < len) {
					iFrom = iFrom->ReAllocL(len);
				}
				*iFrom=value;
			}
		}
		iStack->AppendL(EStackMessage);
	} else if (element.Compare(KSubject)==0 && currentState == EStackMessage) {
		iStack->AppendL(EStackSubject);
	} else if (element.Compare(KBody)==0 && currentState == EStackMessage) {
		iStack->AppendL(EStackBody);
	} else if (element.Compare(KStatus) == 0) {
		// Case: <status>
		iStack->AppendL(EStackStatus);	
	} else {
		// Case: unhandled tag
		iStack->AppendL(EStackIgnoreElement);
	}
}

void  CSocketsEngine::endElement(void *userData, const XML_Char *name)
{

	CSocketsEngine* eng=(CSocketsEngine*)userData;
	eng->endElement(name);
}

void  CSocketsEngine::endElement(const XML_Char *name)
{
	CALLSTACKITEM(_L("CSocketsEngine::endElement"));

	RDebug::Print(_L("-%s"),name);
	switch (iStack->iCurrent->Item)
	{
		
	case EStackIgnoreElement:
		// just pop the last item
		iStack->DeleteLast();
		break;
		
	case EStackIdentFailure:
		// post last item and initiate disconnection
		iStack->DeleteLast();
		
		if (iIdAttempt > KMaxIdentificationRetry)
		{
			iSocketsReader->SetIssueRead(EFalse);
			SendDisconnectionL();	
			ReportError(EIdentificationFailed, 0);
		}
		else
		{
			SendIdentificationL();
		}
		break;
		
	case EStackStatus:
		iStack->DeleteLast();
		break;
		
	case EStackPresenceInfo:
		{
			TInt timestamp_len=15;
			if (iPresenceInfo->Des().Length() > 0)
			{
				RDebug::Print(iPresenceInfo->Des().Left(timestamp_len));
				TTime stamp=CPresenceData::ParseTimeL(iPresenceInfo->Des().Left(timestamp_len));
				
				if (stamp == TTime(_L("20000101:000000.000000")))
				{
					iContextNotifier.NotifyNewPresenceInfo(*iFrom, 
						*iPresenceInfo, stamp);
				}
				else
				{
					iContextNotifier.NotifyNewPresenceInfo(*iFrom, 
						iPresenceInfo->Mid(timestamp_len), stamp);
				}
			}
			
			// reset values
			
			iFrom->Des().Zero();
			
			iPresenceInfo->Des().Zero();
			iStack->DeleteLast();
			break;
		}
	case EStackMessage:
		{
			if (iMessage->Des().Length() > 0 || iSubject->Des().Length()>0) {
				iContextNotifier.NotifyNewMessage(*iFrom, *iSubject, *iMessage);
			}
			iMessage->Des().Zero();
			iFrom->Des().Zero();
			iSubject->Des().Zero();
			iStack->DeleteLast();
		}
		break;
	case EStackBody:
	case EStackSubject:
		iStack->DeleteLast();
		break;
	case EStackOfflinePresenceInfo:
		//send something to context server to notify the offline status of contact
		
		//reset values
		
		delete iFrom;
		iFrom = iFrom = HBufC::NewL(0);
		
		iPresenceInfo->Des().Zero();
		
		iStack->DeleteLast();
		break;
		
	case EStackConnected:
		iStack->DeleteLast();
		if (iUserPresenceInfo->Length() >0)
		{
			SendPresenceInfoL(*iUserPresenceInfo);	
		}
		break;
		
	case EStackSession:
		// disconnect
		iStack->DeleteLast();
		if (Connected())
		{
			DisconnectSocket();
		}
		break;
		
	case EStackError:
		iStack->DeleteLast();
		break;
		
	case EStackStreamError:
		iStack->DeleteLast();
		iSocketsReader->SetIssueRead(EFalse);
		ReportError(EStreamError,0);
		break;
	}
}

void CSocketsEngine::charData(const XML_Char *s, int len)
{
	CALLSTACKITEM(_L("CSocketsEngine::charData"));

	if (!iStack->iCurrent) return;
	
	if (iStack->iCurrent->Item == EStackStatus) {
		TPtrC t = TPtrC((TUint16*)s,len);
		while ( iPresenceInfo->Length()+len > iPresenceInfo->Des().MaxLength()) {
			iPresenceInfo=iPresenceInfo->ReAllocL(iPresenceInfo->Des().MaxLength()*2);
		}
		iPresenceInfo->Des().Append(t);
		
		//Log(*iPresenceInfo);
	} else if (iStack->iCurrent->Item == EStackError)	{
		iErrorValue = iErrorValue->ReAllocL(iErrorValue->Length()+len);
		iErrorValue->Des().Append((TUint16*)s,len);
	} else if (iStack->iCurrent->Item == EStackBody) {
		TPtrC t = TPtrC((TUint16*)s,len);
		while ( iMessage->Length()+len > iMessage->Des().MaxLength()) {
			iMessage=iMessage->ReAllocL(iMessage->Des().MaxLength()*2);
		}
		iMessage->Des().Append(t);
	} else if (iStack->iCurrent->Item == EStackSubject) {
		TPtrC t = TPtrC((TUint16*)s,len);
		while ( iSubject->Length()+len > iSubject->Des().MaxLength()) {
			iSubject=iSubject->ReAllocL(iSubject->Des().MaxLength()*2);
		}
		iSubject->Des().Append(t);
	}
}

void CSocketsEngine::charData(void *userData, const XML_Char *s, int len)
{

	CSocketsEngine* eng=(CSocketsEngine*)userData;
	eng->charData(s, len);
}


// from MConnectionOpenerObserver --------------------------------------------------------------------
void CSocketsEngine::success(CBase* /*source*/)
{
	CALLSTACKITEM(_L("CSocketsEngine::success"));

	TRAPD(err, ConnectL());
	if (err!=KErrNone) {
		ReportError(EUnknownError, err);
	}
}

void CSocketsEngine::error(CBase* source, TInt code, const TDesC& reason)
{
	CALLSTACKITEM(_L("CSocketsEngine::error"));

	Log(reason);
	ReportError(ENetworkConnectError, code);
}

void CSocketsEngine::info(CBase* source, const TDesC& msg)
{
	CALLSTACKITEM(_L("CSocketsEngine::info"));

	Log(msg);
}
