/*
* $Id: RecognizerView.cpp,v 1.2 2004/10/01 08:41:52 mraento Exp $
*
* Context
*
* Copyright (C) 2004 Beat Gfeller, Michael Rohs (rohs@inf.ethz.ch)
* Copyright (C) 2004 Mika Raento
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

// INCLUDE FILES
#include "RecognizerView.h"
#include <recognizer.rsg>
#include <symbian_auto_ptr.h>
#include <app_context.h>
#include <avkon.hrh>
#include <aknviewappui.h>
#include "VisualCodeSystem.h"
#include <bautils.h>

enum KEYCODES {
	JOY_LEFT = 0xF807,
	JOY_RIGHT = 0xF808,
	JOY_UP = 0xF809,
	JOY_DOWN = 0xF80A,
	JOY_CLICK = 0xF845,
	KEY_CALL = 0xF862,
	KEY_CANCEL = 0xF863
};

class CRecognizerContainer : public CCoeControl, MVisualCodeObserver
{
public: // Constructors and destructor
	void ConstructL(const TRect& aRect, CAknView* View);
	~CRecognizerContainer();

public: // New functions

	CCodeInfo* GetCode();
private:
	void DrawImage(const CFbsBitmap* aImage) const;
	inline void DrawCross(CWindowGc& gc, const TPoint& center) const;
	void ShowCodes(const TRect& aDrawRect, const TSize& aBmpSize) const;

private:
	void SizeChanged();

	TInt CountComponentControls() const;

	CCoeControl* ComponentControl(TInt aIndex) const;

	void Draw(const TRect& aRect) const;

	void ViewFinderStartStop();

        virtual TKeyResponse OfferKeyEventL(
		const TKeyEvent& aKeyEvent,TEventCode aType);
private:
    
	CVisualCodeSystem* iVCS;
	CFbsBitmap* iBitmap; bool owns_bitmap;
	TBool iTakingPicture;
	TInt iCurrentCode;
	CCodeInfo* iCurrentCodeInfo;
	bool	iInvalid;
	RPointerArray<CCodeInfo>* iCurrentCodeArray;
	TPoint iTranslation;
	TPoint iTarget;
	TRealPoint iTilting;
	TInt iMagnification;
	bool	started;
	CAknView*	iView;
	RPointerArray<CCodeInfo>* iCodes;

private: // MVisualCodeObserver

	virtual void PictureUpdate(
		CFbsBitmap& aBitmap, 
		RPointerArray<CCodeInfo>* aCodes, 
		const TPoint& aTranslation,
			TInt aMinDiffTranslation,
			TInt aRotation,
			TInt aMinDiffRotation);

	virtual void PictureTaken(
		CFbsBitmap* aBitmap, 
		RPointerArray<CCodeInfo>* aCodes);

};

class CRecognizerViewImpl : public CRecognizerView
{
public: // // Constructors and destructor
	
	CRecognizerViewImpl(MRecognizerCallback* aCallback, TUid aId, TUid LocalDefaultView, TVwsViewId* NextViewId);
        void ConstructL();
        virtual ~CRecognizerViewImpl();
        
private:
        void DoActivateL(const TVwsViewId& aPrevViewId,
		TUid aCustomMessageId,
		const TDesC8& aCustomMessage);
        void DoDeactivate();
	
        void HandleCommandL(TInt aCommand);
	
	TUid Id() const;

private: //Data
        CRecognizerContainer* iContainer; 
	MRecognizerCallback* iCallback;
	TVwsViewId	iPrevView;
	TUid		iId;
	TInt		iResource;
	TUid iDefaultView;
	TVwsViewId* iNextViewId;
};


EXPORT_C CRecognizerView* CRecognizerView::NewL(MRecognizerCallback* aCallback, TUid aId, 
						TUid LocalDefaultView, TVwsViewId* NextViewId)
{
	auto_ptr<CRecognizerViewImpl> ret(new (ELeave) CRecognizerViewImpl(aCallback, aId, LocalDefaultView, NextViewId));
	ret->ConstructL();
	return ret.release();
}

EXPORT_C CRecognizerView::~CRecognizerView()
{
}

CRecognizerViewImpl::CRecognizerViewImpl(MRecognizerCallback* aCallback, TUid aId, TUid LocalDefaultView, TVwsViewId* NextViewId) : 
	iCallback(aCallback), iId(aId), iDefaultView(LocalDefaultView), iNextViewId(NextViewId)
{
}

TUid CRecognizerViewImpl::Id() const
{
	return iId;
}

void CRecognizerContainer::ConstructL(const TRect& aRect, CAknView* View)
{
	iView=View;
	iCurrentCodeArray=new RPointerArray<CCodeInfo>;

	iTakingPicture = FALSE;
	iCurrentCode = -1;
	iBitmap = NULL;
	iTranslation.SetXY(0, 0);
	iTarget.SetXY(320, 240);
	iTilting.SetXY(0.0, 0.0);
	iMagnification = 2;

	CreateWindowL();
	SetRect(aRect);
	ActivateL();

	ViewFinderStartStop();
}



// Destructor
CRecognizerContainer::~CRecognizerContainer()
{
	if (owns_bitmap) delete iBitmap;
	delete iVCS;
	delete iCurrentCodeInfo;
	if (iCurrentCodeArray) iCurrentCodeArray->Close();
	delete iCurrentCodeArray;
}

void CRecognizerContainer::SizeChanged()
{
}


TInt CRecognizerContainer::CountComponentControls() const
{
	return 0;
}

CCoeControl* CRecognizerContainer::ComponentControl(TInt /*aIndex*/) const
{
	return NULL;
}

void CRecognizerContainer::DrawImage(const CFbsBitmap* aImage) const
{
	CWindowGc& gc = SystemGc();
	gc.Activate(Window());
	gc.Clear();

	TSize si = aImage->SizeInPixels();
	TInt wi = si.iWidth;
	TInt hi = si.iHeight;

	TInt wr = Rect().iBr.iX;
	TInt hr = Rect().iBr.iY;
	TPoint tl;
	tl.iX = (wr - wi) >> 1;
	tl.iY = (hr - hi) >> 1;
	gc.BitBlt(tl, aImage);

	if (iTranslation.iX == 0 && iTranslation.iY == 0) {
		ShowCodes(TRect(tl.iX, tl.iY, tl.iX + wi, tl.iY + hi), si);
	}

	// draw cross
	gc.SetPenStyle(CGraphicsContext::ESolidPen);
	gc.SetPenColor(KRgbWhite);
	TPoint center = Rect().Center();
	DrawCross(gc, center);

	gc.Deactivate();
	iCoeEnv->WsSession().Flush();
}



inline void CRecognizerContainer::DrawCross(CWindowGc& gc, const TPoint& center) const 
{
	gc.DrawLine(TPoint(center.iX-5, center.iY), TPoint(center.iX+6, center.iY));
	gc.DrawLine(TPoint(center.iX, center.iY-5), TPoint(center.iX, center.iY+6));
}



// ---------------------------------------------------------
// CRecognizerContainer::Draw(const TRect& aRect) const
// ---------------------------------------------------------
//
void CRecognizerContainer::Draw(const TRect& aRect) const
{
	CWindowGc& gc = SystemGc();

	if (iBitmap != NULL) {
		TRect drawRect(0,1,176,133); // keep aspect ratio 160/120 = 640/480 = 4/3
		gc.DrawBitmap(drawRect, iBitmap);
		ShowCodes(drawRect, iBitmap->SizeInPixels());
	} else {
		CWindowGc& gc = SystemGc();
		gc.SetPenStyle(CGraphicsContext::ENullPen);
		if (!started) {
			gc.SetBrushColor(KRgbGray);
		} else {
			gc.SetBrushColor(KRgbWhite);
		}
		gc.SetBrushStyle(CGraphicsContext::ESolidBrush);
		gc.DrawRect(aRect);
	}

}



void CRecognizerContainer::ShowCodes(const TRect& aDrawRect, const TSize& aBmpSize) const
{
	RPointerArray<CCodeInfo>* Codes=0;
	TInt CurrentCode; bool draw_rect=true;
	if (!iVCS || iVCS->GetCodes() == NULL || iVCS->GetCodes()->Count()==0 || iInvalid) {
		if (!iCurrentCodeInfo) return;
		iCurrentCodeArray->Reset();
		User::LeaveIfError(iCurrentCodeArray->Append(iCurrentCodeInfo));
		Codes=iCurrentCodeArray;
		CurrentCode=0;
		if (iVCS || iInvalid) draw_rect=false;
	} else {
		Codes=iVCS->GetCodes();
		CurrentCode=iCurrentCode;
	}

	TInt xd = aDrawRect.iTl.iX;
	TInt yd = aDrawRect.iTl.iY;
	TInt hd = aDrawRect.Height();
	TInt wd = aDrawRect.Width();

	TInt hi = aBmpSize.iHeight;
	TInt wi = aBmpSize.iWidth;

	CWindowGc& gc = SystemGc();
	gc.SetPenStyle(CGraphicsContext::ESolidPen);

	TInt n = Codes->Count();

	// draw frames around the codes

	if (draw_rect) {
		for (TInt i = 0; i < n; i++) {
			CCodeInfo* ci = (*Codes)[i];

			gc.SetPenColor(ci->IsCodeValid() ? KRgbYellow : KRgbRed);
			gc.SetPenSize(i != CurrentCode ? TSize(1,1) : TSize(2,2));

			TPoint rb = ci->GetImageCoordinates(TPoint(10,10));

			gc.DrawLine(TPoint(xd + rb.iX*wd/wi, yd + rb.iY*hd/hi), 
						TPoint(xd + ci->x2*wd/wi, yd + ci->y2*hd/hi));
			gc.DrawLine(TPoint(xd + rb.iX*wd/wi, yd + rb.iY*hd/hi), 
						TPoint(xd + ci->x4*wd/wi, yd + ci->y4*hd/hi));
			gc.DrawLine(TPoint(xd + ci->x1*wd/wi, yd + ci->y1*hd/hi), 
						TPoint(xd + ci->x2*wd/wi, yd + ci->y2*hd/hi));
			gc.DrawLine(TPoint(xd + ci->x4*wd/wi, yd + ci->y4*hd/hi), 
						TPoint(xd + ci->x1*wd/wi, yd + ci->y1*hd/hi));
		}
	}

	gc.SetPenSize(TSize(1,1));

	// display value of current code

	if (CurrentCode >= 0 && CurrentCode < Codes->Count()) {
		CCodeInfo* ci = (*Codes)[CurrentCode];
		TBuf<64> info;

		// code value

		ci->code->AppendToString(info);
		gc.SetPenColor(KRgbYellow);
		gc.SetBrushStyle(CGraphicsContext::ENullBrush);
		const CFont* fontUsed = iEikonEnv->DenseFont();
		gc.UseFont(fontUsed);
		TInt baseline = aDrawRect.Height() - 2 * fontUsed->AscentInPixels();
		gc.DrawText(info, aDrawRect, baseline, CGraphicsContext::ECenter);

	}
}



void CRecognizerContainer::ViewFinderStartStop()
{
	started=true;

	if (!iVCS || !iTakingPicture) {
		delete iVCS; iVCS=0; iBitmap=0;
		iVCS=new (ELeave) CVisualCodeSystem();
		iVCS->ConstructL();
		iVCS->AddObserver(this);
	}
	if (iTakingPicture) {
		if (iBitmap) {
			CFbsBitmap* temp=iBitmap;
			iBitmap=new (ELeave) CFbsBitmap;
			owns_bitmap=true;
			User::LeaveIfError(iBitmap->Duplicate(temp->Handle()));
		}
		delete iVCS; iVCS=0;
		iTakingPicture=FALSE;
	} else {
		iTakingPicture = TRUE;
		// start with: updates on, translation off, rotation off, low-quality recognition on
		iVCS->Start(TRUE, FALSE, FALSE, TRUE);
	}
}

void CRecognizerContainer::PictureUpdate(
	CFbsBitmap& aBitmap, 
	RPointerArray<CCodeInfo>* aCodes, 
	const TPoint& aTranslation,
	TInt /*aMinDiffTranslation*/,
	TInt /*aRotation*/,
	TInt /*aMinDiffRotation*/)
{
	iTranslation.SetXY(aTranslation.iX, aTranslation.iY);
	if (aCodes != NULL) {
		iCurrentCode = iVCS->FindClosestCode(iTarget);
		if (iCurrentCode >= 0) {
			CCodeInfo* ci = (*aCodes)[iCurrentCode];
			TPoint p = ci->GetTilting();
			iTilting.SetXY(p.iX, p.iY);

			if (ci->IsCodeValid()) {
				iInvalid=false;
				delete iCurrentCodeInfo; iCurrentCodeInfo=0;
				iCurrentCodeInfo=new (ELeave) CCodeInfo;
				iCurrentCodeInfo->ConstructL(*ci);
			} else {
				iInvalid=true;
			}
		}
	}

	iBitmap=&aBitmap;
	DrawImage(&aBitmap);
}

CCodeInfo* CRecognizerContainer::GetCode()
{
	return iCurrentCodeInfo;
}

void CRecognizerContainer::PictureTaken(
	CFbsBitmap* aBitmap, 
	RPointerArray<CCodeInfo>* /*aCodes*/)
{
	iTakingPicture = FALSE;
	iBitmap = aBitmap;
	iCurrentCode = iVCS->FindClosestCode(iTarget);
	DrawNow();
}


void CRecognizerViewImpl::ConstructL()
{
#ifndef __WINS__
	TFileName resfile=_L("c:\\System\\data\\recognizer.rsc");
#else
	TFileName resfile=_L("z:\\System\\data\\recognizer.rsc");
#endif
	BaflUtils::NearestLanguageFile(iEikonEnv->FsSession(), resfile); //for localization
	User::LeaveIfError(iResource=iEikonEnv->AddResourceFileL(resfile));

	BaseConstructL(R_RECOGNIZER_VIEW);
}

void CRecognizerViewImpl::DoActivateL(const TVwsViewId& aPrevViewId, TUid /*aCustomMessageId*/,
		const TDesC8& /*aCustomMessage*/)
{
	iPrevView=aPrevViewId;
	if (!iContainer) {
		iContainer = new (ELeave) CRecognizerContainer;
		iContainer->SetMopParent( this );
		iContainer->ConstructL( ClientRect(), this );
		AppUi()->AddToStackL( *this, iContainer );
	}
}

void CRecognizerViewImpl::DoDeactivate()
{
	if ( iContainer )
        {
		AppUi()->RemoveFromViewStack( *this, iContainer );
        }
	
	delete iContainer;
	iContainer = 0;
}

// ----------------------------------------------------
// CRecognizerViewImpl::~CRecognizerViewImpl()
// Destructor
// Frees reserved resources
// ----------------------------------------------------
//
CRecognizerViewImpl::~CRecognizerViewImpl()
{
	if (iContainer)
        {
		AppUi()->RemoveFromViewStack( *this, iContainer );
		delete iContainer;
        }
	if (iResource) iEikonEnv->DeleteResourceFile(iResource);
}

// ----------------------------------------------------
// CRecognizerViewImpl::HandleKeyEventL(
//     const TKeyEvent& aKeyEvent,TEventCode /*aType*/)
// takes care of key event handling
// ----------------------------------------------------
//
TKeyResponse CRecognizerContainer::OfferKeyEventL(
						  const TKeyEvent& aKeyEvent, TEventCode aType)
{
	if (aKeyEvent.iCode==JOY_CLICK && aType==EEventKey) {
		if (iTakingPicture==TRUE) {
			ViewFinderStartStop();
		} else {
			iView->HandleCommandL(EAknSoftkeyOk);
		}
		return EKeyWasConsumed;
	} else {
		return EKeyWasNotConsumed;
	}
}

// ----------------------------------------------------
// CRecognizerViewImpl::HandleCommandL(TInt aCommand)
// takes care of command handling
// ----------------------------------------------------
//
void CRecognizerViewImpl::HandleCommandL(TInt aCommand)
{
	switch ( aCommand )
        {
	case EAknSoftkeyOk:
		{
		CCodeInfo* code=iContainer->GetCode();
		if (!code) return;
		iCallback->CodeSelected(*code);
		}
		break;
	case EAknSoftkeyCancel:
		{
		iCallback->Cancelled();
		}
		break;
        default:
		return;
		break;      
        }
	AppUi()->ActivateLocalViewL(iDefaultView);
	*iNextViewId=iPrevView;
}

// End of File  
