/*
 * $Id: CodeInfo.h,v 1.1.1.1 2004/08/06 10:53:19 mraento Exp $
 *
 * Visual Codes for Symbian OS
 * Copyright (C) 2004 Beat Gfeller, Michael Rohs (rohs@inf.ethz.ch)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef CODEINFO_H
#define CODEINFO_H

// INCLUDES

#include <e32base.h>
#include "BigInteger.h"
#include "Warper.h"
#include "CodeChecker.h"

// CONSTANTS

#define INVALID_ROTATION 0xffff

// FORWARD DECLARATIONS

// CLASS DECLARATION

/**
 * Represents a single code found in the image.
 */
class CCodeInfo : public CBase {

public:

	/** Value of the code. */
	CBigInteger* code;

	/** Coordinates of upper left corner stone (code coordinate (0,0)). */
	TInt x1, y1;

	/** Coordinates of upper right corner stone (code coordinate (10,0)). */
	TInt x2, y2;

	/** Coordinates of shorter guide bar (code coordinate (8,10)). */
	TInt x3, y3;

	/** Coordinates of lower left corner stone (code coordinate (0,10)). */
	TInt x4, y4;

	/** Coordinates of longer guide bar (code coordinate (10,5)). */
	TInt x5, y5;

	/** Image width and height (in pixels). */
	TInt iw, ih;

	/**
	 * Projective warper for this code. Used to transform
	 * image to code coordinates and vice versa.
	 */
	TWarper* warper;

	/**
	 * True, iff a successful error check was performed on this code.
	 * Applicable only to codes that are protected by an error correcting code.
	 */
	TBool errorCheckPassed;

public:

	/** Create object and initialize fields. */
	CCodeInfo(TInt x1, TInt y1, TInt x2, TInt y2, 
			  TInt x3, TInt y3, TInt x4, TInt y4,
			  TInt x5, TInt y5, TInt iw, TInt ih, 
			  CBigInteger* c, TWarper* w);

	/** Default constructor. For use with copy constructor. */
	CCodeInfo();

	/** 2nd phase copy constructor. */
	void ConstructL(const CCodeInfo& aCi);

	/** Virtual destructor for clean up. */
	virtual ~CCodeInfo();

	/** Get coordinates of image center in code coordinate system. */
	inline TPoint GetCodeCoordinatesImageCenter();

	/** Get the code coordinates corresponding to the given of image coordinates. */
	inline TPoint GetCodeCoordinates(const TPoint& imageCoordinates);

	/** Get the image coordinates corresponding to the given of code coordinates. */
	inline TPoint GetImageCoordinates(const TPoint& codeCoordinates);

	/** Get the image coordinates corresponding to the given of code coordinates. */
	inline TPoint GetImageCoordinates(TReal aX, TReal aY) const;

	/** Get the precise code coordinates corresponding to the given of image coordinates. */
	inline TRealPoint GetCodeCoordinatesPrecise(const TRealPoint& imageCoordinates);

	/** Get the precise code coordinates corresponding to the given of image coordinates. */
	inline TRealPoint GetCodeCoordinatesPrecise(const TReal& aX, const TReal& aY);

	/** Get the precise image coordinates corresponding to the given of code coordinates. */
	inline TRealPoint GetImageCoordinatesPrecise(const TRealPoint& codeCoordinates);

	/** Get the precise image coordinates corresponding to the given of code coordinates. */
	inline TRealPoint GetImageCoordinatesPrecise(const TReal& aX, const TReal& aY);

	/** Get amount of rotation of the code in the image (in degrees counterclockwise). */
	TInt GetRotation();

	/** 
	 * Tilting of the code plane relative to the image plane.
	 * iX represents the tilting of the x-axis of the code,
	 * iY represents the tilting of the y-axis of the code.
	 * iX < 0 means tilting to the left, iX > 0 means tilting to the right.
	 * iY < 0 means tilting to the top, iY > 0 means tilting to the bottom.
	 */
	TPoint GetTilting();

	/** Return error checking result. */
	inline TBool IsCodeValid() { return errorCheckPassed; }

private:

	/** Rotation of the code in degrees counterclockwise. */
	TInt iRotation;

	/** 
	 * Tilting of the code plane relative to the image plane.
	 * iX represents the tilting of the x-axis of the code,
	 * iY represents the tilting of the y-axis of the code.
	 * iX < 0 means tilting to the left, iX > 0 means tilting to the right.
	 * iY < 0 means tilting to the top, iY > 0 means tilting to the bottom.
	 */
	TPoint iTilting;

	/** Compute the distance between the two given points. */
	TReal Distance(const TRealPoint& p, const TRealPoint& q);

};



inline TPoint CCodeInfo::GetCodeCoordinatesImageCenter()
{
	return warper->Backwarp(iw >> 1, ih >> 1);
}



inline TPoint CCodeInfo::GetCodeCoordinates(const TPoint& imageCoordinates)
{
	return warper->Backwarp(imageCoordinates.iX, imageCoordinates.iY);
}



inline TPoint CCodeInfo::GetImageCoordinates(const TPoint& codeCoordinates)
{
	return warper->Warp(codeCoordinates.iX, codeCoordinates.iY);
}



inline TPoint CCodeInfo::GetImageCoordinates(TReal aX, TReal aY) const
{
	return warper->Warp(aX, aY);
}



inline TRealPoint CCodeInfo::GetCodeCoordinatesPrecise(const TRealPoint& imageCoordinates)
{
	return warper->BackwarpPrecise(imageCoordinates.iX, imageCoordinates.iY);
}



inline TRealPoint CCodeInfo::GetCodeCoordinatesPrecise(const TReal& aX, const TReal& aY)
{
	return warper->BackwarpPrecise(aX, aY);
}



inline TRealPoint CCodeInfo::GetImageCoordinatesPrecise(const TRealPoint& codeCoordinates)
{
	return warper->WarpPrecise(codeCoordinates.iX, codeCoordinates.iY);
}



inline TRealPoint CCodeInfo::GetImageCoordinatesPrecise(const TReal& aX, const TReal& aY)
{
	return warper->WarpPrecise(aX, aY);
}



#endif

// End of File