/*
 * $Id: VisualCodeSystem.h,v 1.1.1.1 2004/08/06 10:53:19 mraento Exp $
 *
 * Visual Codes for Symbian OS
 * Copyright (C) 2004 Beat Gfeller, Michael Rohs (rohs@inf.ethz.ch)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef VISUALCODESYSTEM_H
#define VISUALCODESYSTEM_H

// INCLUDES

#include <e32base.h>

#include "CodeInfo.h"
#include "VisualCodeObserver.h"
#include "PictureTaker.h"
#include "RecognitionAlgorithm.h"


// CONSTANTS

// FORWARD DECLARATIONS

class CPictureTaker;

// CLASS DECLARATION


/**
 * Provides the main interface to the visual code system.
 */
class CVisualCodeSystem : public CBase 
{

public:

	/**
	 * Construct visual code system with the given observer.
	 * @param aObserver observes recognized visual codes
	 */
	CVisualCodeSystem();

	/**
	 * Two-phase construction.
	 */
	void ConstructL();

	/**
	 * Virtual destructor.
	 */
	virtual ~CVisualCodeSystem();

	/** Add observer to the set of visual code observers. */
	void AddObserver(const MVisualCodeObserver* aObserver);

	/** Remove observer from the set of visual code observers. */
	void RemoveObserver(const MVisualCodeObserver* aObserver);

	/**
	 * Start view finder, optionally send low-quality picture updates, optionally
	 * do motion and rotation detection, and optionally perform visual code 
	 * recognition on the low quality pictures. Periodically (for each frame) calls 
	 * PictureUpdate on the observer.
	 *
	 * @param aSendUpdates send low-quality pictures for view finder
	 * @param aSendTranslation compute (x,y)-motion of consecutive views
	 * @param aSendRotation compute rotation values of consecutive views
	 * @param aDoLQRecognition perform code recognition on low-quality pictures 
	 *                         while view finder is active (degrades view finder performance)
	 * 
	 */
	void Start(
		TBool aSendUpdates, 
		TBool aSendTranslation,
		TBool aSendRotation,
		TBool aDoLQRecognition);
			/* const CCoeCtrl* aDrawTarget, 
			   TBool aShowCross,
			   TBool aShowMagnification,*/

	/**
	 * Stop view finder, take a final high-quality picture, 
	 * and recognize visual codes it contains. As a result
	 * PictureTaken is called on the observer.
	 */
	inline void Stop() {
		iPictureTaker->StopL();
	}

	inline void SetSendTranslation(TBool aSendTranslation) {
		iPictureTaker->SetSendTranslation(aSendTranslation);
	}

	inline void SetSendRotation(TBool aSendRotation) {
		iPictureTaker->SetSendRotation(aSendRotation);
	}

	inline void SetDoLQRecognition(TBool aDoLQRecognition) {
		iDoLQRecognition = aDoLQRecognition;
	}

	//	void DrawCodes(const CWindowGC & aGc, const TRect& aArea);

	//	void Unwarp(CFbsBitmap& aBmpResult, const TRect& aArea);

	/**
	 * Gets the most recently recognized codes. Only valid if the recognition 
	 * process is not active. The array has to be duplicated if necessary.
	 *
	 * @return most recently recognized codes
	 */
	inline RPointerArray<CCodeInfo>* GetCodes() const { 
		return iRecognizer->GetCodes(); 
	};

	/**
	 * Gets the most recently captured picture. The bitmap has to be duplicated 
	 * if necessary.
	 *
	 * @return most recently recognized codes
	 */
	inline CFbsBitmap* GetBitmap() const { 
		return iPictureTaker->GetBitmap(); 
	};

	/**
	 * Returns the index of the code closest to the given image coordinates
	 * or -1 if no code is available.
	 *
	 * @param target image coordinates relative to which closest code 
	 *               is to be found
	 * @return closest code to given image coordinates
	 */
	TInt FindClosestCode(const TPoint& target) const;

private: // for use by CPictureTaker, private "friend" access

	/** 
	 * Callback method used by iPictureTaker for sending 
	 * low-quality picture updates during view finding.
	 */
	void PictureUpdateL(
		CFbsBitmap& aPicture, 
		const TPoint& aTranslation,
		TInt aMinDiffTranslation,
		TInt aRotation,
		TInt aMinDiffRotation);

	/** 
	 * Callback method used by iPictureTaker for sending the final 
	 * high-quality picture at the end of the view finding process.
	 */
	void PictureTakenL(CFbsBitmap* aPicture);

	friend class CPictureTaker;

private:

	/** Observers of code sightings. */
	RPointerArray<MVisualCodeObserver> iObservers;

	/** Takes pictures for us. */
	CPictureTaker* iPictureTaker;

	/** The actual heart of the system: the visual code recognizer. */
	CRecognitionAlgorithm* iRecognizer;

	/** If set, even in the view finding process, visual codes are recognized. */
	TBool iDoLQRecognition;

};

#endif
