/*
 * $Id: CodeChecker.cpp,v 1.1.1.1 2004/08/06 10:53:19 mraento Exp $
 *
 * Visual Codes for Symbian OS
 * Copyright (C) 2004 Beat Gfeller, Michael Rohs (rohs@inf.ethz.ch)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <eikenv.h>
#include "CodeChecker.h"
#include "BigInteger.h"
#include "CodeInfo.h"



const TUint32 CCodeChecker::H[7][3] = 
{
	{ 0x0, 0x7f, 0x3ffffc0 },
	{ 0x3f, 0xffffff80, 0xfc000020 },
	{ 0x1fffc0, 0x7fff80, 0xfc000790 },
	{ 0xfe03fc0, 0x7f807f80, 0xfc07f808 },
	{ 0x71e3c3c7, 0x8787878f, 0x3c787804 },
	{ 0xb66cccd9, 0x999999b3, 0xcd999982 },
	{ 0xdab5556a, 0xaaaaaad5, 0x56aaaa81 }
};



CCodeChecker::CCodeChecker()
{
	mask1 = NULL;
	mask2 = NULL;
	mask3 = NULL;
	tmp1 = NULL;
	tmp2 = NULL;
}



void CCodeChecker::ConstructL() 
{
	if (mask1 != NULL) return;

	CBigInteger* m1 = new (ELeave) CBigInteger();
	CleanupStack::PushL(m1);
	m1->ConstructL(83);

	CBigInteger* m2 = new (ELeave) CBigInteger();
	CleanupStack::PushL(m2);
	m2->ConstructL(83);
		
	CBigInteger* m3 = new (ELeave) CBigInteger();
	CleanupStack::PushL(m3);
	m3->ConstructL(83);
		
	CBigInteger* t1 = new (ELeave) CBigInteger();
	CleanupStack::PushL(t1);
	t1->ConstructL(83);

	CBigInteger* t2 = new (ELeave) CBigInteger();
	CleanupStack::PushL(t2);
	t2->ConstructL(83);

	CleanupStack::Pop(5); // t2, t1, m3, m2, m1

	mask1 = m1;
	mask2 = m2;
	mask3 = m3;
	tmp1 = t1;
	tmp2 = t2;

	mask1->Set(_L("7ff800000000000000000"));
	mask2->Set(_L("00000ffffffffffffffff"));
	mask3->Set(_L("7fa7ea7fa7ea7fa7ea7f")); // 10-byte mask
}



CCodeChecker::~CCodeChecker()
{
	delete mask1;
	delete mask2;
	delete mask3;
	delete tmp1;
	delete tmp2;
}



TInt CCodeChecker::ErrorPosition(CBigInteger* code) const 
{
	if (code == NULL || code->GetBitCount() != 83) {
		return EFalse;
	}

	// code (83 bits = 32 + 32 + 19 bits)
	TUint32 c[] = {0, 0, 0};
	const TUint8* x = code->GetBuffer();

	c[0]  = (x[3] & 0xff) << 24;
	c[0] |= (x[2] & 0xff) << 16;
	c[0] |= (x[1] & 0xff) << 8;
	c[0] |=  x[0] & 0xff;

	c[1]  = (x[7] & 0xff) << 24;
	c[1] |= (x[6] & 0xff) << 16;
	c[1] |= (x[5] & 0xff) << 8;
	c[1] |=  x[4] & 0xff;

	c[2]  = (x[10] & 0xff) << 16;
	c[2] |= (x[9] & 0xff) << 8;
	c[2] |=  x[8] & 0xff;

	TInt errorPos = ErrorPosition(c);

	// no attempt is made to correct the error
	/*
	if (errorPos != -1) {
		TInt a = errorPos / 32;
		TInt b = (a<2? 31:18) - (errorPos % 32);
		if (a >= 0 && a < x.length) {
			x[a] ^= (1 << b);
		}
	}
	*/

	return errorPos;
}



TInt CCodeChecker::ErrorPosition(TUint32* code) const 
{
	TInt rows = 7;
	TInt cols = 3;

	TUint32 syndrom = 0;
	
	for (TInt y=0; y<rows; y++) {
		TInt onesCount = 0;
		for (TInt x=0; x<cols; x++) {
			TUint32 v = code[x] & H[y][x];
			while (v != 0) {
				onesCount += (v & 1);
				v = (v >> 1) & 0x7fffffff;
			}
		}
		syndrom = (syndrom << 1) | (onesCount & 1);
	}

	if (syndrom == 0) return -1;
	
	TInt log2 = 0;
	TInt onesCount = 0;
	TUint32 v = syndrom;
	while (v != 0) {
		log2++;
		onesCount += (v & 1);
		v = (v >> 1) & 0x7fffffff;
	}
	if (onesCount == 1) {
		// error in parity part
		return 83 - log2;
	}

	return syndrom - log2 - 1;
}



void CCodeChecker::Decode(CBigInteger* code) 
{
	tmp1->Copy(code);
	tmp1->And(mask1);
	tmp1->ShiftRight(7);
	code->And(mask2);
	code->Or(tmp1);
}



TBool CCodeChecker::DoErrorCheck(CCodeInfo* ci) 
{
	tmp2->Copy(ci->code);
	tmp2->Xor(mask3);

	if (ErrorPosition(tmp2) == -1) {
		Decode(tmp2);
		ci->code->Copy(tmp2);
		ci->errorCheckPassed = ETrue;
		return ETrue;
	}

	return EFalse;
}


