/*
 * $Id: RecognitionAlgorithm.cpp,v 1.1.1.1 2004/08/06 10:53:19 mraento Exp $
 *
 * Visual Codes for Symbian OS
 * Copyright (C) 2004 Beat Gfeller, Michael Rohs (rohs@inf.ethz.ch)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

// INCLUDE FILES

#include <e32math.h>

#include "RecognitionAlgorithm.h"
#include "ClusterInfo.h"
#include "Warper.h"
#include "BigInteger.h"
#include "CodeInfo.h"
#include "CodeChecker.h"

// PREPROCESSOR CONSTANTS

// set this to 1 if you wish that the bitmap is changed to black and white after processing.
#define DRAW_BW 0

#define SHOW_CODES 1
#define SHOW_CODE_VALUES 0
#define SHOW_CODE_DOTS 0

/** Constant used to mark background pixels. */
#define BACKGROUND -1

/** Constant used to mark foreground (object) pixels. */
#define FOREGROUND 0

// CONSTANTS

// these are all important parameters to be set for the recognition

/**
 * Used to correct the "erosion" effect (stretch for MINOR axis).
 * Find good constant -> 1.02 seems OK (the effect is not too strong though)
 */
const TReal CRecognitionAlgorithm::stretch = 1.02;

/**
 *  Number of units which are allowed for searching the corners.
 * (units = max. distance from estimated position)
 */
const TReal CRecognitionAlgorithm::units = 2.0;

/**
 * Number of units that the corners may be corrected.
 * If the jump would be larger, no correction is applied.
 */
const TReal CRecognitionAlgorithm::maxcorrect = 3.0;

/** Moment range for major guide bar. */
const TReal CRecognitionAlgorithm::majorbar_min = 0.00;
const TReal CRecognitionAlgorithm::majorbar_max = 0.10;

/** Moment range for minor guide bar. */
const TReal CRecognitionAlgorithm::minorbar_min = 0.00;
const TReal CRecognitionAlgorithm::minorbar_max = 0.30;

/** Moment range for corner stones (other than guide bars). */
const TReal CRecognitionAlgorithm::corner_min = 0.20;
const TReal CRecognitionAlgorithm::corner_max = 1.00;

/** Most extreme ratio allowed for length of left edge to length of top edge. */
const TReal CRecognitionAlgorithm::minratio = 0.5;

/** Set this in DEGREES. */
const TReal CRecognitionAlgorithm::anglediff = 30.0;

/** Range of angles allowed for top left corner (undistorted: 90). */
const TReal CRecognitionAlgorithm::minangle = (90.0-anglediff) * KDegToRad;
const TReal CRecognitionAlgorithm::maxangle = (90.0+anglediff) * KDegToRad;



/**************************************************************************/

/** Default constructor. */
CRecognitionAlgorithm::CRecognitionAlgorithm() 
{
	iBmp = NULL;
	w = 0; h = 0;
	p = NULL;

	pSize = 0;
	prev_gn = NULL;
	prev_gnSize = 0;
	scanline = NULL;
	scanlineSize = 0;
	equals = NULL;
	equalsSize = 2000; // 2000 should suffice in most cases, dynamic reallocation if too small

	clusters = NULL;
	currentCluster = 0;
	clusterCount = 0;
	codes = NULL;
	codeChecker = NULL;

	debugCounter = 0;
}



void CRecognitionAlgorithm::ConstructL() 
{
}



CRecognitionAlgorithm::~CRecognitionAlgorithm() {
	// DON'T DELETE bitmap, because the calling class 
	// still needs it and will delete it

	delete [] p;
	delete [] prev_gn;
	delete [] scanline;
	delete [] equals;

	delete [] clusters;

	if (codes != NULL) {
		codes->ResetAndDestroy();
		delete codes;	
	}

	delete codeChecker;
}



/**************************************************************************/

/**
 * Looks for codes in the image. 
 * Note: The returned array and its components will only be valid until 
 * the next call to RecognizeL and until the RecognitionAlgorithm is destroyed.
 * @param aBmp image to find visual codes in
 * @return set of CodeInfo objects, only valid until next call to RecognizeL
 */
RPointerArray<CCodeInfo>* CRecognitionAlgorithm::RecognizeL(const CFbsBitmap* aBmp) 
{
	TSize s = aBmp->SizeInPixels();
	return RecognizeL(aBmp, TRect(0, 0, s.iWidth, s.iHeight));
}



/**
 * Looks for codes in the image. 
 * Note: The returned array and its components will only be valid until 
 * the next call to RecognizeL and until the RecognitionAlgorithm is destroyed.
 * @param aBmp image to find visual codes in
 * @param aArea the image area in which to look for codes
 * @return set of CodeInfo objects, only valid until next call to RecognizeL
 */
RPointerArray<CCodeInfo>* CRecognitionAlgorithm::RecognizeL(const CFbsBitmap* aBmp, const TRect& aArea) 
{
	TTime t1;
	t1.UniversalTime();

	// main steps

	// 1) set the image

	iBmp = aBmp;
	iArea.SetRect(aArea.iTl.iX, aArea.iTl.iY, aArea.iBr.iX, aArea.iBr.iY);
	w = aArea.iBr.iX - aArea.iTl.iX;
	h = aArea.iBr.iY - aArea.iTl.iY;

	// 2) correct barrel distortion

	CorrectBarrelDistortion();

	// 3) do the thresholding and the first phase of cluster finding
	//    (recording cluster equivalences)

	DoThresholdingL();

	// 4) do the second phase of cluster finding (resolving equivalences),
	//    calculate the sizes, centers, and bounding boxes of all clusters

	FindClustersL();

	// 5) do the actual code recognition (using the clusters found before)

	DoRecognitionL();

	// 6) do error checks

	if (codeChecker == NULL) {
		codeChecker = new (ELeave) CCodeChecker();
		CleanupStack::PushL(codeChecker);
		codeChecker->ConstructL();
		CleanupStack::Pop();
	}

	TInt n = codes->Count();
	for (TInt i = 0; i < n; i++) {
		CCodeInfo* ci = (*codes)[i];
		codeChecker->DoErrorCheck(ci);
	}

	// 7) return the result

	TTime t2;
	t2.UniversalTime();
	TTimeIntervalMicroSeconds mus = t2.MicroSecondsFrom(t1);
	TInt ms = mus.Int64().Low() / 1000;
	debugCounter = ms;

#if 0
	// save black and white image

	CBigInteger* bi = new (ELeave) CBigInteger(pSize);
	bi->ConstructL();
	for (TInt i = 0; i < pSize; i++) {
		if (p[i] != BACKGROUND) bi->SetBit(i);
	}
	RFs fs;
	User::LeaveIfError(fs.Connect());
	RFile f;
	User::LeaveIfError(f.Replace(fs, _L("C:\\Nokia\\Images\\rl.gif"), 
		EFileWrite | EFileShareExclusive | EFileStream));
	f.Write(*(bi->iBuffer));
	f.Close();
	fs.Close();
	delete bi;
#endif

#if 0
	// just for saving memory

	delete [] p;
	p = NULL;

	delete [] prev_gn;
	prev_gn = NULL;

	delete [] scanline;
	scanline = NULL;

	delete [] equals;
	equals = NULL;

	delete [] clusters;
	clusters = NULL;
#endif

	return codes;
}



/**************************************************************************/

/**
 * Step 2: Correct barrel distortion.
 */
void CRecognitionAlgorithm::CorrectBarrelDistortion() 
{
	/*
	BarrelDistortion bd = new BarrelDistortion();
	img = bd.correctBarrelDistortion(img);
	*/
}



/**************************************************************************/

/**
 * Step 3: Do the thresholding and the first phase of cluster finding.
 * <br>
 * Uses a slightly modified version of Wellner's adaptive thresholding:<br>
 * Pierre D. Wellner: Thresholding for the DigitalDesk. Technical
 * Report EPC-93-110, Rank Xerox Research Centre, Cambridge, UK, 1993.
 */
void CRecognitionAlgorithm::DoThresholdingL() 
{
	/**           /
	 *            | FOREGROUND, if pn < ((gs(n) + gs(n-w)) / (2*s)) *
	 * color(n) = |                     ((100-t)/100)
	 *            | BACKGROUND, otherwise
	 *            \
	 * where pn = gray value of current pixel,
	 *        s = width of moving average, and
	 *        t = threshold percentage of brightness range
	 *    gs(n) = gs(n-1) * (1-1/s) + pn
	 *    gs(n-w) = gs-value of pixel above current pixel
	 *
	 */

	TInt s = w >> 3; // s: number of pixels in the moving average (w = image width)
	if (s <= 0) s = 1;
	TInt t = 15;     // t: percentage of brightness range considered white

	const TInt S = 9; // integer shift, needed to avoid floating point operations
	const TInt power2S = 1 << S;

	// for speedup: multiply all values by 2^s, and use integers instead of doubles

	TInt factor = power2S * (100-t) / (100*s); // multiplicand for threshold

	TInt gn = 127 * s; // initial value of the moving average (127 = average gray value)
	TInt q = power2S - power2S / s; // constant needed for average computation

	// allocate space for the p[] array
	// only allocate if not already done (reuse)
	if (p != NULL && pSize != h * w) { // reallocate if w or h change //@todo only change if bigger
		delete [] p;
		p = NULL;
	}
	if (p == NULL) {
		pSize = h * w;
		p = new (ELeave) TInt[pSize];
		// alternatively: p = (TInt*) User::AllocL(pSize * sizeof(TInt));
	}

	// store previous scanline of gn-values
	// required to avoid artifacts in each other line
	if (prev_gn != NULL && prev_gnSize != w) { // reallocate if w changes
		delete [] prev_gn;
		prev_gn = NULL;
	}
	if (prev_gn == NULL) {
		prev_gnSize = w;
		prev_gn = new (ELeave) TInt[prev_gnSize];
	}

	// initalize prev_gn with initial gn value
	TInt i;
	for (i = 0; i < w; i++) {
		prev_gn[i] = gn;
	}

	// used to store the label equivalences (index 0 unused)
	if (equals == NULL) {
		equals = new (ELeave) TInt[equalsSize];
	}
	for (i = 0; i < equalsSize; i++) {
		equals[i] = 0;
	}
	currentCluster = 0;

	// buffer for scanline, assume 3 bytes per pixel!
	if (scanline != NULL && scanlineSize != 3 * w) { // reallocate if w changes
		delete [] scanline;
		scanline = NULL;
	}
	if (scanline == NULL) {
		scanlineSize = 3 * w;
		scanline = new (ELeave) TUint8[scanlineSize];
	}
	TPtr8 scanlineTPtr8 = TPtr8(scanline, scanlineSize);

	TDisplayMode mode = EColor16M; // iBmp->DisplayMode()

	TInt pn, hn;
	TInt x, y, y2;

	// run through all pixels in a zig-zag fashion
	// this is the most time consuming operation of the recognition process

	TInt offset;
	TInt ytop = iArea.iTl.iY;
	TInt xleft = iArea.iTl.iX;

	for (y = 0, y2 = 0; y < h; y++, y2 += w) {

		iBmp->GetScanLine(scanlineTPtr8, TPoint(xleft, y + ytop), w, mode);

		offset = 1;
		for (x = 0; x < w; x++) { // left-to-right

			// get pixel color and transform to grey: (red + green) / 2
			pn = (scanline[offset] + scanline[offset+1]) >> 1;
			offset += 3;

			gn = ((gn * q) >> S) + pn; // update average

			// average of current pixel and the one above it
			hn = (gn + prev_gn[x]) >> 1;
			prev_gn[x] = gn; // update previous scanline memory

			if (pn < (hn * factor) >> S) {
				TInt top = (y > 0) ? p[y2-w + x] : BACKGROUND;
				TInt left = (x > 0) ? p[y2 + x-1] : BACKGROUND;
				ProcessForegroundPixelL(y2 + x, top, left);
			} else {
				p[y2 + x] = BACKGROUND;
			}
		}

		y++;
		y2 += w;
		if (y == h) break;

		iBmp->GetScanLine(scanlineTPtr8, TPoint(xleft, y + ytop), w, mode);

		offset = 3 * (w-1) + 1;
		for (x = w-1; x >= 0; x--) { // right-to-left

			// get pixel color and transform to grey: (red + green) / 2
			pn = (scanline[offset] + scanline[offset+1]) >> 1;
			offset -= 3;

			gn = ((gn * q) >> S) + pn; // update average

			// average of current pixel and the one above it
			hn = (gn + prev_gn[x]) >> 1;
			prev_gn[x] = gn; // update previous scanline memory

			if (pn < (hn * factor) >> S) {
				TInt top = (y > 0) ? p[y2-w + x] : BACKGROUND;
				TInt right = (x < w-1) ? p[y2 + x+1] : BACKGROUND;
				ProcessForegroundPixelL(y2 + x, top, right);
			} else {
				p[y2 + x] = BACKGROUND;
			}
		}

	}

}



/**************************************************************************/

/**
 * Enter a label for foreground pixel (x,y) in the p-array
 * and record cluster equivalences in equals-array.
 *
 * @param pIndex array index of foreground pixel (y*w+x)
 * @param y y-coordinate of foreground pixel
 * @param top label of pixel above (x,y)
 * @param left label of pixel left from (x,y)
 */
inline void CRecognitionAlgorithm::ProcessForegroundPixelL(TInt pIndex, TInt top, TInt left) 
{
	if (top != BACKGROUND && left != BACKGROUND) {
		p[pIndex] = top;
		if (top != left) { // record equivalence in equality-chain
			// look for left in equals-chain of top
			TInt eq = equals[top];
			while (eq != left && eq != top) {
				eq = equals[eq];
			}
			if (eq == top) { // left not in equality-chain of top, merge both chains
				TInt equalsTop = equals[top];
				TInt equalsLeft = equals[left];
				equals[top] = equalsLeft;
				equals[left] = equalsTop;
			}
		}
	} else {
		if (top != BACKGROUND) { // use top label
			p[pIndex] = top;
		} else if (left != BACKGROUND) { // use left label
			p[pIndex] = left;
		} else { // new label
			currentCluster++;
			if (currentCluster >= equalsSize) { // equals-array too small, reallocate
				TInt equalsSizeNew = 2 * equalsSize;
				TInt* equalsNew = new (ELeave) TInt[equalsSizeNew];
				TInt i;
				for (i = 0; i < equalsSize; i++) {
					equalsNew[i] = equals[i];
				}
				for (; i < equalsSizeNew; i++) {
					equalsNew[i] = 0;
				}
				delete [] equals;
				equals = equalsNew;
				equalsSize = equalsSizeNew;
			}
			equals[currentCluster] = currentCluster; // cluster is equivalent to itself
			p[pIndex] = currentCluster;
		}
	}
}



/**************************************************************************/

/**
 * Step 4: Do the second phase of cluster finding.
 * Uses the splinter information to resolve cluster equivalences.
 * Calculates the sizes, centers, bounding boxes, eccentricity of all clusters.
 */
void CRecognitionAlgorithm::FindClustersL() 
{
	// resolve cluster equivalences and
	// produce continuous (1,2,3,...) cluster labels

	// array of cluster labels, index 0 unused
	TInt* label = (TInt*) User::AllocLC((currentCluster + 1) * sizeof(TInt));

	TInt i;
	for (i = 1; i <= currentCluster; i++) { // index 0 unused
		label[i] = 0;
	}

	TInt currentLabel = 0;
	for (i = 1; i <= currentCluster; i++) { // index 0 unused
		if (label[i] == 0) { // cluster i has no label yet
			currentLabel++;
			// run through equality-chain and set current label
			TInt j = i;
			while (label[j] == 0) {
				label[j] = currentLabel;
				j = equals[j];
			}
		}
	}

	clusterCount = currentLabel; // the number of clusters found

	// now, the label entries are correct and can be used to
	// resolve equivalences in the p-array

	// the center, bounding box, and size are stored in the cluster_... arrays
	TInt s32 = (currentLabel + 1) * sizeof(TInt);
	TInt s64 = (currentLabel + 1) * sizeof(TInt64);
	TInt* cluster_size = (TInt*) User::AllocLC(s32);
	TInt64* cluster_x_sum = (TInt64*) User::AllocLC(s64); // for center calculation
	TInt64* cluster_y_sum = (TInt64*) User::AllocLC(s64);
	TInt* cluster_min_x = (TInt*) User::AllocLC(s32);  // bounding boxes
	TInt* cluster_max_x = (TInt*) User::AllocLC(s32);
	TInt* cluster_min_y = (TInt*) User::AllocLC(s32);
	TInt* cluster_max_y = (TInt*) User::AllocLC(s32);

	// initialize
	for (i = 1; i <= currentLabel; i++) {
		cluster_min_x[i] = w-1;
		cluster_max_x[i] = 0;
		cluster_min_y[i] = h-1;
		cluster_max_y[i] = 0;
		cluster_size[i]  = 0;
		cluster_x_sum[i] = 0;
		cluster_y_sum[i] = 0;
	}

	// run through the whole image again, calculating center, bounding box, size,
	// meanwhile setting all cluster labels in p correctly

	TInt pIndex = 0;
	for (TInt y = 0; y < h; y++) {
		for (TInt x = 0; x < w; x++) {
			TInt i = p[pIndex++]; // p contains unresolved cluster labels
			if (i == BACKGROUND) continue;
			i = label[i]; // use new label as index

			cluster_size[i]++;
			cluster_x_sum[i] += x;
			cluster_y_sum[i] += y;
			if (cluster_min_x[i] > x) cluster_min_x[i] = x;
			if (cluster_max_x[i] < x) cluster_max_x[i] = x;
			if (cluster_min_y[i] > y) cluster_min_y[i] = y;
			if (cluster_max_y[i] < y) cluster_max_y[i] = y;

			// update p with the correct (resolved) label
			p[pIndex - 1] = i;
		}
	}

	// create cluster objects

	if (clusters != NULL) { // cleanup
		delete [] clusters;
		clusters = NULL;
	}
	// index 0 unused, clusterCount entries
	clusters = new (ELeave) TClusterInfo[currentLabel+1]; 

	for (i = 1; i <= currentLabel; i++) {

		// center

		/**
		 *  c_x = 1/size * sum[for all (x,y) in cluster]{ x }
		 *  c_y = 1/size * sum[for all (x,y) in cluster]{ y }
		 */

		TReal tmpR;
		TReal clusterSizeInv = 1.0 / cluster_size[i];
		
		tmpR = cluster_x_sum[i].GetTReal() * clusterSizeInv;
		Math::Round(tmpR, tmpR, 0);
		TInt c_x = (TInt) tmpR;

		tmpR = cluster_y_sum[i].GetTReal() * clusterSizeInv;
		Math::Round(tmpR, tmpR, 0);
		TInt c_y = (TInt) tmpR;

		// calculate second-order moments of cluster i

		/**
		 *  m_x = 1/size * sum[for all (x,y) in cluster]{ (x - c_x)^2 }
		 *  m_y = 1/size * sum[for all (x,y) in cluster]{ (y - c_y)^2 }
		 * m_xy = 1/size * sum[for all (x,y) in cluster]{ (x - c_x)*(y - c_y) }
		 */

		TInt64 mx = 0;
		TInt64 my = 0;
		TInt64 mxy = 0;

		// iterate over bounding box

		TInt minY = cluster_min_y[i];
		TInt maxY = cluster_max_y[i];
		TInt minX = cluster_min_x[i];
		TInt maxX = cluster_max_x[i];
		for (TInt y = minY, y2 = minY * w; y <= maxY; y++, y2 += w) {
			TInt ydiff = y - c_y;
			TInt ydiff2 = ydiff * ydiff;
			for (TInt x = minX; x <= maxX; x++) {
				if (p[y2 + x] == i) { // if point belongs to cluster
					TInt xdiff = x - c_x;
					mx += xdiff * xdiff;
					my += ydiff2;
					mxy += xdiff * ydiff;
				}
			}
		}

		// divide by size

		TReal m_x = mx.GetTReal() * clusterSizeInv;
		TReal m_y = my.GetTReal() * clusterSizeInv;
		TReal m_xy = mxy.GetTReal() * clusterSizeInv;

		// calculate parameters of ellipse

		/**
		 * E = { (x,y) | d*x^2 + 2*e*x*y + f*y^2 <= 1}
		 * with
		 * | d e | = 1 / (4*m_x*m_y - m_xy^2) * |  m_y  -m_xy |
		 * | e f |                              | -m_xy   m_x |
		 */

		TReal factor = (4 * m_x * m_y - m_xy * m_xy);

		if (factor != 0.0) factor = 1.0 / factor;

		TReal d =  m_y  * factor;
		TReal e = -m_xy * factor;
		TReal f =  m_x  * factor;

		// calculate (absolute) eigenvalues

		TReal root;
		Math::Sqrt(root, (d-f)*(d-f) + 4*e*e);

		TReal lambda1 = (d + f + root) * 0.5;
		if (lambda1 < 0) lambda1 = -lambda1;

		TReal lambda2 = (d + f - root) * 0.5;
		if (lambda2 < 0) lambda2 = -lambda2;

		// calculate eccentricity

		TReal ratio;
		if (lambda1 != 0.0 && lambda2 != 0.0) {
			ratio = (lambda1 < lambda2) ? lambda1/lambda2 : lambda2/lambda1;
		} else {
			ratio = 0.0;
		}

		// calculate direction

		TReal alpha; // =  0.5 * Math.atan2(2*e, d-f);
		
		if (e != 0.0 && d != f) {
			Math::ATan(tmpR, 2*e, d-f);
		} else {
			tmpR = 0.0;
		}
		
		alpha = 0.5 * tmpR;

		// store values in cluster object

		clusters[i].SetValues(i, c_x, c_y,
			 cluster_min_x[i], cluster_max_x[i], 
			 cluster_min_y[i], cluster_max_y[i],
			 ratio, alpha, cluster_size[i]);

	} // end for all clusters

	// delete all buffers

	CleanupStack::Pop(7);
	User::Free(cluster_size);
	User::Free(cluster_x_sum);
	User::Free(cluster_y_sum);
	User::Free(cluster_min_x);
	User::Free(cluster_max_x);
	User::Free(cluster_min_y);
	User::Free(cluster_max_y);

	CleanupStack::Pop();
	User::Free(label);
}



/**************************************************************************/

/**
 * Step 5: Do the code recognition (using the clusters found before).
 * @return the set of <code>CodeInfo</code> objects determined
 */
void CRecognitionAlgorithm::DoRecognitionL() 
{
	debugPoints.Reset();
	debugCounter = 0;

	ClearL();

	TInt i, j, orient;
	TReal tmpR;

	// look at the clusters, one at a time
	debugTReal1 = 0.0;
	debugTReal2 = 0.0;

	for (TInt ci = 1; ci <= clusterCount; ci++) {

		TClusterInfo& cl = clusters[ci];

		// SPEEDUP: ignore small clusters
		// tests showed that the smallest recognized major guide bars have size >= 30
		if (cl.size < 20) continue;
		//debugCounter++;

		// make sure the cluster investigated could be a kernel
		if (!IsKernel(cl)) continue; 

		/************************************************************************/

		// look for major guide bar

		TInt cut_x[2]; // cut coordinates
		TInt cut_y[2];
		TInt cut_r[2];

		TClusterInfo* corner[2];
		corner[0] = NULL;
		corner[1] = NULL;

		// follow cluster's major axis
		TReal sinAlpha;
		TReal cosAlpha;
		Math::Sin(sinAlpha, cl.alpha);
		Math::Cos(cosAlpha, cl.alpha);
		TReal xst = -sinAlpha; 
		TReal yst = cosAlpha;

		// do 2 cuts along major axis or major guide bar, binary search

		for (i = 0, orient = 1; i < 2; i++, orient *= -1) {
			TInt r = ((cl.max_x - cl.min_x) + (cl.max_y - cl.min_y)) >> 1;
			TInt low = 0;
			TInt high = r;
			TInt x = KMinTInt;
			TInt y = KMinTInt;
			TInt xold, yold;

			while (ETrue) {
				r = (low + high) >> 1;
				xold = x;
				yold = y;

				x = cl.c_x + (TInt)(orient * r * xst);
				if (x < 0) x = 0;
				else if (x >= w) x = w-1;
				y = cl.c_y + (TInt)(orient * r * yst);
				if (y < 0) y = 0;
				else if (y >= h) y = h-1;

				if (low >= high || xold == x && yold == y) {
					// debugPoints.Append(TPoint(x, y));
					cut_x[i] = x;
					cut_y[i] = y;
					cut_r[i] = r;
					break;
				}

				TBool insideBounds = x >= cl.min_x && x <= cl.max_x &&
									 y >= cl.min_y && y <= cl.max_y;
				TBool onCluster = insideBounds && p[y*w + x] == cl.label;

				if (onCluster) {
					low = r + 1;
				} else {
					high = r;
				}
			}

		}

		// the cut points are on the edge of the assumed major guide bar now
		// move further until another cluster (a corner stone) is reached or r is too large

		for (i = 0, orient = 1; i < 2; i++, orient *= -1) {
			TInt r = cut_r[i]; // Math.sqrt(d2(cl.c_x, cl.c_y, cut_x[i], cut_y[i]));
			TInt max_r = r * 11 / 7; // 11 : 7 == dist(cornerstones) : length(major guide bar)
			TInt x = cut_x[i];
			TInt y = cut_y[i];

			while (r <= max_r &&
				x >= 0 && x < w && y >= 0 && y < h &&
				(p[y * w + x] == BACKGROUND || p[y * w + x] == cl.label))
			{
				// debugPoints.Append(TPoint(x, y));
				x = cl.c_x + (TInt)(orient * r * xst);
				y = cl.c_y + (TInt)(orient * r * yst);
				r++;
			}

			// if no corner stone was found, exit
			if (r > max_r || x < 0 || x >= w || y < 0 || y >= h) break;

			cut_x[i] = x;
			cut_y[i] = y;

			// postcondition: neighboring cluster found (within distance max_r)

			TInt index = p[y * w + x];
			if (index >= 0) { // found corner stone, store its cluster
				corner[i] = &clusters[index];
				// debugPoints.Append(TPoint(corner[i]->c_x, corner[i]->c_y));
			} else {
				break;
			}
		}

		// we need two corner stones next to the major guide bar
		if (corner[0] == NULL || corner[1] == NULL) continue;

		// now, we have two of the corner stones and can calculate
		// ystep and major axis
		// the cut points are located on the closest edges of the cornerstones
		// this corresponds to a distance of 9 units in the code coordinate system
		TReal ystep = D2(cut_x[0], cut_y[0], cut_x[1], cut_y[1]);
		Math::Sqrt(ystep, ystep);
		ystep /= 9.0;
		TReal major_x = -sinAlpha * ystep;
		TReal major_y = cosAlpha * ystep;

		/************************************************************************/

		// one of the corner stones just located should be the minor
		// guide bar: find  minor guide bar and its axis (which ideally
		// is oriented orthogonally to major guide bar axis)

		TReal oldminor_x = cosAlpha * ystep;
		TReal oldminor_y = sinAlpha * ystep;

		if (corner[0]->size < corner[1]->size) { // axis must be switched
			major_x = -major_x;
			major_y = -major_y;
			oldminor_x = -oldminor_x;
			oldminor_y = -oldminor_y;

			// exchange corners, too
			TClusterInfo* temp_cl = corner[0];
			corner[0] = corner[1];
			corner[1] = temp_cl;
		}

		// check whether corner[0] is a plausible minor guide bar and
		// whether corner[1] is a plausible upper right corner stone

		if (!IsLongCorner(*corner[0], cl) || !IsCorner(*corner[1], cl)) continue;

		// check whether the major guide bar (cl) is larger
		// than the minor guide bar (corner[0]), drop kernel otherwise
		// @todo: check if this is ok for tilted codes

//		if (cl.size <= corner[0]->size) continue;

		// postcondition: major axis is orientated correctly
		// now, calculate minor axis and xstep

		// corner[0] is lower corner (i.e. it is the minor guide bar),
		// corner[1] is the upper corner

		// debugPoints.Append(TPoint(corner[0]->c_x, corner[0]->c_y));
		// debugPoints.Append(TPoint(corner[1]->c_x, corner[1]->c_y));

		/************************************************************************/

		// now, cut the major axis of the lower corner with its boundary

		// follow major axis of the minor guide bar
		Math::Sin(sinAlpha, corner[0]->alpha);
		Math::Cos(cosAlpha, corner[0]->alpha);
		xst = -sinAlpha; 
		yst = cosAlpha;

		// do 2 cuts along major axis of minor guide bar, linear search

		TInt c_x = corner[0]->c_x;
		TInt c_y = corner[0]->c_y;
		TInt min_x = corner[0]->min_x;
		TInt max_x = corner[0]->max_x;
		TInt min_y = corner[0]->min_y;
		TInt max_y = corner[0]->max_y;
		TInt label = corner[0]->label;

		for (i = 0, orient = 1; i < 2; i++, orient *= -1) {
			TInt r = 1;

			tmpR = orient * r * xst;
			Math::Round(tmpR, tmpR, 0);
			TInt x = c_x + (TInt) tmpR;

			tmpR = orient * r * yst;
			Math::Round(tmpR, tmpR, 0);
			TInt y = c_y + (TInt) tmpR;

			while (x >= min_x && x <= max_x && y >= min_y && y <= max_y
				&& p[y * w + x] == label)
			{
				// debugPoints.Append(TPoint(x, y));

				r++;

				tmpR = orient * r * xst;
				Math::Round(tmpR, tmpR, 0);
				x = c_x + (TInt) tmpR;

				tmpR = orient * r * yst;
				Math::Round(tmpR, tmpR, 0);
				y = c_y + (TInt) tmpR;
			}

			// calculate last point on cluster
			r--;

			tmpR = orient * r * xst;
			Math::Round(tmpR, tmpR, 0);
			cut_x[i] = c_x + (TInt) tmpR;

			tmpR = orient * r * yst;
			Math::Round(tmpR, tmpR, 0);
			cut_y[i] = c_y + (TInt) tmpR;

			// debugPoints.Append(TPoint(x, y));
		}

		// length of minor guide bar: 5 units in code coordinate system
		TReal xstep = D2(cut_x[0], cut_y[0], cut_x[1], cut_y[1]);
		Math::Sqrt(xstep, xstep);
		xstep /= 5.0;

		// oldminor is the direction of the minor axis of the larger guide bar
		// (xst,yst) is the direction of major axis of the smaller guide bar
		TReal scalarprod = oldminor_x * xst + oldminor_y * yst;
		if (scalarprod < 0) {
			xst = -xst; // flip axis
			yst = -yst;
		}

		TReal minor_x = xst * xstep;
		TReal minor_y = yst * xstep;

		// stretch: used to correct the "erosion" effect

		minor_x *= stretch;
		minor_y *= stretch;

		/************************************************************************/

		// now, find the cornerstone positions

		// coordinates of the cornerstones
		// (upper left, upper right, minor guide bar, lower left)
		TInt x1, y1, x2, y2, x3, y3, x4, y4;

		tmpR = corner[1]->c_x - 10 * minor_x;
		Math::Round(tmpR, tmpR, 0);
		x1 = (TInt) tmpR;

		tmpR = corner[1]->c_y - 10 * minor_y;
		Math::Round(tmpR, tmpR, 0);
		y1 = (TInt) tmpR;
		// debugPoints.Append(TPoint(x1, y1));

		x2 = corner[1]->c_x;
		y2 = corner[1]->c_y;
		// debugPoints.Append(TPoint(x2, y2));

		x3 = corner[0]->c_x;
		y3 = corner[0]->c_y;
		// debugPoints.Append(TPoint(x3, y3));

		tmpR = corner[0]->c_x - 8 * minor_x;
		Math::Round(tmpR, tmpR, 0);
		x4 = (TInt) tmpR;

		tmpR = corner[0]->c_y - 8 * minor_y;
		Math::Round(tmpR, tmpR, 0);
		y4 = (TInt) tmpR;
		// debugPoints.Append(TPoint(x4, y4));

		// add a check here that investigates the shape of the corners: e.g. check for the distortion
		// (axis ratio of corner square) and exit if too large. -> reduces "false positives"
		// note: this could be done also after the correction of the cornerpoints

		// decision: do these two checks:
		// - length ratio of left and upper lines of visual code square
		// - angle of top left corner

		// ratio

		TReal ratio2 = (x1 != x4 || y1 != y4) ? 
			D2(x1, y1, x2, y2) / D2(x1, y1, x4, y4) : 0.0; // avoid division by zero
		if (ratio2 < minratio * minratio || ratio2 * minratio * minratio > 1.0) {
			continue;
		}
//		debugCounter++;

		// angle: a*b = |a|*|b|*cos(angle)

		TReal edge1_x = x2 - x1;
		TReal edge1_y = y2 - y1;
		TReal size = edge1_x * edge1_x + edge1_y * edge1_y;
		Math::Sqrt(size, size);
		if (size == 0.0) size = 1.0; // avoid division by zero
		edge1_x /= size;
		edge1_y /= size;

		TReal edge2_x = x4 - x1;
		TReal edge2_y = y4 - y1;
		size = edge2_x * edge2_x + edge2_y * edge2_y;
		Math::Sqrt(size, size);
		if (size == 0.0) size = 1.0; // avoid division by zero
		edge2_x /= size;
		edge2_y /= size;

		scalarprod = edge1_x * edge2_x + edge1_y * edge2_y;
		TReal angle;
		Math::ACos(angle, scalarprod);

		if (angle < minangle || angle > maxangle) continue;
//		debugCounter++;

		// check whether the cornerstones are present

		TClusterInfo* co2 = corner[1]; // right top
		TClusterInfo* co3 = corner[0]; // right bottom (minor guide bar)

		// check whether co2 and co3 were found (drop kernel otherwise)

		if (co2 == NULL || co3 == NULL) continue;
//		debugCounter++;

		// look for left corners in some area (pass kernel cluster as well)

		/*
		// size ratio of (upper right corner stone) to (major guide bar / 7)
		TReal r1 = (TReal) co2->size / (TReal) cl.size * 7.0;
		Math::Sqrt(r1, r1);
		r1 = (r1 - 1.0) * 0.5 + 1.0;
		if (r1 < 0.80) r1 = 0.80;
		else if (r1 > 1.25) r1 = 1.25;

		// size ratio of (minor guide bar / 5) to (major guide bar / 7)
		TReal r2 = (co3->size * 7.0) / (cl.size * 5.0);
		Math::Sqrt(r2, r2);
		r2 = (r2 - 1.0) * 1.5 + 1.0;
		if (r2 < 0.80) r2 = 0.80;
		else if (r2 > 1.25) r2 = 1.25;

		TReal r = r2 * r1;
		if (r < 0.80) r = 0.80;
		else if (r > 1.25) r = 1.25;

		tmpR = co3->c_x - 8 * minor_x * r;
		Math::Round(tmpR, tmpR, 0);
		x4 = (TInt) tmpR;

		tmpR = co3->c_y - 8 * minor_y * r;
		Math::Round(tmpR, tmpR, 0);
		y4 = (TInt) tmpR;
		*/

		// debugPoints.Append(TPoint(x4, y4));

		tmpR = units * ystep;
		Math::Round(tmpR, tmpR, 0);
		TInt MAX_S = (TInt) tmpR;

		TReal max = xstep > ystep ? xstep : ystep;
		TReal maxcorr = maxcorrect * maxcorrect * max * max;

		TClusterInfo* co4 = SearchCorner(x4, y4, cl, MAX_S, maxcorr); // left bottom
		
		// check if the corner was found (drop kernel otherwise)
		if (co4 == NULL) continue;
//		debugCounter++;

		// size ratio of (lower left corner stone) to (major guide bar / 7)
		TReal r = (TReal) co4->size / (TReal) cl.size * 7.0;
		Math::Sqrt(r, r);
		r = (r - 1.0) * 0.5 + 1.0;
		if (r < 0.80) r = 0.80;	else if (r > 1.25) r = 1.25;
//		r = 1.0;
//		debugTReal2 = r;

		tmpR = co4->c_x - 10 * major_x * r;
		Math::Round(tmpR, tmpR, 0);
		x1 = (TInt) tmpR;

		tmpR = co4->c_y - 10 * major_y * r;
		Math::Round(tmpR, tmpR, 0);
		y1 = (TInt) tmpR;

		// debugPoints.Append(TPoint(x1, y1));

		TClusterInfo* co1 = SearchCorner(x1, y1, cl, MAX_S, maxcorr); // left top

		// check if the corner was found (drop kernel otherwise)
		if (co1 == NULL) continue;
//		debugCounter++;

		// set x1..y4 to the centers of the corners,

		x1 = co1->c_x;
		y1 = co1->c_y;
		x2 = co2->c_x;
		y2 = co2->c_y;
		x3 = co3->c_x;
		y3 = co3->c_y;
		x4 = co4->c_x;
		y4 = co4->c_y;

		/************************************************************************/

		// read the code

		// use a TWarper object to "undistort" the image
		// use the following points for the warping
		// (0,0) (10,0) (8,10) (0,10)

		TWarper* warper = new (ELeave) TWarper(x1, y1, x2, y2, x3, y3, x4, y4, EFalse);
		CleanupStack::PushL(warper);

		CBigInteger* code = new (ELeave) CBigInteger();
		CleanupStack::PushL(code);
		code->ConstructL(83);
		TInt bitIndex = code->GetBitCount();

		TPoint bit; // used to get the warped point

		// bottom

		for (j = 10; j > 8; j--) {
			for (i = 4; i > 1; i--) {
				bitIndex--;
				bit = warper->Warp(i,j);
				TInt x = bit.iX;
				TInt y = bit.iY;
				if (x >= 0 && y >= 0 && x < w && y < h) {
					if (p[y*w + x] != BACKGROUND) {
						code->SetBit(bitIndex);
					}
				}
			}
		}

		// middle

		for (j = 8; j > 1; j--) {
			for (i = 8; i >= 0; i--) {
				bitIndex--;
				bit = warper->Warp(i,j);
				TInt x = bit.iX;
				TInt y = bit.iY;
				if (x >= 0 && y >= 0 && x < w && y < h) {
					if (p[y*w + x] != BACKGROUND) {
						code->SetBit(bitIndex);
					}
				}
			}
		}

		// top

		for (j = 1; j >= 0; j--) {
			for (i = 8; i > 1; i--) {
				bitIndex--;
				bit = warper->Warp(i,j);
				TInt x = bit.iX;
				TInt y = bit.iY;
				if (x >= 0 && y >= 0 && x < w && y < h) {
					if (p[y*w + x] != BACKGROUND) {
						code->SetBit(bitIndex);
					}
				}
			}
		}

		// add the code found to the container
//		debugCounter++;
//		if (debugCounter == 0 || cl.size < debugCounter) debugCounter = cl.size;

		CCodeInfo* ci = new (ELeave) CCodeInfo(
			x1, y1, x2, y2, x3, y3, x4, y4, 
			cl.c_x, cl.c_y, // cl is the major guide bar of the code
			w, h, 
			code, // just debugging information on this line :)
			warper);

		codes->Append(ci);

//		if (co3->ratio < debugTReal1) debugTReal1 = co3->ratio;
//		if (co3->ratio > debugTReal2) debugTReal2 = co3->ratio;
//		debugTReal2 = (TReal)cl.size / (TReal)co3->size;

//		if (cl.ratio < debugTReal1) debugTReal1 = cl.ratio;
//		if (cl.ratio > debugTReal2) debugTReal2 = cl.ratio;

		CleanupStack::Pop(2); // warper, code

	} // end for alls clusters

}



/**************************************************************************/

/**
 * Search for cluster in a spiral, starting in (x,y), until one is found
 * or the distance from (x,y) is too large.
 *
 * @param clusters the vector of clusters found
 * @param x x-coordinate of starting point
 * @param y y-coordinate of starting point
 * @param kernel the assumed kernel (major guide bar) of this visual code
 * @param max_s maximum distance from (x,y)
 * @param maxcorr squared maximum correction distance
 */
TClusterInfo* CRecognitionAlgorithm::SearchCorner(TInt x, TInt y, const TClusterInfo& kernel, TInt max_s, TReal maxcorr) const
{
	TClusterInfo* cl = NULL;
	max_s *= 2; // *2 because s/2 is the distance to the origin
	TInt x0 = x;
	TInt y0 = y;

	for (TInt s = 0; s <= max_s; s++) {
		TInt m = s % 4; // direction

		for (TInt iter = 0; iter < s; iter++) {

			if (x >= 0 && y >= 0 && x < w && y < h && p[y*w + x] > 0) {
				// debugPoints.Append(TPoint(x, y)); // note: remove "const" after method declaration when using this call
				cl = &clusters[p[y*w + x]];
				if (IsCorner(*cl, kernel)) {
					// test if the correction distance is not too large
					if (D2(x0, y0, cl->c_x, cl->c_y) <= maxcorr) {
						return cl;
					}
				}
			}

			if (m == 0) {
				y--;
			} else if (m == 1) {
				x++;
			} else if (m == 2) {
				y++;
			} else if (m == 3) {
				x--;
			}

		}
	}

	return NULL;
}



/**************************************************************************/

// SMALL HELPER FUNCTIONS:

/** Check for the third (bottom right) corner (minor guide bar). */
TBool CRecognitionAlgorithm::IsLongCorner(const TClusterInfo& cl, const TClusterInfo& kernel) const
{
	// special case for very small kernels
	if (kernel.size < 150) return ETrue;

	// check size
	TInt expectedSize = kernel.size * 5 / 7;
	if (2 * cl.size < expectedSize || cl.size > 2 * expectedSize) {
		return EFalse;
	}

	// check shape (ratio: 1.0 for perfect squares and circles, 0.0 for ideal lines)
	if (cl.ratio < minorbar_min || cl.ratio > minorbar_max) return EFalse;

	return ETrue;
}



/** Check whether cl is a potential corner stone. */
TBool CRecognitionAlgorithm::IsCorner(const TClusterInfo& cl, const TClusterInfo& kernel) const
{
	// special case for very small kernels
	if (kernel.size < 150) return ETrue;

	// check size
	TInt expectedSize = kernel.size / 7;
	if (2 * cl.size < expectedSize || cl.size > 2 * expectedSize) {
		return EFalse;
	}

	// check shape (ratio: 1.0 for perfect squares and circles, 0.0 for ideal lines)
	if (cl.ratio < corner_min || cl.ratio > corner_max) return EFalse;

	// another possibility:
	// check if there is enough BACKGROUND around the cluster

	return ETrue;
}



/** Check whether cl is a potential kernel (major guide bar). */
TBool CRecognitionAlgorithm::IsKernel(const TClusterInfo& cl) const
{
	// check shape (ratio: 1.0 for perfect squares and circles, 0.0 for ideal lines)
	return cl.ratio >= majorbar_min && cl.ratio <= majorbar_max;
}



void CRecognitionAlgorithm::ClearL()
{
	if (codes == NULL) {
		codes = new (ELeave) RPointerArray<CCodeInfo>;
	} else {
		codes->ResetAndDestroy();
	}
}
