/*
 * $Id: PictureTaker.cpp,v 1.1.1.1 2004/08/06 10:53:19 mraento Exp $
 *
 * Visual Codes for Symbian OS
 * Copyright (C) 2004 Beat Gfeller, Michael Rohs (rohs@inf.ethz.ch)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

// INCLUDES

#include <e32math.h>

#include "PictureTaker.h"
#include "VisualCodeSystem.h"

// CONSTANTS

// FORWARD DECLARATIONS

// CLASS DEFINITION



CPictureTaker::CPictureTaker(CVisualCodeSystem& aObserver)
: iObserver(aObserver)
{
	iCamera = NULL;
	iCameraState = EReleased;

	iBitmap = NULL;

	iFrameCounter = 0;
	iTranslation.SetXY(0,0);
	iRotation = 0;
	iMinDiffTranslation = 0;
	iMinDiffRotation = 0;

	iStopRequest = FALSE;

	debugTInt = 0;
}



void CPictureTaker::ConstructL() 
{
	if (iCamera != NULL) return;

	iCamera = CCamera::NewL(*this, 0);

	scanline = new (ELeave) TUint8[3*160];
	blocks1 = new  (ELeave) TInt[7][10];
	blocks2 = new  (ELeave) TInt[7][10];
	currentBlock = 0;
}



CPictureTaker::~CPictureTaker() 
{
	delete iCamera;
	delete iBitmap;
	delete scanline;
	delete [] blocks1;
	delete [] blocks2;
}



void CPictureTaker::StartL(
	TBool aSendUpdates, 
	TBool aSendTranslation,
	TBool aSendRotation
) 
{
	if (iCameraState != EReleased || iCamera == NULL) return;
	iCameraState = EStarted;

	iStopRequest = FALSE;

	iSendUpdates = aSendUpdates;
	iSendTranslation = aSendTranslation;
	iSendRotation = aSendRotation;

	iFrameCounter = 0;

	iTranslation.SetXY(0,0);
	iRotation = 0;
	iMinDiffTranslation = 0;
	iMinDiffRotation = 0;

	currentBlock = 0;

	// debugTBuf.Copy(_L(" 0 "));

	/*
	TCameraInfo ci;
	iCamera->CameraInfo(ci);
	debugTBuf.Append(ci.iHardwareVersion.Name());
	*/

	iCamera->Reserve();
}



void CPictureTaker::StopL() 
{
	if (iCameraState == EReleased || iCamera == NULL) return;

	if (iCameraState == EPowerOnComplete) {
		iCameraState = EStopping;

		if (iCamera->ViewFinderActive()) {
			iCamera->StopViewFinder();
		}

		// capture final high-quality (640 x 480) bitmap
		CCamera::TFormat format = CCamera::EFormatFbsBitmapColor16M;
		iCamera->PrepareImageCaptureL(format, 0);
		iCamera->CaptureImage();
	} else {
		iStopRequest = TRUE;
	}

}



void CPictureTaker::ReserveComplete(TInt aError)
{
	// debugTBuf.Append(_L(" 1 "));

	if (aError == KErrNone) {
		iCameraState = EReserveComplete;
		iCamera->PowerOn();
	} else {
		iCameraState = EReleased;
	}
}



void CPictureTaker::PowerOnComplete(TInt aError)
{
	// debugTBuf.Append(_L(" 2 "));

	if (aError == KErrNone) {
		iCameraState = EPowerOnComplete;
		/*
		TCameraInfo ci;
		iCamera->CameraInfo(ci);
		CCamera::TFormat format = CCamera::EFormatFbsBitmapColor16M;

		for (TInt i = 0; i < ci.iNumImageSizesSupported; i++) {
			iCamera->EnumerateCaptureSizes(size, i, format);
			debugTBuf.AppendNum(size.iWidth);
			debugTBuf.Append(_L(","));
			debugTBuf.AppendNum(size.iHeight);
			debugTBuf.Append(_L(" "));
		}

		// iCamera->PrepareImageCaptureL(format, 0);
		// iCamera->CaptureImage();
		*/

		// capture low-quality (160 x 112) bitmaps
		/*
		iCamera->PrepareImageCaptureL(format, 1);
		*/

		if (iStopRequest) {
			if (iCamera->ViewFinderActive()) {
				iCamera->StopViewFinder();
			}
			// capture final high-quality (640 x 480) bitmap
			CCamera::TFormat format = CCamera::EFormatFbsBitmapColor16M;
			iCamera->PrepareImageCaptureL(format, 0);
			iCamera->CaptureImage();

			iStopRequest = FALSE;
		} else if (iSendUpdates) {
			CCamera::TFormat format = CCamera::EFormatFbsBitmapColor16M;
			TSize size;
			iCamera->EnumerateCaptureSizes(size, 1, format);
			iCamera->StartViewFinderBitmapsL(size);
		}
	} else {
		iCamera->Release();
		iCameraState = EReleased;
	}
}



void CPictureTaker::ViewFinderFrameReady(CFbsBitmap& aFrame)
{
	if (iSendTranslation || iSendRotation) {
		iFrameCounter++;
		if ((iFrameCounter % 2) == 0) {
			GetImageMovement(aFrame);
		}
	} else {
		iObserver.PictureUpdateL(aFrame, TPoint(0,0), 0, 0, 0);
	}
}



void CPictureTaker::ImageReady(CFbsBitmap* aBitmap, HBufC8* /*aData*/, TInt aError)
{
	// debugTBuf.Append(_L(" 3"));

	iCamera->PowerOff(); // @todo: check if necessary
	iCamera->Release();
	iCameraState = EReleased;

	if (aError == KErrNone) {
		delete iBitmap;
		iBitmap = aBitmap;
		iObserver.PictureTakenL(aBitmap);
	} 
}



void CPictureTaker::FrameBufferReady(MFrameBuffer* /*aFrameBuffer*/, TInt /*aError*/)
{
	// no implementation required
}



void CPictureTaker::SetSendTranslation(TBool aSendTranslation)
{
	iSendTranslation = aSendTranslation;
	if (!iSendTranslation) {
		iTranslation.SetXY(0,0);
		iMinDiffTranslation = 0;
		if (!iSendRotation) {
			iFrameCounter = 0;
			currentBlock = 0;
		}
	}
}



void CPictureTaker::SetSendRotation(TBool aSendRotation)
{
	iSendRotation = aSendRotation;
	if (!iSendRotation) {
		iRotation = 0;
		iMinDiffRotation = 0;
		if (!iSendTranslation) {
			iFrameCounter = 0;
			currentBlock = 0;
		}
	}
}



TBool CPictureTaker::GetImageMovement(CFbsBitmap& aBitmap) 
{
	// assert: iBmp.size = 160 x 120

	TDisplayMode mode = EColor16M; //iBmp.DisplayMode();

	TInt w = 160; // 160/16 = 10 blocks //@todo: make dynamic
	TInt h = 112; // 112/16 = 7 blocks
	TPtr8 scanlineTPtr8 = TPtr8(scanline, 3*160);

	// TInt blockSize = 16;
	TInt bw = 10;
	TInt bh = 7;

	TInt (*b1)[10];
	TInt (*b2)[10];

	if ((currentBlock % 2) == 0) {
		b1 = blocks1;
		b2 = blocks2;
	} else {
		b1 = blocks2;
		b2 = blocks1;
	}
	currentBlock++;

	// 7*10 = 70

	for (TInt j = 0; j < bh; j++) {
		for (TInt i = 0; i < bw; i++) {
			b1[j][i] = 0;
		}
	}
	// memset(block, 0, ...)?

	// compute gray values of blocks
#if 1

	// (160/4) * (112/4) = 1120

	for (TInt y = 0; y < h; y += 4) {
		aBitmap.GetScanLine(scanlineTPtr8, TPoint(0, y+4), w, mode);
		TInt by = y >> 4;
		for (TInt px = 1, x = 0; px < w; px += 4) {
			x++; // skip blue color component: TInt blue = (*scanline)[x++];
			TInt green = scanline[x++] & 0xff;
			TInt red = scanline[x++] & 0xff;
			x += 9; // skip three pixels (nine bytes)
			TInt gray = (red + green) >> 1;
			b1[by][px>>4] += gray;
		}
	}
#else
	for (TInt y = 0; y < h; y += 2) {
		aBitmap.GetScanLine(scanlineTPtr8, TPoint(0, y+4), w, mode);
		TInt by = y >> 4;
		for (TInt px = 1, x = 0; px < w; px += 2) {
			x += 4; // skip pixel and skip blue color component: TInt blue = (*scanline)[x++];
			TInt green = scanline[x++] & 0xff;
			TInt red = scanline[x++] & 0xff;
			TInt gray = (red + green) >> 1;
			b1[by][px>>4] += gray;
		}
	}
#endif

	if (currentBlock == 1) return FALSE; // other block not valid yet
	
	// compute differences, find minimal difference

	// translational differences

	if (iSendTranslation) {
		TInt minDiff = -1;
		TInt minDiffX = 0;
		TInt minDiffY = 0;

		// 5 * 5 * 7 * 10 = 700

		for (TInt oy = -2; oy <= 2; oy++) {
			for (TInt ox = -2; ox <= 2; ox++) {
				TInt count = 0;
				TInt diff = 0;
				for (TInt y1 = 0; y1 < bh; y1++) {
					for (TInt x1 = 0; x1 < bw; x1++) {
						TInt x2 = x1 + ox;
						TInt y2 = y1 + oy;
						if (x2 >= 0 && x2 < bw && y2 >= 0 && y2 < bh) { 
							diff += Abs(b1[y1][x1] - b2[y2][x2]);
							count++;
						}
					}
				}
				diff /= count;
				if (minDiff < 0 || diff < minDiff) {
					minDiff = diff;
					minDiffX = ox;
					minDiffY = oy;
				}
			}
		}

		iTranslation.SetXY(minDiffX, minDiffY);
		iMinDiffTranslation = minDiff;

	} else {
		iTranslation.SetXY(0,0);
		iMinDiffTranslation = 0;
	}

	// rotational differences

	if (iSendRotation) {
		TInt minDiffRotation = -1;
		TInt minDiffAngle = 0;

		// -24, -18, -12, -6, 0, 6, 12, 18, 24
		// -32, -24, -16, -8, 0, 8, 16, 24, 32
		// 9 * 7 * 10 = 630

		for (TInt alpha = -24; alpha <= 24; alpha += 6) {
			TInt count = 0;
			TInt diff = 0;
			
			TReal alphaRad = alpha * KDegToRad;
			TReal sinAlpha;
			TReal cosAlpha;
			Math::Sin(sinAlpha, alphaRad);
			Math::Cos(cosAlpha, alphaRad);
			TInt bw2 = bw >> 1;
			TInt bh2 = bh >> 1;
			
			for (TInt y1 = 0; y1 < bh; y1++) {
				for (TInt x1 = 0; x1 < bw; x1++) {
					TInt dx = x1 - bw2;
					TInt dy = y1 - bh2;
					TReal xr = (dx * cosAlpha - dy * sinAlpha);
					TReal yr = (dy * cosAlpha + dx * sinAlpha);
					Math::Round(xr, xr, 0);
					Math::Round(yr, yr, 0);
					TInt x2 = bw2 + (TInt)xr;
					TInt y2 = bh2 + (TInt)yr;

					if (x2 >= 0 && x2 < bw && y2 >= 0 && y2 < bh) { 
						diff += Abs(b1[y1][x1] - b2[y2][x2]);
						count++;
					}
				}
			}
			diff /= count;
			if (minDiffRotation < 0 || diff < minDiffRotation) {
				minDiffRotation = diff;
				minDiffAngle = alpha;
			}

		}

		iRotation = minDiffAngle;
		iMinDiffRotation = minDiffRotation;

	} else {
		iRotation = 0;
		iMinDiffRotation = 0;
	}

	iObserver.PictureUpdateL(aBitmap, iTranslation, iMinDiffTranslation, iRotation, iMinDiffRotation);

	return TRUE;
}
