/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/




// INCLUDE FILES
#include "Context_logContainer.h"

#include <eiklabel.h>  // for example label control
#include <eikenv.h>

#include "cl_settings.h"
#include "basestack.h"
#include <symbian_auto_ptr.h>
#include <aknmessagequerydialog.h> 
#include <context_log.rsg>

enum KEYCODES {
	JOY_LEFT = 0xF807,
	JOY_RIGHT = 0xF808,
	JOY_UP = 0xF809,
	JOY_DOWN = 0xF80A,
	JOY_CLICK = 0xF845,
	KEY_CALL = 0xF862,
	KEY_CANCEL = 0xF863
};

// ================= MEMBER FUNCTIONS =======================

void CContext_logContainer::register_source(const TDesC& name, const TDesC& initial_value, const TTime& time)
{
	CALLSTACKITEM(_L("CContext_logContainer::register_source"));

	if (! name.Compare(KLog_cellid) ) {
		register_source(_L("cell name"), _L(""), time);
	}

	int this_label_index=label_store->Count()+1;

	auto_ptr<CEikLabel> label(new (ELeave) CEikLabel);
	label->SetContainerWindowL( *this );
	label->SetFont(CEikonEnv::Static()->DenseFont());
	label->SetTextL( name );
	log_labels->AppendL(name);
	label->SetExtent( TPoint(10, this_label_index*12), TSize(50, 12));
	label_store->AppendL(label.get());
	label.release();
	
	label.reset(new (ELeave) CEikLabel);
	label->SetContainerWindowL( *this );
	label->SetFont(CEikonEnv::Static()->DenseFont());
	label->SetTextL( initial_value );
	label->SetExtent( TPoint(65, this_label_index*12), TSize(100, 12));
	value_store->AppendL(label.get());
	label.release();

	if (iCursorPos==-1) iCursorPos=0;

	if (initial_value.Length()) {
		new_value(Mlogger::VALUE, name, initial_value, time);
	}
	DrawNow();
}

TKeyResponse CContext_logContainer::OfferKeyEventL(const TKeyEvent &aKeyEvent, TEventCode aType)
{
	if (iCursorPos==-1) return EKeyWasNotConsumed;

	if (aKeyEvent.iCode==JOY_CLICK) {
		CEikLabel* label=value_store->At(iCursorPos);
		const TDesC *text=label->Text();
		HBufC* buf=HBufC::NewLC(text->Length());
		*buf=*text;
		CAknMessageQueryDialog *note=CAknMessageQueryDialog::NewL(*buf);
		note->ExecuteLD(R_LOGVIEW_EVENT_DIALOG);
		CleanupStack::PopAndDestroy();

	} else if (aKeyEvent.iCode==JOY_UP) {
		if (iCursorPos>0) --iCursorPos;
		DrawNow();
	} else if (aKeyEvent.iCode==JOY_DOWN ) {
		if (iCursorPos < value_store->Count()-1 ) ++iCursorPos;
		DrawNow();
	} else {
		return EKeyWasNotConsumed;
	}
	return EKeyWasConsumed;

}

void delete_bufc(void* data)
{
	HBufC* p;
	p=(HBufC*)data;
	delete p;
}

CContext_logContainer::CContext_logContainer(MApp_context& Context, CCellMap& CellMap) : MContextBase(Context),
	iCellMap(CellMap), iCursorPos(-1)
{

}

void CContext_logContainer::new_value(log_priority priority, const TDesC& name, const TDesC& value, const TTime& time)
{
	CALLSTACKITEM(_L("CContext_logContainer::new_value"));


	CEikLabel* label=find_value_label(name);
	if (!label) return;
		
	if (! name.Compare(KLog_cellid) && priority==Mlogger::VALUE) {
		TInt id=iCellMap.GetId(value);
		HBufC* user_given_name=0;
		user_given_name=(HBufC*)cellid_names->GetData(id);
		CEikLabel* name_label=find_value_label(_L("cell name"));
		if (user_given_name) {
			current_name=*user_given_name;
			if (name_label) {
				name_label->SetTextL(*user_given_name);
				if (id==iCurrentLoc->CurrentBaseId()) 
					name_label->SetUnderlining(ETrue);
				else
					name_label->SetUnderlining(EFalse);
			}
		} else {
			TBuf<200> tmp;
			if (current_name.Length()) {
				tmp.Format(_L("last: %S"), &current_name);
			}
			if (name_label) name_label->SetTextL(tmp);
			name_label->SetUnderlining(EFalse);
		}
	}
	if (! name.Compare(_L("base")) && priority==Mlogger::VALUE) {
#ifdef __WINS__
		TBuf<20> t;
		_LIT(KFormatTxt,"%04d%02d%02dT%02d%02d%02d ");
		TDateTime dt;
		dt=time.DateTime();
		t.Format(KFormatTxt, dt.Year(), (TInt)dt.Month()+1, (TInt)dt.Day()+1,
			dt.Hour(), dt.Minute(), dt.Second());
		RDebug::Print(t); RDebug::Print(value);
#endif
		if (value.Length()==0 && !(label->Text()->Left(5).Compare(_L("last:")))) return;
	}
	label->SetTextL(value);
	DrawNow();
}

void CContext_logContainer::unregister_source(const TDesC& /*name*/, const TTime& /*time*/)
{
	CALLSTACKITEM(_L("CContext_logContainer::unregister_source"));

	//TODO
}

CEikLabel*	CContext_logContainer::find_value_label(const TDesC& name)
{
	TInt pos;
	TInt ret=log_labels->Find(name, pos);
	if (ret==0) 
		return value_store->At(pos);
	return 0;
}

void CContext_logContainer::add_cellid_name(const TDesC& cellid, const TDesC& name)
{
	CALLSTACKITEM(_L("CContext_logContainer::add_cellid_name"));

	HBufC* n=HBufC::NewL(name.Length());
	CleanupStack::PushL(n);
	*n=name;
	TInt id=iCellMap.GetId(cellid);
	cellid_names->AddDataL(id, (void*)n, true); 
	CleanupStack::Pop(); // n

	CEikLabel* label=find_value_label(_L("cell name"));
	CEikLabel* id_label=find_value_label(KLog_cellid);
	if (label && id_label && (! id_label->Text()->Compare(cellid))) {
		label->SetTextL(name);
		DrawNow();
	}
	TInt id_for_cell=iCellMap.GetId(cellid);
	label=find_value_label(_L("base"));
	if (id_for_cell==iCurrentLoc->CurrentBaseId() && label) {
		label->SetTextL(name);
		DrawNow();
	}
	TTime now; now.HomeTime();

	if (id_for_cell==iCurrentLoc->CurrentBaseId()) {
		TTime now; GetTime(now);
		iCurrentLoc->now_at_location(cellid, id_for_cell, true, false, now);
	}
}

bool CContext_logContainer::is_named(const TInt id)
{
	CALLSTACKITEM(_L("CContext_logContainer::is_named"));

	if (cellid_names->GetData(id)) return true;
	return false;
}


CCircularLog*	CContext_logContainer::BaseLog()
{
	return iCurrentLoc->BaseLog();
}

void CContext_logContainer::ConstructL(const TRect& aRect)
{
	CALLSTACKITEM(_L("CContext_logContainer::ConstructL"));

	cellid_names=CGenericIntMap::NewL();
	cellid_names->SetDeletor(delete_bufc);

	iCurrentLoc=CCurrentLoc::NewL(AppContext(), cellid_names);

	CreateWindowL();
	
	label_store=new (ELeave) CArrayFixFlat<CEikLabel*>(1);
	value_store=new (ELeave) CArrayFixFlat<CEikLabel*>(1);

	log_labels=new (ELeave) CDesCArrayFlat(8);
	
	status_label=new (ELeave) CEikLabel;
	status_label->SetContainerWindowL(*this);
	status_label->SetFont(CEikonEnv::Static()->DenseFont());
	status_label->SetTextL( _L("starting") );
	status_label->SetExtent(TPoint(10, 108), status_label->MinimumSize());
	
	err_label=new (ELeave) CEikLabel;
	err_label->SetContainerWindowL(*this);
	err_label->SetFont(CEikonEnv::Static()->DenseFont());
	err_label->SetTextL( _L("    ") );
	err_label->SetExtent(TPoint(10, 120), err_label->MinimumSize());

	SetRect(aRect);
	ActivateL();
}

// Destructor
CContext_logContainer::~CContext_logContainer()
{
	CALLSTACKITEM(_L("CContext_logContainer::~CContext_logContainer"));

	delete iLabel;
	delete status_label;
	delete err_label;
	if (label_store) {
		TInt count=label_store->Count();
		for (int i=count-1; i>=0; i--)  {
			delete (*label_store)[i];
		}
		delete label_store;
	}
	if (value_store) {
		TInt count=value_store->Count();
		for (int i=count-1; i>=0; i--)  {
			delete (*value_store)[i];
		}
		delete value_store;
	}
	delete log_labels;
	delete cellid_names;
	delete iCurrentLoc;
}

void CContext_logContainer::set_status(const TDesC& status)
{
	CALLSTACKITEM(_L("CContext_logContainer::set_status"));

	status_label->SetTextL(status);
	status_label->SetExtent(TPoint(10, 108), status_label->MinimumSize());
	DrawNow();
}

void CContext_logContainer::set_error(const TDesC& err)
{
	CALLSTACKITEM(_L("CContext_logContainer::set_error"));

	err_label->SetTextL(err);
	err_label->SetExtent(TPoint(10, 120), err_label->MinimumSize());
	DrawNow();
}

// ---------------------------------------------------------
// CContext_logContainer::SizeChanged()
// Called by framework when the view size is changed
// ---------------------------------------------------------
//
void CContext_logContainer::SizeChanged()
{
	CALLSTACKITEM(_L("CContext_logContainer::SizeChanged"));

	status_label->SetExtent(TPoint(10, 108), status_label->MinimumSize());
	err_label->SetExtent(TPoint(10, 120), status_label->MinimumSize());
	
	if (!label_store) return;
	
	for (int i=0; i<label_store->Count(); i++)  {
		(*label_store)[i]->SetExtent( TPoint(10, (i)*12), TSize(50, 12) );
		(*value_store)[i]->SetExtent( TPoint(65, (i)*12), 
			(*value_store)[i]->MinimumSize() );
	}	
}

// ---------------------------------------------------------
// CContext_logContainer::CountComponentControls() const
// ---------------------------------------------------------
//
TInt CContext_logContainer::CountComponentControls() const
{
	CALLSTACKITEM(_L("CContext_logContainer::CountComponentControls"));

	if (!label_store) return 2;
	
	return label_store->Count()+value_store->Count()+2; // return nbr of controls inside this container
}

// ---------------------------------------------------------
// CContext_logContainer::ComponentControl(TInt aIndex) const
// ---------------------------------------------------------
//
CCoeControl* CContext_logContainer::ComponentControl(TInt aIndex) const
{
	CALLSTACKITEM(_L("CContext_logContainer::ComponentControl"));

	if (aIndex==0) {
		return status_label;
	}
	--aIndex;

	if (aIndex==0) {
		return err_label;
	} else if (label_store && aIndex>0 && aIndex <= label_store->Count()) {
		return (*label_store)[aIndex-1];
	} else if (label_store && aIndex>label_store->Count() && aIndex <= 
		label_store->Count()+value_store->Count()) {
		return (*value_store)[aIndex-label_store->Count()-1];
	} else {
		return NULL;
	}
}


// ---------------------------------------------------------
// CContext_logContainer::Draw(const TRect& aRect) const
// ---------------------------------------------------------
//
void CContext_logContainer::Draw(const TRect& aRect) const
{
	CALLSTACKITEM(_L("CContext_logContainer::Draw"));

	CWindowGc& gc = SystemGc();
	// TODO: Add your drawing code here
	// example code...
	gc.SetPenStyle(CGraphicsContext::ENullPen);
	gc.SetBrushColor(KRgbGray);
	gc.SetBrushStyle(CGraphicsContext::ESolidBrush);
	gc.DrawRect(aRect);

	if (iCursorPos!=-1) {
		TRect r(TPoint(0, (iCursorPos+1)*12), TSize( Rect().Width(), 12));
		gc.SetBrushColor(KRgbWhite);
		gc.DrawRect(r);
	}

}

// ---------------------------------------------------------
// CContext_logContainer::HandleControlEventL(
//     CCoeControl* aControl,TCoeEvent aEventType)
// ---------------------------------------------------------
//
void CContext_logContainer::HandleControlEventL(
						CCoeControl* /*aControl*/,TCoeEvent /*aEventType*/)
{
	CALLSTACKITEM(_L("CContext_logContainer::HandleControlEventL"));

	// TODO: Add your control event handler code here
}

_LIT(CLASS_NAME, "CContext_logContainer");

const TDesC& CContext_logContainer::name() const
{
	CALLSTACKITEM(_L("CContext_logContainer::name"));

	return CLASS_NAME;
}

CCurrentLoc* CContext_logContainer::CurrentLoc()
{
	return iCurrentLoc;
}
// End of File  
