/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "discover.h"
#include "cl_settings.h"
#include "local_defaults.h"

#include <e32base.h>
#include <f32file.h>
#include <e32std.h>

#include "log_base_impl.h"
#include <timeout.h>

#include <es_sock.h>
#include <bt_sock.h>
#include <btmanclient.h>

#ifndef NO_BTENG_H
#include <bteng.h>
#else
// reverse engineered from BTENG.LIB:
enum TBTDiscoverabilityMode { EBTDiscoverabilityMode0, EBTDiscoverabilityMode1 };
enum TBTSearchMode { EBTSearchMode0, EBTSearchMode1 };
class CBTMCMSettings : public CBase {
public:
        IMPORT_C static int  GetAllSettings(int &, enum TBTDiscoverabilityMode &, enum TBTSearchMode &, class TDes16 &, int &);
        IMPORT_C static int  GetDiscoverabilityModeL(enum TBTDiscoverabilityMode &);
        IMPORT_C static int  GetLocalBDAddress(class TBTDevAddr &);
        IMPORT_C static int  GetLocalBTName(class TDes16 &);
        IMPORT_C static int  GetPowerStateL(int &);
        IMPORT_C static int  GetSearchModeL(enum TBTSearchMode &);
        IMPORT_C static int  IsLocalNameModified(int &);
        IMPORT_C static class CBTMCMSettings *  NewL(class MBTMCMSettingsCB *);
        IMPORT_C static class CBTMCMSettings *  NewLC(class MBTMCMSettingsCB *);
        IMPORT_C int  SetDefaultValuesL(void);
        IMPORT_C int  SetDiscoverabilityModeL(enum TBTDiscoverabilityMode, int);
        IMPORT_C int  SetLocalBTName(class TDesC16 const &);
        IMPORT_C int  SetPowerStateL(int, int);
        IMPORT_C int  SetSearchModeL(enum TBTSearchMode);
};
#endif

class CDiscoverImpl: public CDiscover, public CCheckedActive, public MTimeOut, public MSettingListener
{
public:
	~CDiscoverImpl();
	
	const TDesC& get_value();
	
	static CDiscoverImpl* NewL(MApp_context& Context);
private:
	void ConstructL();
	CDiscoverImpl(MApp_context& Context);
	virtual void DoCancel();
	virtual void CheckedRunL();
	virtual const TDesC& GetOwnAddress() { return own_address; }
	virtual void SettingChanged(TInt Setting);
private:
	enum comm_state { IDLE, GETTING_LOCAL, FIRST_SEARCH, NEXT_SEARCH };
	comm_state current_state;
	TBuf<2048> names;
	TBuf<2048> prev_env;

	RHostResolver hr; bool hr_is_open;
	RSocket socket; bool socket_is_open;
	TBuf<50> own_address; TBTDevAddrPckg iDevAddrPckg;
	RSocketServ socket_server; bool socket_server_is_open;
	TInquirySockAddr addr;
	TNameEntry name;
	TProtocolDesc pInfo;
	TInt	iInterval;

	RTimer timer; bool timer_is_open;
	CTimeOut*	iTimeOut;
	TInt		iTimeOutCount;
	TBool		iFirst;

#ifndef NO_BTENG_H
	RBTDeviceHandler iBtHandler;
#endif

	void release();
	void finished();
	void Restart();
	void expired(CBase* source);
	
	friend class CDiscover;
};

CDiscover* CDiscover::NewL(MApp_context& Context)
{
	CALLSTACKITEM2(_L("CDiscoverImpl::NewL"), &Context);

	auto_ptr<CDiscoverImpl> ret(new (ELeave) CDiscoverImpl(Context));
	ret->ConstructL();
	return ret.release();
}

CDiscover::~CDiscover()
{
}

CDiscover::CDiscover(MApp_context& Context) : Mlog_base_impl(Context)
{
}

CDiscoverImpl::CDiscoverImpl(MApp_context& Context) : CDiscover(Context),
CCheckedActive(CCheckedActive::EPriorityIdle, _L("CDiscoverImpl")), 
current_state(IDLE), iFirst(ETrue)
{
	CALLSTACKITEM(_L("CDiscoverImpl::CDiscoverImpl"));

}

void CDiscoverImpl::ConstructL() 
{
	CALLSTACKITEM(_L("CDiscoverImpl::ConstructL"));

	Settings().NotifyOnChange(SETTING_BT_SCAN_INTERVAL, this);

	Mlog_base_impl::ConstructL(KLog_bt, _L(""));
	iTimeOut=CTimeOut::NewL(*this);

	CActiveScheduler::Add(this); // add to scheduler

	Restart();
}

void CDiscoverImpl::Restart()
{
	if (prev_env.Length()>0) post_new_value(_L(""));
	prev_env.Zero();
	Cancel();
	release();
	iTimeOut->Reset();
	names.Zero();
	current_state=IDLE;

	iInterval=5*60;
	Settings().GetSettingL(SETTING_BT_SCAN_INTERVAL, iInterval);
	if (iInterval==0) return;

	User::LeaveIfError(timer.CreateLocal()); timer_is_open=true;
#ifndef __WINS__
	User::LeaveIfError(socket_server.Connect()); socket_server_is_open=true;
	TProtocolName aName=_L("BTLinkManager");
	User::LeaveIfError(socket_server.FindProtocol(aName, pInfo));
	User::LeaveIfError(socket.Open(socket_server, KBTAddrFamily, KSockSeqPacket, KL2CAP));	
	User::LeaveIfError(hr.Open(socket_server,pInfo.iAddrFamily,pInfo.iProtocol)); hr_is_open=true;

	auto_ptr<CBTMCMSettings> BtSettings(CBTMCMSettings::NewL(0));
	TBool power;
	BtSettings->GetPowerStateL(power);
	if (!power) {
		post_new_value(_L("Powering up Bt"), Mlogger::INFO);
		BtSettings->SetPowerStateL(ETrue, EFalse);
	}

	socket.Ioctl(KHCILocalAddressIoctl, iStatus, &iDevAddrPckg, KSolBtHCI);
	SetActive();
	current_state=GETTING_LOCAL;
#else // __WINS__
	TTimeIntervalMicroSeconds32 w(iInterval*1000*1000);
	timer.After(iStatus, w);
	SetActive();
#endif
}

CDiscoverImpl* CDiscoverImpl::NewL(MApp_context& Context)
{
	CALLSTACKITEM2(_L("CDiscoverImpl::NewL"), &Context);

	auto_ptr<CDiscoverImpl> ret(new (ELeave) CDiscoverImpl(Context));
	ret->ConstructL();
	return ret.release();
}

void CDiscoverImpl::release()
{	
	CALLSTACKITEM(_L("CDiscoverImpl::release"));

	if (socket_is_open) socket.Close();
	socket_is_open=false;

	if (timer_is_open) timer.Close();
	timer_is_open=false;
	if (hr_is_open) hr.Close();
	hr_is_open=false;
	if (socket_server_is_open) socket_server.Close();
	socket_server_is_open=false;
}

CDiscoverImpl::~CDiscoverImpl()
{
	CALLSTACKITEM(_L("CDiscoverImpl::~CDiscoverImpl"));

	Settings().CancelNotifyOnChange(SETTING_BT_SCAN_INTERVAL, this);

	delete iTimeOut;
	Cancel();
	release();
}	

void CDiscoverImpl::DoCancel()
{
	CALLSTACKITEM(_L("CDiscoverImpl::DoCancel"));

	switch(current_state) {
	case IDLE:
		timer.Cancel();
		break;
	case GETTING_LOCAL:
		socket.CancelIoctl();
		break;
	default:
		hr.Cancel();
		break;
	}
}

const TDesC& CDiscoverImpl::get_value()
{
	CALLSTACKITEM(_L("CDiscoverImpl::get_value"));

	return names;
}


void CDiscoverImpl::CheckedRunL()
{
	CALLSTACKITEM(_L("CDiscoverImpl::CheckedRunL"));

	iTimeOut->Reset();
	TBuf<30> msg;
	if (iStatus.Int()!=KErrNone && iStatus.Int()!=KErrHostResNoMoreResults ) {
		_LIT(stat_err, "error: %d");
		msg.Format(stat_err, iStatus.Int());
		post_new_value(_L(""), Mlogger::VALUE);
		post_new_value(msg, Mlogger::ERR);
		prev_env.Zero();
		Restart();
		return;
	}
#ifndef __WINS__
	switch(current_state) {
	case GETTING_LOCAL:
		{
		if (socket_is_open) socket.Close();
		socket_is_open=false;

		if (iInterval>0) {
			TTimeIntervalMicroSeconds32 w;
			if (iFirst)
				w=TTimeIntervalMicroSeconds32(3*1000*1000);
			else
				w=TTimeIntervalMicroSeconds32(iInterval*1000*1000);
			iFirst=EFalse;
			timer.After(iStatus, w);
			SetActive();
			current_state=IDLE;
		}

		TBuf<50> msg=_L("Own address: ");
		iDevAddrPckg().GetReadable(own_address);
		msg.Append(own_address);
		post_new_value(msg, Mlogger::INFO);
		break;
		}
	case FIRST_SEARCH:
		{
		if (iStatus.Int()!=KErrHostResNoMoreResults) {
			names.Zero();
			TBuf<50> addr;
			TInquirySockAddr btaddr(name().iAddr);
			btaddr.BTAddr().GetReadable(addr);
			names.Append(addr);
			names.Append(_L(" ["));
			names.Append(name().iName);
			names.Append(_L(","));
			names.AppendNum(btaddr.MajorClassOfDevice());
			names.Append(_L(":"));
			names.AppendNum(btaddr.MinorClassOfDevice());
			names.Append(_L(":"));
			names.AppendNum(btaddr.MajorServiceClass());
			names.Append(_L("] "));
			hr.Next(name, iStatus);
			iTimeOut->Wait(3*60);
			SetActive();
			current_state=NEXT_SEARCH;
		} else {
			if (prev_env.Length()>0) {
				post_new_value(_L(""));
				prev_env.Zero();
			}
			current_state=IDLE;
			iTimeOutCount=0;
			if (iInterval>0) {
				TTimeIntervalMicroSeconds32 w(iInterval*1000*1000);
				timer.After(iStatus, w);
				SetActive();
			}
		}
		}
		break;
	case NEXT_SEARCH:
		{
		if (iStatus.Int()!=KErrHostResNoMoreResults) {
			if (! (names.Length()+62+name().iName.Length() > names.MaxLength()) ) {
				TBuf<50> addr;
				TInquirySockAddr btaddr(name().iAddr);
				btaddr.BTAddr().GetReadable(addr);
				names.Append(addr);
				names.Append(_L(" ["));
				names.Append(name().iName);
				names.Append(_L(","));
				names.AppendNum(btaddr.MajorClassOfDevice());
				names.Append(_L(":"));
				names.AppendNum(btaddr.MinorClassOfDevice());
				names.Append(_L(":"));
				names.AppendNum(btaddr.MajorServiceClass());
				names.Append(_L("] "));
			}
			hr.Next(name, iStatus);
			iTimeOut->Wait(3*60);
			SetActive();
		} else {
			if (prev_env.Compare(names)) {
				post_new_value(names);
				prev_env=names;
			} else {
				post_new_value(names, Mlogger::UNCHANGED_VALUE);
			}
			current_state=IDLE;
			iTimeOutCount=0;
			if (iInterval>0) {
				TTimeIntervalMicroSeconds32 w(iInterval*1000*1000);
				timer.After(iStatus, w);
				SetActive();
			}
		}
		}
		break;
	case IDLE:
		{
		TBool connections=EFalse;
#ifndef NO_BTENG_H
		iBtHandler.AreExistingConnectionsL(connections);
#endif
		if (connections) {
			if (iInterval>0) {
				TTimeIntervalMicroSeconds32 w(iInterval*1000*1000);
				timer.After(iStatus, w);
				SetActive();
			}
			names=_L("BT connections in use");
			post_new_value(names);
		} else {
			if (iInterval>0) {
				addr.SetIAC(KGIAC);
				addr.SetAction(KHostResInquiry|KHostResName|KHostResIgnoreCache);
				hr.GetByAddress(addr, name, iStatus);

				SetActive();
			
				iTimeOut->Wait(3*60);
				current_state=FIRST_SEARCH;
			}
		}
		break;
		}
	}
#else
	own_address=_L("0020e04c71b9");
	post_new_value(_L("0020e04c71b8 [GLOMGOLD-25,1:1:2] 0002ee51c437 [Reno7650,2:3:4] 0002ee51c438 [Reno7651,2:3:4]"));

#endif

}

void CDiscoverImpl::expired(CBase* /*source*/)
{
	CALLSTACKITEM(_L("CDiscoverImpl::expired"));
	if (iTimeOutCount>5) {
		post_new_value(_L("too many timeouts in Bt discovery"), Mlogger::ERR);
		User::Leave(-1029);
	}
	iTimeOutCount++;
	post_new_value(_L(""), Mlogger::VALUE);
	post_new_value(_L("timeout in Bt discover"), Mlogger::ERR);
	Restart();
}

void CDiscoverImpl::SettingChanged(TInt /*Setting*/)
{
	TInt prev=iInterval;
	iInterval=5*60;
	Settings().GetSettingL(SETTING_BT_SCAN_INTERVAL, iInterval);
	if (prev==0 && iInterval!=0 && current_state==IDLE) Restart();
}