/* 
	Context - software for storing, publishing and receiving context information
	on Nokia Series 60 version 1 and version 2 smartphones

    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "locationing.h"
#include <PLPVAR.H>
#include <PLPVARIANT.H>

#include "cl_settings.h"

#ifdef __S60V2__
#include <etelmm.h>
#endif

#ifndef __WINS__
#define WAIT_TIME	90
#else
#define WAIT_TIME	1
#endif

#ifndef __WINS__
#define WAIT_TIME_DISCONNECTED	90
#else
#define WAIT_TIME_DISCONNECTED	1
#endif 

void Clocationing::ConstructL(i_status_notif* i_cb, MNaming* i_naming, sms* i_sms_handler, CPresencePublisher * aPublisher)
{
	CALLSTACKITEM(_L("Clocationing::ConstructL"));

	iName=(TText*)L"Clocationing";	
	iNaming=i_naming;
	cb=i_cb;
	
	smsh = i_sms_handler;

	iPublisher = aPublisher;

	iTimer=CTimeOut::NewL(*this);

	TInt enable;
	if (Settings().GetSettingL(SETTING_LOCATIONSERVICE_ENABLE, enable)) {
		iEnabled=enable;
	} else {
		iEnabled=true;
	}
#ifndef __WINS__

	// FIXME: refactor
	bool success=false;
#if !defined(__S60V2__)
#  if !defined(NO_ETELAGSM_H)
	MAdvGsmPhoneInformation::TSubscriberId i;
	if (Phone().GetSubscriberId(i)==KErrNone) {
		imsi=i;
		success=true;
	}
#  endif
#else
	RMobilePhone phone;
	RTelServer::TPhoneInfo info;
	TInt ret;
	if ( (ret=TelServer().GetPhoneInfo( 0, info ))==KErrNone) {
		if ((ret=phone.Open( TelServer(), info.iName ))==KErrNone) {
			CleanupClosePushL(phone);
			RMobilePhone::TMobilePhoneSubscriberId i;
			TRequestStatus r;
			phone.GetSubscriberId(r, i);
			User::WaitForRequest(r);
			if (r==KErrNone) {
				imsi=i;
				success=true;
			} else {
				TBuf<80> msg=_L("error getting imsi ");
				msg.AppendNum(r.Int());
				msg.Append(_L(" ")); msg.Append(i);
				if (cb) cb->status_change(msg);
			}
			CleanupStack::PopAndDestroy();
		} else {
			TBuf<40> msg=_L("error opening phone ");
			msg.AppendNum(ret);
			if (cb) cb->status_change(msg);
		}
	} else {
		TBuf<40> msg=_L("error getting phone info ");
		msg.AppendNum(ret);
		if (cb) cb->status_change(msg);
	}

#endif
	if (!success) {
		TPlpVariantMachineId machineId;
		PlpVariant::GetMachineIdL(machineId);
		imsi=machineId;
	}

#else
	// Return a fake IMEI when working on emulator
	_LIT(KEmulatorImsi, "244050000000000");
	imsi.Copy(KEmulatorImsi);
#endif
	smsh->AddHandler(this);

	Settings().NotifyOnChange(SETTING_LOCATIONSERVICE_ENABLE, this);

}

Clocationing::Clocationing(MApp_context& Context) : MContextBase(Context)
{
	CALLSTACKITEM(_L("Clocationing::Clocationing"));

}

Clocationing* Clocationing::NewL(MApp_context& Context, i_status_notif* i_cb, MNaming* i_naming, sms * i_sms_handler, CPresencePublisher * aPublisher)
{
	CALLSTACKITEMSTATIC(_L("Clocationing::NewL"));

	auto_ptr<Clocationing> ret(new (ELeave) Clocationing(Context));

	ret->ConstructL(i_cb, i_naming, i_sms_handler, aPublisher);
	return ret.release();
}

void Clocationing::handle_error(const TDesC& descr)
{
	CALLSTACKITEM(_L("Clocationing::handle_error"));
	
	// pending sms location reception
	iTimer->Reset();

	//resume presence
	if (iPublisher) iPublisher->ResumeConnection();

	cb->error(descr);
}

void Clocationing::handle_move(const TMsvId& /*msg_id*/, const TMsvId& /*from_folder*/, const TMsvId& /*to_folder*/, const TDesC& /*sender*/)
{
}

void Clocationing::handle_delete(const TMsvId& /*msg_id*/, const TMsvId& /*parent_folder*/, const TDesC& /*sender*/)
{
}

void Clocationing::handle_sending(const TMsvId& /*entry_id*/, const TDesC& /*sender*/, const TDesC & /*body*/)
{
}
void Clocationing::handle_change(const TMsvId& /*msg_id*/, const TDesC& /*sender*/)
{
}

void Clocationing::handle_read(const TMsvId& /*msg_id*/, const TDesC& /*sender*/)
{
}


bool Clocationing::handle_reception(const TMsvId& /*entry_id*/, const TMsvId& /*folder_id*/, const TDesC& sender, const TDesC& body)
{
	CALLSTACKITEM(_L("Clocationing::handle_reception"));

#ifndef __WINS__
	if (iStyle==ENone) return false;
	if (iStyle==ERadiolinja) {
		if (sender.Compare(_L("16507")) ) {
			return false;
		}
	}
	if (iStyle==ESonera) {
		if (sender.Compare(_L("15400")) ) {
			return false;
		}
	}
#endif
	
	iTimer->Reset();

	if (iToDiscard) {
		--iToDiscard;
		cb->status_change(_L("discarded unreliable result"));
		return true;
	}
	if (!iToName.SentRequest) return false;

	auto_ptr<HBufC> content(0);
	bool error_reply=false;

	if (sender.Compare(_L("16507"))==0) {
		if (body.Left(8).Compare(_L("itse  on"))) {
			error_reply=true;
		} else {
			content.reset(body.Mid(19, body.Length()-19-1).AllocL());
			content->Des().Trim();
		}
	} else if (sender.Compare(_L("15400"))==0) {
		TInt colon_pos=body.Find(_L(":"));
		if (colon_pos==KErrNotFound || body.Find(_L("sijainti")) == KErrNotFound ) {
			error_reply=true;
		} else {
			TBuf<30> city, district;

			TLex l(body);
			while (l.Get() != ':');
			l.Inc();
			l.Mark();

			while (l.Get() != ',');
			l.UnGet();
			city=l.MarkedToken().Left(30);

			l.Inc(2);
			l.Mark();
			while (l.Get() != ',');
			l.UnGet();
			district=l.MarkedToken().Left(30);

			content.reset(HBufC::NewL(60));
			content->Des().Append(district);
			content->Des().Append(_L(", "));
			content->Des().Append(city);
		}

	}

	if (error_reply) {
		if (true || iRetryCount>5) 
		{
			/*
			 * It seems that if the locationing doesn't
			 * work, it won't work again before a cell change.
			 * The Radiolinja service probably sometimes doesn't
			 * get updated correctly, and doesn't retry.
			 * It's not worth trying again before we go
			 * to a new cell.
			 */
			iToName.Reset();
			cb->error(_L("Error result"));

			//resume presence
			if (iPublisher) iPublisher->ResumeConnection();
		} 
		else 
		{
			iToName.SentRequest=false;
			iRetryCount++;
			iTimer->Wait(WAIT_TIME*4);
		}
		
		return true;
	}

	cb->status_change(*content);

	iNaming->add_cellid_name(iToName.CellId, *content);
	iToName.Reset();

	//resume presence
	if (iPublisher) iPublisher->ResumeConnection();
	return true;
}

void Clocationing::GetNameL(const TDesC& CellId)
{
	CALLSTACKITEM(_L("Clocationing::GetNameL"));

	if (!iLocationingAvailable || !iEnabled) return;

	if (iToName.SentRequest) {
		return;
	}
	iToName.CellId=CellId;
	iToName.SentRequest=false;
	
	//suspend connection
	if (iPublisher) iPublisher->SuspendConnection();
	iTimer->Wait(WAIT_TIME_DISCONNECTED);
	iRetryCount=0;
}

void Clocationing::expired(CBase*)
{
	CALLSTACKITEM(_L("Clocationing::expired"));
	TBuf<20> msg; volatile TInt err;

	if (iToName.SentRequest == false)
	{	
		err=KErrNotFound;
		if (iStyle==ERadiolinja) 
			err=smsh->send_message(_L("16507"), _L("PAIKKA itse"), false);
		else if (iStyle==ESonera)
			err=smsh->send_message(_L("15400"), _L("MISS"), false);

		iToName.Time.HomeTime();
		if (err!=KErrNone) {
			msg.Format(_L("sms error %d"), err);
			cb->error(msg);
			
		} else {
			iToName.SentRequest=true;
			iTimer->Wait(WAIT_TIME);
			return;
		}
	}
	if (iPublisher) iPublisher->ResumeConnection();;
}

void Clocationing::test()
{
	CALLSTACKITEM(_L("Clocationing::test"));

	if (!iLocationingAvailable) {
		cb->error(_L("Locationing not available"));
		return;
	}
	if (!iEnabled) {
		cb->error(_L("Locationing not enabled"));
		return;
	}

	volatile TInt err=0;
	TBuf<20> msg;
	smsh->send_message(_L("16507"), _L("PAIKKA itse"), false);
	if (err!=KErrNone) {
		msg.Format(_L("sms error %d"), err);
		cb->error(msg);
	}
}

Clocationing::~Clocationing()
{
	CALLSTACKITEM(_L("Clocationing::~Clocationing"));

	Settings().CancelNotifyOnChange(SETTING_LOCATIONSERVICE_ENABLE, this);
	delete iTimer;
}

void Clocationing::now_at_location(const TDesC& cell, TInt id, bool is_base, bool loc_changed, TTime time)
{
	CALLSTACKITEM(_L("Clocationing::now_at_location"));

	if ( cell.FindC(_L("RADIOLINJA"))!=KErrNotFound) {
		iLocationingAvailable=true;
		iStyle=ERadiolinja;
#if 1
	} else if ( cell.FindC(_L("SONERA"))!=KErrNotFound) {
		iLocationingAvailable=true;
		iStyle=ESonera;
#endif
	} else {
		iLocationingAvailable=false;
		iStyle=ENone;
	}

	if (loc_changed) {
		if (iToName.SentRequest) {
			iToName.SentRequest=false;
			++iToDiscard;
			if (iPublisher) iPublisher->ResumeConnection();
		} else if (iTimer->IsActive()) {
			if (iPublisher) iPublisher->ResumeConnection();
		}
		iTimer->Reset();
	}
}

void Clocationing::SettingChanged(TInt /*Setting*/)
{
	CALLSTACKITEM(_L("Clocationing::SettingChanged"));

	TInt enable;
	if (Settings().GetSettingL(SETTING_LOCATIONSERVICE_ENABLE, enable)) {
		iEnabled=enable;
	} else {
		iEnabled=true;
	}
}