#include "log_gps.h"
#include <symbian_auto_ptr.h>
#include <btmanclient.h>
#include <btextnotifiers.h>
#include <btsdp.h>
#include <bt_sock.h>

#include <timeout.h>
#include "cl_settings.h"

#define KBTServiceSerial 0x1101
#define READ_INTERVAL 10

_LIT8(KStartCmd, "");
_LIT8(KStopCmd, "");

_LIT(KName, "CLogGpsImpl");

class CLogGpsImpl : public CLogGps, public MTimeOut, 
	public MSdpAgentNotifier, public MSdpAttributeValueVisitor, public MSettingListener {
private:
	virtual void SelectDevice();
	virtual void TestDevice();
	CLogGpsImpl(MApp_context& Context);
	virtual ~CLogGpsImpl();

	enum comm_state { IDLE, SELECTING, GETTING_SERVICE, CONNECTING, SENDING_START, SENDING_STOP, CLOSING, READING, RETRY };
	comm_state current_state;
	RNotifier iNotifier; bool not_is_open;
	TBTDeviceSelectionParams selectionFilter;
	TUUID targetServiceClass;
	TBTDeviceSelectionParamsPckg pckg;
	TBTDeviceResponseParamsPckg result;

	CTimeOut*	iTimeOut;
	TInt		iLogTime;
	TTime		iLogStarted;
	bool		iFirst;

	CSdpAgent* agent;
	CSdpSearchPattern* list;

	TBTSockAddr	iBtAddr;
	bool		iAddrSet; 

	TUint port;
	bool seen_rfcomm;

	RSocket socket; bool socket_is_open; bool socket_is_connected;
	RSocketServ socket_server; bool socket_server_is_open;

	TBuf8<100> buf;
	TBuf8<100> line; TBuf<100> line16;
	TBuf<200> whole_line;
	int read_count;

	// parsing
	bool nl; bool cmd;
	bool getting_line;
	TBuf8<6> cmdstr; TBuf<6> cmdstr16;

	void CheckedRunL();
	void DoCancel();
	void ConstructL();

	void get_service();
	void connect_to_service();
	void release();
	void handle_buffer();
	void retry(TInt WaitTime=30);
	void calculate_time();

	//MSdpAgentNotifier
	virtual void AttributeRequestComplete(TSdpServRecordHandle aHandle, TInt aError);
	virtual void AttributeRequestResult(TSdpServRecordHandle aHandle, 
		TSdpAttributeID aAttrID, CSdpAttrValue* aAttrValue);
	virtual void NextRecordRequestComplete(TInt aError, 
		TSdpServRecordHandle aHandle, TInt aTotalRecordsCount);
	//MSdpAttributeValueVisitor
	virtual void VisitAttributeValueL(CSdpAttrValue &aValue, TSdpElementType aType);
	virtual void StartListL(CSdpAttrValueList &aList);
	virtual void EndListL();

	// MTimeOut
	virtual void expired(CBase* Source);

	// Mlogger
	virtual void register_source(const TDesC& name, const TDesC& initial_value, const TTime& time);
	virtual void new_value(log_priority priority, const TDesC& name, const TDesC& value, const TTime& time);
	virtual void unregister_source(const TDesC& name, const TTime& time);
	virtual const TDesC& name() const;

	// MSettingListener
	virtual void SettingChanged(TInt Setting);

	friend class CLogGps;
	friend class auto_ptr<CLogGpsImpl>;
};

CLogGps* CLogGps::NewL(MApp_context& Context)
{
	CALLSTACKITEM2(_L("CLogGps::NewL"), &Context);

	auto_ptr<CLogGpsImpl> ret(new (ELeave) CLogGpsImpl(Context));
	ret->ConstructL();
	return ret.release();
}

CLogGps::CLogGps(MApp_context& Context) : CCheckedActive(EPriorityNormal, _L("CLogGps")), 
	Mlog_base_impl(Context)
{

}

CLogGps::~CLogGps()
{

}

CLogGpsImpl::CLogGpsImpl(MApp_context& Context) : CLogGps(Context), targetServiceClass(0x2345), iFirst(true)
{
}

void CLogGpsImpl::ConstructL()
{
	CALLSTACKITEM(_L("CLogGpsImpl::ConstructL"));

	Settings().NotifyOnChange(SETTING_GPS_LOG_TIME, this);
	Settings().NotifyOnChange(SETTING_BT_SCAN_INTERVAL, this);

	iTimeOut=CTimeOut::NewL(*this);
	Mlog_base_impl::ConstructL(KLog_gps, _L(""));

	iLogTime=15;
	calculate_time();
	TBuf8<10> dabuf;
	iAddrSet=Settings().GetSettingL(SETTING_GPS_BT_ADDR, dabuf);
	if (iAddrSet && dabuf.Length()==6) {
		TBTDevAddr  da(dabuf);
		TInt p;
		iAddrSet=Settings().GetSettingL(SETTING_GPS_BT_PORT, p);
		port=p;
		iBtAddr.SetBTAddr(da);
		iBtAddr.SetPort(port);
	}

	CActiveScheduler::Add(this); // add to scheduler
}

void CLogGpsImpl::SelectDevice()
{
	CALLSTACKITEM(_L("CLogGpsImpl::SelectDevice"));

	release();

	TInt ret;
	if ((ret=iNotifier.Connect())!=KErrNone) {
		TBuf<30> msg=_L("Error connecting to notifier: ");
		msg.AppendNum(ret);

		post_new_value(msg, Mlogger::ERR);
		return;
	}
	not_is_open=true;

	selectionFilter.SetUUID(targetServiceClass);
	iNotifier.StartNotifierAndGetResponse(iStatus, KDeviceSelectionNotifierUid, pckg, result);
	
	SetActive();	
	current_state=SELECTING;
}

CLogGpsImpl::~CLogGpsImpl()
{
	CALLSTACKITEM(_L("CLogGpsImpl::~CLogGpsImpl"));

	release();
	delete iTimeOut;
	Settings().CancelNotifyOnChange(SETTING_BT_SCAN_INTERVAL, this);
	Settings().CancelNotifyOnChange(SETTING_GPS_LOG_TIME, this);
}

void CLogGpsImpl::release()
{
	CALLSTACKITEM(_L("CLogGpsImpl::release"));

	iTimeOut->Reset();

	Cancel();

	if (not_is_open) iNotifier.Close(); not_is_open=false;

	delete agent; agent=0;
	delete list; list=0;
	
	if (socket_is_connected) {
		TRequestStatus s;
		socket.Shutdown(RSocket::EImmediate, s);
		User::WaitForRequest(s);
		socket_is_connected=false;
	}
	if (socket_is_open) socket.Close(); socket_is_open=false;
	if (socket_server_is_open) socket_server.Close(); socket_server_is_open=false;

	current_state=IDLE;

}

void CLogGpsImpl::DoCancel()
{
	CALLSTACKITEM(_L("CLogGpsImpl::DoCancel"));

	switch(current_state) {
	case SELECTING:
		iNotifier.CancelNotifier(KDeviceSelectionNotifierUid);
		break;
	case CONNECTING:
		socket.CancelConnect();
		break;
	case READING:
		socket.CancelRead();
		break;
	default:
		break;
	}
	current_state=IDLE;
}

void CLogGpsImpl::get_service()
{
	CALLSTACKITEM(_L("CLogGpsImpl::get_service"));

	release();
	agent = CSdpAgent::NewL(*this, result().BDAddr());
	list = CSdpSearchPattern::NewL();
	list->AddL(KBTServiceSerial);
	agent->SetRecordFilterL(*list);
	agent->NextRecordRequestL();
}

void CLogGpsImpl::retry(TInt WaitTime)
{
	CALLSTACKITEM(_L("CLogGpsImpl::retry"));

	if (iFirst) {
		TBuf<20> addr;
		iBtAddr.BTAddr().GetReadable(addr);
		TBuf<100> msg=_L("using dev ");
		msg.Append(addr);
		msg.Append(_L(" port "));
		msg.AppendNum(iBtAddr.Port());
		post_new_value(msg, Mlogger::INFO);
	}
	iFirst=false;

	TTime curr;
	curr.HomeTime();
	if (curr + TTimeIntervalSeconds(WaitTime) > iLogStarted+TTimeIntervalSeconds(iLogTime) ) {
		release();
	} else {
		release();
		current_state=RETRY;

		iTimeOut->Wait(WaitTime);
	}

	line.Zero(); cmdstr.Zero();
	cmd=nl=getting_line=false;
}

void CLogGpsImpl::expired(CBase*)
{
	CALLSTACKITEM(_L("CLogGpsImpl::expired"));

	if (current_state==RETRY) {
		current_state=IDLE;
		connect_to_service();
	} else {
		Cancel();
		if (KStopCmd().Length()>0) {
			socket.Write(KStopCmd, iStatus);
			current_state=SENDING_STOP;
		} else {
			socket.Shutdown(RSocket::EImmediate, iStatus);
			current_state=CLOSING;
		}
		SetActive();
	}
}

void CLogGpsImpl::CheckedRunL()
{
	CALLSTACKITEM(_L("CLogGpsImpl::CheckedRunL"));

	TBuf<30> msg;
	if (iStatus.Int()!=KErrNone) {
		_LIT(stat_err, "error: %d %d");
		msg.Format(stat_err, iStatus.Int(), TInt(current_state));
		post_new_value(msg, Mlogger::ERR);
		if (current_state==READING || current_state==CONNECTING)
			retry();
		else
			release();
		return;
	}
	switch(current_state) {
	case SELECTING:
		if(!result().IsValidBDAddr()) {
			msg=_L("cancelled");
			post_new_value(msg, Mlogger::INFO);
			current_state=IDLE;
		} else {
			msg=_L("selected ");
			msg.Append(result().DeviceName());
			post_new_value(msg, Mlogger::INFO);
			port=0; seen_rfcomm=false;
			get_service();
		}
		if (not_is_open) iNotifier.Close(); not_is_open=false;
		break;
	case SENDING_START:
		socket.Read(buf, iStatus);
		current_state=READING;
		SetActive();
		break;
	case SENDING_STOP:
		socket.Shutdown(RSocket::EImmediate, iStatus);
		current_state=CLOSING;
		SetActive();
		break;
	case CLOSING:
		release();
		break;
	case CONNECTING:
		{
			socket_is_connected=true;
			post_new_value(_L("connected"), Mlogger::INFO);
			read_count=0;
			if ( KStartCmd().Length()>0) {
				socket.Write(KStartCmd, iStatus);
				current_state=SENDING_START;
			} else {
				socket.Read(buf, iStatus);
				current_state=READING;
			}
			SetActive();
		}
		break;
	case RETRY:
	case IDLE:
	case GETTING_SERVICE:
		msg.Format(_L("Unexpected state %d"), current_state);
		post_new_value(msg, Mlogger::ERR);
		break;
	case READING:
		{
		handle_buffer();
		buf.Zero();
		socket.Read(buf, iStatus);
		current_state=READING;
		SetActive();
		}
		break;
	}
}

void CLogGpsImpl::handle_buffer()
{
	CALLSTACKITEM(_L("CLogGpsImpl::handle_buffer"));

	int pos=0;
	int len=buf.Length();
	bool incremented=false;
	TText8 cur;
	while (pos<len) {
		if (pos<len && getting_line) {
			cur=buf[pos];
			if (! ( cur== '\r' || cur == '\n') )  {
				line.Append(cur);
				if (line.Length()==line.MaxLength()) {
					// overflow: garbled
					line.Zero();
					getting_line=false;
				}
				if (cur=='\r') {
					++pos; incremented=true;
				}
			} else {
				whole_line=_L("$");
				TInt s;
				CC()->ConvertToUnicode(cmdstr16, cmdstr, s);
				whole_line.Append(cmdstr16);
				whole_line.Append(_L(","));
				CC()->ConvertToUnicode(line16, line, s);
				whole_line.Append(line16);
				post_new_value(whole_line);
			}
		}
		if (pos<len && (cur=buf[pos])=='\n') {
			nl=true; cmd=false; getting_line=false;
			++pos; incremented=true;
		}
		if (pos<len && nl) {
			if ((cur=buf[pos])=='$') {
				cmd=true;
				cmdstr.Zero();
			} // else: garbled line
			nl=false;
			++pos; incremented=true;
			//cb->status_change(_L("nl"));
		}
		if (pos<len && cmd) {
			if ((cur=buf[pos])==',') {
				cmd=false;
				if (! cmdstr.Compare(_L8("GPRMC") ) ) {
					++read_count;
					if (read_count==READ_INTERVAL) {
						getting_line=true;
						line.Zero();
						post_new_value(_L("reading line"), Mlogger::INFO);
						read_count=0;
					}
				}
			} else {
				cmdstr.Append(cur);
				//TBuf<6> msg;
				//cc->ConvertToUnicode(msg, cmdstr, state);
				//cb->status_change(msg);
				if (cmdstr.Length()==cmdstr.MaxLength()) {
					// overflow: garbled
					cmdstr.Zero();
					cmd=false;
				}
			}
			++pos; incremented=true;
		}
		if (!incremented) ++pos;
		incremented=false;
	}
}


void CLogGpsImpl::AttributeRequestComplete(TSdpServRecordHandle /*aHandle*/, TInt /*aError*/)
{
	CALLSTACKITEM(_L("CLogGpsImpl::AttributeRequestComplete"));

	if (port==0) {
		post_new_value(_L("didn't find service"), Mlogger::ERR);
	} else {
		TBuf<30> msg=_L("found port ");
		msg.AppendNum(port);
		post_new_value(msg, Mlogger::INFO);
		iBtAddr.SetBTAddr(result().BDAddr());
		iBtAddr.SetPort(port);
		iAddrSet=true;
		msg=_L("writing setting, len ");
		msg.AppendNum(iBtAddr.Length());
		post_new_value(msg, Mlogger::INFO);
		Settings().WriteSettingL(SETTING_GPS_BT_ADDR, iBtAddr.BTAddr().Des());
		Settings().WriteSettingL(SETTING_GPS_BT_PORT, iBtAddr.Port());
	}
	current_state=IDLE;
}

void CLogGpsImpl::AttributeRequestResult(TSdpServRecordHandle /*aHandle*/, 
				      TSdpAttributeID /*aAttrID*/, CSdpAttrValue* aAttrValue)
{
	CALLSTACKITEM(_L("CLogGpsImpl::AttributeRequestResult"));

	
	if (aAttrValue) 
		aAttrValue->AcceptVisitorL(*this);	
}

void CLogGpsImpl::VisitAttributeValueL(CSdpAttrValue &aValue, TSdpElementType aType)
{
	CALLSTACKITEM(_L("CLogGpsImpl::VisitAttributeValueL"));

	if (aType==ETypeUUID) {
		if (aValue.UUID()==KRFCOMM) {
			seen_rfcomm=true;
		} else {
			seen_rfcomm=false;
		}
	} else if (aType==ETypeUint && seen_rfcomm) {
		port=aValue.Uint();
	}
}

void CLogGpsImpl::StartListL(CSdpAttrValueList &/*aList*/)
{
	CALLSTACKITEM(_L("CLogGpsImpl::StartListL"));

}

void CLogGpsImpl::EndListL()
{
	CALLSTACKITEM(_L("CLogGpsImpl::EndListL"));

}

void CLogGpsImpl::connect_to_service()
{
	CALLSTACKITEM(_L("CLogGpsImpl::connect_to_service"));

	TInt ret;
	TBuf<30> err;
	if ( (ret=socket_server.Connect()) != KErrNone) {
		_LIT(f, "RSocketServer.Connect: %d");
		err.Format(f, ret);
		post_new_value(err, Mlogger::ERR);
		retry();
		return;
	}
	socket_server_is_open=true;

	//if ( (ret=socket.Open(socket_server, KBTAddrFamily, KSockStream, KRFCOMM )) != KErrNone) {
	if ( (ret=socket.Open(socket_server, _L("RFCOMM"))) != KErrNone ) {
		_LIT(f, "RSocket.Open: %d");
		err.Format(f, ret);
		post_new_value(err, Mlogger::ERR);
		retry();
		return;
	}
	socket_is_open=true;

	// only log for set time
	TTime curr; curr.HomeTime();
	TTimeIntervalSeconds secs;
	TTime stop_at;
	stop_at=iLogStarted+TTimeIntervalSeconds(iLogTime);
	if (stop_at > curr ) {
		stop_at.SecondsFrom(curr, secs);
		iTimeOut->Wait(secs.Int());

		socket.Connect(iBtAddr, iStatus);
   
		current_state=CONNECTING;
		SetActive();
	} else {
		release();
	}


}


void CLogGpsImpl::NextRecordRequestComplete(TInt aError, 
					 TSdpServRecordHandle aHandle, TInt aTotalRecordsCount)
{
	CALLSTACKITEM(_L("CLogGpsImpl::NextRecordRequestComplete"));

	TBuf<30> msg;
	if (aError!=KErrNone && aError!=KErrEof) {
		_LIT(err, "service error: %d");
		msg.Format(err, aError);
		post_new_value(msg, Mlogger::ERR);
	} else if (aError==KErrEof) {
		_LIT(err, "disconnected");
		post_new_value(err, Mlogger::ERR);
	} else if(aTotalRecordsCount==0) {
		_LIT(err, "no Serial service");
		post_new_value(err, Mlogger::ERR);
	} else {
		_LIT(st, "found service");
		post_new_value(st, Mlogger::INFO);
		agent->AttributeRequestL(aHandle, KSdpAttrIdProtocolDescriptorList);
		return;
	}
	current_state=IDLE;
}

void CLogGpsImpl::register_source(const TDesC& name, const TDesC& initial_value, const TTime& time)
{
	CALLSTACKITEM(_L("CLogGpsImpl::register_source"));

	if (name.Compare(KLog_bt)) return;

	new_value(Mlogger::VALUE, name, initial_value, time);
}

void CLogGpsImpl::new_value(log_priority priority, const TDesC& name, const TDesC& value, const TTime& /*time*/)
{
	CALLSTACKITEM(_L("CLogGpsImpl::new_value"));

	if (priority!=Mlogger::VALUE && priority!=Mlogger::UNCHANGED_VALUE) return;

	if (!iAddrSet || iLogTime<=0) return;
	if (name.Compare(KLog_bt)) return;

	TBuf<20> addr;
	iBtAddr.BTAddr().GetReadable(addr);
	if (value.Find(addr)!=KErrNotFound) {
		iLogStarted.HomeTime();
		retry(3);
	}
}

void CLogGpsImpl::unregister_source(const TDesC& , const TTime& )
{
	CALLSTACKITEM(_L("CLogGpsImpl::unregister_source"));

}

const TDesC& CLogGpsImpl::name() const
{
	CALLSTACKITEM(_L("CLogGpsImpl::name"));

	return KName;
}

void CLogGpsImpl::SettingChanged(TInt)
{
	CALLSTACKITEM(_L("CLogGpsImpl::SettingChanged"));

	calculate_time();
}

void CLogGpsImpl::calculate_time()
{
	TInt bt_int;
	Settings().GetSettingL(SETTING_GPS_LOG_TIME, iLogTime);
	Settings().GetSettingL(SETTING_BT_SCAN_INTERVAL, bt_int);
	if (iLogTime > bt_int-5) iLogTime=bt_int-5;
}

void CLogGpsImpl::TestDevice()
{
	iLogStarted.HomeTime();
	retry(1);
}
