/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "picture_publisher.h"
#include "transfer.h"
#include "cl_settings.h"
#include <bautils.h>

_LIT(KClassName, "CPicturePublisherImpl");

class TOldPrompt : public MUploadPrompt {
public:
	TOldPrompt(bool Delete=true);
	virtual void Prompt(const TDesC& FileName, MUploadCallBack* CallBack);
private:
	bool	iDelete;
};

class CPicturePublisherImpl : public CPicturePublisher, public MContextBase,
	public i_status_notif, public MSettingListener  {
public:
	~CPicturePublisherImpl();
private:
	CPicturePublisherImpl(MApp_context& Context, i_status_notif& notif, MUploadPrompt& Prompt,
		CTransferDir*	aTransferDir);
	void ConstructL(const TDesC& Path, const TDesC& Files, TInt Setting,
		const TDesC& AdditionalFiles=KNullDesC);

	virtual void PublishOld();

	void CheckedRunL();
	TInt CheckedRunError(TInt aError);
	void DoCancel();

	virtual void register_source(const TDesC&, const TDesC& initial_value, const TTime&);
	virtual void new_value(log_priority priority, const TDesC&, const TDesC& value, const TTime&);
	virtual void unregister_source(const TDesC&, const TTime&);
	virtual const TDesC& name() const;

	virtual void finished();
	virtual void error(const TDesC& descr);
	virtual void status_change(const TDesC& status);

	virtual void SettingChanged(TInt Setting);

	void Transfer(const TDesC& aSubDir, MUploadPrompt* aPrompt, const TTime& aFromTime);
	void Transfer();
	bool IsEnabled();
	void StartL();
	void Stop();

	i_status_notif& iNotif;
	TBuf<200>	iLoc;
	bool		iNoPics;
	TFileName	iPath;
	TBuf<30>	iFiles, iAddFiles;
	TInt		iSetting;
	CTransferDir*	iTransferDir;
	TOldPrompt*	iOldPrompt;
	TBool		iPathExists;

#ifdef __WINS__test
	RTimer		iTimer;
#endif
	MUploadPrompt&	iPrompt;

	friend class CPicturePublisher;
};

CPicturePublisher* CPicturePublisher::NewL(MApp_context& Context, i_status_notif& notif,
					const TDesC& Path, const TDesC& Files, TInt Setting, 
					MUploadPrompt& Prompt, CTransferDir*	aTransferDir,
					const TDesC& AdditionalFiles)
{
	CALLSTACKITEM(_L("CPicturePublisher::NewL"));

	auto_ptr<CPicturePublisherImpl> ret(new (ELeave) 
		CPicturePublisherImpl(Context, notif, Prompt, aTransferDir));
	ret->ConstructL(Path, Files, Setting, AdditionalFiles);
	return ret.release();
}

CPicturePublisherImpl::~CPicturePublisherImpl()
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::~CPicturePublisherImpl"));

	Settings().CancelNotifyOnChange(iSetting, this);
	Stop();
#ifdef __WINS__test
	iTimer.Close();
#endif
	delete iOldPrompt;
}

CPicturePublisher::CPicturePublisher() : CCheckedActive(EPriorityNormal, KClassName)
{
}

CPicturePublisherImpl::CPicturePublisherImpl(MApp_context& Context, i_status_notif& notif, MUploadPrompt& Prompt, CTransferDir*	aTransferDir) :
	MContextBase(Context), iNotif(notif), iTransferDir(aTransferDir), iPrompt(Prompt)
{
}

void CPicturePublisherImpl::ConstructL(const TDesC& Path, const TDesC& Files, TInt Setting,
				const TDesC& AdditionalFiles)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::ConstructL"));

	TBool delete_after=ETrue;
	Settings().GetSettingL(SETTING_DELETE_UPLOADED, delete_after);
	iOldPrompt=new (ELeave) TOldPrompt(delete_after);

	iPath=Path; iFiles=Files; iSetting=Setting; iAddFiles=AdditionalFiles;
	Settings().NotifyOnChange(iSetting, this);
#ifdef __WINS__test
	iTimer.CreateLocal();
#endif
	CActiveScheduler::Add(this);

	if (IsEnabled()) {
		StartL();
	}
}

void CPicturePublisherImpl::Stop()
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::Stop"));

	Cancel();
}

void CPicturePublisherImpl::StartL()
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::StartL"));

	Stop();

	iStatus=KRequestPending;
#ifndef __WINS__test
	TFileName t=iPath;
	//
	// only listen on directories that exist, otherwise
	// we seem to get a lot of spurious events
	//
	t.Replace(0, 1, _L("c"));
	t.Append(_L("\\"));
	iPathExists=BaflUtils::PathExists(Fs(), t);
	if (!iPathExists) {
		t.Replace(0, 1, _L("e"));
		iPath.Replace(0, 1, _L("e"));
		iPathExists=BaflUtils::PathExists(Fs(), t);
	} else {
		t.Replace(0, 1, _L("e"));
		if (! (BaflUtils::PathExists(Fs(), t))) iPath.Replace(0, 1, _L("c"));
	}
	if (iPathExists) {
		Fs().NotifyChange(ENotifyEntry, iStatus, iPath);
	} else {
		return;
	}
#else
	iTimer.After( iStatus, 30*1000*1000 );
#endif
	SetActive();
}

bool CPicturePublisherImpl::IsEnabled()
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::IsEnabled"));

	TBool enabled;
	if ( Settings().GetSettingL(iSetting, enabled) && enabled) return true;
	return false;
}

void CPicturePublisherImpl::SettingChanged(TInt)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::SettingChanged"));

	if (IsEnabled()) {
		TRAPD(err, StartL());
		if (err!=KErrNone) {
			TBuf<100> msg;
			msg.Format(_L("Error starting PicturePublisher: %d"), err);
			iNotif.error(msg);
		}
	} else {
		Stop();
	}
}
			

void CPicturePublisherImpl::CheckedRunL()
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::CheckedRunL"));

	if (iStatus==KErrNone) {
		Transfer();
	} else {
		TBuf<30> msg=_L("error ");
		msg.AppendNum(iStatus.Int());
		iNotif.error(msg);
	}
	iStatus=KRequestPending;
#if 1
	Fs().NotifyChange(ENotifyEntry, iStatus, iPath);
#else
	iTimer.After( iStatus, 10*1000*1000 );
#endif
	SetActive();
}

void CPicturePublisherImpl::Transfer()
{
	/*
	TBuf<200> msg=_L("new pictures noticed ");
	msg.Append(iPath);

	iNotif.status_change(msg);
	*/

	TTime now; now.HomeTime(); 

#ifndef __WINS__
	now-=TTimeIntervalMinutes(2);
#else
	now-=TTimeIntervalDays(2);
#endif

	Transfer(_L(""), &iPrompt, now);
}

void CPicturePublisherImpl::Transfer(const TDesC& aSubDir, MUploadPrompt* aPrompt, const TTime& aFromTime)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::Transfer"));

	iNoPics=true;
	if (iLoc.Length()==0) iLoc=_L("Unknown");
	TFileName p;
	if (iPath.Left(1).Compare(_L("?"))) {
		p=iPath;
		p.Append(_L("\\"));
		p.Append(aSubDir);
		p.Append(iFiles);
		iTransferDir->ProcessDir(p, SETTING_PUBLISH_URLBASE, SETTING_PUBLISH_SCRIPT,
			iLoc, aPrompt, aFromTime);
		if (iAddFiles.Length()>0) {
			p=iPath;
			p.Append(_L("\\"));
			p.Append(aSubDir);
			p.Append(iAddFiles);
			iTransferDir->ProcessDir(p, SETTING_PUBLISH_URLBASE, SETTING_PUBLISH_SCRIPT,
				iLoc, aPrompt, aFromTime);
		}

	} else {
		p=iPath;
		p.Replace(0, 1, _L("c"));
		p.Append(_L("\\"));
		p.Append(aSubDir);
		p.Append(iFiles);
		iTransferDir->ProcessDir(p, SETTING_PUBLISH_URLBASE, SETTING_PUBLISH_SCRIPT,
			iLoc, aPrompt, aFromTime);
		if (iAddFiles.Length()>0) {
			p=iPath;
			p.Replace(0, 1, _L("c"));
			p.Append(_L("\\"));
			p.Append(aSubDir);
			p.Append(iAddFiles);
			iTransferDir->ProcessDir(p, SETTING_PUBLISH_URLBASE, SETTING_PUBLISH_SCRIPT,
				iLoc, aPrompt, aFromTime);
		}

		p=iPath;
		p.Replace(0, 1, _L("e"));
		p.Append(_L("\\"));
		p.Append(aSubDir);
		p.Append(iFiles);
		iTransferDir->ProcessDir(p, SETTING_PUBLISH_URLBASE, SETTING_PUBLISH_SCRIPT,
			iLoc, aPrompt, aFromTime);
		if (iAddFiles.Length()>0) {
			p=iPath;
			p.Replace(0, 1, _L("e"));
			p.Append(_L("\\"));
			p.Append(aSubDir);
			p.Append(iAddFiles);
			iTransferDir->ProcessDir(p, SETTING_PUBLISH_URLBASE, SETTING_PUBLISH_SCRIPT,
				iLoc, aPrompt, aFromTime);
		}
	}

	iNoPics=false;
}

TInt CPicturePublisherImpl::CheckedRunError(TInt aError)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::CheckedRunError"));

	return aError;
}

void CPicturePublisherImpl::DoCancel()
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::DoCancel"));

#ifndef __WINS__test
	Fs().NotifyChangeCancel();
#else
	iTimer.Cancel();
#endif
}

void CPicturePublisherImpl::register_source(const TDesC&, const TDesC& initial_value, const TTime&)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::register_source"));

	if (initial_value.Length()>0) iLoc=initial_value;
}

void CPicturePublisherImpl::new_value(log_priority priority, const TDesC&, const TDesC& value, const TTime&)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::new_value"));

	if (priority!=Mlogger::VALUE) iLoc=_L("");
	else iLoc=value;
}

void CPicturePublisherImpl::unregister_source(const TDesC&, const TTime&)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::unregister_source"));

}

const TDesC& CPicturePublisherImpl::name() const
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::name"));

	return KClassName;
}

void CPicturePublisherImpl::finished()
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::finished"));

	if (! iNoPics ) {
		iNotif.status_change(_L("uploaded pics"));
	}
}

void CPicturePublisherImpl::error(const TDesC& descr)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::error"));

	iNotif.error(descr);
}

void CPicturePublisherImpl::status_change(const TDesC& status)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::status_change"));

	iNotif.status_change(status);
}

TOldPrompt::TOldPrompt(bool Delete) : iDelete(Delete)
{
}

void TOldPrompt::Prompt(const TDesC& /*FileName*/, MUploadCallBack* CallBack)
{
	CALLSTACKITEM(_L("TOldPrompt::Prompt"));

	auto_ptr<CXmlBuf> b(CXmlBuf::NewL(50));
	b->Leaf(_L("packet"), _L(""));
	CallBack->Back(true, iDelete, b.get());
}

void CPicturePublisherImpl::PublishOld()
{
	if (!iPathExists) return;

	TTime now(0);
	Transfer(_L(""), iOldPrompt, now);
	Transfer(_L("Old\\"), iOldPrompt, now);
}
