/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "presence_publisher.h"
#include "symbian_auto_ptr.h"
#include "cl_settings.h"
#include "i_logger.h"
#include "cellmap.h"
#include "eikenv.h"
#include "status_notif.h"
#include <context_log.mbg>

#ifndef __WINS__
_LIT(KIconFile, "c:\\system\\data\\context_log.mbm");
#else
_LIT(KIconFile, "z:\\system\\data\\context_log.mbm");
#endif

#pragma warning(disable: 4706) // assignment withing conditional expression

_LIT(KPresencePublisher, "PresencePublisher");

CPresencePublisher* CPresencePublisher::NewL(MApp_context& Context, i_status_notif* CallBack, 
					     MPresencePublisherListener& aListener, CBTDeviceList* aBuddyList)
{
	CALLSTACKITEMSTATIC(_L("CPresencePublisher::NewL"));

	auto_ptr<CPresencePublisher> ret( new (ELeave) CPresencePublisher(Context, CallBack, aListener, aBuddyList) );
	ret->ConstructL();
	return ret.release();
}

CPresencePublisher::~CPresencePublisher()
{
	CALLSTACKITEM(_L("CPresencePublisher::~CPresencePublisher"));

	delete ctx;
	Settings().CancelNotifyOnChange(SETTING_PRESENCE_ENABLE, this);
	Settings().CancelNotifyOnChange(SETTING_CURRENT_AP, this);
	Cancel();
	delete iWait;
	iSession.Close();
	delete iSendPresence;
	delete iPresence;
	delete iPresenceRunning;
}

void CPresencePublisher::SuspendConnection()
{
	iCallBack->status_change(_L("Suspend Presence"));
	CALLSTACKITEM(_L("CPresencePublisher::SuspendConnection"));

	if (iNextOp==EDisable || iNextOp==ESuspend) return;
	iWait->Cancel();

	switch (iCurrentState) {
	case EDisabled:
	case EDisabling:
		break;
	case ESuspended:
	case ESuspending:
		break;
	case EReconnecting:
		iCurrentState=ESuspended;
		break;
	default:
		if (IsActive()) {
			iNextOp=ESuspend;
		} else {
			iSession.MsgSuspendConnection(iStatus);
			SetActive();
			iCurrentState=ESuspending;
		}
		break;
	}
}

void CPresencePublisher::QueueUpdate()
{
	iNewValue=true;
	if (IsActive()) return;
	iCurrentState=EQueuingUpdate;
	TRequestStatus *s=&iStatus;
	User::RequestComplete(s, KErrNone);
	SetActive();
}
	
bool CPresencePublisher::Disabled()
{
	CALLSTACKITEM(_L("CPresencePublisher::Disabled"));

	if (iCurrentState==EDisabling || iCurrentState==EDisabled || iNextOp==EDisable) {
		return true;
	}
	return false;
}

void CPresencePublisher::ResumeConnection()
{
	CALLSTACKITEM(_L("CPresencePublisher::ResumeConnection"));
	iCallBack->status_change(_L("Resume Presence"));

	if (IsActive() && (iNextOp==EResume || iNextOp==EEnable)) return;
	iWait->Cancel();

	switch (iCurrentState) {
	default:
		if (IsActive()) {
			iNextOp=EResume;
		} else {
			Restart();
		}
		break;
	}
}

bool CPresencePublisher::ConnectionSuspended()
{
	CALLSTACKITEM(_L("CPresencePublisher::ConnectionSuspended"));

	if (iCurrentState==ESuspending || iCurrentState==ESuspended || iNextOp==ESuspend ||
		iCurrentState==EDisabling || iCurrentState==EDisabled || iNextOp==EDisable) return true;
	return false;
}

CPresencePublisher::CPresencePublisher(MApp_context& Context, i_status_notif* CallBack , 
				       MPresencePublisherListener& aListener, CBTDeviceList* aBuddyList): 
CCheckedActive(EPriorityIdle, _L("CPresencePublisher")), MContextBase(Context), iCallBack(CallBack), iListener(aListener),
iBuddyList(aBuddyList)

{
	CALLSTACKITEM(_L("CPresencePublisher::CPresencePublisher"));

}

void CPresencePublisher::ConstructL()
{
	CALLSTACKITEM(_L("CPresencePublisher::ConstructL"));

	{
		CALLSTACKITEM(_L("CPresencePublisher::ConstructL::0"));
		TRAPD(err, iPresenceRunning=CNotifyState::NewL(AppContext(), KIconFile));
		if (iPresenceRunning) iPresenceRunning->SetCurrentState(EMbmContext_logP_not, EMbmContext_logP_not);
	}

	{
		CALLSTACKITEM(_L("CPresencePublisher::ConstructL::1"));
		ctx=CApp_context::NewL(true, _L("ContextServer"));
	}

	TInt enabled=1;
	{
		CALLSTACKITEM(_L("CPresencePublisher::ConstructL::2"));
		iWait=CTimeOut::NewL(*this);
		iPresence=CXmlBuf::NewL(256);

		iCurrentState=EDisabled;
		Settings().NotifyOnChange(SETTING_PRESENCE_ENABLE, this);
		Settings().NotifyOnChange(SETTING_CURRENT_AP, this);

		if (! Settings().GetSettingL(SETTING_PRESENCE_ENABLE, enabled) ) {
			enabled=1;
		}
	}

	{
		CALLSTACKITEM(_L("CPresencePublisher::ConstructL::3"));
		CActiveScheduler::Add(this);
		Settings().GetSettingL(SETTING_CURRENT_AP, iAP);
		if (enabled && iAP>0) {
			iWaitTime=1;
			Restart();
		} else {
			iCurrentState=EDisabled;
		}
	}
}

void CPresencePublisher::SettingChanged(TInt /*Setting*/)
{
#ifdef __WINS__
	RDebug::Print(_L("CPresencePublisher::SettingChanged"));
#endif

	CALLSTACKITEM(_L("CPresencePublisher::SettingChanged"));
	//iCallBack->status_change(_L("settingchanged"));

	TInt enabled=1;
	if (! Settings().GetSettingL(SETTING_PRESENCE_ENABLE, enabled) ) {
		enabled=1;
	}
	Settings().GetSettingL(SETTING_CURRENT_AP, iAP);
	if (enabled && iAP>0) {
		iWaitTime=1;
		Restart();
	} else {
		if (iCurrentState==EDisabled || iCurrentState==EDisabling) return;
		iWait->Cancel();
		if (iCurrentState==EReconnecting || iCurrentState==ESuspended) {
			Cancel();
			iCurrentState=EDisabled;
			return;
		}
		if (iCurrentState!=ESuspending) {
			Cancel();
			iSession.MsgSuspendConnection(iStatus);
			iCurrentState=ESuspending;
			SetActive();
		}
		iNextOp=EDisable;
	}
}

// Mlogger
void CPresencePublisher::register_source(const TDesC& name, const TDesC& initial_value, const TTime& time)
{
	CALLSTACKITEM(_L("CPresencePublisher::register_source"));

	if (initial_value.Length()>0) {
		new_value(VALUE, name, initial_value, time);
	}
}

void CPresencePublisher::new_value(log_priority priority, const TDesC& name, const TDesC& value, const TTime& time)
{
	CALLSTACKITEM(_L("CPresencePublisher::new_value"));

	if (priority!=VALUE) return;

	if (! name.Compare(KLog_cellid) ) {
		CCellMap::Parse(value, iCell.iCellId, iCell.iLAC, iCell.iNetwork);
		iCell.iUpdated=time;
	} else if (! name.Compare(KLog_profile) ) {
		TLex16 lex(value);
		lex.Val(iProfile.iProfileId);
		lex.SkipCharacters();
		lex.SkipSpace(); lex.Mark();
		lex.SkipCharacters();
		iProfile.iProfileName=lex.MarkedToken();
		lex.SkipSpace(); lex.Inc(1); lex.SkipSpace();
		lex.Val(iProfile.iRingingType);
		lex.SkipCharacters();
		lex.SkipSpace();
		lex.Val(iProfile.iRingingVolume);
		lex.SkipCharacters();
		lex.SkipSpace(); lex.Mark();
		lex.SkipCharacters();
		TBuf<10> tmp=lex.MarkedToken();
		if (! tmp.Left(2).Compare( _L("On") ) ) {
			iProfile.iVibra=ETrue;
		} else {
			iProfile.iVibra=EFalse;
		}

		iProfile.iUpdated=time;
	} else if (! name.Compare(KLog_activity) ) {
		if ( ! value.Compare(_L("idle")) ) {
			iActivity.iMode=TActivityInfo::EIdle;
		} else {
			iActivity.iMode=TActivityInfo::EActive;
		}
		iActivity.iUpdated=time;
	} else if (! name.Compare(KLog_gps) ) {
		iGpsInfo.iGpsData=value;
		iGpsInfo.iUpdated=time;
	} else if (! name.Compare(KLog_cellname) ) {
		TInt fieldsep;
		TInt beg=0;
		TInt valuesep;
		fieldsep=value.Locate(';');
		if (fieldsep==KErrNotFound) fieldsep=value.Length();

		TInt iPrevFlags=iLocation.iFlags;
		iLocation.iUpdated=time;
		iLocation.iFlags &= TLocInfo::EPrevious;
		while( fieldsep!=KErrNotFound ) {
			valuesep=value.Mid(beg, fieldsep-beg).Locate(':');
			if (valuesep==KErrNotFound) valuesep=beg;
			else valuesep+=2;
			TPtrC16 field=value.Mid(beg, valuesep-beg);
			TPtrC16 fieldval=value.Mid(beg+valuesep, fieldsep-beg-valuesep);

			if (! field.Left(4).Compare(_L("last")) ) {
				if (! (iPrevFlags & TLocInfo::EPrevious) || 
						iLocation.iPrevious->Des().Compare(fieldval) ) {
					delete iLocation.iPrevious; iLocation.iPrevious=0;
					iLocation.iPrevious=fieldval.AllocL();
					iLocation.iPUpdated=time;
				}
				iLocation.iFlags|=TLocInfo::EPrevious;
			} else if (! field.Left(4).Compare(_L("next")) ) {
				if (! (iPrevFlags & TLocInfo::ENext) || 
						iLocation.iNext->Des().Compare(fieldval) ) {
					delete iLocation.iNext; iLocation.iNext=0;
					iLocation.iNext=fieldval.AllocL();
					iLocation.iNUpdated=time;
				}
				iLocation.iFlags|=TLocInfo::ENext;
			} else {
				if (! (iPrevFlags & TLocInfo::ECurrent) || 
						iLocation.iCurrent->Des().Compare(fieldval) ) {
					delete iLocation.iCurrent; iLocation.iCurrent=0;
					iLocation.iCurrent=fieldval.AllocL();
					iLocation.iCUpdated=time;
				}
				iLocation.iFlags|=TLocInfo::ECurrent;
			}
			beg=fieldsep;
			fieldsep=value.Mid(beg).Locate(';');
		}
	} else if (! name.Compare(KLog_bt)) {

		iNeighbourhoodInfo.iBuddies=0;
		iNeighbourhoodInfo.iOtherPhones=0;

		if (! iBluetoothInfo.iNodes ) {
			iBluetoothInfo.iNodes=CList<TBluetoothInfo::TBTNode>::NewL();
		} else {
			iBluetoothInfo.iNodes->reset();
		}
		iBluetoothInfo.iUpdated=time;
		TInt sep;
		TInt beg=0;
		sep=value.Locate(' ');
		TBluetoothInfo::TBTNode node;
		while (sep!=KErrNotFound) {
			node.iAddr=value.Mid(beg).Left(sep);
			beg+=sep+2;
			if (beg>value.Length()) sep=KErrNotFound;
			else sep=value.Mid(beg).Locate(',');
			if (sep==KErrNotFound) break;
			node.iNick=value.Mid(beg).Left(sep);

			{
			beg+=sep+1;
			if (beg>value.Length()) sep=KErrNotFound;
			else sep=value.Mid(beg).Locate(':');
			if (sep==KErrNotFound) break;
			TLex16 lex(value.Mid(beg).Left(sep));
			lex.Val(node.iMajorClass);
			}

			{
			beg+=sep+1;
			if (beg>value.Length()) sep=KErrNotFound;
			else sep=value.Mid(beg).Locate(':');
			if (sep==KErrNotFound) break;
			TLex16 lex2(value.Mid(beg).Left(sep));
			lex2.Val(node.iMinorClass);
			}

			{
			beg+=sep+1;
			if (beg>value.Length()) sep=KErrNotFound;
			else sep=value.Mid(beg).Locate(']');
			if (sep==KErrNotFound) break;
			TLex16 lex3(value.Mid(beg).Left(sep));
			lex3.Val(node.iServiceClass);
			}

			beg+=sep+2;
			if (beg>value.Length()) sep=KErrNotFound;
			else sep=value.Mid(beg).Locate(' ');

			iBluetoothInfo.iNodes->AppendL(node);
			if ( iBuddyList->FindbyReadable(node.iAddr ) !=KErrNotFound ) {
				++iNeighbourhoodInfo.iBuddies;
			} else if ( node.iMajorClass == 2 ) {
				++iNeighbourhoodInfo.iOtherPhones;
			}

		}
	}

	MakePresence();
	if (iCurrentState==EConnected) {
		// wait one turn of the active scheduler
		QueueUpdate();
	} else {
		iNewValue=true;
	}
}

void CPresencePublisher::MakePresence()
{
	CALLSTACKITEM(_L("CPresencePublisher::MakePresence"));

	iPresence->Zero();
	iPresence->BeginElement(_L("presence"));

	MXmlInfo* events[] = { &iCell, &iLocation, &iActivity, &iProfile, &iNeighbourhoodInfo, 0 };

	MXmlInfo* event;
	TBuf<20> buffer;
	for (int i=0; (event=events[i]); i++) {

		iPresence->BeginElement(_L("event"));
		_LIT(KFormatTxt,"%04d%02d%02dT%02d%02d%02d");
		TDateTime dt=event->iUpdated.DateTime();

		buffer.Format(KFormatTxt, dt.Year(), (TInt)dt.Month()+1, (TInt)dt.Day()+1,
			dt.Hour(), dt.Minute(), dt.Second());
		iPresence->Leaf(_L("datetime"), buffer);
		event->Persist(iPresence);
		iPresence->EndElement(_L("event"));
	}
	iPresence->EndElement(_L("presence"));
#ifdef __WINS__
	//RDebug::Print(iPresence->Buf());
#endif

}

void CPresencePublisher::SendUpdate()
{
	CALLSTACKITEM(_L("CPresencePublisher::SendUpdate"));

	delete iSendPresence; iSendPresence=0;
	iSendPresence=iPresence->Buf().AllocL();

	iSession.MsgUpdateUserPresence(*iSendPresence, iStatus);

	SetActive();
	iCurrentState=ESendingUpdate;
	iNewValue=false;

	iSentTimeStamp.HomeTime();

#ifdef __WINS__
	RDebug::Print(_L("CPresencePublisher::SendUpdate()"));
#endif

	//User::Leave(1003);

}

void CPresencePublisher::unregister_source(const TDesC& /*name*/, const TTime& /*time*/)
{
	CALLSTACKITEM(_L("CPresencePublisher::unregister_source"));

}

const TDesC& CPresencePublisher::name() const
{
	CALLSTACKITEM(_L("CPresencePublisher::name"));

	return KPresencePublisher;
}

void CPresencePublisher::Restart()
{
	CALLSTACKITEM(_L("CPresencePublisher::Restart"));

	//iCallBack->status_change(_L("restart"));

	Cancel();
	iSession.Close();
	iCurrentState=EReconnecting;
	iNextOp=ENone;
	iWait->Wait(iWaitTime);
#ifdef __WINS__
	RDebug::Print(_L("CPresencePublisher::Restart()"));
#endif
}

void CPresencePublisher::CheckedRunL()
{
	CALLSTACKITEM(_L("CPresencePublisher::CheckedRunL"));

#ifdef __WINS__
	TBuf<150> msg;
	msg.Format(_L("CPresencePublisher::CheckedRunL %d, state"), iStatus.Int(), iCurrentState);
	RDebug::Print(msg);
	msg.Zero();
#else
	TBuf<50> msg;
#endif

	if (iStatus<0) {
		iWaitTime=(int)(iWaitTime*1.5);

		if(iPresenceRunning) iPresenceRunning->SetCurrentState(EMbmContext_logP_not, EMbmContext_logP_not);
		// depending on the error, we should not always restart
		//Restart();
		if (iStatus == -1 /*EIdentificationError*/) 
		{
			TBuf<100> error;
			error.Append(_L("ID Error: "));
			error.Append(iUser);
			
			iCallBack->error(error);
			CEikonEnv::Static()->AlertWin(error,iPass);
		}
		else 
		{
			msg.Format(_L("Presence error %d, restarting"), iStatus.Int());
			iCallBack->status_change(msg);
			if (iStatus.Int() == KErrServerTerminated) {
				// ContextServer crashed
				auto_ptr<HBufC> stack(0);
				
				TRAPD(err, stack.reset(ctx->GetFormattedCallStack(_L("Presence"))));
				if (err==KErrNone && stack.get()!=0) {
					iCallBack->status_change(*stack);
				}
			}
			
			
			Restart();
		}
		return;
	}

	iWaitTime=10;

	switch (iCurrentState) {
	case EConnecting:
	case EResuming:
		msg=_L("Presence Connected");
		if(iPresenceRunning) iPresenceRunning->SetCurrentState(EMbmContext_logP, EMbmContext_logP);
	case ESendingUpdate:
		iCurrentState=EConnected;
		iListener.NotifyNewPresence(this);
		break;
	case ESuspending:
		msg=_L("Presence Suspended");
		if(iPresenceRunning) iPresenceRunning->SetCurrentState(EMbmContext_logP_not, EMbmContext_logP_not);
		iCurrentState=ESuspended;
		iListener.NotifyNewPresence(this);
		break;
	case EQueuingUpdate:
		iCurrentState=EConnected;
		break;
	default:
		msg=_L("Presence Restarting");
		Restart();
		break;
	}

	if (msg.Length()>0) iCallBack->status_change(msg);

	switch (iNextOp) {
	case ESuspend:
		iSession.MsgSuspendConnection(iStatus);
		SetActive();
		iCurrentState=ESuspending;
		break;
	case EResume:
		iSession.MsgResumeConnection(iStatus);
		SetActive();
		iCurrentState=EResuming;
		break;
	case EDisable:
		iSession.Close();
		iCurrentState=EDisabled;
		break;
	default:
		break;
	};
	iNextOp=ENone;

	if (iNewValue && iCurrentState==EConnected) {
		SendUpdate();
	}
}
		
void CPresencePublisher::DoCancel()
{
	CALLSTACKITEM(_L("CPresencePublisher::DoCancel"));

	iSession.Cancel();
}

TInt CPresencePublisher::CheckedRunError(TInt /*aError*/)
{
	CALLSTACKITEM(_L("CPresencePublisher::CheckedRunError"));

	Restart();
	
	return KErrNone;
}

void CPresencePublisher::expired(CBase*)
{
	CALLSTACKITEM(_L("CPresencePublisher::expired"));

	//iCallBack->status_change(_L("expired"));
	TBuf<50> tmp;

	TInt enabled=1;
	if (! Settings().GetSettingL(SETTING_PRESENCE_ENABLE, enabled) ) {
		enabled=1;
	}
	if (!enabled) {
		iCurrentState=EDisabled;
		return;
	}

	if (Settings().GetSettingL(SETTING_JABBER_NICK, tmp) && Settings().GetSettingL(SETTING_JABBER_PASS, iPass)) {
		TInt sep=tmp.Locate('@');
		if (sep!=KErrNotFound) {
			TInt ret=iSession.ConnectToContextServer();
			if (ret!=KErrNone) {
				TBuf<30> msg;
				msg.Format(_L("ConnectToContextServer %d"), ret);
				iCallBack->error(msg);
				Restart();
				return;
			}
			iUser=tmp.Mid(0, sep);
			iServer=tmp.Mid(sep+1);

			iCallBack->status_change(_L("Presence Connecting"));
			iSession.MsgConnectToPresenceServer(iUser, iPass,
				iServer, iAP, iStatus);
			iCurrentState=EConnecting;
			SetActive();
		}
	}
}

const TCellInfo& CPresencePublisher::CellInfo() const
{
	return iCell;
}

const TGpsInfo& CPresencePublisher::GpsInfo() const
{
	return iGpsInfo;
}

const TLocInfo& CPresencePublisher::LocInfo() const
{
	return iLocation;
}

const TNeighbourhoodInfo& CPresencePublisher::NeighbourhoodInfo() const
{
	return iNeighbourhoodInfo;
}


const TActivityInfo& CPresencePublisher::ActivityInfo() const
{
	return iActivity;
}

const TProfileInfo& CPresencePublisher::ProfileInfo() const
{
	return iProfile;
}

const TTime& CPresencePublisher::SendTimeStamp() const
{
	return iSentTimeStamp;
}

bool CPresencePublisher::IsSent() const
{
	if (iCurrentState==ESendingUpdate || iCurrentState==EConnected) return true;

	return false;
}

const TBluetoothInfo& CPresencePublisher::BluetoothInfo() const
{
	return iBluetoothInfo;
}
