/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "transfer.h"
#include <context_log.rsg>
#include "symbian_auto_ptr.h"
#include <plpvariant.h>
#include "cl_settings.h"
#include "local_defaults.h"
#include <bautils.h>
#include "file_output_base.h"

#include <raii_f32file.h>

#define BASE_BUSY_WAIT	20

#if !defined(__WINS__) && defined(__S60V2__)
#include <etelmm.h>
#endif

#ifdef __WINS__
#define WAIT_PERIOD 1*60
#else
#define WAIT_PERIOD 60*60
#endif

#define UPLOAD_FROM 2
#define UPLOAD_TO 5

//const TTimeIntervalMinutes	KUploadInterval=5;
//const TTimeIntervalMinutes	KUploadIntervalForce=15;

TDummyPrompt::TDummyPrompt(bool Delete) : iDelete(Delete)
{
	CALLSTACKITEM(_L("TDummyPrompt::TDummyPrompt"));
}

void ToPacket(const TDesC& filen, TDes& packet)
{
	CALLSTACKITEM(_L("TDummyPrompt::TDummyPrompt"));

	TInt dirpos=filen.LocateReverse('\\');
	packet=filen.Left(dirpos);
	packet.Append(_L("\\context"));
	packet.Append(filen.Mid(dirpos));
	TInt extpos=packet.LocateReverse('.');
	if (extpos!=KErrNotFound) {
		packet[extpos]='_';
	}
	packet.Append(_L(".xml"));
}

void ToDel(const TDesC& filen, TDes& del)
{
	CALLSTACKITEM(_L("TDummyPrompt::TDummyPrompt"));

	TInt dirpos=filen.LocateReverse('\\');
	del=filen.Left(dirpos);
	del.Append(_L("\\context"));
	del.Append(filen.Mid(dirpos));

	TInt extpos=del.LocateReverse('.');
	if (extpos!=KErrNotFound) {
		del[extpos]='_';
	}
	del.Append(_L(".del"));
}

void TDummyPrompt::Prompt(const TDesC& /*FileName*/, MUploadCallBack* CallBack)
{
	CALLSTACKITEM(_L("TDummyPrompt::Prompt"));

	CallBack->Back(true, iDelete, 0);
}

CTransferBase::CTransferBase(MApp_context& Context, MUploadPrompt& Prompt, bool move_to_mmc) : 
		MContextBase(Context), iPrompt(Prompt), move_to_memory_card(move_to_mmc)
{
	CALLSTACKITEM(_L("CTransferBase::CTransferBase"));

}

CSendUITransfer::CSendUITransfer(MApp_context& Context) : CTransferBase(Context, iDummyPrompt)
{
	CALLSTACKITEM(_L("CSendUITransfer::CSendUITransfer"));
}


void CTransferBase::ConstructL(i_status_notif* callback, const TDesC& dir1, const TDesC& dir2)
{
	CALLSTACKITEM(_L("CTransferBase::ConstructL"));

	send_dirs=new CDesC16ArrayFlat(5);
	send_leave=new CArrayFixFlat<bool>(5);

	TBool use_mmc=ETrue;
	Settings().GetSettingL(SETTING_USE_MMC, use_mmc);
	has_memory_card=false;
	TDriveInfo i;
	if (use_mmc && Fs().Drive(i, EDriveE)==KErrNone) {
		// memory card
		has_memory_card=true;
	}

	dir_prefixes=CList<TFileName>::NewL();
	TFileName prefix;
	if (dir1.Length()) {
		prefix=dir1;
		if (dir1.Right(1).Compare(_L("\\")) ) { prefix.Append(_L("\\")); }
		dir_prefixes->AppendL(prefix);
		if (has_memory_card) {
			prefix.Replace(0, 1, _L("e"));
			dir_prefixes->AppendL(prefix);
		}
	}

	if (dir2.Length()) {
		prefix=dir2;
		if (dir2.Right(1).Compare(_L("\\")) ) { prefix.Append(_L("\\")); }
		dir_prefixes->AppendL(prefix);
		if (has_memory_card) {
			prefix.Replace(0, 1, _L("e"));
			dir_prefixes->AppendL(prefix);
		}
	}

	CList<TFileName>::Node *n=dir_prefixes->iFirst;
	TFileName contextdir;
	while(n) {
		prefix=n->Item;
		if (prefix.Length()) {
			contextdir=prefix;
			contextdir.Append(_L("context\\"));
			TRAPD(err, BaflUtils::EnsurePathExistsL(Fs(), contextdir));
			// ignore errors, since the drive might not exists, of course
			// there might be other errors which will cause a failure
			// in the packet/del files
		}
		n=n->Next;
	}

	cb=callback;
	iCallBack=CTimeOut::NewL(*this);
}

void CSendUITransfer::ConstructL(i_status_notif* callback, TInt cmdid, const TDesC& dir1, const TDesC& dir2)
{
	CALLSTACKITEM(_L("CSendUITransfer::ConstructL"));

	CTransferBase::ConstructL(callback, dir1, dir2);
	sendui=CSendAppUi::NewL(cmdid, NULL);
}

CSendUITransfer* CSendUITransfer::NewL(MApp_context& Context, i_status_notif* callback, TInt cmdid, 
				       const TDesC& dir1, const TDesC& dir2)
{
	CALLSTACKITEM2(_L("CSendUITransfer::NewL"), &Context);

	auto_ptr<CSendUITransfer> ret(new (ELeave) CSendUITransfer(Context));
	ret->ConstructL(callback, cmdid, dir1, dir2);
	return ret.release();
}

CTransferBase::~CTransferBase()
{
	CALLSTACKITEM(_L("CTransferBase::~CTransferBase"));

	delete dir_prefixes;
	delete dir;
	delete iPacket8;
	delete send_dirs;
	delete send_leave;
	delete file_names; delete list_file_names;
	delete iCallBack;
}

CSendUITransfer::~CSendUITransfer()
{
	CALLSTACKITEM(_L("CSendUITransfer::~CSendUITransfer"));

	delete sendui;
}

void CSendUITransfer::DisplayMenuL(CEikMenuPane& aMenuPane)
{
	CALLSTACKITEM(_L("CSendUITransfer::DisplayMenuL"));

	sendui->DisplaySendCascadeMenuL(aMenuPane);
}

void CSendUITransfer::DisplaySendMenuL(CEikMenuPane& aMenuPane, TInt pos)
{
	CALLSTACKITEM(_L("CSendUITransfer::DisplaySendMenuL"));

	//TSendingCapabilities c(0, 100000, 0);
	TSendingCapabilities c(0, 0, 0);
	sendui->DisplaySendMenuItemL(aMenuPane, pos, 
		c);
}

void CTransferBase::add_filesL(const TDesC& file, bool leave_last)
{
	CALLSTACKITEM(_L("CTransferBase::add_filesL"));

	CList<TFileName>::Node *n=dir_prefixes->iFirst;
	TFileName filen;
	while (n) {
		filen=n->Item;
		filen.Append(file);
		send_dirs->AppendL(filen);
		send_leave->AppendL(leave_last);
		n=n->Next;
	}

}

void CTransferBase::FileStep()
{
	CALLSTACKITEM(_L("CTransferBase::FileStep"));

	TEntry fe;
	TBuf<250> msg;

next_step:
	msg=_L("");
	while (dir_i >= dir_count) {
		if (send_dir_i >= send_dirs->Count()) {
			if (again) {
				again=false;
				busy_files=false;
				send_dir_i=0;
			} else {
				GotFiles();
				return;
			}
		}
		dirname=(*send_dirs)[send_dir_i];
		leave=(*send_leave)[send_dir_i];
		send_dir_i++;

		delete dir; dir=0;
		TInt err=Fs().GetDir(dirname, KEntryAttNormal, ESortByName, dir);
		if (err != KErrNone) {
			// msg.Format(_L("error %d getting dir %S"), err, &dirname);
			// cb->status_change(msg);
			goto next_step;
		}
		dir_count=dir->Count();
		if (leave) --dir_count;
		p.Set(dirname, 0, 0);
		dir_i=0;
	}

	filen=p.DriveAndPath();
	filen.Append((*dir)[dir_i].iName);
	TBool use_mmc=ETrue;
	if (move_to_memory_card && has_memory_card && filen.Left(1).CompareF(_L("c"))==0 && 
			Settings().GetSettingL(SETTING_USE_MMC, use_mmc) && use_mmc &&
			(*dir)[dir_i].iName.Left(12).Compare(_L("cellid_names")) ) {
		filen2=filen;
		filen2.Replace(0, 1, _L("e"));
		if (BaflUtils::CopyFile(Fs(), filen, filen2) == KErrNone) {
			BaflUtils::DeleteFile(Fs(), filen);
			filen=filen2;
		}
	}
	dir_i++;
	TInt err=Fs().Entry(filen, fe);
	if (err==KErrNone) {
		if (fe.iSize>0 && fe.IsArchive()) {
			TFileName packet; ToPacket(filen, packet);
			if (BaflUtils::FileExists(Fs(), packet) ) {
				if (file_names) {
					//msg=_L("in progress ");
					//msg.Append(filen);
				} else {
					RFile f;
					TInt err;
					err=f.Open(Fs(), filen, EFileRead|EFileShareAny);
					f.Close();
					if (err==KErrAccessDenied || err==KErrInUse) {
						msg=_L("skipping busy ");
						msg.Append(filen);
						busy_files=true;
					} else {
						list_file_names->AppendL(filen);
						msg=_L("adding file ");
						msg.Append(filen);
					}
				}
			} else {
				if (fe.iModified>iAfterTime) {
					msg=_L("prompting file");
					in_call=true;
					iPrompt.Prompt(filen, this);
					in_call=false;
					return;
				}
			}
		} else if (fe.iSize==0) {
			//msg=_L("size 0");
			Fs().Delete(filen);
		} else {
			//msg=_L("skip file ");
			//msg.Append(filen);
		}
	} else {
		msg.Format(_L("error getting file attr %d"), err);
	}
	if (msg.Length()>0) {
		cb->status_change(msg);
	}
	goto next_step;
}

void CTransferBase::PutIntoFilesL(bool Upload, bool DeleteFromPhone,
		CXmlBuf* Packet, const TDesC& filename)
{
	CALLSTACKITEM(_L("CTransferBase::PutIntoFilesL"));
	if (Upload) {
		if (Packet) {
			TInt len=Packet->Buf().Length()+50;
			if (!iPacket8 || iPacket8->Length() < len) {
				delete iPacket8; iPacket8=0;
				iPacket8=HBufC8::NewL(len);
			} else {
				iPacket8->Des().Zero();
			}
			TPtr8 p=iPacket8->Des();
			CC()->ConvertFromUnicode(p, Packet->Buf());

			TFileName packet; ToPacket(filename, packet);
			{
				RAFile f; f.ReplaceLA(Fs(), packet, EFileWrite|EFileShareAny);
				User::LeaveIfError(f.Write(*iPacket8));
			}
		}
		if (DeleteFromPhone) {
			TFileName del; ToDel(filename, del);
			RAFile f; f.ReplaceLA(Fs(), del, EFileWrite|EFileShareAny);
		}
	} else if (DeleteFromPhone) {
		Fs().Delete(filen);
	} else {
		Fs().SetAtt(filen, 0, KEntryAttArchive);
	}
}

void CTransferBase::DoBackL(bool Upload, bool DeleteFromPhone,
	CXmlBuf* Packet)
{
	CALLSTACKITEM(_L("CTransferBase::DoBackL"));

	PutIntoFilesL(Upload, DeleteFromPhone, Packet, filen);
	if (Upload) {
		RFile f;
		TInt err;
		err=f.Open(Fs(), filen, EFileRead|EFileShareAny);
		f.Close();
		if (err==KErrAccessDenied || err==KErrInUse) {
			busy_files=true;
		} else {
			list_file_names->AppendL(filen);
		}
	}
}

void CTransferBase::Back(bool Upload, bool DeleteFromPhone,
	CXmlBuf* Packet)
{
	CALLSTACKITEM(_L("CTransferBase::Back"));

	TRAPD(err, DoBackL(Upload, DeleteFromPhone, Packet));

	//TODO: report to user
	if (err!=KErrNone) {
		TBuf<50> msg;
		msg.Format(_L("Error %d in Back"), err);
		cb->error(msg);
	}

	if (in_call) {
		iCallBack->Wait(0);
	} else {
		FileStep();
	}
}

void CTransferBase::expired(CBase* source)
{
	CALLSTACKITEM(_L("CTransferBase::expired"));

	if (source==iCallBack) 
		FileStep();
}

void CTransferBase::GetFiles(TTime AfterTime)
{
	CALLSTACKITEM(_L("CTransferBase::GetFiles"));

	if (listing_files) {
		again=true;
		return;
	}
	busy_files=false;

	if (!file_names) file_index=0;
	listing_files=true;

	if (send_dirs->Count()==0) {
		GotFiles();
		return;
	}

	iAfterTime=AfterTime;

	delete list_file_names; list_file_names=0;
	list_file_names=new (ELeave) CDesC16ArrayFlat(10);

	send_dir_i=dir_i=0;
	dir_count=-1; delete dir; dir=0;

	FileStep();
}

bool CSendUITransfer::transfer_files(TInt cmdid)
{
	CALLSTACKITEM(_L("CSendUITransfer::transfer_files"));

	if (in_progress) {
		cb->status_change(_L("busy 1"));
		return false;
	}
	if (file_names) {
		cb->status_change(_L("busy 2"));
		return false;
	}
	if (list_file_names) {
		cb->status_change(_L("busy 3"));
		return false;
	}

#if __WINS__
	TUid MtmUid=sendui->MtmForCommand(cmdid);
	RDebug::Print(MtmUid.Name());
#endif

	iCmdId=cmdid;

	in_progress=true;

	GetFiles();

	return true;
}

void CSendUITransfer::GotFiles()
{
	CALLSTACKITEM(_L("CSendUITransfer::GotFiles"));

	if (listing_files) {
		listing_files=false;
		if (!file_names) {
			file_names=list_file_names;
			list_file_names=0;
		} else {
			return;
		}
	}

	if (!file_names || file_names->Count()==0) {
		delete file_names; file_names=0;
		in_progress=false;
		cb->finished();
		return;
	}

	sendui->CreateAndSendMessageL (iCmdId, 0, file_names);

	CAknQueryDialog* dlg = new(ELeave) CAknQueryDialog(CAknQueryDialog::ENoTone);
	CleanupStack::PushL(dlg);
	
	_LIT(pr, "Delete files?");
	dlg->SetPromptL(pr);

	CleanupStack::Pop(); //dlg

	if (dlg->ExecuteLD(R_CL_CONFIRMATION_QUERY_DIALOG)) {
		if (cb) cb->status_change(_L("deleting files"));
		for (int i=0; i<file_names->Count(); i++) {
			Fs().Delete((*file_names)[i]);
			TFileName d;
			ToDel((*file_names)[i], d); Fs().Delete(d);
			ToPacket((*file_names)[i], d); Fs().Delete(d);

		}
		if (cb) cb->status_change(_L("deleted files"));
	}

	delete file_names; file_names=0;
	in_progress=false;
	cb->finished();

	return;
}

void CFtpTransfer::expired(CBase* source)
{
	CALLSTACKITEM(_L("CFtpTransfer::expired"));

	if (source==iBusyTimer) {
		iBusyRetryWait=(TInt)(iBusyRetryWait*1.5);
		iBusyRetryCount++;
		TransferFiles(iMeta, iAfterTime, iUrlBase);
	} else {
		CTransferBase::expired(source);
	}
}

CFtpTransfer* CFtpTransfer::NewL(MApp_context& Context, 
	i_status_notif* callback, const TDesC& dir1, const TDesC& dir2,
		TConnType ConnType, MUploadPrompt& Prompt, bool move_to_mmc)
{
	CALLSTACKITEM2(_L("CFtpTransfer::NewL"), &Context);

	auto_ptr<CFtpTransfer> ret(new (ELeave) CFtpTransfer(Context, Prompt, move_to_mmc));
	ret->ConstructL(callback, dir1, dir2, ConnType);
	return ret.release();
}

void CFtpTransfer::ConstructL(i_status_notif* callback, const TDesC& dir1, const TDesC& dir2,
		TConnType ConnType)
{
	CALLSTACKITEM(_L("CFtpTransfer::ConstructL"));

	iBusyTimer=CTimeOut::NewL(*this);
	iBusyRetryWait=BASE_BUSY_WAIT;
	CTransferBase::ConstructL(callback, dir1, dir2);
	iConnType=ConnType;
}

CFtpTransfer::CFtpTransfer(MApp_context& Context, MUploadPrompt& Prompt, bool move_to_mmc) : CTransferBase(Context, Prompt, move_to_mmc)
{
	CALLSTACKITEM(_L("CFtpTransfer::CFtpTransfer"));

	current_state=IDLE;
}

void CFtpTransfer::TransferFiles(const TDesC& Meta, TTime AfterTime, const TDesC& aUrlBase, const TDesC& aScript)
{
	CALLSTACKITEM(_L("CFtpTransfer::TransferFiles"));

	iBusyTimer->Reset();
	iMeta=Meta;

	if (aUrlBase.Length()==0) {
		if (! Settings().GetSettingL(SETTING_UPLOAD_URLBASE, iUrlBase)) {
			User::Leave(-1026);
		}
	} else {
		iUrlBase=aUrlBase;
	}
	if (aScript.Length()==0) {
		if (! Settings().GetSettingL(SETTING_UPLOAD_SCRIPT, iScript)) {
			User::Leave(-1026);
		}
	} else {
		iScript=aScript;
	}
	GetFiles(AfterTime);
}

void CFtpTransfer::GotFiles()
{
	CALLSTACKITEM(_L("CFtpTransfer::GotFiles"));

	if (busy_files) {
		if (iBusyRetryCount>=10) {
			cb->status_change(_L("giving up on busy files"));
			iBusyTimer->Reset();
			iBusyRetryWait=BASE_BUSY_WAIT;
			iBusyRetryCount=0;
		} else {
			cb->status_change(_L("waiting on busy files"));
			iBusyTimer->Wait(iBusyRetryWait);
		}
	} else {
		iBusyTimer->Reset();
		iBusyRetryWait=BASE_BUSY_WAIT;
		iBusyRetryCount=0;
	}

	if (listing_files) {
		listing_files=false;
		if (!file_names) {
			file_names=list_file_names;
			list_file_names=0;
			file_index=0;
		} else {
			return;
		}
	}

	delete http; http=0;

	if (!file_names || file_names->Count()==0 || file_index>=file_names->Count()) {
		delete file_names; file_names=0;
		current_state=IDLE;
		cb->finished();
		return;
	}

	TInt Iap;
	if (!Settings().GetSettingL(SETTING_CURRENT_AP, Iap) || Iap<0) {
		cb->error(_L("no access point set"));
		delete file_names; file_names=0;
		current_state=IDLE;
		return;
	}

#ifndef __WINS__
#ifndef __S60V2__
	RBasicGsmPhone::TRegistrationStatus r;
	Phone().GetNetworkRegistrationStatus(r);
	if (r!=RBasicGsmPhone::ERegisteredOnHomeNetwork) {
#else
	RMobilePhone phone;
	RTelServer::TPhoneInfo info;
	RMobilePhone::TMobilePhoneRegistrationStatus r=RMobilePhone::ERegistrationUnknown;
	if (TelServer().GetPhoneInfo( 0, info )==KErrNone) {
		if (phone.Open( TelServer(), info.iName )==KErrNone) {
			CleanupClosePushL(phone);
			TRequestStatus s;
			phone.GetNetworkRegistrationStatus(s, r);
			User::WaitForRequest(s);
			CleanupStack::PopAndDestroy();
		}
	}
	if (r!=RMobilePhone::ERegisteredOnHomeNetwork) {
#endif
		cb->error(_L("not allowed while roaming"));
		delete file_names; file_names=0;
		current_state=IDLE;
		return;
	}
#endif


	http=CHttp::NewL(AppContext(), *this);
	current_state=CONNECTING;

	TBuf<200> url=iUrlBase;
	TBuf<50> script=iScript;
	url.Append(script);

	TBuf<50> Proxy;
	Settings().GetSettingL(SETTING_PROXY, Proxy);
	TInt	ProxyPort;
	Settings().GetSettingL(SETTING_PROXY_PORT, ProxyPort);
	http->Connect(Iap, url, Proxy, ProxyPort);
}

CFtpTransfer::~CFtpTransfer()
{
	CALLSTACKITEM(_L("CFtpTransfer::~CFtpTransfer"));

	delete iBusyTimer;
	delete iFtpPacket8;
	delete http;
}

void CFtpTransfer::GetFileInfoL(HBufC8*& this_packet, TDes& datetime)
{
	CALLSTACKITEM(_L("CFtpTransfer::GetFileInfoL"));

	TFileName filen=(*file_names)[file_index];

	TFileName del; ToDel(filen, del);
	if (BaflUtils::FileExists(Fs(), del)) {
		iDeleteThisFile=true;
	} else {
		iDeleteThisFile=false;
	}

	TEntry fe;
	User::LeaveIfError(Fs().Entry(filen, fe));
	format_datetime(datetime, fe.iModified);

	TFileName packet; ToPacket(filen, packet);
	if (BaflUtils::FileExists(Fs(), packet)) {
		User::LeaveIfError(Fs().Entry(packet, fe));
		if (fe.iSize!=0) {
			if (!iFtpPacket8 || iFtpPacket8->Des().MaxLength() < fe.iSize) {
				delete iFtpPacket8; iFtpPacket8=0;
				iFtpPacket8=HBufC8::NewL(fe.iSize);
			} else {
				iFtpPacket8->Des().Zero();
			}
			{
				RAFile f; f.OpenLA(Fs(), packet, EFileRead|EFileShareAny);
				TPtr8 ptr=iFtpPacket8->Des();
				User::LeaveIfError(f.Read(ptr, fe.iSize));
			}
		} else { 
			if (!iFtpPacket8) iFtpPacket8=HBufC8::NewL(2048);
			else iFtpPacket8->Des().Zero();
		}
		this_packet=iFtpPacket8;
	} else {
		this_packet=0;
	}
}

void CFtpTransfer::SendFile()
{
	CALLSTACKITEM(_L("CFtpTransfer::SendFile"));

	current_state=SENDING_FILE;
	HBufC8* this_packet=0;

	TBuf<30> dt;
	TRAPD(err, GetFileInfoL(this_packet, dt));
	if (iMeta.Length()==0) dt.Zero();

	if (err==KErrNone) {
		TBuf<40> msg;
		msg.Format(_L("Sending %d"), file_index);
		cb->status_change(msg);

		http->Store((*file_names)[file_index], dt, this_packet);
	} else {
		iDeleteThisFile=false;
		success(0);
	}
}

void CFtpTransfer::success(CBase* /*source*/)
{
	CALLSTACKITEM(_L("CFtpTransfer::success"));

	TBuf<40> msg;
again:
	switch(current_state) {
	case IDLE:
		delete file_names; file_names=0;
		cb->error(_L("inconsistent state"));
		break;
	case CWDING:
		msg.Format(_L("Sending %d"), file_index);
		cb->status_change(msg);
		SendFile();
		break;
	case SENDING_FILE:
		if (iDeleteThisFile) {
			TFileName d;
			Fs().Delete((*file_names)[file_index]);
			ToDel((*file_names)[file_index], d); Fs().Delete(d);
			ToPacket((*file_names)[file_index], d); Fs().Delete(d);
		} else {
			TFileName d;
			Fs().SetAtt((*file_names)[file_index], 0, KEntryAttArchive);
			ToPacket((*file_names)[file_index], d); Fs().Delete(d);
		}
		++file_index;
		if (file_index<file_names->Count()) {
			msg.Format(_L("Sending %d"), file_index);
			cb->status_change(msg);
			SendFile();
		} else {
			if (list_file_names && list_file_names->Count()>0 && ! listing_files) {
				file_index=0;
				delete file_names; file_names=0;
				file_names=list_file_names; list_file_names=0;
				msg.Format(_L("Sending next %d"), file_index);
				cb->status_change(msg);
				SendFile();
			} else {
				cb->status_change(_L("closing"));
				current_state=CLOSING;
				if (http) http->Close();
			}
		}
		break;
	case CLOSING:
		delete http; http=0;
		current_state=IDLE;
		delete file_names; file_names=0;
		if (list_file_names && ! listing_files) {
			GotFiles();
		} else {
			cb->finished();
		}
		break;
	case CONNECTING:
		if (GetImsi(iImsi)) {
			http->Cwd(iImsi);
			current_state=CWDING;
			goto again;
		} else {
			current_state=CLOSING;
			if (http) http->Close();
			delete file_names; file_names=0;
			cb->error(_L("Cannot get IMSI"));
		}
		break;
	default:
		break;
	}

}

bool CFtpTransfer::GetImsi(TDes& aImsi)
{
	CALLSTACKITEM(_L("CFtpTransfer::GetImsi"));

#ifndef __WINS__
	bool success=false;
#ifndef __S60V2__
#  ifndef NO_ETELAGSM_H
	MAdvGsmPhoneInformation::TSubscriberId i;
	if (Phone().GetSubscriberId(i)==KErrNone) {
		aImsi=i;
		success=true;
	}
#  endif
#else
	RMobilePhone phone;
	RTelServer::TPhoneInfo info;
	if (TelServer().GetPhoneInfo( 0, info )==KErrNone) {
		if (phone.Open( TelServer(), info.iName )==KErrNone) {
			CleanupClosePushL(phone);
			RMobilePhone::TMobilePhoneSubscriberId i;
			TRequestStatus r;
			phone.GetSubscriberId(r, i);
			User::WaitForRequest(r);
			if (r==KErrNone) {
				aImsi=i;
				success=true;
			}
			CleanupStack::PopAndDestroy();
		}
	}

#endif
	if (!success) {
		TPlpVariantMachineId machineId;
		PlpVariant::GetMachineIdL(machineId);
		aImsi=machineId;
	}
#else
	// Return a fake IMEI when working on emulator
	_LIT(KEmulatorImsi, "244050000000000");
	aImsi.Copy(KEmulatorImsi);
#endif
	return true;
}

void CFtpTransfer::error(CBase* source, TInt /*code*/, const TDesC& reason)
{
	CALLSTACKITEM(_L("CFtpTransfer::error"));

	if (current_state==MKDING) {
		// probably exists already, let's try CWDing
		success(source);
	} else if (current_state==SENDING_FILE) {
		// let's try the next file
		file_index++;
		GotFiles();
	} else {
		delete file_names; file_names=0;
		cb->error(reason);
		delete http; http=0;
		current_state=IDLE;
	}
}

void CFtpTransfer::info(CBase* /*source*/, const TDesC& msg)
{
	CALLSTACKITEM(_L("CFtpTransfer::info"));

	cb->status_change(msg);
}

CPeriodicTransfer* CPeriodicTransfer::NewL(MApp_context& Context, int hours, MSocketObserver* callback)
{
	CALLSTACKITEM2(_L("CPeriodicTransfer::NewL"), &Context);

	auto_ptr<CPeriodicTransfer> ret(new (ELeave) 
		CPeriodicTransfer(Context, hours, callback));
	ret->ConstructL();
	return ret.release();
}

CPeriodicTransfer::~CPeriodicTransfer()
{
	CALLSTACKITEM(_L("CPeriodicTransfer::~CPeriodicTransfer"));

	delete iTimer;
	delete iFtp;
	delete iCommLog;
}

CPeriodicTransfer::CPeriodicTransfer(MApp_context& Context, int hours, MSocketObserver* callback) : 
MContextBase(Context), iHours(hours), iCb(callback)
{
	CALLSTACKITEM(_L("CPeriodicTransfer::CPeriodicTransfer"));

}


void CPeriodicTransfer::ConstructL()
{
	CALLSTACKITEM(_L("CPeriodicTransfer::ConstructL"));

	iTimer=CTimeOut::NewL(*this);
	iFtp=CFtpTransfer::NewL(AppContext(), this, DataDir(), AppDir(), CFtpTransfer::HTTP, iDummyPrompt);
	iFtp->add_filesL(_L("log*txt"), true);
	iFtp->add_filesL(_L("book*txt"), true);
	iFtp->add_filesL(_L("calllog*txt"), true);
	iFtp->add_filesL(_L("starter*txt"), true);
	iFtp->add_filesL(_L("comm*txt"), false);
	iFtp->add_filesL(_L("rec*amr"), false);
	iFtp->add_filesL(_L("cellid_names.txt"), false);
	iCommLog=Clog_comm::NewL(AppContext(), this);
	current_state=IDLE;
	iTimer->Wait(WAIT_PERIOD);
}

void CPeriodicTransfer::expired(CBase*)
{
	CALLSTACKITEM(_L("CPeriodicTransfer::expired"));

const TTimeIntervalHours	KUploadInterval=22;
const TTimeIntervalHours	KUploadIntervalForce=72;

	TTime now;
	now.HomeTime();

	TInt hour=now.DateTime().Hour();
	TTime prev_sending(0);
	Settings().GetSettingL(SETTING_LAST_COMMLOG_UPLOAD, prev_sending);

	if (++iCount>=iHours || (iHours==24 && (
			(hour>=UPLOAD_FROM && hour<UPLOAD_TO && prev_sending+KUploadInterval<now) ||
			(prev_sending+KUploadIntervalForce<now) ) )
			) {
		RDebug::Print(_L("Periodic transfer!"));
		Transfer();
	} else {
		iTimer->Wait(WAIT_PERIOD);
	}
}

void CPeriodicTransfer::Transfer(bool MakeCommLog)
{
	CALLSTACKITEM(_L("CPeriodicTransfer::Transfer"));

	iTimer->Reset();
	iCb->info(this, _L("Getting comm log"));
	current_state=GETTING_COMM_LOG;
	if (MakeCommLog && ! NoSpaceLeft()) {
		iCommLogFrom=0;
		TTime existing(0);
		TRAPD(err, Settings().GetSettingL(SETTING_LAST_COMMLOG_UPLOAD, iCommLogFrom));
		iCommLog->LogsExist(existing);
		if (existing>iCommLogFrom) iCommLogFrom=existing;
		if (! iCommLog->write_comm_log(iCommLogFrom)) {
			finished();
		}
	} else {
		finished();
	}
}

void CPeriodicTransfer::finished()
{
	CALLSTACKITEM(_L("CPeriodicTransfer::finished"));

	if (current_state==GETTING_COMM_LOG) {
		TBool enabled;
		if (! Settings().GetSettingL(SETTING_LOG_UPLOAD_ENABLE, enabled) || ! enabled) {
			iCb->info(this, _L("not uploading"));
		} else {
			iCb->info(this, _L("FTPing"));
			current_state=FTPING;		
			iFtp->TransferFiles();
			return;
		}
	} else {
		iCb->info(this, _L("Transferred"));
	}
	iCount=0;
	current_state=IDLE;
	TRAPD(err, Settings().WriteSettingL(SETTING_LAST_COMMLOG_UPLOAD, iCommLogFrom));
	iTimer->Wait(WAIT_PERIOD);
}

void CPeriodicTransfer::error(const TDesC& reason)
{
	CALLSTACKITEM(_L("CPeriodicTransfer::error"));

	current_state=IDLE;
	iTimer->Wait(WAIT_PERIOD);
	iCb->error(this, 1, reason);
}

void CPeriodicTransfer::status_change(const TDesC& msg)
{
	iCb->info(this, msg);
}

class CMultiPromptImpl : public CMultiPrompt, public MSettingListener, public MContextBase {
private:
	virtual void AddPromptL(TInt Code, MUploadPrompt* Prompt);
	CMultiPromptImpl(MApp_context& Context);
	void ConstructL();

	virtual void Prompt(const TDesC& FileName, MUploadCallBack* CallBack);

	virtual void SettingChanged(TInt Setting);

	struct TPromptItem {
		MUploadPrompt	*iPrompt;
		TInt		iCode;

		TPromptItem() : iPrompt(0), iCode(-1) { }
		TPromptItem(TInt aCode, MUploadPrompt *aPrompt) : iPrompt(aPrompt), iCode(aCode) { }
	};
	CList<TPromptItem>	*iPrompts;
	TInt			iCurrentCode;
	MUploadPrompt		*iCurrentPrompt;

	friend class CMultiPrompt;

public:
	virtual ~CMultiPromptImpl();
};

CMultiPrompt* CMultiPrompt::NewL(MApp_context& Context)
{
	auto_ptr<CMultiPromptImpl> ret(new (ELeave) CMultiPromptImpl(Context));
	ret->ConstructL();
	return ret.release();
}

CMultiPrompt::~CMultiPrompt()
{
}

CMultiPromptImpl::CMultiPromptImpl(MApp_context& Context) : MContextBase(Context)
{
}

CMultiPromptImpl::~CMultiPromptImpl()
{
	Settings().CancelNotifyOnChange(SETTING_UPLOAD_PROMPT_TYPE, this);
	delete iPrompts;
}

void CMultiPromptImpl::ConstructL()
{
	iPrompts=CList<TPromptItem>::NewL();
	Settings().GetSettingL(SETTING_UPLOAD_PROMPT_TYPE, iCurrentCode);
	Settings().NotifyOnChange(SETTING_UPLOAD_PROMPT_TYPE, this);
}

void CMultiPromptImpl::AddPromptL(TInt Code, MUploadPrompt* Prompt)
{
	iPrompts->AppendL( TPromptItem(Code, Prompt) );
	if (Code==iCurrentCode) iCurrentPrompt=Prompt;
}

void CMultiPromptImpl::SettingChanged(TInt /*Setting*/)
{
	Settings().GetSettingL(SETTING_UPLOAD_PROMPT_TYPE, iCurrentCode);
	CList<TPromptItem>::Node *i=iPrompts->iFirst;
	while (i) {
		if (i->Item.iCode==iCurrentCode) {
			iCurrentPrompt=i->Item.iPrompt;
			break;
		}
		i=i->Next;
	}
}

void CMultiPromptImpl::Prompt(const TDesC& FileName, MUploadCallBack* CallBack)
{
	if (iCurrentPrompt) {
		iCurrentPrompt->Prompt(FileName, CallBack);
	} else {
		CallBack->Back(false, false, 0);
	}
}