#include "transfer2.h"

#include <db.h>
#include <symbian_auto_ptr.h>
#include "http.h"
#include "cl_settings.h"
#include <bautils.h>
#include <file_output_base.h>
#include <plpvariant.h>
#include "transfer.h"

#include <raii_f32file.h>

_LIT(KUploadDir, "ToBeUploaded\\");
_LIT(KNoUploadDir, "NotShared\\");
_LIT(KUploadedDir, "Shared\\");
_LIT(KOldDir, "Old\\");

class CHttpTransfer2Impl : public CHttpTransfer2, public MContextBase, public MDBStore, public MSocketObserver {
private:
	// CHttpTransfer2
	virtual void AddFileToQueueL(const TDesC& aFileName, TInt UrlSetting,
			TInt ScriptSetting, TBool DeleteAfter, const TDesC& aMeta,
			CXmlBuf* aPacket, TInt aWaitTime=0);
	virtual TInt DeleteFile(const TDesC& aFileName);
	virtual TInt DoNotTransfer(const TDesC& aFileName);
	virtual TInt MoveToOld(const TDesC& aFileName);
	virtual TBool FileIsQueued(const TDesC& aFileName);
	virtual void Trigger(TBool Force=EFalse);

	CHttpTransfer2Impl(MApp_context& Context,
		RDbDatabase& Db, MSocketObserver* aStatusNotif);
	void ConstructL();

	friend class CHttpTransfer2;

	// CheckedActive
	void CheckedRunL();
	TInt CheckedRunError(TInt aError);
	void DoCancel();

	// MSocketObserver
	virtual void success(CBase* source);
	virtual void error(CBase* source, TInt code, const TDesC& reason);
	virtual void info(CBase* source, const TDesC& msg);

	// internal
	void StartFileTransferL();
	void ConnectL();
	void HandleSent();
	TBool MoveToCurrent();
	void Close();
	void GetFileInfoL(const TDesC& aFileName, const TDesC& PacketFileName, HBufC8*& this_packet, TDes& datetime);
	void Reset();
	void MakeFileName();
	void AddFileToQueueL(const TDesC& aFileName, TInt UrlSetting,
			TInt ScriptSetting, TBool DeleteAfter, const TDesC& aMeta,
			const TDesC& PacketFileName, TInt aWaitTime);

	enum state { IDLE, CONNECTING, MKDING, CWDING, SENDING_FILE, CLOSING };
	state current_state;
	enum TOp { UPLOAD, DELETE, DONTSHARE, MOVETOOLD };

	void QueueOpL(const TDesC& aFileName, TOp Op);
	void DoNotTransferL(const TDesC& aFileName);
	void MoveToOldL(const TDesC& aFileName);

	CHttp*		iHttp;
	TFileName	iTempFileName, iPacketFileName;
	TDbBookmark	iCurrentFileMark;
	MSocketObserver* iStatusNotif;
	RTimer		iTimer;
	TInt		iWaitTime;
	HBufC8*		iPacket8;
	TInt		iRetries;
	TBuf<20>	iImsi;
	TBuf<20>	iDT, iPrevDT;
	TBuf<30>	iFileNameBuf;
	TInt		iDTCount;
	TInt		iFileNameIndexLen;
	bool		iNoTransfer;
public:
	~CHttpTransfer2Impl();
};


CHttpTransfer2* CHttpTransfer2::NewL(MApp_context& Context,
	RDbDatabase& Db, MSocketObserver* aStatusNotif)
{
	CALLSTACKITEM(_L("CHttpTransfer2::NewL"));


	auto_ptr<CHttpTransfer2Impl> ret(new (ELeave) CHttpTransfer2Impl(Context, Db, aStatusNotif));
	ret->ConstructL();
	return ret.release();
}

CHttpTransfer2::CHttpTransfer2() : CCheckedActive(EPriorityLow, _L("CHttpTransfer2"))
{
	CALLSTACKITEM(_L("CHttpTransfer2::CHttpTransfer2"));

}

CHttpTransfer2::~CHttpTransfer2()
{
	CALLSTACKITEM(_L("CHttpTransfer2::~CHttpTransfer2"));

}


void CHttpTransfer2Impl::MakeFileName()
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::MakeFileName"));

	TTime now; now.HomeTime();
	format_datetime(iDT, now);
	iDT.Trim();

	iFileNameBuf=iDT;
	if (! (iDT.Compare(iPrevDT)) ) {
		++iDTCount;
		iFileNameBuf.Append(_L("-"));
		iFileNameBuf.AppendNum(iDTCount);
	} else {
		iDTCount=0;
	}
	iPrevDT=iDT;
}

void CHttpTransfer2Impl::AddFileToQueueL(const TDesC& aFileName, TInt UrlSetting,
	TInt ScriptSetting, TBool DeleteAfter, const TDesC& aMeta,
	const TDesC& PacketFileName, TInt aWaitTime) 
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::AddFileToQueueL::file"));

	iTable.Cancel();

	TParse p; p.Set(aFileName, 0, 0);

	iTempFileName = p.DriveAndPath();
	if (! iTempFileName.Right( KOldDir().Length() ).Compare( KOldDir )) {
		iTempFileName = p.DriveAndPath().Left(p.DriveAndPath().Length()-KOldDir().Length());
	}
	iTempFileName.Append(KUploadDir);
	BaflUtils::EnsurePathExistsL(Fs(), iTempFileName);
	iTempFileName.Append(iFileNameBuf);
	iTempFileName.Append(p.Ext());

	iTable.InsertL();
	iTable.SetColL(1, aFileName);
	iTable.SetColL(2, UrlSetting);
	iTable.SetColL(3, ScriptSetting);
	iTable.SetColL(4, DeleteAfter);
	iTable.SetColL(5, aMeta.Left(KMaxFileName));
	iTable.SetColL(6, PacketFileName);
	iTable.SetColL(7, UPLOAD);
	PutL();

	if (Fs().Rename(aFileName, iTempFileName)==KErrNone) {
		iTable.UpdateL();
		iTable.SetColL(1, iTempFileName);
		PutL();
	}

	iWaitTime=aWaitTime;
}

void CHttpTransfer2Impl::AddFileToQueueL(const TDesC& aFileName, TInt UrlSetting,
	TInt ScriptSetting, TBool DeleteAfter, const TDesC& aMeta,
	CXmlBuf* aPacket, TInt aWaitTime)
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::AddFileToQueueL::packet"));


	{
	TParse p; p.Set(aFileName, 0, 0);

	MakeFileName();

	RDebug::Print(aFileName);
	if (aPacket) {
		TInt len=aPacket->Buf().Length()+50;
		auto_ptr<HBufC8> packet8(HBufC8::NewL(len));
		TPtr8 pp=packet8->Des();
		CC()->ConvertFromUnicode(pp, aPacket->Buf());

		iPacketFileName = p.DriveAndPath();
		RDebug::Print(iPacketFileName);
		if (! iPacketFileName.Right( KOldDir().Length() ).Compare( KOldDir )) {
			iPacketFileName = p.DriveAndPath().Left(p.DriveAndPath().Length()-KOldDir().Length());
		}
		iPacketFileName.Append(KUploadDir);
		BaflUtils::EnsurePathExistsL(Fs(), iPacketFileName);
		iPacketFileName.Append(iFileNameBuf);
		iPacketFileName.Append(_L(".xml"));

		RAFile f; f.ReplaceLA(Fs(), iPacketFileName, EFileWrite|EFileShareAny);
		User::LeaveIfError(f.Write(*packet8));
	} else {
		iPacketFileName.Zero();
	}
	}

	AddFileToQueueL(aFileName, UrlSetting, ScriptSetting, DeleteAfter, aMeta, iPacketFileName, aWaitTime);

	Trigger();
}

void CHttpTransfer2Impl::QueueOpL(const TDesC& aFileName, TOp Op)
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::QueueOpL"));

	iTable.Cancel();

	iTable.InsertL();
	iTable.SetColL(1, aFileName);
	iTable.SetColL(2, 0);
	iTable.SetColL(3, 0);
	iTable.SetColL(4, 0);
	iTable.SetColL(5, _L(""));
	iTable.SetColL(6, _L(""));
	iTable.SetColL(7, Op);
	PutL();

	iWaitTime=5;
	Trigger();
}

TInt CHttpTransfer2Impl::DeleteFile(const TDesC& aFileName)
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::DeleteFile"));


	if(Fs().Delete(aFileName)==KErrNone) return KErrNone;

	TRAPD(err, QueueOpL(aFileName, DELETE));
	return err;
}

void CHttpTransfer2Impl::DoNotTransferL(const TDesC& aFileName)
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::DoNotTransferL"));

	TFileName filen;
	TParse p; p.Set(aFileName, 0, 0);
	filen = p.DriveAndPath();
	if (! filen.Right( KOldDir().Length() ).Compare( KOldDir )) {
		filen = p.DriveAndPath().Left(p.DriveAndPath().Length()-KOldDir().Length());
	}
	filen.Append(KNoUploadDir);
	BaflUtils::EnsurePathExistsL(Fs(), filen);

	MakeFileName();
	filen.Append(iFileNameBuf);
	filen.Append(p.Ext());

	User::LeaveIfError(Fs().Rename(aFileName, filen));
	RDebug::Print(_L("moved %S to %S"), &aFileName, &filen);
}

TInt CHttpTransfer2Impl::DoNotTransfer(const TDesC& aFileName)
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::DoNotTransfer"));


	TInt err;
	TRAP(err, DoNotTransferL(aFileName));
	if (err==KErrNone) return err;
	TRAP(err, QueueOpL(aFileName, DONTSHARE));
	return err;
}

void CHttpTransfer2Impl::MoveToOldL(const TDesC& aFileName)
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::MoveToOldL"));

	TFileName fn;
	TBool del=EFalse;
	ToDel(aFileName, fn);
	if (BaflUtils::FileExists(Fs(), fn)) {
		del=ETrue;
	}
	ToPacket(aFileName, fn);
	if (BaflUtils::FileExists(Fs(), fn)) {
		// compatibility with old upload
		MakeFileName();
		TRAPD(err,
			AddFileToQueueL(aFileName, SETTING_PUBLISH_URLBASE, 
			SETTING_PUBLISH_SCRIPT, del, _L("unknown"), fn, 0));
		if (err==KErrNone) {
			ToDel(aFileName, fn);
			Fs().Delete(fn);
		}
		User::LeaveIfError(err);
		return;
	}
	TParse p; p.Set(aFileName, 0, 0);
	fn = p.DriveAndPath();
	if (! fn.Right( KOldDir().Length() ).Compare( KOldDir ) ) {
		return;
	}
	fn.Append(KOldDir);
	BaflUtils::EnsurePathExistsL(Fs(), fn);
	MakeFileName();
	fn.Append(iFileNameBuf);
	fn.Append(p.Ext());

	User::LeaveIfError(Fs().Rename(aFileName, fn));
	RDebug::Print(_L("moved %S to %S"), &aFileName, &fn);
}

TInt CHttpTransfer2Impl::MoveToOld(const TDesC& aFileName)
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::MoveToOld"));

	TInt err;
	TRAP(err, MoveToOldL(aFileName));
	if (err==KErrNone) return err;
	TRAP(err, QueueOpL(aFileName, MOVETOOLD));
	return err;
}

CHttpTransfer2Impl::CHttpTransfer2Impl(MApp_context& Context,
				       RDbDatabase& Db, MSocketObserver* aStatusNotif) : 
	MContextBase(Context), MDBStore(Db), iStatusNotif(aStatusNotif)
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::CHttpTransfer2Impl"));


}

void CHttpTransfer2Impl::ConstructL()
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::ConstructL"));

	iFileNameIndexLen=50;

	TInt cols[] = { EDbColText, EDbColInt32, EDbColInt32, EDbColBit, EDbColText, EDbColText, EDbColInt32, -1 };
	TInt idx[] = { 1, -1 };

#ifndef __WINS__
	TPlpVariantMachineId machineId;
	PlpVariant::GetMachineIdL(machineId);
	iImsi=machineId;
#else
	// Return a fake IMEI when working on emulator
	_LIT(KEmulatorImsi, "244050000000000");
	iImsi.Copy(KEmulatorImsi);
#endif
	iTimer.CreateLocal();

	SetTextLen(KMaxFileName);

	MDBStore::ConstructL(cols, idx, false, _L("TRANSFER2"), ETrue);

	CActiveScheduler::Add(this);

	Trigger();
}

void CHttpTransfer2Impl::CheckedRunL()
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::CheckedRunL"));

	iTable.Cancel();

	if (current_state==IDLE) {
		if (iTable.FirstL()) {
			iCurrentFileMark=iTable.Bookmark();
			ConnectL();
			iWaitTime=0;
		}
	} else if (current_state==SENDING_FILE) {
		StartFileTransferL();
	}
}

TInt CHttpTransfer2Impl::CheckedRunError(TInt aError)
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::CheckedRunError"));

	if (iStatusNotif) {
		if (iLatestErrorStack)
			iStatusNotif->error(this, aError, *iLatestErrorStack);
		else
			iStatusNotif->error(this, aError, _L("http2runl"));
	}
	if (iWaitTime==0) {
		iWaitTime=10;
	} else {
		iWaitTime=(TInt)(iWaitTime*1.5);
	}
	Reset();
	Trigger();
	return KErrNone;
}

void CHttpTransfer2Impl::DoCancel()
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::DoCancel"));

	iTimer.Cancel();
}

void CHttpTransfer2Impl::success(CBase* /*source*/)
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::success"));

	iTable.Cancel();

	switch(current_state) {
	case CONNECTING:
		iHttp->Cwd(iImsi);
		StartFileTransferL();
		break;
	case SENDING_FILE:
		{
		if (iStatusNotif) iStatusNotif->success(this);
		if (iStatusNotif) iStatusNotif->info(this, _L("transfered file"));

		HandleSent();
		if (iTable.NextL() || iTable.FirstL()) {
			iCurrentFileMark=iTable.Bookmark();
			Trigger(ETrue);
		} else {
			if (iStatusNotif) iStatusNotif->info(this, _L("all files transfered"));
			Close();
		}
		}
		break;
	case CLOSING:
		Reset();
		iWaitTime=1;
		Trigger();
		break;
	}
}

TBool CHttpTransfer2Impl::MoveToCurrent()
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::MoveToCurrent"));


	TRAPD(err, iTable.GotoL(iCurrentFileMark));
	if (err!=KErrNone) return EFalse;
	return ETrue;
}

void CHttpTransfer2Impl::error(CBase* /*source*/, TInt code, const TDesC& reason)
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::error"));

	iTable.Cancel();

	if (iStatusNotif) iStatusNotif->error(this, code, reason);

	if (current_state==SENDING_FILE) {
		if (iTable.NextL()) {
			iCurrentFileMark=iTable.Bookmark();
			iWaitTime=1;	
		} else if (iTable.FirstL()) {
			iCurrentFileMark=iTable.Bookmark();
			++iRetries;
			iWaitTime=300;
		} else {
			Reset();
			return;
		}
	} else {
		Reset();
		++iRetries;
		iWaitTime=300;
	}
	Trigger(ETrue);
}

void CHttpTransfer2Impl::info(CBase* /*source*/, const TDesC& msg)
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::info"));


	if (iStatusNotif) iStatusNotif->info(this, msg);
}

void CHttpTransfer2Impl::Trigger(TBool Force)
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::Trigger"));


	if (!Force && current_state!=IDLE) return;

	Cancel();

	if (iWaitTime==0) {
		TRequestStatus *s=&iStatus;
		User::RequestComplete(s, KErrNone);
	} else {
		TTimeIntervalMicroSeconds32 w(iWaitTime*1000*1000);
		iTimer.After(iStatus, w);
	}
	SetActive();
}

void CHttpTransfer2Impl::ConnectL()
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::ConnectL"));

	iTable.Cancel();
	TInt Iap;
	Settings().GetSettingL(SETTING_CURRENT_AP, Iap);

	if (Iap==-1) {
		// 'No connection'
		current_state=SENDING_FILE;
		iNoTransfer=true;
		Trigger(ETrue);
	} else {
		iNoTransfer=false;

		if (!iHttp) {
			iHttp=CHttp::NewL(AppContext(), *this);
		}

		if (!MoveToCurrent()) {
			if (iStatusNotif) iStatusNotif->error(this, -1031, _L("current record disappeared"));
			Reset();
			return;
		}
		iTable.GetL();

		TInt url_setting, script_setting;

		url_setting=iTable.ColInt32(2);
		script_setting=iTable.ColInt32(3);

		TBuf<100> url;
		TBuf<50> script;
		Settings().GetSettingL(url_setting, url);
		Settings().GetSettingL(script_setting, script);
		url.Append(script);

		TBuf<50> Proxy;
		Settings().GetSettingL(SETTING_PROXY, Proxy);
		TInt	ProxyPort;
		Settings().GetSettingL(SETTING_PROXY_PORT, ProxyPort);

		current_state=CONNECTING;
		iHttp->Connect(Iap, url, Proxy, ProxyPort);
	}
}

void CHttpTransfer2Impl::Close()
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::Close"));

	current_state=CLOSING;
	iHttp->Close();
}

CHttpTransfer2Impl::~CHttpTransfer2Impl()
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::~CHttpTransfer2Impl"));

	Cancel();
	iTimer.Close();

	delete iHttp;
	delete iPacket8;
}

void CHttpTransfer2Impl::GetFileInfoL(const TDesC& aFileName,
				      const TDesC& PacketFileName, HBufC8*& this_packet, TDes& datetime)
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::GetFileInfoL"));

	TEntry fe;
	User::LeaveIfError(Fs().Entry(aFileName, fe));
	format_datetime(datetime, fe.iModified);

	if (BaflUtils::FileExists(Fs(), PacketFileName)) {
		User::LeaveIfError(Fs().Entry(PacketFileName, fe));
		if (fe.iSize!=0) {
			if (!iPacket8 || iPacket8->Des().MaxLength() < fe.iSize) {
				delete iPacket8; iPacket8=0;
				iPacket8=HBufC8::NewL(fe.iSize);
			} else {
				iPacket8->Des().Zero();
			}
			RAFile f; f.OpenLA(Fs(), PacketFileName, EFileRead|EFileShareAny);
			TPtr8 ptr=iPacket8->Des();
			User::LeaveIfError(f.Read(ptr, fe.iSize));
		} else { 
			if (!iPacket8) iPacket8=HBufC8::NewL(2048);
			else iPacket8->Des().Zero();
		}
		this_packet=iPacket8;
	} else {
		this_packet=0;
	}
}

void CHttpTransfer2Impl::StartFileTransferL()
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::StartFileTransferL"));

	iTable.Cancel();

	if (! MoveToCurrent()) {
		if (iStatusNotif) iStatusNotif->error(this, -1031, _L("current record disappeared"));
		Reset();
		return;
	}

	bool done=false;
	while (!done) {
		// skip files in use
		iTable.GetL();
		iTempFileName=iTable.ColDes16(1);
		RFile f;
		TInt err;
		err=f.Open(Fs(), iTempFileName, EFileRead|EFileShareAny);
		TOp Op=(TOp)iTable.ColInt32(7);
		bool goto_next=false;
		if (err==KErrNone) {
			f.Close();
			switch (Op) {
			case UPLOAD:
				if (iTempFileName.Find(KUploadDir)==KErrNotFound) {
					iPacketFileName=iTable.ColDes16(6);
					_LIT(Kcontext, "context");
					TFileName filen; TParse p;
					if (iPacketFileName.FindC(Kcontext)!=KErrNotFound) {
						p.Set(iTempFileName, 0, 0);
						MakeFileName();
						filen=p.DriveAndPath();
						filen.Append(KUploadDir);
						filen.Append(iFileNameBuf);
						filen.Append(p.Ext());
					} else {
						p.Set(iTempFileName, 0, 0);
						filen=iPacketFileName;
						filen.Replace( filen.Length()-4, 4, p.Ext());
					}
					iTable.UpdateL();
					if ( Fs().Rename(iTempFileName, filen) == KErrNone ) {
						iTempFileName=filen;
						iTable.SetColL(1, iTempFileName);
					}
					PutL();
				}
				break;
			case DELETE:
				User::LeaveIfError(Fs().Delete(iTempFileName));
				goto_next=true;
				break;
			case DONTSHARE:
				DoNotTransferL(iTempFileName);
				goto_next=true;
				break;
			case MOVETOOLD:
				MoveToOldL(iTempFileName);
				goto_next=true;
				break;
			}
		}
		if (goto_next) {
			iTable.DeleteL();
			if (iTable.NextL()) {
				current_state=SENDING_FILE;
				iCurrentFileMark=iTable.Bookmark();
				iWaitTime=0;
				Trigger(ETrue);
			} else {
				Reset();
				iWaitTime=15;
				Trigger();
			}
			return;
		}
		if (err==KErrAccessDenied || err==KErrInUse) {
		} else if (err==KErrNotFound) {
			iPacketFileName=iTable.ColDes16(6);
			Fs().Delete(iPacketFileName);
			iTable.DeleteL();
		} else {
			iCurrentFileMark=iTable.Bookmark();
			done=true;
		}
		if (!done) {
			if (! iTable.NextL()) {
				Reset();
				iWaitTime=15;
				Trigger();
				return;
			}
		}
	}

	if (iNoTransfer) {
		if (iTable.NextL()) {
			iCurrentFileMark=iTable.Bookmark();
			Trigger();
		} else {
			Reset();
		}
	} else {
		if (!iHttp) {
			ConnectL();
		} else {
			iPacketFileName=iTable.ColDes16(6);
			TFileName iMeta;
			iMeta=iTable.ColDes16(5);

			HBufC8* packet=0; TBuf<20> dt;
			GetFileInfoL(iTempFileName, iPacketFileName, packet, dt);
			if (iMeta.Length()==0) dt.Zero();

			current_state=SENDING_FILE;
			iHttp->Store(iTempFileName, dt, packet);
		}
	}
}

void CHttpTransfer2Impl::Reset()
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::Reset"));

	delete iHttp; iHttp=0;
	current_state=IDLE;
}

void CHttpTransfer2Impl::HandleSent()
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::HandleSent"));

	iTable.Cancel();

	if (! MoveToCurrent() ) {
		if (iStatusNotif) iStatusNotif->error(this, -1031, _L("current record disappeared"));
		Reset();
		return;
	}

	iTable.GetL();

	iTempFileName=iTable.ColDes16(1);
	iPacketFileName=iTable.ColDes16(6);

	Fs().Delete(iPacketFileName);

	TBool del=iTable.ColInt(4);
	if(del) {
		Fs().Delete(iTempFileName);
	} else {
		TFileName moveto=iTempFileName;
		TParse p;
		if (moveto.Find(KUploadDir)) {
			moveto.Replace(moveto.Find(KUploadDir), KUploadDir().Length(), KUploadedDir);
		} else {
			p.Set(iTempFileName, 0, 0);
			moveto=p.DriveAndPath();
			moveto.Append(KUploadedDir);
			MakeFileName();
			moveto.Append(iFileNameBuf);
			moveto.Append(p.Ext());
		}
		p.Set(moveto, 0, 0);
		BaflUtils::EnsurePathExistsL(Fs(), p.DriveAndPath());
		Fs().Rename(iTempFileName, moveto);
	}
	iTable.DeleteL();

}

TBool CHttpTransfer2Impl::FileIsQueued(const TDesC& aFileName)
{
	CALLSTACKITEM(_L("CHttpTransfer2Impl::FileIsQueued"));
	iTable.Cancel();

	TDbSeekKey k(aFileName);
	TBool ret=iTable.SeekL(k);
	while(ret) {
		iTable.GetL();
		if ( aFileName.Left(iFileNameIndexLen).Compare(iTable.ColDes(1).Left(iFileNameIndexLen)) ) return EFalse;
		if ( aFileName.Compare(iTable.ColDes(1)) ) return ETrue;
		ret=iTable.NextL();
	}
	return EFalse;
}
