#include "transferdir2.h"
#include "transfer2.h"
#include <list.h>
#include <symbian_auto_ptr.h>
#include <db.h>

class CTransferDirImpl : public CTransferDir, public MContextBase, public MUploadCallBack {
private:
	TInt ProcessDir(const TDesC& aDirName, TInt UrlSetting,
		TInt ScriptSetting, const TDesC& aMeta,  MUploadPrompt* Prompt,
		TTime TimeLimit);
	CTransferDirImpl(MApp_context& Context, MSocketObserver& aObserver);
	void ConstructL();
	void Back(bool Upload, bool DeleteFromPhone, CXmlBuf* Packet);
	virtual CHttpTransfer2* Transferer();

	struct TDirItem {
		TFileName	iDirName;
		TInt		iUrlSetting;
		TInt		iScriptSetting;
		TFileName	iMeta;
		MUploadPrompt*  iPrompt;
		TTime		iTimeLimit;

		TDirItem(const TDesC& aDirName, TInt UrlSetting,
			TInt ScriptSetting, const TDesC& aMeta, MUploadPrompt* Prompt, TTime TimeLimit) : 
				iDirName(aDirName),
				iUrlSetting(UrlSetting), iScriptSetting(ScriptSetting), iPrompt(Prompt),
				iTimeLimit(TimeLimit)
		{
			iMeta=aMeta.Left(KMaxFileName);
		}
	};

	CList<TDirItem*>	*iList;
	TDirItem		*iCurrentDir;
	TInt			iCurrentItem;
	CDir			*iDir;
	CDb			*iDb;
	CHttpTransfer2		*iTransfer;
	TFileName		iCurrentDirName;
	MSocketObserver&	iObserver;
	enum TState { EIdle, EProcessing };
	TState			iCurrentState;

	void CheckedRunL();
	TInt CheckedRunError(TInt aError);
	void DoCancel();

	void FileStep();
	void Async();
	void DoBackL(bool Upload, bool DeleteFromPhone, CXmlBuf* Packet);

	friend class CTransferDir;
public:
	~CTransferDirImpl();
};


CTransferDir* CTransferDir::NewL(MApp_context& Context, MSocketObserver& aObserver)
{
	CALLSTACKITEM(_L("CTransferDir::NewL"));


	auto_ptr<CTransferDirImpl> ret(new (ELeave) CTransferDirImpl(Context, aObserver));
	ret->ConstructL();
	return ret.release();
}

CTransferDir::CTransferDir() : CCheckedActive(EPriorityNormal, _L("CTransferDir"))
{
	CALLSTACKITEM(_L("CTransferDir::CTransferDir"));


}

CTransferDir::~CTransferDir()
{
	CALLSTACKITEM(_L("CTransferDir::~CTransferDir"));


}

TInt CTransferDirImpl::ProcessDir(const TDesC& aDirName, TInt UrlSetting,
	TInt ScriptSetting, const TDesC& aMeta, MUploadPrompt* Prompt, TTime TimeLimit)
{
	CALLSTACKITEM(_L("CTransferDirImpl::ProcessDir"));


	TDirItem* i=new (ELeave) TDirItem(aDirName, UrlSetting, ScriptSetting, aMeta, Prompt, TimeLimit);
	TRAPD(err, iList->AppendL(i));
	if (err!=KErrNone) delete i;
	else {
		if (iCurrentState!=EIdle) return KErrNone;
		Async();
	}
	return err;
}

CTransferDirImpl::CTransferDirImpl(MApp_context& Context, MSocketObserver& aObserver) : 
		MContextBase(Context), iObserver(aObserver)
{
	CALLSTACKITEM(_L("CTransferDirImpl::CTransferDirImpl"));


}

void CTransferDirImpl::ConstructL()
{
	CALLSTACKITEM(_L("CTransferDirImpl::ConstructL"));


	iList=CList<TDirItem*>::NewL();
	iDb=CDb::NewL(AppContext(), _L("TRANSFER"), EFileRead|EFileWrite|EFileShareAny);

	CActiveScheduler::Add(this);

	iTransfer=CHttpTransfer2::NewL(AppContext(), iDb->Db(), &iObserver);
}

void CTransferDirImpl::CheckedRunL()
{
	CALLSTACKITEM(_L("CTransferDirImpl::CheckedRunL"));


	if (iCurrentState==EIdle) {
		delete iCurrentDir; iCurrentDir=0;
		iCurrentDir=iList->Pop();
		if (!iCurrentDir) return;

		delete iDir; iDir=0;
		TInt err=Fs().GetDir(iCurrentDir->iDirName, KEntryAttNormal, ESortByName, iDir);
		{
			TParse p; p.Set(iCurrentDir->iDirName, 0, 0);
			iCurrentDirName = p.DriveAndPath();
			RDebug::Print(iCurrentDirName);
		}
		if (err!=KErrNone) {
			Async();
			return;
		}
		iCurrentItem=0;
		FileStep();
	} else {
		FileStep();
	}
}

TInt CTransferDirImpl::CheckedRunError(TInt aError)
{
	CALLSTACKITEM(_L("CTransferDirImpl::CheckedRunError"));


	return aError;
}

void CTransferDirImpl::DoCancel()
{
	CALLSTACKITEM(_L("CTransferDirImpl::DoCancel"));


}

void CTransferDirImpl::DoBackL(bool Upload, bool DeleteFromPhone, CXmlBuf* Packet)
{
	CALLSTACKITEM(_L("CTransferDirImpl::DoBackL"));

	TEntry e=(*iDir)[iCurrentItem-1];
	TFileName filen=iCurrentDirName; filen.Append(e.iName);
	if (Upload) {
		TRAPD(err, iTransfer->AddFileToQueueL(filen, iCurrentDir->iUrlSetting,
			iCurrentDir->iScriptSetting, DeleteFromPhone,
			iCurrentDir->iMeta, Packet));
		if (err!=KErrNone) {
			User::Leave(err);
		}
	} else {
		if (DeleteFromPhone) {
			iTransfer->DeleteFile(filen);
		} else {
			iTransfer->DoNotTransfer(filen);
		}
	}
}

void CTransferDirImpl::Back(bool Upload, bool DeleteFromPhone, CXmlBuf* Packet)
{
	CALLSTACKITEM(_L("CTransferDirImpl::Back"));


	TRAPD(err, DoBackL(Upload, DeleteFromPhone, Packet));
	if (err!=KErrNone) {
		TAknGlobalNoteType notetype=EAknGlobalErrorNote;
		TBuf<100> msg=_L("Failed to queue file for upload: ");
		msg.AppendNum(err);
		AppContext().ShowGlobalNote(notetype, msg);
	}
	Async();
}

void CTransferDirImpl::FileStep()
{
	CALLSTACKITEM(_L("CTransferDirImpl::FileStep"));


	if (iCurrentItem < iDir->Count()) {
		iCurrentState=EProcessing;
		TEntry e=(*iDir)[iCurrentItem++];
		TFileName filen=iCurrentDirName; filen.Append(e.iName);
		if (iTransfer->FileIsQueued(filen)) {
			Async();
		} else {
			if (e.iModified > iCurrentDir->iTimeLimit) {
				iCurrentDir->iPrompt->Prompt( filen, this );
			} else {
				iTransfer->MoveToOld( filen );
				Async();
			}
		}
	} else {
		iCurrentState=EIdle;
		Async();
	}
}

void CTransferDirImpl::Async()
{
	CALLSTACKITEM(_L("CTransferDirImpl::Async"));


	if (IsActive()) return;

	TRequestStatus *s=&iStatus;
	User::RequestComplete(s, KErrNone);
	SetActive();
}

CTransferDirImpl::~CTransferDirImpl()
{
	CALLSTACKITEM(_L("CTransferDirImpl::~CTransferDirImpl"));


	Cancel();
	delete iTransfer;
	delete iDb;

	if (iList) {
		TDirItem* i;
		while (i=iList->Pop()) delete i;
	}
	delete iCurrentDir;
	delete iList;
	delete iDir;
}

CHttpTransfer2* CTransferDirImpl::Transferer()
{
	return iTransfer;
}