#include "bberrors.h"
#include <badesca.h>
#include "bbdata.h"
#include "bbtypes.h"
#include "concretedata.h"
#include "symbian_tree.h"
#include <bafindf.h>
#include "context_uids.h"
#include <e32uid.h>
#include "stringarg.h"
#include "errorhandling.h"

_LIT(KAttModule, "module");
_LIT(KAttId, "id");
_LIT(KAttMajorVersion, "major_version");
_LIT(KAttMinorVersion, "minor_version");

EXPORT_C TTypeName TTypeName::IdFromAttributes(const XML_Char ** atts) 
{
	TTypeName ret={ { 0 }, -1, -1, -1};
	if (!atts) return ret;

	const XML_Char** p=&atts[0];
	while ( *p && *(p+1)) {
		TPtrC att((const TUint16*)*p), val((const TUint16*)*(p+1));
		TLex l;

		TUint32 uvalue; TInt value;
		if (val.Left(2).CompareF(_L("0x"))==0) {
			l.Assign(val.Mid(2));
			l.Val(uvalue, EHex);
			value=uvalue;
		} else {
			l.Assign(val);
			l.Val(value);
			uvalue=value;
		}
		if (att.Compare(KAttModule)==0) {
			ret.iModule.iUid=uvalue;
		} else if (att.Compare(KAttId)==0) {
			ret.iId=value;
		} else if (att.Compare(KAttMajorVersion)==0) {
			ret.iMajorVersion=value;
		} else if (att.Compare(KAttMinorVersion)==0) {
			ret.iMinorVersion=value;
		}
		p+=2;
	}
	return ret;
}

EXPORT_C void TTypeName::ExternalizeL(RWriteStream& aStream) const
{
	aStream.WriteInt32L(iModule.iUid);
	aStream.WriteInt32L(iId);
	aStream.WriteInt32L(iMajorVersion);
	aStream.WriteInt32L(iMinorVersion);
}

EXPORT_C TTypeName TTypeName::IdFromStreamL(RReadStream& aStream)
{
	TTypeName ret={ {0}, -1, -1, -1};
	ret.iModule.iUid=aStream.ReadInt32L();
	ret.iId=aStream.ReadInt32L();
	ret.iMajorVersion=aStream.ReadInt32L();
	ret.iMinorVersion=aStream.ReadInt32L();
	return ret;
}

EXPORT_C void TTypeName::IntoStringL(TDes& aString) const
{
	CheckStringSpaceL(aString, 45);
	aString.Append(_L("[ 0x"));
	aString.AppendNum(iModule.iUid, EHex);
	aString.Append(_L(" "));
	aString.AppendNum(iId);
	aString.Append(_L(" "));
	aString.AppendNum(iMajorVersion);
	aString.Append(_L(" "));
	aString.AppendNum(iMinorVersion);
	aString.Append(_L(" ]"));
}

EXPORT_C bool TTypeName::operator==(const TTypeName& rhs) const
{
	return (
		(iModule.iUid == rhs.iModule.iUid) &&
		(iId == rhs.iId) &&
		(iMajorVersion == rhs.iMajorVersion) &&
		(iMinorVersion == rhs.iMinorVersion)
		);
}

EXPORT_C bool TTupleName::operator==(const TTupleName& rhs) const
{
	if (iModule.iUid!=rhs.iModule.iUid) return false;
	if (iId!=rhs.iId) return false;
	return true;
}

EXPORT_C void TTypeName::CompareMajorL(const TTypeName& aOther) const
{
	if (aOther.iModule!=iModule || aOther.iId!=iId) 
		InputErr(_L("Blackboard datatype doesn't match. Given %1 expected %2")).
			UserMsg(aOther, *this).ErrorCode(BBErrorCode(KTypeDoesNotMatch)).
			Raise();
	if (aOther.iMajorVersion!=iMajorVersion) 
		InputErr(_L("Blackboard datatype version not supported. Given %1 expected %2")).
			UserMsg(aOther, *this).ErrorCode(BBErrorCode(KTypeDoesNotMatch)).
			Raise();
}

EXPORT_C MDesCArray* TTypeName::MakeAttributesLC() const
{
	CDesCArrayFlat *ret=new (ELeave) CDesCArrayFlat(8);
	CleanupStack::PushL(ret);
	TBuf<20> buf;
	ret->AppendL(KAttModule);
	buf.Append(_L("0x")); buf.AppendNum((TInt)iModule.iUid, EHex); ret->AppendL(buf);

	ret->AppendL(KAttId);
	buf.Zero(); buf.AppendNum(iId); ret->AppendL(buf);

	ret->AppendL(KAttMajorVersion);
	buf.Zero(); buf.AppendNum(iMajorVersion); ret->AppendL(buf);

	ret->AppendL(KAttMinorVersion); 
	buf.Zero(); buf.AppendNum(iMinorVersion); ret->AppendL(buf);

	return ret;
}

EXPORT_C MNestedXmlHandler::~MNestedXmlHandler()
{
}

void DeleteBBData(TAny* p) {
	MBBData *m=(MBBData *)p;
	delete m;
}

void DeleteBBFactory(TAny* p) {
	MBBDataFactory *m=(MBBDataFactory*)p;
	delete m;
}

void DeleteRLibrary(TAny* p) {
	RLibrary *m=(RLibrary*)p;
	m->Close();
	delete m;
}

EXPORT_C void CleanupPushBBDataL(MBBData* aData)
{
	CleanupStack::PushL(TCleanupItem(&DeleteBBData, (TAny*)aData));
}

void CleanupPushBBFactoryL(MBBDataFactory* aData)
{
	CleanupStack::PushL(TCleanupItem(&DeleteBBFactory, (TAny*)aData));
}

/*
 * Concepts:
 * !Using polymorphic dlls!
 */

class CBBDataFactoryImpl : public CBBDataFactory {
private:
	CBBDataFactoryImpl();
	void ConstructL();
	~CBBDataFactoryImpl();
	void Reset();

	virtual MBBData* CreateBBDataL(const TTypeName& aType, const TDesC& aName, MBBDataFactory* aTopLevelFactory);

	friend class CBBDataFactory;
	friend class auto_ptr<CBBDataFactoryImpl>;

	CGenericIntMap*	iFactories;
	CGenericIntMap*	iLibraries;
	RFs		iFs; bool fs_open;
};

IMPORT_C CBBDataFactory* CBBDataFactory::NewL()
{
	auto_ptr<CBBDataFactoryImpl> ret(new (ELeave) CBBDataFactoryImpl);
	ret->ConstructL();
	return ret.release();
}

IMPORT_C CBBDataFactory::~CBBDataFactory()
{
}

IMPORT_C void CBBDataFactory::ConstructL()
{
}

CBBDataFactoryImpl::CBBDataFactoryImpl()
{
}

void CBBDataFactoryImpl::ConstructL()
{
	iFactories=CGenericIntMap::NewL();
	iFactories->SetDeletor(&DeleteBBFactory);
	iLibraries=CGenericIntMap::NewL();
	iLibraries->SetDeletor(&DeleteRLibrary);
	User::LeaveIfError(iFs.Connect());
}

CBBDataFactoryImpl::~CBBDataFactoryImpl()
{
	delete iFactories;
	delete iLibraries;
	iFs.Close();
}

MBBData* CBBDataFactoryImpl::CreateBBDataL(const TTypeName& aType, const TDesC& aName, MBBDataFactory* )
{
	MBBDataFactory* f=(MBBDataFactory*)iFactories->GetData((uint32)aType.iModule.iUid);
	if (!f) {
		RLibrary *l=new (ELeave) RLibrary;
		TInt err;
		TRAP(err, iLibraries->AddDataL((uint32)aType.iModule.iUid, l, true));
		if (err!=KErrNone) {
			delete l;
			User::Leave(err);
		}
		auto_ptr<CFindFileByType> findf(new (ELeave) CFindFileByType(iFs));
		TUidType uid(KDynamicLibraryUid, KUidBlackBoardData, aType.iModule);
#ifdef __WINS__
		err=findf->FindFirst(_L("contextmediafactory.dll"), _L("z:\\system\\libs\\"), 
			TUidType(KNullUid, KNullUid, KNullUid));
		err=findf->FindFirst(_L("contextmediafactory.dll"), _L("z:\\system\\libs\\"), 
			TUidType(KNullUid, KNullUid, aType.iModule));
		err=findf->FindFirst(_L("*.dll"), _L("z:\\system\\libs\\"), uid);
		//err=findf->FindFirst(_L("blackboardfactory.dll"), _L("z:\\system\\libs\\"), 
		//	TUidType(KNullUid, KNullUid, aType.iModule));
#else
		err=findf->FindFirst(_L("*.dll"), _L("\\system\\libs\\"), uid);
#endif
		if (err!=KErrNone) {
			EnvErr(gettext("A library installation is missing, please reinstall the software.")).
			TechMsg(_L("Cannot find dll with UID %1, error: %2"), aType.iModule.iUid, err)
				.ErrorCode(BBErrorCode(KNoDllForType)).Raise();
		}

		err=l->Load( findf->Entry().iName, TUidType(KNullUid, KNullUid, aType.iModule) );
		if (err!=KErrNone) {
			EnvErr(gettext("A library installation is faulty, please reinstall the software.")).
				TechMsg(_L("Cannot load dll %1 with UID %2, error: %3"), 
					findf->Entry().iName, aType.iModule.iUid,
					err).
				ErrorCode(BBErrorCode(KNoDllForType)).
				Raise();
		}
		TLibraryFunction entry=l->Lookup(1);
		f=(MBBDataFactory*)entry();
		if (!f) User::Leave(KErrNoMemory);
		CleanupPushBBFactoryL(f);
		f->ConstructL();
		iFactories->AddDataL((uint32)aType.iModule.iUid, f, true);
		CleanupStack::Pop();
	}
	return f->CreateBBDataL(aType, aName, this);
}

void CBBDataFactoryImpl::Reset()
{
	iLibraries->Reset();
	iFactories->Reset();
}

EXPORT_C bool TComponentName::operator==(const TComponentName& rhs) const
{
	return (rhs.iModule.iUid==iModule.iUid && rhs.iId==iId);
}

EXPORT_C CBBGeneralHolder::CBBGeneralHolder(MBBDataFactory* aFactory, MBBData* aValue) : iValue(aValue), iFactory(aFactory),
	iOwnsValue(ETrue)
{
}

EXPORT_C void CBBGeneralHolder::SetOwnsValue(TBool aOwns)
{
	iOwnsValue=aOwns;
}

EXPORT_C CBBGeneralHolder::~CBBGeneralHolder()
{
	if (iOwnsValue) delete iValue;
}

EXPORT_C void CBBGeneralHolder::IntoStringL(TDes& aString) const
{
	if (iValue) iValue->IntoStringL(aString);
}

EXPORT_C void CBBGeneralHolder::IntoXmlL(MBBExternalizer* aBuf, TBool ) const
{
	if (iValue) iValue->IntoXmlL(aBuf, ETrue);
}

EXPORT_C void CBBGeneralHolder::ExternalizeL(RWriteStream& aStream) const
{
	if (iValue) {
		iValue->Type().ExternalizeL(aStream);
		iValue->ExternalizeL(aStream);
	} else {
		KNullType.ExternalizeL(aStream);
	}
}

EXPORT_C void CBBGeneralHolder::FromStringL(const TDesC& )
{
	Bug(_L("CBBGeneralHolder::FromStringL called")).Raise();
}

EXPORT_C MNestedXmlHandler* CBBGeneralHolder::FromXmlL(MNestedXmlHandler* aParent, CXmlParser* aParser,
				HBufC*& aBuf, TBool aCheckType)
{
	return iValue->FromXmlL(aParent, aParser, aBuf, aCheckType);
}

// FIXME: should this be a parameter to the class?
_LIT(KGeneral, "general");

EXPORT_C void CBBGeneralHolder::InternalizeL(RReadStream& aStream)
{
	TTypeName t=TTypeName::IdFromStreamL(aStream);
	if (! (t==KNullType) ) {
		iValue=iFactory->CreateBBDataL(t, KGeneral, iFactory);
		iValue->InternalizeL(aStream);
	}
}

EXPORT_C const TTypeName& CBBGeneralHolder::Type() const
{
	if (iValue) return iValue->Type();
	else return KNullType;
}

EXPORT_C TBool CBBGeneralHolder::Equals(const MBBData* aRhs) const
{
	if (!aRhs && !iValue) return ETrue;
	if (aRhs && iValue) return iValue->Equals(aRhs);
	return EFalse;
}

const TDesC& CBBGeneralHolder::Name() const
{
	if (!iValue) return KNullDesC;
	else return iValue->Name();
}

EXPORT_C MBBData::~MBBData()
{
}

EXPORT_C MBBData* CBBGeneralHolder::CloneL(const TDesC& Name) const
{
	MBBData* val=0;
	if (iValue) {
		val=iValue->CloneL(Name);
		CleanupPushBBDataL(val);
	}
	CBBGeneralHolder* ret=new (ELeave) CBBGeneralHolder(iFactory, iValue);
	if (val) CleanupStack::Pop();
	return ret;
}

EXPORT_C TBool CBBGeneralHolder::OwnsValue() const
{
	return iOwnsValue;
}

void AppendComponentNameToString(const void* aComponentName, TDes& aString)
{
	const TComponentName& n=*( (const TComponentName*)aComponentName );
	aString.Append(_L("[ 0x"));
	aString.AppendNum(n.iModule.iUid, EHex);
	aString.Append(_L(" "));
	aString.AppendNum(n.iId);
	aString.Append(_L("]"));
}

EXPORT_C TComponentName::operator TStringArg() const
{
	return TStringArg(this, 2+12+1+2+10, AppendComponentNameToString);
}

void AppendTypeNameToString(const void* aTypeName, TDes& aString)
{
	const TTypeName& n=*( (const TTypeName*)aTypeName );
	n.IntoStringL(aString);
}

IMPORT_C TTypeName::operator TStringArg() const
{
	return TStringArg(this, 45, AppendTypeNameToString);
}