#include "bbxml.h"
#include "bberrors.h"


class CXmlBufExternalizerImpl : public CXmlBufExternalizer
{
private:
	virtual void BeginList(const TDesC& aName, TBool aIncludeBBType, const TTypeName& aBBType);
	virtual void BeginCompound(const TDesC& aName, TBool aIncludeBBType, const TTypeName& aBBType);
	virtual void Field(const TDesC& aName, TBasicType aBasicTyype, const TDesC& aValue,
		TBool aIncludeBBType, const TTypeName& aBBType);

	virtual void EndCompound(const TDesC& aName);
	virtual void EndList(const TDesC& aName);

	void BeginElement(const TDesC& aName, TBool aIncludeBBType, const TTypeName& aBBType);
};

EXPORT_C CXmlBufExternalizer* CXmlBufExternalizer::NewL(TInt aInitialSize)
{
	auto_ptr<CXmlBufExternalizerImpl> ret(new (ELeave) CXmlBufExternalizerImpl);
	ret->ConstructL(aInitialSize);
	return ret.release();
}

void CXmlBufExternalizerImpl::BeginElement(const TDesC& aName, TBool aIncludeBBType, const TTypeName& aBBType)
{
	if (aIncludeBBType) {
		MDesCArray* attrs=aBBType.MakeAttributesLC();
		CXmlBuf::BeginElement(aName, attrs);
		CleanupStack::PopAndDestroy();
	} else {
		CXmlBuf::BeginElement(aName);
	}
}

void CXmlBufExternalizerImpl::BeginList(const TDesC& aName, TBool aIncludeBBType, const TTypeName& aBBType)
{
	BeginElement(aName, aIncludeBBType, aBBType);
}

void CXmlBufExternalizerImpl::BeginCompound(const TDesC& aName, TBool aIncludeBBType, const TTypeName& aBBType)
{
	BeginElement(aName, aIncludeBBType, aBBType);
}

void CXmlBufExternalizerImpl::Field(const TDesC& aName, TBasicType, const TDesC& aValue,
	TBool aIncludeBBType, const TTypeName& aBBType)
{
	BeginElement(aName, aIncludeBBType, aBBType);
	CXmlBuf::Characters(aValue);
	CXmlBuf::EndElement(aName);
}

void CXmlBufExternalizerImpl::EndCompound(const TDesC& aName)
{
	EndElement(aName);
}
void CXmlBufExternalizerImpl::EndList(const TDesC& aName)
{
	EndElement(aName);
}

EXPORT_C TSingleXml::TSingleXml(MNestedXmlHandler* aParent, CXmlParser* aParser, HBufC*& aBuf, MBBData& aValue, TBool aCheckType) : TBBXml(aParent, aParser, aBuf, aValue, aCheckType)
{
}

EXPORT_C TCheckingXml::TCheckingXml(MNestedXmlHandler* aParent, CXmlParser* aParser) :
		iParent(aParent), iIgnore(this, aParser), iIgnoreElement(EFalse), iParser(aParser)
{
}

EXPORT_C TBBXml::TBBXml(MNestedXmlHandler* aParent, CXmlParser* aParser, 
		HBufC*& aBuf, MBBData& aValue, TBool aCheckType) : 
		TCheckingXml(aParent, aParser), iBuf(aBuf), 
		iValue(aValue), iCheckType(aCheckType)
{
	if (iBuf) {
		iBuf->Des().Zero();
	} 
}
EXPORT_C CContainerXml::CContainerXml(MNestedXmlHandler* aParent, CXmlParser* aParser,
		HBufC*& aBuf, MBBData& aValue, TBool aCheckType) :
		TBBXml(aParent, aParser, aBuf, aValue, aCheckType)
{
}

EXPORT_C void TCheckingXml::StartElement(const XML_Char * name,
 	const XML_Char ** atts)
{
	TRAPD(err, StartElementL(name, atts));
	if (err!=KErrNone) {
		ErrorInHandler(err);
		// ErrorInHandler throws, if the top-level parser thinks that's
		// necessary. If not, we can ignore the element
		iIgnoreElement=ETrue;
		iIgnore.StartElement(name, atts);
		iParser->SetHandler(&iIgnore);
	}
}

EXPORT_C void TSingleXml::StartElementL(const XML_Char * name,
 	const XML_Char ** atts)
{
	TPtrC n((const TUint16*)name);
	if (n.Compare(iValue.Name()) ) {
		User::Leave(KNameDoesNotMatch);
	}
	if (iCheckType) {
		TTypeName read_type=TTypeName::IdFromAttributes(atts);
		read_type.CompareMajorL(iValue.Type());
	}
}

EXPORT_C void TCheckingXml::EndElement(const XML_Char * name)
{
	TRAPD(err, EndElementL(name));
	if (err!=KErrNone) {
		iIgnoreElement=ETrue;
		ErrorInHandler(err);
	}
	iParent->EndElement(name);
	iParser->SetHandler(iParent);
}


EXPORT_C void TSingleXml::EndElementL(const XML_Char * /*name*/)
{
	if (!iIgnoreElement) {
		if (iBuf) {
			iBuf->Des().Trim();
			iValue.FromStringL(*iBuf);
		} else {
			iValue.FromStringL(KNullDesC);
		}
	}
}

EXPORT_C void TCheckingXml::CharacterData(const XML_Char *s,
 	    int len)
{
	TRAPD(err, CharacterDataL(s, len));
	if (err!=KErrNone) {
		ErrorInHandler(err);
		iIgnoreElement=ETrue;
		iIgnore.StartElement(0, 0);
		iParser->SetHandler(&iIgnore);
	}
}

EXPORT_C void TCheckingXml::ErrorInHandler(TInt aError)
{
	iParent->SetError(aError);
}

EXPORT_C void TSingleXml::CharacterDataL(const XML_Char *s,
 	    int len)
{
	if (!iBuf) {
		iBuf=HBufC::NewL(len*2);
	}
	while (iBuf->Length() + len > iBuf->Des().MaxLength()) {
		iBuf=iBuf->ReAllocL(iBuf->Des().MaxLength() * 2);
	}
	TPtrC p((const TUint16*)s, len);
	iBuf->Des().Append(p);
}

EXPORT_C void TSingleXml::Error(XML_Error Code, const XML_LChar * String, long ByteIndex)
{
	iParent->Error(Code, String, ByteIndex);
}

EXPORT_C void TSingleXml::SetError(TInt aError)
{
	iParent->SetError(aError);
}

EXPORT_C void CContainerXml::StartElementL(const XML_Char *name,
			const XML_Char **atts)
{
	// the depth has to be before doing possible Leave's, so that if an error
	// occurs and TCheckingXml sets an TIgnore to eat the element,
	// the EndElement will work on the right depth
	if (iDepth==0) {
		++iDepth;
		TPtrC n((const TUint16*)name);
		if (n.Compare(iValue.Name()) ) {
			User::Leave(KNameDoesNotMatch);
		}
		if (iCheckType) {
			TTypeName read_type=TTypeName::IdFromAttributes(atts);
			read_type.CompareMajorL(iValue.Type());
		}
	} else {
		++iDepth;
		StartInnerElementL(name, atts);
	}
}

EXPORT_C void CContainerXml::CharacterDataL(const XML_Char * /*s*/, int /*len*/)
{
	// ignore
}

EXPORT_C void CContainerXml::EndElementL(const XML_Char *s)
{
	if (iDepth==2) {
		EndInnerElementL(s);
	}
}

EXPORT_C void CContainerXml::EndElement(const XML_Char *s)
{
	TRAPD(err, EndElementL(s));
	if (err!=KErrNone) {
		ErrorInHandler(err);
	}
	--iDepth;
	if (iDepth==0) {
		iParent->EndElement(s);
		iParser->SetHandler(iParent);
	}
}


EXPORT_C void CContainerXml::SetCurrentHandler(MNestedXmlHandler* aHandler)
{
	delete iCurrentHandler;
	iCurrentHandler=aHandler;
	iParser->SetHandler(aHandler);
}

EXPORT_C MNestedXmlHandler* CContainerXml::GetCurrentHandler()
{
	return iCurrentHandler;
}

EXPORT_C CContainerXml::~CContainerXml()
{
	delete iCurrentHandler;
}

EXPORT_C CPartContainerXml::CPartContainerXml(MNestedXmlHandler* aParent, CXmlParser* aParser,
					      HBufC*& aBuf, TBBCompoundData& aValue, TBool aCheckType) :
	CContainerXml(aParent, aParser, aBuf, aValue,  aCheckType), iCompoundValue(aValue), iErrorsInCurrent(EFalse)
{
}

EXPORT_C void CPartContainerXml::StartInnerElementL(const XML_Char *name, const XML_Char **atts)
{
	//RDebug::Print(_L("CPartContainerXml::StartInnerElementL"));

	TPtrC namep((const TUint16*)name);

	TUint partno;
	MBBData *p=iCompoundValue.GetPart(namep, TTypeName::IdFromAttributes(atts), partno);
	iCurrentPart=partno;
	if (p) {
		SetCurrentHandler(p->FromXmlL(this, iParser, iBuf, iCheckType) );
		GetCurrentHandler()->StartElement(name, atts);
	} else {
		User::Leave(KUnexpectedElement);
	}
}

EXPORT_C void CPartContainerXml::EndInnerElementL(const XML_Char *)
{
	//RDebug::Print(_L("CPartContainerXml::EndInnerElementL"));

	if (iCurrentPart>=0 && !iIgnoreElement) iCompoundValue.ReadPart(iCurrentPart, iErrorsInCurrent);
	iErrorsInCurrent=EFalse;
}

EXPORT_C void CPartContainerXml::Error(XML_Error Code, const XML_LChar * /*String*/, long /*ByteIndex*/)
{
	SetError(Code);
}

EXPORT_C void CPartContainerXml::SetError(TInt aError)
{
	iErrorsInCurrent=ETrue;
	if (!iParent) User::Leave(aError);
	else iParent->SetError(aError);
}

class CSingleParserImpl : public CSingleParser, public MNestedXmlHandler {
public:
	MBBData* iData; 
	MNestedXmlHandler* iCurrentHandler;
	CXmlParser* iParser;
	HBufC* iBuf;
	TBool  iCheckType;
	TBool	iIgnoreUnknown;
	
	void SetParser(CXmlParser* aParser) { iParser=aParser; }
	CSingleParserImpl(MBBData* aData, TBool  aCheckType,
		TBool aIgnoreUnknown) : iData(aData), iCheckType(aCheckType),
		iIgnoreUnknown(aIgnoreUnknown) { }
	void ConstructL() {
		iParser=CXmlParser::NewL(*this);
	}
	~CSingleParserImpl() { delete iBuf; delete iCurrentHandler; delete iParser; }

	virtual void ParseL(const TDesC8& aXml) {
		iParser->Parse( (char*)(aXml.Ptr()), aXml.Size(), 0);
	}

	virtual void StartElement(const XML_Char *name,
				const XML_Char **atts) {

		if (!iCurrentHandler) iCurrentHandler=iData->FromXmlL(this, iParser, iBuf, iCheckType);
		else User::Leave(KErrGeneral);

		if (iCurrentHandler) iCurrentHandler->StartElement(name, atts);
		iParser->SetHandler(iCurrentHandler);
	}
		
	virtual void EndElement(const XML_Char * /*name*/) {
	}

	virtual void CharacterData(const XML_Char * /*s*/,
				    int /*len*/) {
		User::Leave(KErrGeneral);
	}

	virtual void Error(XML_Error /*Code*/, const XML_LChar * /*String*/, long /*ByteIndex*/) {
		User::Leave(KErrInvalidXml);
	}
	virtual void SetError(TInt aError) { 
		if (iIgnoreUnknown && aError==KUnexpectedElement) return;
		User::Leave(aError); 
	}
};

EXPORT_C CSingleParser* CSingleParser::NewL(MBBData* aParseInto, TBool  aCheckType,
					    TBool aIgnoreUnknown)
{
	CSingleParserImpl *ret=new (ELeave) CSingleParserImpl(aParseInto, aCheckType, 
		aIgnoreUnknown);
	CleanupStack::PushL(ret);
	ret->ConstructL();
	CleanupStack::Pop();
	return ret;
}


IMPORT_C TIgnoreXml::TIgnoreXml(MNestedXmlHandler* aParent, CXmlParser* aParser) : 
	iDepth(0), iParent(aParent), iParser(aParser) { }

void TIgnoreXml::SetError(TInt aError)
{
	// cannot get called
}

void TIgnoreXml::StartElement(const XML_Char *name,
			const XML_Char **atts)
{
	++iDepth;
}

void TIgnoreXml::EndElement(const XML_Char *name)
{
	--iDepth;
	if (iDepth==0) {
		iParser->SetHandler(iParent);
		iParent->EndElement(name);
	}
}

void TIgnoreXml::CharacterData(const XML_Char *s,
			    int len)
{
}

void TIgnoreXml::Error(XML_Error Code, const XML_LChar * String, long ByteIndex)
{
}
