/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "blackboard_cs.h"
#include <e32math.h>
#include "context_uids.h"

#include "blackboardclientsession.h"

static const TUint KDefaultMessageSlots = 2;
static const TUid KServerUid3 = { CONTEXT_UID_BLACKBOARDSERVER }; // matches UID in server/group/contextserver.mbm file

_LIT(KBlackBoardServerFilename, "BlackBoardServer");

#ifdef __WINS__
static const TUint KServerMinHeapSize =  0x1000;  //  4K
static const TUint KServerMaxHeapSize = 0x10000;  // 64K
#endif

static TInt StartServer();
static TInt CreateServerProcess();


EXPORT_C RBBClient::RBBClient() : RSessionBase(), iArgs(0), iArgPckg(0),
	iTupleArg(0), iTupleArgPckg(0), iIdPtr(0, 0, 0),
	iFullPtr(0, 0, 0), iNotifyFullPtr(0, 0, 0), iTupleNamePckg(iTupleName),
	iComponentNamePckg(iComponentName)
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("RBBClient"));

	
}

EXPORT_C TInt RBBClient::Connect()
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("Connect"));

	TInt result;
	
	result = ::StartServer();
	if (result == KErrNone)
	{
		result = CreateSession(KBlackBoardServerName,
			Version(),
			KDefaultMessageSlots);	
	}
	if (result==KErrNone) {
		iArgs=new TFullArgs;
		if (iArgs) iArgPckg=new TPckg<TFullArgs>(*iArgs);
		iTupleArg=new TTupleArgs;
		if (iTupleArg) iTupleArgPckg=new TPckg<TTupleArgs>(*iTupleArg);
		if (!iArgs || !iArgPckg || !iTupleArg || !iTupleArgPckg) {
			Close();
			result=KErrNoMemory;
		}
	}

	return result;
}

EXPORT_C TVersion RBBClient::Version() const
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("Version"));

	return(TVersion(KBlackBoardServMajorVersionNumber,
		KBlackBoardServMinorVersionNumber,
		KBlackBoardServBuildVersionNumber));
}

//----------------------------------------------------------------------------------------

EXPORT_C void RBBClient::CancelOther()
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("CancelOther"));

	SendReceive(ECancelOther, NULL);
}

EXPORT_C void RBBClient::CancelNotify()
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("CancelNotify"));

	SendReceive(ECancelNotify, NULL);
}

EXPORT_C void RBBClient::TerminateBlackBoardServer(TRequestStatus& aStatus)
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("TerminateBlackBoardServer"));

	SendReceive(ETerminateBlackBoardServer, NULL, aStatus);
}

void	RBBClient::MakeIdPtr(TUint& aId)
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("MakeIdPtr"));

	iIdPtr.Set((TUint8*)&aId, sizeof(TUint), sizeof(TUint));
}

void	RBBClient::MakeFullPtr(TFullArgs& aFull, TPtr8& aInto)
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("MakeFullPtr"));

	aInto.Set((TUint8*)&aFull, sizeof(TFullArgs), sizeof(TFullArgs));
}

EXPORT_C void RBBClient::Put(TTupleName aTupleName, const TDesC& aSubName, 
		const TComponentName aComponent,
		const TDesC8& aSerializedData, TBBPriority aPriority, 
		TBool aReplace, TUint& aIdInto, TRequestStatus& aStatus,
		TBool aPersist, TBool aNotifySender, 
		TBool aIsReply)
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("Put"));

	if (aSubName.Length() > KMaxTupleSubNameLength) {
		TRequestStatus* s=&aStatus;
		User::RequestComplete(s, KErrTooBig);
		return;
	}
	iArgs->iTupleName=aTupleName;
	iArgs->iSubName=aSubName;
	iArgs->iComponentName=aComponent;
	iArgs->iPriority=aPriority;
	if (aIsReply) {
		iArgs->iTupleType=ETupleReply;
	} else {
		iArgs->iTupleType=ETupleDataOrRequest;
	}

	TInt flags=0;
	if (aReplace) flags|=KArgFlagReplace;
	if (!aPersist) flags|=KArgFlagDontPersist;
	if (!aNotifySender) flags|=KArgFlagDoNotNotifySender;

	MakeIdPtr(aIdInto);

	// IN flags, IN TFullArgs, OUT_OPT TUint id, IN data
	TAny* params[KMaxMessageArguments];
	params[0]=(void*)flags;
	params[1]=(void*)iArgPckg;
	params[2]=(void*)&iIdPtr;
	params[3]=(void*)&aSerializedData;

	SendReceive(EPut, &params, aStatus);
}

EXPORT_C void RBBClient::Get(const TTupleName& aName, const TDesC& aSubName, 
		TFullArgs& aMeta, TDes8& aSerializedData,
		TRequestStatus& aStatus)
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("Get"));

	if (aSubName.Length() > KMaxTupleSubNameLength) {
		TRequestStatus* s=&aStatus;
		User::RequestComplete(s, KErrTooBig);
		return;
	}

	// flags, IN TTupleName|TTupleArgs, OUT TFullArgs, OUT data
	MakeFullPtr(aMeta, iFullPtr);
	TAny* params[KMaxMessageArguments];
	TInt flags=0;
	if (aSubName.Length()==0) {
		flags=KArgFlagNoSubName;
		iTupleName=aName;
		params[1]=(void*)&iTupleNamePckg;
	} else {
		iTupleArg->iTupleName=aName;
		iTupleArg->iSubName=aSubName;
		params[1]=iTupleArgPckg;
	}
	params[0]=(void*)flags;
	params[2]=(void*)&iFullPtr;
	params[3]=(void*)&aSerializedData;

	SendReceive(EGetByTuple, params, aStatus);
}

EXPORT_C void RBBClient::Get(const TComponentName& aName,
		TFullArgs& aMeta, TDes8& aSerializedData,
		TRequestStatus& aStatus)
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("Get"));

	// flags, IN TComponentName, OUT TFullArgs, OUT data
	MakeFullPtr(aMeta, iFullPtr);
	TAny* params[KMaxMessageArguments];
	TInt flags=0;
	iComponentName=aName;
	params[1]=(void*)&iComponentNamePckg;
	params[0]=(void*)flags;
	params[2]=(void*)&iFullPtr;
	params[3]=(void*)&aSerializedData;

	SendReceive(EGetByComponent, params, aStatus);
}

EXPORT_C void RBBClient::AddNotificationL(const TTupleName& aTupleName, 
		TBool aGetExisting, TBBPriority aPriority,
		TRequestStatus& aStatus)
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("AddNotificationL"));

	// flags,  IN TTupleName.iModule.iUid, TTupleName.iId, TBBPriority
	TInt flags=0;
	if (aGetExisting) flags=KArgFlagGetExisting;
	TAny* params[KMaxMessageArguments];

	params[0]=(void*)flags;
	params[1]=(void*)aTupleName.iModule.iUid;
	params[2]=(void*)aTupleName.iId;
	params[3]=(void*)aPriority;

	SendReceive(EAddNotifyByTupleFilter, params, aStatus);
}

EXPORT_C void RBBClient::AddNotificationL(const TComponentName& aComponentName, 
		TRequestStatus& aStatus)
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("AddNotificationL"));

	// not in use,  IN TComponentName.iModule.iUid, IN TComponentName.iId, not in use
	TAny* params[KMaxMessageArguments];

	params[0]=(void*)0;
	params[1]=(void*)aComponentName.iModule.iUid;
	params[2]=(void*)aComponentName.iId;
	params[3]=(void*)0;

	SendReceive(EAddNotifyByComponentFilter, params, aStatus);
}

EXPORT_C void RBBClient::WaitForNotify(TFullArgs& aMeta, TDes8& aSerializedData, 
		TRequestStatus& aStatus)
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("WaitForNotify"));

	// not used, not used, OUT TFullArgs, OUT data
	MakeFullPtr(aMeta, iNotifyFullPtr);
	TAny* params[KMaxMessageArguments];
	params[0]=0;
	params[1]=0;
	params[2]=(void*)&iNotifyFullPtr;
	params[3]=(void*)&aSerializedData;

	SendReceive(ENotifyOnChange, params, aStatus);
}

EXPORT_C void RBBClient::Delete(TUint id, 
			TRequestStatus& aStatus)
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("Delete"));

	// not in use,  IN TUint, not in use, not in use

	TAny* params[KMaxMessageArguments];
	params[0]=0;
	params[1]=(void*)id;
	params[2]=0;
	params[3]=0;

	SendReceive(EDeleteById, params, aStatus);
}


EXPORT_C void RBBClient::Close()
{
	CALLSTACKITEM_N(_CL("RBBClient"), _CL("Close"));

	delete iArgs;
	delete iArgPckg;
	delete iTupleArg;
	delete iTupleArgPckg;

	RSessionBase::Close();
}

static TInt StartServer()
{
	CALLSTACKITEM_N(_CL("RSessionBase"), _CL("Close"));

	RMutex mutex;

	_LIT(KMutexName,  "blackboardclientsessionmutex");
	TInt result;
	result = mutex.CreateGlobal(KMutexName);
	if (result != KErrNone)
        {
		result=mutex.OpenGlobal(KMutexName);
        }
	if (result != KErrNone)
        {
		return  result;
	}
	mutex.Wait();
	
	TFindServer findBlackBoardServer(KBlackBoardServerName);
	TFullName name;
	
	result = findBlackBoardServer.Next(name);
	if (result == KErrNone)
        {
		mutex.Signal();
		// Server already running
		return KErrNone;
        }
	
	result = CreateServerProcess();
	if (result != KErrNone)
        {
		mutex.Signal();
		return  result;
        }
	
	RSemaphore semaphore;
	if (semaphore.CreateGlobal(KBlackBoardServerSemaphoreName, 0)!=KErrNone) {
		User::LeaveIfError(semaphore.OpenGlobal(KBlackBoardServerSemaphoreName));
	}
	
	// Semaphore opened ok
	semaphore.Wait();
	semaphore.Close();

	mutex.Signal();
	
	return  KErrNone;
}

static TInt CreateServerProcess()
{
	CALLSTACKITEM_N(_CL("User"), _CL("LeaveIfError"));

	TInt result;
	
	const TUidType serverUid(KNullUid, KNullUid, KServerUid3);
	
#ifdef __WINS__
	
	RLibrary lib;
	result = lib.Load(KBlackBoardServerFilename, _L("z:\\system\\programs"), serverUid);
	if (result != KErrNone)
        {
		return  result;
        }
	
	//  Get the WinsMain function
	TLibraryFunction functionWinsMain = lib.Lookup(1);
	
	//  Call it and cast the result to a thread function
	TThreadFunction serverThreadFunction = reinterpret_cast<TThreadFunction>(functionWinsMain());
	
	TName threadName(KBlackBoardServerName);
	
	// Append a random number to make it unique
	threadName.AppendNum(Math::Random(), EHex);
	
	RThread server;
	
	result = server.Create(threadName,   // create new server thread
		serverThreadFunction, // thread's main function
		KDefaultStackSize,
		NULL,
		&lib,
		NULL,
		KServerMinHeapSize,
		KServerMaxHeapSize,
		EOwnerProcess);
	
	lib.Close();    // if successful, server thread has handle to library now
	
	if (result != KErrNone)
        {
		return  result;
        }
	
	server.SetPriority(EPriorityMore);
	
	
#else
	
	RProcess server;
	
	result = server.Create(KBlackBoardServerFilename, _L(""), serverUid);
	if (result != KErrNone)
        {
		return  result;
        }
	
#endif
	
	server.Resume();
	server.Close();
	
	return KErrNone;
}

