#include "cbbsession.h"
#include "subscriptions.h"

#include "bberrors.h"
#include "symbian_refcounted_ptr.h"
#include "bbtypes.h"
#include "concretedata.h"
#include <s32mem.h>

class CRefCountedData : public MRefCounted {
public:
	MBBData* Get() { return iData; }
	const MBBData* Get() const { return iData; }
	void AddRef() const { ++iRefCount; }
	void Release() const { --iRefCount; if (iRefCount<=0) delete this; }
	static CRefCountedData* NewL(MBBData* aData) {
		CRefCountedData* c=new CRefCountedData(aData);
		if (!c) {
			delete aData;
			User::Leave(KErrNoMemory);
		}
		c->AddRef();
		return c;
	}
	~CRefCountedData() { delete iData; }
private:
	CRefCountedData(MBBData *aData) : iData(aData), iRefCount(0) { }
	MBBData* iData;
	mutable TInt iRefCount;
};

class MBlackBoardObserver {
// needs to be named like this for use with CSubscriptions
public:
	virtual void NewValue(TUint aId, const TTupleName& aTupleName, const TDesC& aSubName, 
		const TComponentName& aComponentName, 
		const CRefCountedData* aData) = 0;
};

class CBBSessionImpl : public CBBSession {
private:
	CBBSessionImpl(MApp_context& Context, MBBDataFactory* aFactory);
	void ConstructL();
	~CBBSessionImpl();
	CBBSubSession* CreateSubSessionL(MBBObserver* aObserver);

public:
	void AddNotificationL(const TTupleName& aTupleName, 
			MBlackBoardObserver* anObserver);
	void AddNotificationL(const TComponentName& aComponentName, 
			MBlackBoardObserver* anObserver);
	void DeleteNotifications(MBlackBoardObserver* anObserver);

	void GetL(const TTupleName& aName, const TDesC& aSubName, 
			MBBData*& aDataInto);
	void DeleteL(TUint id);
	//own
	void DoPutL(const TTupleName& aTupleName, const TDesC& aSubName, 
			const TComponentName& aComponentName,
			const MBBData* aData, TBool aReplace, TBool aIsReply=EFalse);

private:
	virtual MBBDataFactory* GetBBDataFactory() { return iFactory; }

	void CheckedRunL();
	void DoCancel();

	enum TNotifyType {
		EByTuple,
		EByComponent
	};
	void NotifyL(TUint aId, const TTupleName& aName, const TDesC& aSubName, 
		const TComponentName& aComponentName, const CRefCountedData* aData, 
		TNotifyType aNotifyType);

	friend class auto_ptr<CBBSessionImpl>;
	friend class CBBSession;

	CSubscriptions*		iSubscriptions;
	RBBClient		iClientSession;
	HBufC8			*iGetPutBuf, *iNotifyBuf;
	TFullArgs		iFull;
	TPtr8			iNotifyBufP;
	MBBDataFactory*		iFactory;
};

EXPORT_C CBBSession* CBBSession::NewL(MApp_context& Context, MBBDataFactory* aFactory)
{
	CALLSTACKITEM2_N(_CL("CBBSession"), _CL("NewL"), &Context);

	auto_ptr<CBBSessionImpl> ret(new (ELeave) CBBSessionImpl(Context, aFactory));
	ret->ConstructL();
	return ret.release();
}

_LIT(KSession, "CBBSession");

CBBSession::CBBSession(MApp_context& Context) : CCheckedActive(EPriorityNormal, KSession),
	MContextBase(Context) { }

CBBSessionImpl::CBBSessionImpl(MApp_context& Context, MBBDataFactory* aFactory) : 
	CBBSession(Context), iNotifyBufP(0, 0), iFactory(aFactory) { }

void CBBSessionImpl::ConstructL()
{
	CALLSTACKITEM_N(_CL("CBBSessionImpl"), _CL("ConstructL"));

	iSubscriptions=CSubscriptions::NewL();
	User::LeaveIfError(iClientSession.Connect());

	iGetPutBuf=HBufC8::NewL(2048);
	iNotifyBuf=HBufC8::NewL(2048);

	CActiveScheduler::Add(this);
	iNotifyBufP.Set(iNotifyBuf->Des());
	iClientSession.WaitForNotify(iFull, iNotifyBufP, iStatus);
	SetActive();
}

CBBSessionImpl::~CBBSessionImpl()
{
	CALLSTACKITEM_N(_CL("CBBSessionImpl"), _CL("~CBBSessionImpl"));

	Cancel();
	iClientSession.Close();
	delete iSubscriptions;

	delete iGetPutBuf;
	delete iNotifyBuf;
}

void CBBSessionImpl::AddNotificationL(const TTupleName& aTupleName, 
		MBlackBoardObserver* anObserver)
{
	CALLSTACKITEM_N(_CL("CBBSessionImpl"), _CL("AddNotificationL"));

	iSubscriptions->AddNotificationL(anObserver, aTupleName, EBBPriorityNormal);
	TRequestStatus s;

	iClientSession.AddNotificationL(aTupleName, ETrue, EBBPriorityNormal, s);
	// not all sensors post the current value on startup, like base and idle
	//iClientSession.AddNotificationL(aTupleName, EFalse, EBBPriorityNormal, s);
	User::WaitForRequest(s);
	User::LeaveIfError(s.Int());
}

void CBBSessionImpl::DeleteL(TUint id)
{
	CALLSTACKITEM_N(_CL("CBBSessionImpl"), _CL("DeleteL"));

	TRequestStatus s;
	iClientSession.Delete(id, s);

	User::WaitForRequest(s);
	User::LeaveIfError(s.Int());
}

void CBBSessionImpl::AddNotificationL(const TComponentName& aComponentName, 
		MBlackBoardObserver* anObserver)
{
	CALLSTACKITEM_N(_CL("CBBSessionImpl"), _CL("AddNotificationL"));

	iSubscriptions->AddNotificationL(anObserver, aComponentName, EBBPriorityNormal);
	TRequestStatus s;

	iClientSession.AddNotificationL(aComponentName, s);
	User::WaitForRequest(s);
	User::LeaveIfError(s.Int());
}

void CBBSessionImpl::DeleteNotifications(MBlackBoardObserver* anObserver)
{
	CALLSTACKITEM_N(_CL("CBBSessionImpl"), _CL("DeleteNotifications"));

	TRAPD(err, iSubscriptions->DeleteAllSubscriptionsL(anObserver));
	if (err!=KErrNone && err!=KErrNotFound) {
		User::Panic(_L("BBSession"), err);
	}
}

_LIT(KEvent, "event");

void CBBSessionImpl::GetL(const TTupleName& aName, const TDesC& aSubName, 
		MBBData*& aDataInto)
{
	CALLSTACKITEM_N(_CL("CBBSessionImpl"), _CL("GetL"));

	TFullArgs meta;
	delete aDataInto; aDataInto=0;
	TRequestStatus s;
	TPtr8 bufp(iGetPutBuf->Des());
	iClientSession.Get(aName, aSubName, meta, bufp, s);
	User::WaitForRequest(s);
	while (s.Int()==KClientBufferTooSmall) {
		iGetPutBuf=iGetPutBuf->ReAllocL( iGetPutBuf->Des().MaxLength() );
		TPtr8 bufp(iGetPutBuf->Des());
		iClientSession.Get(aName, aSubName, meta, bufp, s);
		User::WaitForRequest(s);
	}
	User::LeaveIfError(s.Int());
	RDesReadStream rs(*iGetPutBuf);
	TTypeName tn=TTypeName::IdFromStreamL(rs);
	aDataInto=iFactory->CreateBBDataL(tn, KEvent, iFactory);
	aDataInto->InternalizeL(rs);
}


void CBBSessionImpl::DoPutL(const TTupleName& aTupleName, const TDesC& aSubName, 
		const TComponentName& aComponentName,
		const MBBData* aData, TBool aReplace, TBool aIsReply)
{
	CALLSTACKITEM_N(_CL("CBBSessionImpl"), _CL("DoPutL"));

	refcounted_ptr<CRefCountedData> p(CRefCountedData::NewL(aData->CloneL(KEvent)));

	TRequestStatus s;
	TInt err=KErrNone;
	{
		TPtr8 bufp(iGetPutBuf->Des());
		RDesWriteStream ws(bufp);
		aData->Type().ExternalizeL(ws);
		TRAP(err, aData->ExternalizeL(ws));
		if (err==KErrNone) ws.CommitL();
	}
	while (err==KErrOverflow) {
		iGetPutBuf=iGetPutBuf->ReAllocL( iGetPutBuf->Des().MaxLength() *2);
		TPtr8 bufp(iGetPutBuf->Des());
		RDesWriteStream ws(bufp);
		aData->Type().ExternalizeL(ws);
		TRAP(err, aData->ExternalizeL(ws));
		if (err==KErrNone) ws.CommitL();
	}
	TUint id;
	iClientSession.Put(aTupleName, aSubName, aComponentName, *iGetPutBuf, EBBPriorityNormal, 
		aReplace, id, s, ETrue, EFalse, aIsReply);

	User::WaitForRequest(s);
	if (! aIsReply ) {
		TRAP(err, NotifyL(id, aTupleName, aSubName, aComponentName, p.get(), EByTuple));
	} else {
		TRAP(err, NotifyL(id, aTupleName, aSubName, aComponentName, p.get(), EByComponent));
	}

	User::LeaveIfError(s.Int());
	User::LeaveIfError(err);
}

void CBBSessionImpl::CheckedRunL()
{
	CALLSTACKITEM_N(_CL("CBBSessionImpl"), _CL("CheckedRunL"));

	if (iStatus.Int()!=KErrNone) {
		if (iStatus.Int()==KClientBufferTooSmall) {
			iNotifyBuf->Des().Zero();
			iNotifyBuf=iNotifyBuf->ReAllocL(iNotifyBuf->Des().MaxLength()*2);
			iNotifyBufP.Set(iNotifyBuf->Des());
			iClientSession.WaitForNotify(iFull, iNotifyBufP, iStatus);
			SetActive();
			return;
		}
		{
			User::Leave(iStatus.Int());
		}
	}
	if (iNotifyBuf->Des().Length()>0) {
		{
			RDesReadStream rs(*iNotifyBuf);
			TTypeName tn;
			TInt err;
			MBBData* datap=0;
			TRAP(err, tn=TTypeName::IdFromStreamL(rs));
			if (err!=KErrNone) {
				TBBLongString* s=new (ELeave) TBBLongString(KEvent);
				datap=s;
				s->Value().Append(_L("error reading datatype: "));
				s->Value().AppendNum(err);
			}
			if (err==KErrNone) {
				TRAP(err, datap=iFactory->CreateBBDataL(tn, KEvent, iFactory));
				if (err!=KErrNone) {
					TBBLongString* s=new (ELeave) TBBLongString(KEvent);
					datap=s;
					s->Value().Append(_L("error creating data: "));
					s->Value().AppendNum(err);
					s->Value().Append(_L(", type: "));
					tn.IntoStringL(s->Value());
				}
			}
			refcounted_ptr<CRefCountedData> data(CRefCountedData::NewL(datap));
			if (err==KErrNone) {
				TRAP(err, data->Get()->InternalizeL(rs));
				if (err!=KErrNone) {
					TBBLongString* s=new (ELeave) TBBLongString(KEvent);
					data.reset(CRefCountedData::NewL(s));
					s->Value().Append(_L("error internalizing data: "));
					s->Value().AppendNum(err);
					s->Value().Append(_L(", type: "));
					tn.IntoStringL(s->Value());
				}
			}
			{
			TNotifyType nt=EByTuple;
			if (iFull.iTupleType==ETuplePermanentSubscriptionEvent || 
				iFull.iTupleType==ETupleReply) {
					nt=EByComponent;
			}
			NotifyL(iFull.iId, iFull.iTupleName, iFull.iSubName, 
				iFull.iComponentName, data.get(), nt);
			}
		}
	}

	iNotifyBuf->Des().Zero();
	iNotifyBufP.Set(iNotifyBuf->Des());
	iClientSession.WaitForNotify(iFull, iNotifyBufP, iStatus);
	SetActive();
}

void CBBSessionImpl::DoCancel()
{
	CALLSTACKITEM_N(_CL("CBBSessionImpl"), _CL("DoCancel"));

	iClientSession.CancelNotify();
}

void CBBSessionImpl::NotifyL(TUint aId, 
			     const TTupleName& aName, const TDesC& aSubName, 
			     const TComponentName& aComponent,
			     const CRefCountedData* aData,
			     TNotifyType aNotifyType)
{
	CALLSTACKITEM_N(_CL("CBBSessionImpl"), _CL("NotifyL"));

	TBBPriority prio;
	if (aNotifyType==EByTuple) {
		for (MBlackBoardObserver* obs=iSubscriptions->FirstL(aName, prio); obs; 
				obs=iSubscriptions->NextL(prio)) {
			obs->NewValue(aId, aName, aSubName, aComponent, aData);
		}
	} else {
		for (MBlackBoardObserver* obs=iSubscriptions->FirstL(aComponent, prio); obs; 
				obs=iSubscriptions->NextL(prio)) {
			obs->NewValue(aId, aName, aSubName, aComponent, aData);
		}
	}
}

class CBBSubSessionImpl : public CBBSubSession, MBlackBoardObserver {
public:
	static CBBSubSessionImpl* NewL(MApp_context& Context, CBBSessionImpl* aSession, MBBObserver* aObserver);
	~CBBSubSessionImpl();
private:
	virtual void AddNotificationL(const TTupleName& aTupleName);
	virtual void AddNotificationL(const TComponentName& aComponentName);
	virtual void DeleteNotifications();

	virtual void GetL(const TTupleName& aName, const TDesC& aSubName, 
			MBBData*& aDataInto);
	virtual void PutL(const TTupleName& aTupleName, const TDesC& aSubName, 
			const MBBData* aData, const TComponentName& aComponentName=KNoComponent);
	virtual void PutRequestL(const TTupleName& aTupleName, const TDesC& aSubName, 
			const MBBData* aData, const TComponentName& aComponentName);
	virtual void PutReplyL(const TTupleName& aTupleName, const TDesC& aSubName, 
			const MBBData* aData, const TComponentName& aComponentName);
	virtual void DeleteL(TUint id);

	CBBSubSessionImpl(MApp_context& Context, CBBSessionImpl* aSession, MBBObserver* aObserver);
	virtual void NewValue(TUint aId, 
		const TTupleName& aTupleName, const TDesC& aSubName, 
		const TComponentName& aComponentName, 
		const CRefCountedData* aData);

	void Async();

	void ConstructL();

	void CheckedRunL();
	void DoCancel();

	struct TItem {
		TUint		iId;
		TTupleName	iName;
		TBuf<128>	 iSubName;
		TComponentName	iComponentName;
		const CRefCountedData* iData;
		TItem(TUint aId, const TTupleName& aName, const TDesC& aSubName, const TComponentName& aComponentName,
			const CRefCountedData* aData) :
			iId(aId), iName(aName), iSubName(aSubName), iComponentName(aComponentName), iData(aData) { }
		TItem() : iId(0), iName(KNoTuple), iComponentName(KNoComponent), iData(0) { }
	};

	CList<TItem> *iPending;
	CBBSessionImpl *iSession;
	MBBObserver*	iObserver;
};

CBBSubSession* CBBSessionImpl::CreateSubSessionL(MBBObserver* aObserver) {
	return CBBSubSessionImpl::NewL(AppContext(), this, aObserver);
}

CBBSubSessionImpl::~CBBSubSessionImpl()
{
	CALLSTACKITEM_N(_CL("CBBSubSessionImpl"), _CL("~CBBSubSessionImpl"));

	Cancel();
	if (iPending) {
		TItem t=iPending->Pop();
		while(t.iData) {
			t.iData->Release();
			t=iPending->Pop();
		}
		delete iPending;
	}
	iSession->DeleteNotifications(this);
}

CBBSubSessionImpl* CBBSubSessionImpl::NewL(MApp_context& Context, CBBSessionImpl* aSession, MBBObserver* aObserver)
{
	CALLSTACKITEM2_N(_CL("CBBSubSessionImpl"), _CL("NewL"), &Context);

	auto_ptr<CBBSubSessionImpl> ret(new (ELeave) CBBSubSessionImpl(Context, aSession, aObserver));
	ret->ConstructL();
	return ret.release();
}

void CBBSubSessionImpl::ConstructL()
{
	CALLSTACKITEM_N(_CL("CBBSubSessionImpl"), _CL("ConstructL"));

	if (iObserver) {
		CActiveScheduler::Add(this);
		iPending=CList<TItem>::NewL();
	}
}

void CBBSubSessionImpl::AddNotificationL(const TTupleName& aTupleName)
{
	CALLSTACKITEM_N(_CL("CBBSubSessionImpl"), _CL("AddNotificationL"));

	if (!iObserver) User::Leave(KErrArgument);
	iSession->AddNotificationL(aTupleName, this);
}

void CBBSubSessionImpl::AddNotificationL(const TComponentName& aComponentName)
{
	CALLSTACKITEM_N(_CL("CBBSubSessionImpl"), _CL("AddNotificationL"));

	if (!iObserver) User::Leave(KErrArgument);
	iSession->AddNotificationL(aComponentName, this);
}

void CBBSubSessionImpl::DeleteNotifications()
{
	CALLSTACKITEM_N(_CL("CBBSubSessionImpl"), _CL("DeleteNotifications"));

	iSession->DeleteNotifications(this);
}

void CBBSubSessionImpl::GetL(const TTupleName& aName, const TDesC& aSubName, 
			MBBData*& aDataInto)
{
	CALLSTACKITEM_N(_CL("CBBSubSessionImpl"), _CL("GetL"));

	iSession->GetL(aName, aSubName, aDataInto);
}
void CBBSubSessionImpl::PutL(const TTupleName& aTupleName, const TDesC& aSubName, 
		const MBBData* aData, const TComponentName& aComponentName)
{
	CALLSTACKITEM_N(_CL("CBBSubSessionImpl"), _CL("PutL"));

	iSession->DoPutL(aTupleName, aSubName, aComponentName, aData, ETrue);
}

void CBBSubSessionImpl::PutRequestL(const TTupleName& aTupleName, const TDesC& aSubName, 
			const MBBData* aData, const TComponentName& aComponentName)
{
	iSession->DoPutL(aTupleName, aSubName, aComponentName, aData, EFalse);
}
void CBBSubSessionImpl::PutReplyL(const TTupleName& aTupleName, const TDesC& aSubName, 
		const MBBData* aData, const TComponentName& aComponentName)
{
	iSession->DoPutL(aTupleName, aSubName, aComponentName, aData, EFalse, ETrue);
}

void CBBSubSessionImpl::NewValue(TUint aId, const TTupleName& aTupleName, const TDesC& aSubName, const TComponentName& aComponentName, 
	const CRefCountedData* aData)
{
	CALLSTACKITEM_N(_CL("CBBSubSessionImpl"), _CL("NewValue"));

	TItem t(aId, aTupleName, aSubName, aComponentName, aData);
	iPending->AppendL(t);
	t.iData->AddRef();
	Async();
}

_LIT(KSubSession, "CBBSubSession");

CBBSubSession::CBBSubSession(MApp_context& Context) : CCheckedActive(EPriorityNormal, KSubSession),
		MContextBase(Context) { }

CBBSubSessionImpl::CBBSubSessionImpl(MApp_context& Context, CBBSessionImpl* aSession, 
				     MBBObserver* aObserver) : CBBSubSession(Context), 
				     iSession(aSession), iObserver(aObserver) { }

void CBBSubSessionImpl::Async()
{
	CALLSTACKITEM_N(_CL("CBBSubSessionImpl"), _CL("Async"));

	if (IsActive()) return;
	TRequestStatus *s=&iStatus;
	User::RequestComplete(s, KErrNone);
	SetActive();
}

void CBBSubSessionImpl::CheckedRunL()
{
	CALLSTACKITEM_N(_CL("CBBSubSessionImpl"), _CL("CheckedRunL"));

	TItem t=iPending->Pop();
	TInt err;
	TRAP(err, iObserver->NewValueL(t.iId, t.iName, t.iSubName, t.iComponentName, t.iData->Get()));
	t.iData->Release();
	User::LeaveIfError(err);
	if (iPending->iCount > 0) Async();
}

void CBBSubSessionImpl::DoCancel()
{
	CALLSTACKITEM_N(_CL("CBBSubSessionImpl"), _CL("DoCancel"));

}

void CBBSubSessionImpl::DeleteL(TUint id)
{
	iSession->DeleteL(id);
}
