/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


//CBlackBoardServer.cpp
#include "BlackBoardServer.h"
#include "BlackBoardServerSession.h"
#include "BlackBoard_cs.h"
#include <E32SVR.H>
#include <basched.h>

#include "app_context.h"
#include "app_context_impl.h"
#include <flogger.h>
#include "symbian_auto_ptr.h"
#include "callstack.h"

#include "permanent.h"

void Log(const TDesC& msg);

bool ClientAlive(TInt ThreadId);

#pragma warning(disable:4706)

CBlackBoardServer::CBlackBoardServer(TInt aPriority, MApp_context& Context) : CServer(aPriority), MContextBase(Context)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("CBlackBoardServer"));


	
}

CBlackBoardServer::~CBlackBoardServer()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("~CBlackBoardServer"));

	delete iSubDuplicate;
	delete iPermanent;
	delete iSubscriptions;
	delete iTupleStore;
	delete iDb;
}


CBlackBoardServer* CBlackBoardServer::NewL(MApp_context& Context, TBool aRunAsServer)
{
	CALLSTACKITEM2_N(_CL("CBlackBoardServer"), _CL("NewL"), &Context);


	CBlackBoardServer* ret = new (ELeave) CBlackBoardServer(EPriorityNormal, Context);
    	CleanupStack::PushL(ret);
    	ret->ConstructL(aRunAsServer);
    	CleanupStack::Pop();
    	return ret;
}

void CBlackBoardServer::ConstructL(TBool aRunAsServer)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("ConstructL"));


	iSubDuplicate=CGenericIntMap::NewL();

	iDb=CDb::NewL(AppContext(), _L("TUPLE"), EFileRead|EFileWrite|EFileShareAny);
	iTupleStore=CTupleStore::NewL(iDb->Db(), AppContext());
	iSubscriptions=CSubscriptions::NewL();
	iPermanent=CPermanentSubscriptions::NewL(AppContext(), this);
	
	TRAPD(err, iPermanent->ReadSubscriptionsL());

	if (err!=KErrNone) {
		//todo: log
	}

	if (aRunAsServer) {
		StartL(KBlackBoardServerName);
	}
}

CSharableSession* CBlackBoardServer::NewSessionL(const TVersion& aVersion) const
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("NewSessionL"));


	// check version
	if (!User::QueryVersionSupported(TVersion(KBlackBoardServMajorVersionNumber,
                                                KBlackBoardServMinorVersionNumber,
                                                KBlackBoardServBuildVersionNumber),
                                                aVersion))
	{
		User::Leave(KErrNotSupported);
	}
	
	// create new session
	RThread client = Message().Client();
	return CBlackBoardServerSession::NewL(client, *const_cast<CBlackBoardServer*> (this));
}


void CBlackBoardServer::IncrementSessions()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("IncrementSessions"));


    	iSessionCount++;
}

void CBlackBoardServer::DecrementSessions()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("DecrementSessions"));


    	iSessionCount--;
    	if (iSessionCount <= 0)
	{
      		iSessionCount =0;
		CActiveScheduler::Stop();
	}    
}

TInt CBlackBoardServer::CheckedRunError(TInt aError)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("CheckedRunError"));


	if (aError == KErrBadDescriptor)
	{
        	// A bad descriptor error implies a badly programmed client, so panic it;
        	// otherwise report the error to the client
        	PanicClient(Message(), EBBBadDescriptor);
	}
	else
	{
		Message().Complete(aError);
	}

	//
	// The leave will result in an early return from CServer::RunL(), skipping
	// the call to request another message. So do that now in order to keep the
	// server running.
	ReStart();

	return KErrNone;	// handled the error fully
}

void CBlackBoardServer::PanicClient(const RMessage& aMessage, TBlackBoardServPanic aPanic)
{
	CALLSTACKITEMSTATIC_N(_CL("CBlackBoardServer"), _CL("PanicClient"));


	aMessage.Panic(KBlackBoardServer, aPanic);
}

void CBlackBoardServer::PanicServer(TBlackBoardServPanic aPanic)
{
	CALLSTACKITEMSTATIC_N(_CL("CBlackBoardServer"), _CL("PanicServer"));


    	User::Panic(KBlackBoardServer, aPanic);
}

void CBlackBoardServer::RunL()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("RunL"));

	AppContext().CallStackMgr().ResetCallStack();


	TRAPD(err, CServer::RunL());
	if (err!=KErrNone) {
		TBuf<40> msg;
		msg.Format(_L("Error in RunL: %d"), err);
		auto_ptr<HBufC> s(AppContext().CallStackMgr().GetFormattedCallStack(_L("BBS")));
		Log(*s);
		//Log(msg);
		CheckedRunError(err);
	}

	AppContext().CallStackMgr().ResetCallStack();
}

void CBlackBoardServer::ThreadFunctionL()
{

	#if defined (__WINS__)
	UserSvr::ServerStarted();
	#endif

	__UHEAP_MARK;

	{
	auto_ptr<CApp_context> c(CApp_context::NewL(false, _L("BlackBoardServer")));
#ifndef __WINS__
	c->SetDataDir(_L("c:\\system\\data\\context\\"), false);
#endif

	// make sure the call stack item is not on the stack
	// after app context is deleted

    	// Construct active scheduler
    	auto_ptr<CActiveScheduler> activeScheduler(new (ELeave) CBaActiveScheduler);

    	// Install active scheduler
    	// We don't need to check whether an active scheduler is already installed
    	// as this is a new thread, so there won't be one
    	CActiveScheduler::Install(activeScheduler.get());

    	// Construct our server
	auto_ptr<CBlackBoardServer> s(CBlackBoardServer::NewL(*c));    

	RSemaphore semaphore;
	if (semaphore.CreateGlobal(KBlackBoardServerSemaphoreName, 0)!=KErrNone) {
		User::LeaveIfError(semaphore.OpenGlobal(KBlackBoardServerSemaphoreName));
	}
	
	// Semaphore opened ok
	semaphore.Signal();
	semaphore.Close();


	// Start handling requests
	CActiveScheduler::Start();
	}
		 
	__UHEAP_MARKEND;

}

TInt CBlackBoardServer::ThreadFunction(TAny* /*aNone*/)
{
    	CTrapCleanup* cleanupStack = CTrapCleanup::New();
	if (cleanupStack == NULL)
	{
      	PanicServer(ECreateTrapCleanup);
	}

	TRAPD(err, ThreadFunctionL());
    	if (err != KErrNone)
	{
        	PanicServer(ESrvCreateServer);
	}
	
    	delete cleanupStack;
    	cleanupStack = NULL;

	return KErrNone;
}

//---------------------------------------------------------------------------

void CBlackBoardServer::GetL(TTupleName& aName, TDes& aSubName,
			     TUint& aIdInto, TTupleType& aTupleTypeInto,
		TComponentName& aComponentInto,
		RADbColReadStream& aDataInto, TUint& aSizeInto)

{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("GetL"));

	if (!iTupleStore->FirstL(ETupleDataOrRequest, aName, aSubName, ETrue)) User::Leave(KErrNotFound);
	iTupleStore->GetCurrentL(aName, aSubName, aIdInto, aTupleTypeInto, aComponentInto, aDataInto, aSizeInto);
}

void CBlackBoardServer::GetL(TComponentName& aComponent, TUint& aIdInto, TTupleType& aTupleTypeInto,
		TTupleName& aNameInto, TDes& aSubNameInto,
		RADbColReadStream& aDataInto, TUint& aSizeInto)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("GetL"));

	if (!iTupleStore->FirstL(ETuplePermanentSubscriptionEvent, aComponent)) {
		if (!iTupleStore->FirstL(ETupleReply, aComponent)) {
			User::Leave(KErrNotFound);
		}
	}
	iTupleStore->GetCurrentL(aNameInto, aSubNameInto, aIdInto, aTupleTypeInto, aComponent, aDataInto, aSizeInto);
}

void CBlackBoardServer::GetL(TUint aId, TTupleType& aTupleTypeInto,
			     TTupleName& aName, TDes& aSubName,
	TComponentName& aComponentInto,
	RADbColReadStream& aDataInto, TUint& aSizeInto)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("GetL"));

	iTupleStore->SeekL(aId);
	iTupleStore->GetCurrentL(aName, aSubName, aId, aTupleTypeInto, aComponentInto, aDataInto, aSizeInto);
}


void CBlackBoardServer::NotifyTupleL(TUint aId,
			const TTupleName& aTupleName, const TDesC& aSubName, 
			const TComponentName& aComponent,
			const TDesC8& aSerializedData)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("NotifyTupleL"));

	MBlackBoardObserver *o=0;
	TBBPriority pr;
	iSubDuplicate->Reset();
	auto_ptr< CList<MBlackBoardObserver*> > to_notify(CList<MBlackBoardObserver*>::NewL());
	for (o=iSubscriptions->FirstL(aTupleName, pr); o; o=iSubscriptions->NextL(pr)) {
		if (! iSubDuplicate->GetData( (uint32) o) ) {
			to_notify->AppendL(o);
			iSubDuplicate->AddDataL((uint32) o, (void*)1);
		}
	}

	TInt err, err_ret=KErrNone;

	while(o=to_notify->Pop()) {
		TRAP(err, o->NotifyL(aId, pr, ETupleDataOrRequest, aTupleName, aSubName, aComponent, aSerializedData));
		if (err!=KErrNone) err_ret=err;
	}
	User::LeaveIfError(err_ret);
}

void CBlackBoardServer::NotifyComponentL(TUint aId, TBBPriority aPriority,
					 TTupleType aTupleType, 
			const TTupleName& aTupleName, const TDesC& aSubName, 
			const TComponentName& aComponent,
			const TDesC8& aSerializedData)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("NotifyComponentL"));

	auto_ptr< CList<MBlackBoardObserver*> > to_notify(CList<MBlackBoardObserver*>::NewL());
	MBlackBoardObserver *o=0;
	TBBPriority pr;
	iSubDuplicate->Reset();
	for (o=iSubscriptions->FirstL(aComponent, pr); o; o=iSubscriptions->NextL(pr)) {
		if (! iSubDuplicate->GetData( (uint32) o) ) {
			to_notify->AppendL(o);
			iSubDuplicate->AddDataL((uint32) o, (void*)1);
		}
	}
	TInt err, err_ret=KErrNone;
	while (o=to_notify->Pop()) {
		TRAP(err, o->NotifyL(aId, aPriority, aTupleType, aTupleName, aSubName, aComponent, aSerializedData));
		if (err!=KErrNone) err_ret=err;
	}
	User::LeaveIfError(err_ret);
}

TInt CBlackBoardServer::PutL(const TTupleName& aTupleName, const TDesC& aSubName, 
					const TComponentName& aComponent,
					const TDesC8& aSerializedData, TBBPriority aPriority, 
					TBool aReplace, TUint& aIdInto,
					TBool aPersist, TTupleType aTupleType)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("PutL"));

	if (aPersist && !aReplace && NoSpaceLeft()) {
		//FIXME reporting?
		//User::Leave(KErrDiskFull);
		return KErrNone;
	}

	if (aPersist) {
		aIdInto=iTupleStore->PutL(aTupleType, aTupleName, aSubName, aComponent,
			aSerializedData, aPriority, aReplace);
	} else {
		aIdInto=0;
	}

	TInt err1=KErrNone, err2=KErrNone;

	if (aTupleType==ETupleDataOrRequest) {
		TRAP(err1, NotifyTupleL(aIdInto, aTupleName, aSubName,
			aComponent, aSerializedData));
	} else {
		TRAP(err2, NotifyComponentL(aIdInto, aPriority, aTupleType, aTupleName, aSubName,
			aComponent, aSerializedData));
	}

	if (err1!=KErrNone) return err1;
	return err2;
}

void CBlackBoardServer::DeleteL(const TTupleName& aName, const TDesC& aSubName)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("DeleteL"));

	iTupleStore->DeleteL(ETupleDataOrRequest, aName, aSubName);
}

void CBlackBoardServer::DeleteL(TUint aId)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("DeleteL"));

	TTupleType aTupleTypeInto; TTupleName aNameInto; TBuf<KMaxTupleSubNameLength> aSubNameInto;
	iTupleStore->DeleteL(aId, aTupleTypeInto, aNameInto, aSubNameInto);
}


void CBlackBoardServer::NotifyExistingL(MBlackBoardObserver *aSession, 
		const TTupleName& aTupleName, TBBPriority aPriority)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("NotifyExistingL"));

	TInt err, err_ret=KErrNone;
	TBool more=iTupleStore->FirstL(ETupleDataOrRequest, aTupleName, _L(""));
	while (more) {
		TRAP(err, aSession->NotifyL(iTupleStore->GetCurrentIdL(), aPriority));
		if (err!=KErrNone) err_ret=err;
		more=iTupleStore->NextL();
	}
	User::LeaveIfError(err_ret);
}

void CBlackBoardServer::NotifyExistingL(MBlackBoardObserver *aSession, 
		const TComponentName& aComponentName, TBBPriority aPriority)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("NotifyExistingL"));

	TInt err, err_ret=KErrNone;
	TBool more=iTupleStore->FirstL(ETupleRequest, aComponentName);
	TInt count=0;
	while (more) {
		TRAP(err, aSession->NotifyL(iTupleStore->GetCurrentIdL(), aPriority));
		if (err!=KErrNone) 
			err_ret=err;
		TRAP(err, more=iTupleStore->NextL());
		if (err!=KErrNone) 
			err_ret=err;
		count++;
	}
	more=iTupleStore->FirstL(ETuplePermanentSubscriptionEvent, aComponentName);
	while (more) {
		TRAP(err, aSession->NotifyL(iTupleStore->GetCurrentIdL(), aPriority));
		if (err!=KErrNone) err_ret=err;
		more=iTupleStore->NextL();
	}
	more=iTupleStore->FirstL(ETupleReply, aComponentName);
	while (more) {
		TRAP(err, aSession->NotifyL(iTupleStore->GetCurrentIdL(), aPriority));
		if (err!=KErrNone) err_ret=err;
		more=iTupleStore->NextL();
	}
	User::LeaveIfError(err_ret);
}

TInt CBlackBoardServer::AddNotificationL(MBlackBoardObserver *aSession, 
		const TTupleName& aTupleName,
		TBool aGetExisting, TBBPriority aPriority)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("AddNotificationL"));

	iSubscriptions->AddNotificationL(aSession, aTupleName, aPriority);
	TInt err=KErrNone;

	if (aGetExisting) 
		TRAP(err, NotifyExistingL(aSession, aTupleName, aPriority));
	return err;
}

void CBlackBoardServer::DeleteNotificationL(MBlackBoardObserver *aSession, 
	const TTupleName& aTupleName)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("DeleteNotificationL"));

	iSubscriptions->DeleteNotificationL(aSession, aTupleName);
}

TInt CBlackBoardServer::AddNotificationL(MBlackBoardObserver *aSession, 
	const TComponentName& aComponentName, 
	TBool aGetExisting, TBBPriority aPriority)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("AddNotificationL"));

	iSubscriptions->AddNotificationL(aSession, aComponentName, aPriority);
	TInt err=KErrNone;

	if (aGetExisting) 
		TRAP(err, NotifyExistingL(aSession, aComponentName, aPriority));
	return err;
}

void CBlackBoardServer::DeleteNotificationL(MBlackBoardObserver *aSession, 
	const TComponentName& aComponentName)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("DeleteNotificationL"));

	iSubscriptions->DeleteNotificationL(aSession, aComponentName);
}

void CBlackBoardServer::DeleteAllNotificationsL(MBlackBoardObserver *aSession)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("DeleteAllNotificationsL"));

	iSubscriptions->DeleteAllSubscriptionsL(aSession);
}

void CBlackBoardServer::DeleteAllNotificationsL()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("DeleteAllNotificationsL"));

	iSubscriptions->DeleteAllSubscriptionsL();
}

void CBlackBoardServer::TerminateServer()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServer"), _CL("TerminateServer"));

	CActiveScheduler::Stop();
}

void LogL(const TDesC& msg)
{
	RFileLogger iLog;
	User::LeaveIfError(iLog.Connect());
	CleanupClosePushL(iLog);
	iLog.CreateLog(_L("Context"),_L("Engine"),EFileLoggingModeAppend);
	TInt i=0;
#ifdef __WINS__
	while (i<msg.Length()) {
#else
	{
#endif
		RDebug::Print(msg.Mid(i));
		iLog.Write(msg.Mid(i));
		i+=128;
	}
	
	iLog.CloseLog();
	CleanupStack::PopAndDestroy();
}

void Log(const TDesC& msg)
{
	TRAPD(err, LogL(msg));
	// ignore error, not much else we can do
	// and not critical
}