/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


//CBlackBoardServerSession.cpp

#include "BlackBoardServerSession.h"
#include "BlackBoard_cs.h"
#include <e32svr.h>
#include <flogger.h>
#include "symbian_auto_ptr.h"
#include "bberrors.h"
#include "app_context_impl.h"
#include "callstack.h"

bool ClientAlive(TInt ThreadId)
{
	CALLSTACKITEM_N(_CL("CleanupStack"), _CL("PopAndDestroy"));


	if (ThreadId==0) return false;

	RThread c; bool ret = false;
	if (c.Open(ThreadId) != KErrNone) return false;
	if (c.ExitType() == EExitPending) ret=true;
	c.Close();
	return ret;
}

class CIdleCallBack : public CActive {
private:
	TBool iEnabled;
	MBack& iBack;
public:

	static CIdleCallBack* NewL(MBack& aBack) {
		auto_ptr<CIdleCallBack> ret(new (ELeave) CIdleCallBack(aBack));
		ret->ConstructL();
		return ret.release();
	}

	CIdleCallBack(MBack& aBack) : CActive(EPriorityIdle), iEnabled(EFalse), iBack(aBack) { }
	void ConstructL() {
		CActiveScheduler::Add(this);
	}
	void RunL() { if(iEnabled) iBack.Back(); }
	void DoCancel() { iEnabled=EFalse; }
	void Reset() { iEnabled=EFalse; }
	void Trigger() { 
		iEnabled=ETrue;
		if (!IsActive()) { 
			TRequestStatus* s=&iStatus;
			User::RequestComplete(s, KErrNone);
			SetActive();
		}
	}
	~CIdleCallBack() { Cancel(); }
};

CBlackBoardServerSession::CBlackBoardServerSession(RThread& aClient, CBlackBoardServer& aServer) : CSession(aClient), iServer(aServer)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("CBlackBoardServerSession"));


    
}

CBlackBoardServerSession::~CBlackBoardServerSession()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("~CBlackBoardServerSession"));


	delete iIdle;

	//FIXME: what happens if server is shutting down?
	iServer.DeleteAllNotificationsL(this);

	/*
	TInt i;
	for (i=0; i<KPriorityCount; i++) {
		delete iNotifications[i];
	}
	*/
	delete iNotifications;

    	iServer.DecrementSessions();
}

void CBlackBoardServerSession::CompleteMessage(TMessageIdxs aWhich, TInt Code)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("CompleteMessage"));


	if (ClientAlive(iMessageThreadId[aWhich]) && iMessage[aWhich]!=RMessage()) {
		iMessage[aWhich].Complete(Code);
	}
	if (aWhich==ENotifyMsgIndex) iWaitingForNotify=EFalse;

	SetMessage(aWhich, RMessage());
}

void CBlackBoardServerSession::ServiceL(const RMessage& aMessage)
{
	GetContext()->CallStackMgr().ResetCallStack();


	TMessageIdxs msgidx;

	msgidx=EOtherMsgIndex;
	if (aMessage.Function()!=ECancelOther && aMessage.Function()!=ECancelNotify) {
		// allow other calls while waiting for notify
		if (aMessage.Function()==ENotifyOnChange) {
			msgidx=ENotifyMsgIndex;	
		}
		SetMessage(msgidx, aMessage);
		ReadFlags(msgidx);
	}

	TInt err;
	
	switch (aMessage.Function())
	{
		case EGetByTuple:
			TRAP(err, GetByTupleL());
			if (err!=KErrNone) CompleteMessage(msgidx, err);
			break;
		case EGetByComponent:
			TRAP(err, GetByComponentL());
			if (err!=KErrNone) CompleteMessage(msgidx, err);
			break;
		case EPut:
			TRAP(err, PutL());
			if (err!=KErrNone) CompleteMessage(msgidx, err);
			break;
		case EDeleteByTuple:
			TRAP(err, DeleteByTupleL());
			if (err!=KErrNone) CompleteMessage(msgidx, err);
			break;
		case EDeleteById:
			TRAP(err, DeleteByIdL());
			if (err!=KErrNone) CompleteMessage(msgidx, err);
			break;
		case EAddNotifyByTupleFilter:
			TRAP(err, AddTupleFilterL());
			if (err!=KErrNone) CompleteMessage(msgidx, err);
			break;
		case EDeleteNotifyByTupleFilter:
			TRAP(err, DeleteTupleFilterL());
			if (err!=KErrNone) CompleteMessage(msgidx, err);
			break;
		case EAddNotifyByComponentFilter:
			TRAP(err, AddComponentFilterL());
			if (err!=KErrNone) CompleteMessage(msgidx, err);
			break;
		case EDeleteNotifyByComponentFilter:
			TRAP(err, DeleteComponentFilterL());
			if (err!=KErrNone) CompleteMessage(msgidx, err);
			break;
		case ENotifyOnChange:
			if (!SendWaitingL()) iWaitingForNotify=ETrue;
			break;
		case ETerminateBlackBoardServer:
			TerminateServer();
			break;
		case ECancelOther:
			// hmm. Since all of these messages get
			// Complete()d when they come in, there isn't
			// a pending message, which makes this is a NOP
			// it'll do something, tho, if we change
			// some message to be really async
			CompleteMessage(EOtherMsgIndex, KErrCancel);
			// but this should be correct even if the
			// above does nothing
			aMessage.Complete(KErrNone);
			break;
		case ECancelNotify:
			iWaitingForNotify=EFalse;
			CompleteMessage(ENotifyMsgIndex, KErrCancel);
			aMessage.Complete(KErrNone);
			break;
		default :
            	PanicClient(EBBBadRequest);
	}

}

void CBlackBoardServerSession::PanicClient(TInt aPanic) const
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("PanicClient"));


	Panic(KBlackBoardServer, aPanic) ; // Note: this panics the client thread, not server
}

CBlackBoardServerSession* CBlackBoardServerSession::NewL(RThread& aClient, CBlackBoardServer& aServer)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("NewL"));


	CBlackBoardServerSession* self = CBlackBoardServerSession::NewLC(aClient, aServer) ;
    	CleanupStack::Pop(self) ;
    	return self ;
}

CBlackBoardServerSession* CBlackBoardServerSession::NewLC(RThread& aClient, CBlackBoardServer& aServer)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("NewLC"));


	CBlackBoardServerSession* self = new (ELeave) CBlackBoardServerSession(aClient, aServer);
    	CleanupStack::PushL(self) ;
    	self->ConstructL();
    	return self ;
}

void CBlackBoardServerSession::ConstructL()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("ConstructL"));

	iServer.IncrementSessions();

	iIdle=CIdleCallBack::NewL(*this);

	/*
	TInt i;
	for (i=0; i<KPriorityCount; i++) {
		iNotifications[i]=CList<TUint>::NewL();
	}
	*/
	iNotifications=new (ELeave) CArrayFixSeg<TUint>(128);
}

void CBlackBoardServerSession::AppendNotificationL(TUint aId)
{
	iNotifications->AppendL(aId);
}

TUint CBlackBoardServerSession::TopNotification()
{
	if (iNotifications->Count() == 0) return 0;
	return iNotifications->At(iNotifications->Count() -1 );
}

TUint CBlackBoardServerSession::PopNotification()
{
	TUint ret=iNotifications->At(iNotifications->Count() -1 );
	iNotifications->Delete(iNotifications->Count() -1 );
	return ret;
}

//------------------------------------------------------------------------

void CBlackBoardServerSession::SetMessage(TMessageIdxs aWhich, const RMessage& aMessage) 
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("SetMessage"));


	if (aMessage==RMessage()) {
		iMessageThreadId[aWhich]=0;
	} else {
		iMessageThreadId[aWhich]=aMessage.Client().Id();
	}
	iMessage[aWhich]=aMessage;
}

void CBlackBoardServerSession::ReadTupleAndSubL(TMessageIdxs aWhich)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("ReadTupleAndSubL"));

	if ( iFlags & KArgFlagNoSubName ) {
		iFullArgs.iSubName.Zero();
		TPckg<TTupleName> p(iFullArgs.iTupleName);
		iMessage[aWhich].ReadL( iMessage[aWhich].Ptr1(), p );
	} else {
		TTupleArgs t;
		TPckg<TTupleArgs> p(t);
		iMessage[aWhich].ReadL( iMessage[aWhich].Ptr1(), p );
		iFullArgs.iTupleName=t.iTupleName;
		iFullArgs.iSubName=t.iSubName;
	}
}

void CBlackBoardServerSession::ReadComponentL(TMessageIdxs aWhich)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("ReadComponentL"));

	iFullArgs.iComponentName.iModule.iUid=iMessage[aWhich].Int1();
	iFullArgs.iComponentName.iId=iMessage[aWhich].Int2();
}


void CBlackBoardServerSession::ReadId(TMessageIdxs aWhich)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("ReadId"));

	iFullArgs.iId=(TUint)iMessage[aWhich].Ptr1();
}

void CBlackBoardServerSession::GetByTupleL()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("GetByTupleL"));

	ReadTupleAndSubL(EOtherMsgIndex);
	RADbColReadStream rs;
	iServer.GetL(iFullArgs.iTupleName, iFullArgs.iSubName, iFullArgs.iId, iFullArgs.iTupleType, iFullArgs.iComponentName, rs, iSize);
	WriteDataL(rs, EOtherMsgIndex);
}

void CBlackBoardServerSession::ReadFlags(TMessageIdxs aWhich)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("ReadFlags"));

	iFlags=(TUint)iMessage[aWhich].Ptr0();
}

void CBlackBoardServerSession::GetByComponentL()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("GetByComponentL"));

	ReadComponentL(EOtherMsgIndex);
	RADbColReadStream rs;
	iServer.GetL(iFullArgs.iComponentName, iFullArgs.iId, iFullArgs.iTupleType, iFullArgs.iTupleName, iFullArgs.iSubName, rs, iSize);
	WriteDataL(rs, EOtherMsgIndex);
}

void CBlackBoardServerSession::ReadFullArgsL(TMessageIdxs aWhich)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("ReadFullArgsL"));

	TPckg<TFullArgs> p(iFullArgs);
	p.SetLength(0);
	TRAPD(err, iMessage[aWhich].ReadL( iMessage[aWhich].Ptr1(), p ));
	User::LeaveIfError(err);
}

HBufC8* CBlackBoardServerSession::ReadDataL(TMessageIdxs aWhich)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("ReadDataL"));

	TInt len;
	len=iMessage[aWhich].Client().GetDesLength(iMessage[aWhich].Ptr3());
	if (len<0) User::Leave(len);
	auto_ptr<HBufC8> ret(HBufC8::NewL(len));
	TPtr8 p=ret->Des();
	iMessage[aWhich].ReadL( iMessage[aWhich].Ptr3(), p );
	return ret.release();
}

void CBlackBoardServerSession::PutL()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("PutL"));

	ReadFullArgsL(EOtherMsgIndex);
	TBool Replace=EFalse;
	if (iFlags & KArgFlagReplace) Replace=ETrue;

	auto_ptr<HBufC8> b(ReadDataL(EOtherMsgIndex));

	// FIXME: return warning if PutL returns an error code
	// TInt ret=
	TBool persist=ETrue;
	if (iFlags & KArgFlagDontPersist) persist=EFalse;

	if (iFlags & KArgFlagDoNotNotifySender)	iInPut=ETrue;

	TRAPD(err, iServer.PutL( iFullArgs.iTupleName, iFullArgs.iSubName, iFullArgs.iComponentName,
		*b, iFullArgs.iPriority, Replace, iFullArgs.iId, persist, iFullArgs.iTupleType ));
	iInPut=EFalse;
	User::LeaveIfError(err);

	if (iMessage[EOtherMsgIndex].Ptr2()) {
		TPckg<TUint> p(iFullArgs.iId);
		iMessage[EOtherMsgIndex].WriteL( iMessage[EOtherMsgIndex].Ptr2(), p );
	}
	CompleteMessage(EOtherMsgIndex, KErrNone);
}

void CBlackBoardServerSession::DeleteByTupleL()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("DeleteByTupleL"));

	ReadTupleAndSubL(EOtherMsgIndex);
	iServer.DeleteL(iFullArgs.iTupleName, iFullArgs.iSubName);
	CompleteMessage(EOtherMsgIndex, KErrNone);
}

void CBlackBoardServerSession::DeleteByIdL()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("DeleteByIdL"));

	ReadId(EOtherMsgIndex);
	iServer.DeleteL(iFullArgs.iId);
	CompleteMessage(EOtherMsgIndex, KErrNone);
}

void CBlackBoardServerSession::ReadPriorityL(TMessageIdxs aWhich)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("ReadPriorityL"));

	iFullArgs.iPriority=(TBBPriority)iMessage[aWhich].Int3();
}

void CBlackBoardServerSession::ReadTupleL(TMessageIdxs aWhich)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("ReadTupleL"));

	iFullArgs.iTupleName.iModule.iUid=iMessage[aWhich].Int1();
	iFullArgs.iTupleName.iId=iMessage[aWhich].Int2();
}

void CBlackBoardServerSession::AddTupleFilterL()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("AddTupleFilterL"));

	ReadTupleL(EOtherMsgIndex);
	ReadPriorityL(EOtherMsgIndex);
	TBool GetExisting=EFalse;
	if (iFlags & KArgFlagGetExisting) GetExisting=ETrue;

	TRAPD(err, iServer.AddNotificationL(this, iFullArgs.iTupleName,
		GetExisting, iFullArgs.iPriority));

	if (err==KErrNone || err==KErrAlreadyExists)
		CompleteMessage(EOtherMsgIndex, KErrNone);
	else
		CompleteMessage(EOtherMsgIndex, err);
}

void CBlackBoardServerSession::DeleteTupleFilterL()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("DeleteTupleFilterL"));

	ReadTupleAndSubL(EOtherMsgIndex);
	iServer.DeleteNotificationL(this, iFullArgs.iTupleName);
	CompleteMessage(EOtherMsgIndex, KErrNone);
}

void CBlackBoardServerSession::AddComponentFilterL()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("AddComponentFilterL"));

	ReadComponentL(EOtherMsgIndex);

	TBool GetExisting=ETrue;
	TRAPD(err, iServer.AddNotificationL(this, iFullArgs.iComponentName,
		GetExisting, EBBPriorityNormal));
	if (err==KErrNone || err==KErrAlreadyExists)
		CompleteMessage(EOtherMsgIndex, KErrNone);
	else
		CompleteMessage(EOtherMsgIndex, err);
}

void CBlackBoardServerSession::DeleteComponentFilterL()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("DeleteComponentFilterL"));

	ReadComponentL(EOtherMsgIndex);
	iServer.DeleteNotificationL(this, iFullArgs.iComponentName);
	CompleteMessage(EOtherMsgIndex, KErrNone);
}

#pragma warning(disable: 4706)
TBool CBlackBoardServerSession::SendWaitingL()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("SendWaitingL"));

	TUint id;
	if ( (id=TopNotification()) ) {
		RADbColReadStream rs;
		iServer.GetL(id, iFullArgs.iTupleType, 
			iFullArgs.iTupleName, iFullArgs.iSubName, 
			iFullArgs.iComponentName, rs, iSize);
		iFullArgs.iId=id;
		WriteDataL(rs, ENotifyMsgIndex);
		PopNotification();
		return ETrue;
	}
	return EFalse;
}

void CBlackBoardServerSession::TerminateServer()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("TerminateServer"));

	CompleteMessage(EOtherMsgIndex, KErrNone);
	iServer.TerminateServer();
}

void CBlackBoardServerSession::NotifyDirectL(TUint aId, TBBPriority aPriority,
		TTupleType aTupleType,
		const TTupleName& aTupleName, const TDesC& aSubName, 
		const TComponentName& aComponent,
		const TDesC8& aSerializedData)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("NotifyDirectL"));

	iFullArgs.iId=aId;
	iFullArgs.iTupleName=aTupleName;
	iFullArgs.iSubName=aSubName;
	iFullArgs.iComponentName=aComponent;
	iFullArgs.iPriority=aPriority;
	iFullArgs.iTupleType;

	WriteDataL(aSerializedData, ENotifyMsgIndex);
}

void CBlackBoardServerSession::NotifyL(TUint aId, TBBPriority aPriority,
			TTupleType aTupleType,
			const TTupleName& aTupleName, const TDesC& aSubName, 
			const TComponentName& aComponent,
			const TDesC8& aSerializedData)

{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("NotifyL"));

	if (iInPut) return;

	TInt err=-1;
	if ( (!aId || aPriority >= KNotifyDirectLimit) && iWaitingForNotify) {
		TRAPD(err, NotifyDirectL(aId, aPriority, aTupleType,
			aTupleName, aSubName,
			aComponent, aSerializedData));
	} 
	if (err!=KErrNone) {
		AppendNotificationL(aId);
		if (err==KClientBufferTooSmall) {
			CompleteMessage(ENotifyMsgIndex, KClientBufferTooSmall);
			iWaitingForNotify=EFalse;
		} else if (aId==0) {
			CompleteMessage(ENotifyMsgIndex, err);
			iWaitingForNotify=EFalse;
		}
		if (aId && iWaitingForNotify) iIdle->Trigger();
	}
}

void CBlackBoardServerSession::NotifyL(TUint aId, TBBPriority aPriority)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("NotifyL"));

#ifdef __WINS__
	TBuf<100> msg=_L("NOTIFYID: ");
	msg.AppendNum(aId);
#endif
	if (iInPut) return;

	AppendNotificationL(aId);

	if (iWaitingForNotify) iIdle->Trigger();
}

void CBlackBoardServerSession::Back()
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("Back"));

	if (!iWaitingForNotify) return;
	TRAPD(err, SendWaitingL());
	if (err!=KErrNone) CompleteMessage(ENotifyMsgIndex, err);
	iWaitingForNotify=EFalse;
}

void CBlackBoardServerSession::WriteDataL(const TDesC8& data, TMessageIdxs aWhich)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("WriteDataL"));

	if ( (iMessage[aWhich]) ==RMessage()) 
		User::Leave(KErrGeneral);

	TPckg<TFullArgs> meta(iFullArgs);
	iMessage[aWhich].WriteL( iMessage[aWhich].Ptr2(), meta ); 

	if (data.Length()==0) {
		TInt x;
		x=0;
	}
	iMessage[aWhich].WriteL( iMessage[aWhich].Ptr3(), data );

	CompleteMessage(aWhich, KErrNone);
}

void CBlackBoardServerSession::WriteDataL(RADbColReadStream& rs, TMessageIdxs aWhich)
{
	CALLSTACKITEM_N(_CL("CBlackBoardServerSession"), _CL("WriteDataL"));

	if ( (iMessage[aWhich])==RMessage()) 
		User::Leave(KErrGeneral);

	TUint len;
	len=iMessage[aWhich].Client().GetDesMaxLength(iMessage[aWhich].Ptr3());
	if (len < iSize ) User::Leave(KClientBufferTooSmall);

	auto_ptr<HBufC8> b(HBufC8::NewL(iSize));
	TPtr8 p=b->Des();
	if (iSize>0) {
		rs.ReadL(p, iSize);
	}

	WriteDataL(p, aWhich);
}