#include "tuplestore.h"
#include "symbian_auto_ptr.h"
#include "bbtypes.h"

static const TInt KVersion=1;

class CTupleStoreImpl : public CTupleStore, public MContextBase, public MDBStore {

	CTupleStoreImpl(RDbDatabase& Db, MApp_context& Context);
	void ConstructL();

	virtual void GetCurrentL(TTupleName& aNameInto, TDes& aSubNameInto, 
		TUint& aIdInto, TTupleType& aTupleTypeInto, 
		TComponentName& aComponentInto, RADbColReadStream& aDataInto,
		TUint& aSizeInto);
	virtual TUint PutL(TTupleType aTupleType, const TTupleName& aTupleName, const TDesC& aSubName, 
		const TComponentName& aComponent,
		const TDesC8& aSerializedData, TBBPriority aPriority, TBool aReplace);
	virtual void DeleteL(TTupleType aTupleType, const TTupleName& aName, const TDesC& aSubName);
	virtual void DeleteL(TUint aId, TTupleType& aTupleTypeInto, TTupleName& aNameInto, TDes& aSubNameInto);

	virtual void SeekL(TUint aId);
	virtual TBool FirstL(TTupleType aTupleType, const TTupleName& aTupleName, const TDesC& aSubName, 
			TBool aExact=ETrue);
	virtual TBool FirstL(TTupleType aTupleType, const TComponentName& aComponentName);
	virtual TBool NextL();
	virtual TUint GetCurrentIdL();

	enum TPartialMatch {
		EMatchNone,
		EMatchPartial,
		EMatchWildCard,
		EMatchFull
	};
	TPartialMatch MatchesL(TTupleType aTupleType, const TTupleName& aTupleName, const TDesC& aSubName);
	TPartialMatch MatchesPartialL(TTupleType aTupleType, const TTupleName& aTupleName, const TDesC& aSubName);
	TBool MatchesL(TTupleType aTupleType, const TComponentName& aComponentName);

	TBool SeekNameL(TTupleType aTupleType, const TTupleName& aName, const TDesC& aSubName);
	TBool SeekNameL(TTupleType aTupleType, const TTupleName& aName, const TComponentName& aComponentName);
	TBool SeekIdL(TUint aId);
	void GetLA(RADbColReadStream& aDataInto, TUint& aSizeInto);

	~CTupleStoreImpl();

	friend class CTupleStore;
	friend class auto_ptr<CTupleStoreImpl>;

	enum TColumns {
		ETupleId=1,
		ETupleType,
		ENameModule,
		ENameId,
		ENameSubName,
		EPriority,
		EComponentModule,
		EComponentId,
		EData 
	};
	enum TIndices {
		EIndexTuple = 0,
		EIndexId,
		EIndexComponent,
	};
	TTupleName	iCurrentSearchTuple;
	TBuf<KMaxTupleSubNameLength> iCurrentSearchSubName;
	TTupleType	iCurrentSearchType;
	TComponentName	iCurrentSearchComponent;
	enum TCurrentSearch { ETuple, EComponent };
	TCurrentSearch	iCurrentSearchMode;
	TInt	iSubNameIndexLength;
};

CTupleStore* CTupleStore::NewL(RDbDatabase& Db, MApp_context& Context)
{
	CALLSTACKITEM_N(_CL("CTupleStore"), _CL("NewL"));

	auto_ptr<CTupleStoreImpl> ret(new (ELeave) CTupleStoreImpl(Db, Context));
	ret->ConstructL();
	return ret.release();
}

CTupleStore::~CTupleStore()
{
	CALLSTACKITEM_N(_CL("CTupleStore"), _CL("~CTupleStore"));

}

CTupleStoreImpl::CTupleStoreImpl(RDbDatabase& Db, MApp_context& Context) :
	MContextBase(Context), MDBStore(Db)
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("CTupleStoreImpl"));

}

void CTupleStoreImpl::ConstructL()
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("ConstructL"));

	iSubNameIndexLength=50;

	// name:      id,           tupletype,   
	TInt cols[]={ EDbColUint32, EDbColInt32, 
	//{     Uid,         Id,          subname }, 
		EDbColInt32, EDbColInt32, EDbColText, 
	//      priority,   
		EDbColInt8,
	// {	Uid,         Id},
		EDbColInt32, EDbColInt32, 
	//      data
		EDbColLongBinary, -1 };

	TInt col_flags[]={ TDbCol::EAutoIncrement, 0, 0, 0, 0, 0,
		0, 0, 
		0 };
	TInt idxs[]={ ETupleType, ENameModule, ENameId, ENameSubName, -2, ETupleId, 
		-2,
		ETupleType, EComponentModule, EComponentId, EPriority, 
		-1 };
	SetTextLen(KMaxTupleSubNameLength);

	MDBStore::ConstructL(cols, idxs, false, _L("TUPLES"), EFalse, col_flags);

	if (SeekIdL(0)) {
		iTable.GetL();
		// check version and migrate if necessary
		TInt version=iTable.ColInt32(ENameId);
		if (version!=KVersion) User::Leave(KErrNotSupported);
	} else {
		iTable.InsertL();
		iTable.SetColL(ETupleType, ETupleSpaceInternal);
		iTable.SetColL(ENameId, KVersion);
		MDBStore::PutL();
	}
}

void CTupleStoreImpl::GetCurrentL(TTupleName& aNameInto, TDes& aSubNameInto, 
	TUint& aIdInto, TTupleType& aTupleTypeInto, TComponentName& aComponentInto, RADbColReadStream& aDataInto,
	TUint& aSizeInto)
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("GetCurrentL"));

	GetLA(aDataInto, aSizeInto);
	aNameInto.iModule.iUid=iTable.ColInt(ENameModule);
	aNameInto.iId=iTable.ColInt(ENameId);
	aSubNameInto=iTable.ColDes16(ENameSubName);
	aComponentInto.iModule.iUid=iTable.ColInt(EComponentModule);
	aComponentInto.iId=iTable.ColInt(EComponentId);
	aIdInto=iTable.ColUint(ETupleId);
	aTupleTypeInto=(TTupleType)iTable.ColInt(ETupleType);
}

void CTupleStoreImpl::GetLA(RADbColReadStream& aDataInto, TUint& aSizeInto)
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("GetLA"));

	iTable.GetL();
	if (iTable.IsColNull(EData)) {
		aSizeInto=0;
	} else {
		aDataInto.OpenLA(iTable, EData);
		aSizeInto=aDataInto.ReadUint32L();
	}
}

TUint CTupleStoreImpl::PutL(TTupleType aTupleType, const TTupleName& aTupleName, const TDesC& aSubName, 
		const TComponentName& aComponent,
		const TDesC8& aSerializedData, TBBPriority aPriority, TBool aReplace)
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("PutL"));

	if (aTupleName.iModule.iUid == KBBAnyUidValue ||
		aTupleName.iId == KBBAnyId ||
		aComponent.iModule.iUid == KBBAnyUidValue ||
		aComponent.iId == KBBAnyId) User::Leave(KErrArgument);

	TUint ret;
	{
		TAutomaticTransactionHolder ath(iTable);

		if (aReplace && SeekNameL(aTupleType, aTupleName, aSubName)) {
			iTable.UpdateL();
		} else {
			RESET_IF_NECESSARY(iTable.InsertL());
			iTable.SetColL(ETupleType, aTupleType);
			iTable.SetColL(ENameModule, aTupleName.iModule.iUid);
			iTable.SetColL(ENameId, aTupleName.iId);
			iTable.SetColL(ENameSubName, aSubName);
			iTable.SetColL(EPriority, aPriority);
			iTable.SetColL(EComponentModule, aComponent.iModule.iUid);
			iTable.SetColL(EComponentId, aComponent.iId);
		}
		ret=iTable.ColUint(ETupleId);

		if (aSerializedData.Length() > 0) {
			RADbColWriteStream w; w.OpenLA(iTable, EData);
			w.WriteUint32L(aSerializedData.Length());
			w.WriteL(aSerializedData);
			w.CommitL();
		} else {
			iTable.SetColNullL(EData);
		}
	}

	MDBStore::PutL();
	return ret;
}
	
void CTupleStoreImpl::DeleteL(TTupleType aTupleType, const TTupleName& aName, const TDesC& aSubName)
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("DeleteL"));

	if (!SeekNameL(aTupleType, aName, aSubName)) {
		User::Leave(KErrNotFound);
	}
	TRAPD(err, MDBStore::DeleteL());
	if (err!=KErrNone) {
		iTable.Reset();
		User::Leave(err);
	}
}

void CTupleStoreImpl::DeleteL(TUint aId, TTupleType& aTupleTypeInto, TTupleName& aNameInto, TDes& aSubNameInto)
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("DeleteL"));

	if (SeekIdL(aId)) {
		iTable.GetL();
		aNameInto.iModule.iUid=iTable.ColInt(ENameModule);
		aNameInto.iId=iTable.ColInt(ENameId);
		aSubNameInto=iTable.ColDes16(ENameSubName);
		aTupleTypeInto=(TTupleType)iTable.ColInt(ETupleType);
		TRAPD(err, MDBStore::DeleteL());
		if (err!=KErrNone) {
			//iTable.Reset();
			User::Leave(err);
		}
	} else {
		User::Leave(KErrNotFound);
	}
}

CTupleStoreImpl::~CTupleStoreImpl()
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("~CTupleStoreImpl"));

}

TBool CTupleStoreImpl::SeekNameL(TTupleType aTupleType, const TTupleName& aName, const TDesC& aSubName)
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("SeekNameL"));

	TDbSeekMultiKey<4> rk;
	rk.Add(aTupleType);
	rk.Add((TInt)aName.iModule.iUid);
	rk.Add(aName.iId);
	rk.Add(aSubName);
	SwitchIndexL(EIndexTuple);
	TBool ret=EFalse;

	RESET_IF_NECESSARY(ret=iTable.SeekL(rk));
	if (!ret) return EFalse;
	TPartialMatch m=MatchesL(aTupleType, aName, aSubName);
	while (m==EMatchPartial) {
		if (!iTable.NextL()) return EFalse;
		m=MatchesL(aTupleType, aName, aSubName);
	}
	if (m>EMatchPartial) return ETrue;
	return EFalse;
}

TBool CTupleStoreImpl::SeekIdL(TUint aId)
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("SeekIdL"));

	TDbSeekKey rk(aId);
	SwitchIndexL(EIndexId);
	TBool ret=EFalse;

	
	RESET_IF_NECESSARY(ret=iTable.SeekL(rk));

	return ret;
}


CTupleStoreImpl::TPartialMatch CTupleStoreImpl::MatchesL(TTupleType aTupleType, const TTupleName& aTupleName, const TDesC& aSubName)
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("TPartialMatch"));

	TPartialMatch m=MatchesPartialL(aTupleType, aTupleName, aSubName);
	if (m == EMatchPartial ) {
		if (aSubName.CompareF(iTable.ColDes(ENameSubName))==0) return EMatchFull;
	}
	return m;
}

CTupleStoreImpl::TPartialMatch CTupleStoreImpl::MatchesPartialL(TTupleType aTupleType, 
								const TTupleName& aTupleName, const TDesC& aSubName)
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("TPartialMatch"));

	iTable.GetL();
	if (iTable.ColInt32(ETupleType) != aTupleType) return EMatchNone;

	if (aTupleName.iModule==KBBAnyUid) {
		return EMatchWildCard;
	}
	if (iTable.ColInt32(ENameModule) != aTupleName.iModule.iUid) return EMatchNone;
	if (aTupleName.iId==KBBAnyId) return EMatchWildCard;
	if (iTable.ColInt32(ENameId) != aTupleName.iId) return EMatchNone;
	if (aSubName.Length()==0) return EMatchWildCard;
	if (aSubName.Left(iSubNameIndexLength).CompareF(iTable.ColDes(ENameSubName).Left(iSubNameIndexLength))) 
		return EMatchNone;
	return EMatchPartial;
}


TBool CTupleStoreImpl::MatchesL(TTupleType aTupleType, const TComponentName& aComponentName)
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("MatchesL"));

	iTable.GetL();
	if (iTable.ColInt32(ETupleType) != aTupleType) return EFalse;

	if (aComponentName.iModule==KBBAnyUid) {
		return ETrue;
	}
	if (iTable.ColInt32(EComponentModule) != aComponentName.iModule.iUid) return EFalse;
	if (aComponentName.iId==KBBAnyId) return ETrue;
	if (iTable.ColInt32(EComponentId) != aComponentName.iId) return EFalse;
	return ETrue;
}

void CTupleStoreImpl::SeekL(TUint aId)
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("SeekL"));

	if (!SeekIdL(aId)) User::Leave(KErrNotFound);
}

TBool CTupleStoreImpl::FirstL(TTupleType aTupleType, const TTupleName& aTupleName, const TDesC& aSubName, 
		TBool aExact)
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("FirstL"));

	iCurrentSearchMode=ETuple;
	iCurrentSearchType=aTupleType;
	iCurrentSearchTuple=aTupleName;
	iCurrentSearchSubName=aSubName;
	TDbSeekMultiKey<4> rk;
	rk.Add(aTupleType);
	if (aTupleName.iModule!=KBBAnyUid) {
		rk.Add((TInt)aTupleName.iModule.iUid);
		if (aTupleName.iId != KBBAnyId) {
			rk.Add(aTupleName.iId);
			if (aSubName.Length()>0 || aExact) {
				rk.Add(aSubName);
			}
		}
	}
	SwitchIndexL(EIndexTuple);
	TBool ret=EFalse;
	RESET_IF_NECESSARY(ret=iTable.SeekL(rk));
	if (!ret) return EFalse;
	TPartialMatch m=MatchesL(aTupleType, aTupleName, aSubName);
	while (m==EMatchPartial) {
		if (!iTable.NextL()) return EFalse;
		m=MatchesL(aTupleType, aTupleName, aSubName);
	}
	if (m>EMatchPartial) return ETrue;
	return EFalse;
}

TBool CTupleStoreImpl::FirstL(TTupleType aTupleType, const TComponentName& aComponentName)
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("FirstL"));

	iCurrentSearchMode=EComponent;
	iCurrentSearchType=aTupleType;
	iCurrentSearchComponent=aComponentName;
	TDbSeekMultiKey<3> rk;
	rk.Add(aTupleType);
	if (aComponentName.iModule.iUid!=KBBAnyUidValue) {
		rk.Add((TInt)aComponentName.iModule.iUid);
		if (aComponentName.iId != KBBAnyId) {
			rk.Add(aComponentName.iId);
		}
	}
	SwitchIndexL(EIndexComponent);
	TBool ret=EFalse;
	RESET_IF_NECESSARY(ret=iTable.SeekL(rk));
	if (ret && MatchesL(aTupleType, aComponentName)) return ETrue;
	return EFalse;
}


TBool CTupleStoreImpl::NextL()
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("NextL"));

	if (! iTable.NextL()) return EFalse;
	if (iCurrentSearchMode==ETuple) {
		if (MatchesL(iCurrentSearchType, iCurrentSearchTuple, iCurrentSearchSubName)>EMatchPartial)
			return ETrue;
		return EFalse;
	} else {
		return MatchesL(iCurrentSearchType, iCurrentSearchComponent);
	}
}
TUint CTupleStoreImpl::GetCurrentIdL()
{
	CALLSTACKITEM_N(_CL("CTupleStoreImpl"), _CL("GetCurrentIdL"));

	iTable.GetL();
	return iTable.ColUint32(ETupleId);
}
