/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/

/*
 * Concepts:
 * !Creating a server!
 */

//CContextClientSession.cpp

#include "ContextCommon.h"
#include "ContextClientSession.h"
#include <e32math.h>

static const TUint KDefaultMessageSlots = 2;
static const TUid KServerUid3 = {0x10200C6B}; // matches UID in server/group/contextserver.mbm file

_LIT(KContextServerFilename, "ContextServer");

#ifdef __WINS__
static const TUint KServerMinHeapSize =  0x1000;  //  4K
static const TUint KServerMaxHeapSize = 0x1000000;  // 256K
#endif

static TInt StartServer();
static TInt CreateServerProcess();


EXPORT_C RContextClientSession::RContextClientSession() : RSessionBase(), iTimeBuffer(NULL, 0, 0)
{
	
}

EXPORT_C TInt RContextClientSession::ConnectToContextServer()
{
	TInt result;
	
	result = ::StartServer();
	if (result == KErrNone)
	{
		result = CreateSession(KContextServerName,
			Version(),
			KDefaultMessageSlots);	
	}
	return result;
}

EXPORT_C TVersion RContextClientSession::Version() const
{
	return(TVersion(KContextServMajorVersionNumber,
		KContextServMinorVersionNumber,
		KContextServBuildVersionNumber));
}

//----------------------------------------------------------------------------------------

EXPORT_C void RContextClientSession::Cancel() const
{
	SendReceive(ECancel, NULL);
}

EXPORT_C void RContextClientSession::MsgTerminateContextServer(TRequestStatus& aStatus)
{
	SendReceive(ETerminateContextServer, NULL, aStatus);
}

EXPORT_C void RContextClientSession::MsgConnectToPresenceServer(const TDesC & username, 
								const TDesC & password, 
								const TDesC & server, TUint32 accessPointID, TRequestStatus& aStatus)
{
	TAny* messageParameters[KMaxMessageArguments];
	messageParameters[0] = (void*)&username;
	messageParameters[1] = (void*)&password;
	messageParameters[2] = (void*)&server;

	TPckgC<TUint32> apPackage(accessPointID);
	messageParameters[3] = &apPackage;
	SendReceive(EConnectToPresenceServer, &messageParameters[0], aStatus);
}

EXPORT_C void RContextClientSession::MsgRequestPresenceInfo(TDes & contact, TDes & presenceInfo, TTime & send_timestamp, TRequestStatus& aStatus)
{
	iTimeBuffer.Set(reinterpret_cast<TUint8*>(&send_timestamp), sizeof(send_timestamp), sizeof(send_timestamp));
	
	TAny* messageParameters[KMaxMessageArguments];
	messageParameters[0] = & contact;
	messageParameters[1] = & presenceInfo;
	messageParameters[2] = static_cast<TAny*>(&iTimeBuffer);
	
	SendReceive(ERequestPresenceInfo, &messageParameters[0], aStatus);	
}

EXPORT_C void RContextClientSession::MsgRequestMessageNotification(TDes & contact, TDes& subject, TDes & message, TRequestStatus& aStatus)
{
	TAny* messageParameters[KMaxMessageArguments];
	messageParameters[0] = & contact;
	messageParameters[1] = & subject;
	messageParameters[2] = & message;
	
	SendReceive(ERequestMessageNotification, &messageParameters[0], aStatus);
}

EXPORT_C void RContextClientSession::MsgRequestPresenceNotification(TDes & contact, TDes & presenceInfo, TTime & send_timestamp, TRequestStatus& aStatus)
{
	iTimeBuffer.Set(reinterpret_cast<TUint8*>(&send_timestamp), sizeof(send_timestamp), sizeof(send_timestamp));
	
	
	TAny* messageParameters[KMaxMessageArguments];
	messageParameters[0] = & contact;
	messageParameters[1] = & presenceInfo;
	messageParameters[2] = static_cast<TAny*>(&iTimeBuffer);

	SendReceive(ERequestPresenceNotification, &messageParameters[0], aStatus);
}

EXPORT_C void RContextClientSession::MsgUpdateUserPresence(TDesC & info, TRequestStatus& aStatus)
{
	TAny* messageParameters[KMaxMessageArguments];
	messageParameters[0] = & info;
	SendReceive(EUpdateUserPresence, &messageParameters[0], aStatus);
}

EXPORT_C void RContextClientSession::MsgSuspendConnection(TRequestStatus& aStatus)
{
	SendReceive(ESuspendConnection, NULL, aStatus);
}

EXPORT_C void RContextClientSession::MsgResumeConnection(TRequestStatus& aStatus)
{
	SendReceive(EResumeConnection, NULL, aStatus);
}


static TInt StartServer()
{
	TInt result;
	
	TFindServer findContextServer(KContextServerName);
	TFullName name;
	
	result = findContextServer.Next(name);
	if (result == KErrNone)
        {
		// Server already running
		return KErrNone;
        }
	
	RSemaphore semaphore;        
	result = semaphore.CreateGlobal(KContextServerSemaphoreName, 0);
	if (result != KErrNone)
        {
		result=semaphore.OpenGlobal(KContextServerSemaphoreName);
        }
	if (result != KErrNone)
        {
		return  result;
	}

	result = CreateServerProcess();
	if (result != KErrNone)
        {
		return  result;
        }
	
	semaphore.Wait();
	semaphore.Close();       
	
	return  KErrNone;
}

static TInt CreateServerProcess()
{
	TInt result;
	
	const TUidType serverUid(KNullUid, KNullUid, KServerUid3);
	
#ifdef __WINS__
	
	RLibrary lib;
	result = lib.Load(KContextServerFilename, _L("z:\\system\\programs"), serverUid);
	if (result != KErrNone)
        {
		return  result;
        }
	
	//  Get the WinsMain function
	TLibraryFunction functionWinsMain = lib.Lookup(1);
	
	//  Call it and cast the result to a thread function
	TThreadFunction serverThreadFunction = reinterpret_cast<TThreadFunction>(functionWinsMain());
	
	TName threadName(KContextServerName);
	
	// Append a random number to make it unique
	threadName.AppendNum(Math::Random(), EHex);
	
	RThread server;
	
	result = server.Create(threadName,   // create new server thread
		serverThreadFunction, // thread's main function
		KDefaultStackSize,
		NULL,
		&lib,
		NULL,
		KServerMinHeapSize,
		KServerMaxHeapSize,
		EOwnerProcess);
	
	lib.Close();    // if successful, server thread has handle to library now
	
	if (result != KErrNone)
        {
		return  result;
        }
	
	server.SetPriority(EPriorityMore);
	
	
#else
	
	RProcess server;
	
	result = server.Create(KContextServerFilename, _L(""), serverUid);
	if (result != KErrNone)
        {
		return  result;
        }
	
#endif
	
	server.Resume();
	server.Close();
	
	return KErrNone;
}
