#ifndef CC_ERRORHANDLING_H_INCLUDED
#define CC_ERRORHANDLING_H_INCLUDED 1

#include <e32std.h>
#include <e32base.h>
#include "symbian_refcounted_ptr.h"
#include "context_uids.h"

/*
 * errorinfo objects are supposed to be used
 * as immutables, since they cost a lot. They are
 * constructed by framework code and should not be
 * modified by others.
 */

enum TSeverity {
	EInfo,
	EWarning,
	EError,
	ECorrupt
};

enum TErrorType {
	EBug,
	EInputData,
	ETemporary,
	ELocalEnvironment,
	ERemote
};

struct TErrorCode {
	TUint	iUid;
	TInt	iCode;
};

IMPORT_C TErrorCode MakeErrorCode(TUint iUid, TInt iCode);

const TErrorCode KErrorBug = { CONTEXT_UID_CONTEXTCOMMON, -1 };
const TErrorCode KErrorUnknown = { CONTEXT_UID_CONTEXTCOMMON, -2 };

class MErrorInfo : public MRefCounted {
public:
	virtual TErrorCode ErrorCode() const = 0;
	virtual const TDesC& StackTrace() const = 0;
	virtual const TDesC& UserMessage() const = 0;
	virtual const TDesC& TechMessage() const = 0;
	virtual TErrorType ErrorType() const = 0;
	virtual TSeverity Severity() const = 0;
	virtual const MErrorInfo* InnerError() const = 0;
	virtual const MErrorInfo* NextError() const = 0;

	virtual ~MErrorInfo() { }
};

class CErrorInfo : public CBase, public MErrorInfo {
public:
	IMPORT_C static CErrorInfo* NewL(TErrorCode aErrorCode,
		HBufC* aUserMsg, HBufC* aTechMessage, HBufC* aCallStack,
		TErrorType aErrorType, TSeverity aSeverity,
		MErrorInfo* aInnerError); // on success, takes ownership of HBufC's
};

#define TRAP_ERRORINFO(ERR, INFO, BLOCK) { TRAP(ERR.iCode, BLOCK); INFO=GetContext()->ErrorInfoMgr().GetLastErrorInfo(ERR); }
#define TRAPD_ERRORINFO(ERR, INFO, BLOCK) { TErrorCode ERR; MErrorInfo* INFO; TRAP(ERR.iCode, BLOCK); INFO=GetContext()->ErrorInfoMgr().GetLastErrorInfo(ERR); }
#endif
