/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/

/*
 * Concepts:
 * !Opening a GPRS connection!
 */

#pragma warning(disable: 4706)

#include "ver.h"

#include "connectioninit.h"
#include <cdbcols.h>
#include <commdb.h>
#include "pointer.h"
#include "app_context.h"

const TInt CConnectionOpener::MAX_RETRIES=10;

#ifndef __S60V2__
EXPORT_C CConnectionOpener* CConnectionOpener::NewL(MSocketObserver& Observer, RSocketServ& Serv)
#else
EXPORT_C CConnectionOpener* CConnectionOpener::NewL(MSocketObserver& Observer, RSocketServ& Serv, RConnection& Connection)
#endif
{
#ifndef __S60V2__
	auto_ptr<CConnectionOpener> ret(new (ELeave) CConnectionOpener(Observer, Serv));
#else
	auto_ptr<CConnectionOpener> ret(new (ELeave) CConnectionOpener(Observer, Serv, Connection));
#endif
	ret->ConstructL();
	return ret.release();
}

void CConnectionOpener::ConstructL()
{
	iWait=CTimeOut::NewL(*this);
	CActiveScheduler::Add(this);
}

#ifndef __S60V2__
CConnectionOpener::CConnectionOpener(MSocketObserver& Observer, RSocketServ &Serv) : 
CCheckedActive(EPriorityIdle, _L("ContextCommon::CConnectionOpener")), iObserver(Observer), iServ(Serv)
#else
CConnectionOpener::CConnectionOpener(MSocketObserver& Observer, RSocketServ &Serv, RConnection& Connection) : 
CCheckedActive(EPriorityIdle, _L("ContextCommon::CConnectionOpener")), iObserver(Observer), iServ(Serv), iConnection(Connection)
#endif
{
}

EXPORT_C void CConnectionOpener::MakeConnectionL(TUint32 IapID)
{
	if (IapID == TUint(-1) ) {
		User::Leave(KErrAccessDenied);
	}
	Cancel();
	iIapId=IapID;

	TTimeIntervalMicroSeconds32 w(45*1000*1000);
	if (iMutex) {
		delete iMutex;
		iMutex=0;
	}
	iMutex=CMutexRequest::NewL(* GetContext(), _L("ContextCommonConnectionInit"),
		w, &iStatus);

	current_state=WAITING_ON_MUTEX;
	SetActive();
}

void CConnectionOpener::DoMakeConnectionL()
{
	TUint32 IapID=iIapId;

	TBuf<40> msg;
	iMadeConnection=false;

	msg.Format(_L("MakeConnectionL, iap %d"), IapID);
	iObserver.info(this, msg);
		
#ifndef __S60V2__

	if (!iInitiator) iInitiator=CIntConnectionInitiator::NewL();
	TUint32 activeiap;
	if (iInitiator->GetActiveIap(activeiap)==KErrNone && activeiap==IapID) {
		iObserver.info(this, _L("connection exists"));
		iObserver.success(this);
		return;
	}
	
	iConnPref.iRanking = 1; 
	iConnPref.iDirection = ECommDbConnectionDirectionOutgoing; 
	iConnPref.iDialogPref = ECommDbDialogPrefDoNotPrompt; 
	CCommsDbConnectionPrefTableView::TCommDbIapBearer bearer; 
	bearer.iBearerSet = KMaxTUint32;
	bearer.iIapId = IapID; 
	iConnPref.iBearer = bearer;
	
	// we cannot really know if the initiator is usable
	// at this point (if retrying/reusing). Let's just
	// recreate it
	delete iInitiator; iInitiator=0;
	iInitiator=CIntConnectionInitiator::NewL();
	
	TInt ret=iInitiator->TerminateActiveConnection(iStatus);
	if (ret==KErrNone) {
		current_state=CLOSING;
		SetActive();
	} else {
		current_state=CONNECTING;
		iInitiator->ConnectL(iConnPref, iStatus);
		SetActive();
#  if defined(__WINS__)
		// we don't seem to get the right return on the
		// emulator :-(
		iStatus=KErrNone;
		return;
#  endif
	}
#else

	if (!iConnectionIsOpen) {
		User::LeaveIfError(iConnection.Open(iServ));
		iConnectionIsOpen=ETrue;
	}
	/*
	auto_ptr<CCommsDatabase> db(CCommsDatabase::NewL(EDatabaseTypeIAP));
	db->SetGlobalSettingL(TPtrC(ASK_USER_BEFORE_DIAL),(TInt)false);
	*/

	iConnPref.SetIapId(IapID);
	iConnPref.SetDirection(ECommDbConnectionDirectionOutgoing);
	iConnPref.SetDialogPreference(ECommDbDialogPrefDoNotPrompt);
	iConnPref.SetBearerSet(ECommDbBearerUnknown);
	
	TBool connected = EFalse;
	
	TUint connectionCount;
	//Enumerate currently active connections across all socket servers
	User::LeaveIfError(iConnection.EnumerateConnections(connectionCount));
	
	if (connectionCount)
	{
		TPckgBuf<TConnectionInfo> connectionInfo;
		for (TUint i = 1; i <= connectionCount; ++i)
		{
			iConnection.GetConnectionInfo(i, connectionInfo);
			msg.Format(_L("existing conn, iap %d"), connectionInfo().iIapId);
			iObserver.info(this, msg);
			
			if (connectionInfo().iIapId == IapID)
			{
				RConnection tmp;
				
				TInt ret;
				User::LeaveIfError(tmp.Open(iServ));
				CleanupClosePushL(tmp);
				ret=tmp.Attach(connectionInfo, RConnection::EAttachTypeNormal);
				if (ret==KErrNone) {
					if (iConnectionIsOpen) {
						iConnection.Close(); 
						iConnectionIsOpen=EFalse;
					}
					User::LeaveIfError(iConnection.Open(iServ));
					iConnectionIsOpen=ETrue;
					ret=iConnection.Attach(connectionInfo, RConnection::EAttachTypeNormal);
				}
				CleanupStack::PopAndDestroy(); //tmp
				if (ret==KErrNone) {
					connected = ETrue;
				} else {
					msg.Format(_L("Error in Attach %d"), ret);
					iObserver.info(this, msg);
				}
				break;
			}
			
		}
	}
	
	if (connected) {
		iMadeConnection=false;
		iWait->Reset();
		delete iMutex; iMutex=0;
		iObserver.info(this, _L("already connected"));
		iObserver.success(this);
	} else {
		Cancel();

		iStatus=KRequestPending;
		iConnection.Start(iConnPref, iStatus);
		SetActive();
		current_state=CONNECTING;
#  ifdef __WINS__
		iWait->Wait(10);
#  else
		iWait->Wait(30);
#  endif
	}	
#endif
}

void CConnectionOpener::expired(CBase*)
{
	delete iMutex; iMutex=0;
	++iRetryCount;
	if (iRetryCount>MAX_RETRIES) {
		if (current_state==WAITING_ON_MUTEX) {
			iObserver.error(this, -1003, _L("Opener retries exceeded in wait"));
		} else {
			iObserver.error(this, -1003, _L("Opener retries exceeded"));
		}
	} else {
		if (current_state==WAITING_ON_MUTEX) {
			iObserver.info(this, _L("Conn retry from wait"));
		} else {
			iObserver.info(this, _L("Conn retry"));
		}
		current_state=IDLE;
#ifdef __WINS__
		iRetrying=ETrue;
#endif
		TRAPD(err, MakeConnectionL(iIapId));
		if (err!=KErrNone) {
			iObserver.error(this, err, _L("Opener retry failed"));
		}
	}
}

void CConnectionOpener::CheckedRunL()
{
	TBuf<50> msg;
	iWait->Reset();
	
	if (current_state==WAITING_ON_MUTEX) {
		TInt err=KErrNone;
		if (iStatus==KErrNone) {
			TRAP(err, DoMakeConnectionL());
		} else if (iStatus==KErrTimedOut) {
			expired(0);
			return;
		} else {
			err=iStatus.Int();
		}
		if (err!=KErrNone) {
			delete iMutex; iMutex=0;
			iObserver.error(this, err, _L("error in makeconnection"));
		}
		return;
	}

#ifndef __S60V2__
	if ( (iStatus==KConnectionTerminated || iStatus==0)
		&& current_state==CLOSING) {
		iObserver.info(this, _L("prev conn closed"));
		// The initiator seems to close its handles after a TerminateActiveConnection
		// so it has to be recreated if we want to call some other methods
		delete iInitiator; iInitiator=0;
		iInitiator=CIntConnectionInitiator::NewL();
		iInitiator->ConnectL(iConnPref, iStatus);
		current_state=CONNECTING;
		SetActive();
		return;
	}
	
	if (iStatus==CONNECTED) return;
	
	if (iStatus!=KErrNone && iStatus!=KConnectionPref1Exists &&
		iStatus!=KConnectionPref1Created) {
		if (iRetryCount>=MAX_RETRIES) {
			msg.Format(_L("Opener error %d"), iStatus.Int());
			iObserver.error(this, iStatus.Int(), msg);
			delete iMutex; iMutex=0;
			return;
		} else {
			msg.Format(_L("Opener error %d (Retry)"), iStatus.Int());
			iObserver.info(this, msg);
			if (current_state==CONNECTING) {
				current_state=RETRYING_CONNECT;
			} else {
				current_state=RETRYING_CLOSE;
			}
			
			// it seems that we sometimes get an error even
			// though the connection gets established. We wait
			// for 15 secs, which should be enough to detect
			// the new connection
			delete iMutex; iMutex=0;
			iWait->Wait(15);
			return;
		}
	}
	
	// If the connection doesn't exist the initiator sends *2* requestcompletes:
	// one with KConnectionPref1Exists and one with KConnectionPref1Created
	// if it does only KConnectionPref1Exists is send.
	
	if (current_state==CONNECTING) {
		iMadeConnection=true;
		delete iMutex; iMutex=0;
		iObserver.success(this);
		current_state=CONNECTED;
	}
	
	if (iStatus==KConnectionPref1Exists) {
		iStatus=KRequestPending;
		SetActive();
	}
#else // __S60V2__
	msg.Format(_L("ConnectionOpener CheckedRunL state %d status %d"), current_state, iStatus.Int());
	iObserver.info(this, msg);
	
	TInt status=iStatus.Int();
#ifdef __WINS__
	iStatus=KRequestPending;
	SetActive();
#endif
	if (status==CONNECTED) {
		return;
	}

	if (status==KErrNone || status==KErrAlreadyExists) {
		if (current_state==CONNECTING) {
			iMadeConnection=true;
			delete iMutex; iMutex=0;
			current_state=CONNECTED;
			iObserver.success(this);
		}
	} else {
		msg.Format(_L("Opener error %d (Retry)"), status);
		iObserver.info(this, msg);
		if (current_state==CONNECTING) {
			current_state=RETRYING_CONNECT;
		} else {
			current_state=RETRYING_CLOSE;
		}
		
		// it seems that we sometimes get an error even
		// though the connection gets established. We wait
		// for 15 secs, which should be enough to detect
		// the new connection
		delete iMutex; iMutex=0;
		iWait->Wait(15);
		return;
	}
#endif
}

EXPORT_C bool CConnectionOpener::MadeConnection()
{
	return iMadeConnection;
}

void CConnectionOpener::DoCancel()
{
	if (current_state==WAITING_ON_MUTEX) {
		delete iMutex; iMutex=0;
	} else {
#ifndef __S60V2__
		iInitiator->Cancel();
#else
		if (iConnectionIsOpen) {
			iConnection.Close(); iConnectionIsOpen=EFalse;
		}
#ifdef __WINS__
		if (!iRetrying) {
#endif
			if (iStatus==KRequestPending) {
				TRequestStatus *s=&iStatus;
				User::RequestComplete(s, KErrCancel);
			}
#ifdef __WINS__
		}
		iRetrying=EFalse;
#endif
		TInt err=iConnection.Open(iServ);
		if (err==KErrNone) {
			iConnectionIsOpen=ETrue;
		}
#endif
	}
}

TInt CConnectionOpener::CheckedRunError(TInt aError)
{
	if (current_state==CONNECTING) {
		TBuf<50> msg;
		msg.Format(_L("CConnectionOpener::CheckedRunError %d"), aError);
		iObserver.error(this, aError, msg);
	}
	iWait->Reset();
	return KErrNone;
}

EXPORT_C void CConnectionOpener::CloseConnection(TBool doStop)
{
	
	iMadeConnection=false;
	iWait->Reset();
	iRetryCount=0;
	Cancel();
	delete iMutex; iMutex=0;
#ifndef __S60V2__
	TRAPD(err,
		if (!iInitiator) iInitiator=CIntConnectionInitiator::NewL();
		iInitiator->TerminateActiveConnection();
		);
		delete iInitiator; iInitiator=0;
#else
		if (doStop && iConnectionIsOpen) {
			//iStatus=KRequestPending;
			iConnection.Stop();
			//SetActive();
		}
		if (iConnectionIsOpen) {
			iConnection.Close();
			iConnectionIsOpen=EFalse;
		}
#endif
	current_state=IDLE;
}

EXPORT_C CConnectionOpener::~CConnectionOpener()
{
	Cancel();
	delete iMutex;
	delete iWait;
#ifndef __S60V2__
	if (iInitiator && iMadeConnection) iInitiator->TerminateActiveConnection();
	delete iInitiator; iInitiator=0;
#else
	if (iConnectionIsOpen) iConnection.Close();
#endif
}