/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


/*
 * Concepts:
 * !Robust database access!
 */
#include "db.h"

#include <bautils.h>
#include <e32std.h>
#include "symbian_auto_ptr.h"
#include "app_context_fileutil.h"

const int COMPACT_BETWEEN_UPDATES=20;
const int COMPACT_BETWEEN_DELETES=20;

EXPORT_C CDb* CDb::NewL(MApp_context& Context, const TDesC& dbname, TInt aFileMode, bool shared)
{
	auto_ptr<CDb> ret(new (ELeave) CDb(Context, shared));
	ret->ConstructL(dbname, aFileMode);
	return ret.release();
}

EXPORT_C CDb::~CDb()
{
	TInt err;
	if (iDbOpen) { 
		TRAP(err, iDb.Compact());
		TRAP(err, iDb.Close());
	}
	delete iStore;
}

EXPORT_C RDbDatabase& CDb::Db()
{
	return iDb;
}

CDb::CDb(MApp_context& Context, bool shared) : MContextBase(Context), iShared(shared)
{
}

void CDb::ConstructL(const TDesC& dbname, TInt aFileMode)
{
	TFileName nameonly;
	nameonly.Format(_L("%S.db"), &dbname);
	MoveIntoDataDirL(AppContext(), nameonly);

	TFileName database_file_install;
	TFileName database_file;

	database_file_install.Format(_L("%S%S_inst.db"), &AppDir(), &dbname);
	database_file.Format(_L("%S%S.db"), &DataDir(), &dbname);

	if (BaflUtils::FileExists(Fs(), database_file_install)) {
		User::LeaveIfError(BaflUtils::CopyFile(Fs(), database_file_install, database_file));
		User::LeaveIfError(BaflUtils::DeleteFile(Fs(), database_file_install));
	}

	bool retry=true;
	while(retry) {
		retry=false;
		TInt store_exists;
		delete iStore; iStore=0;
		if(iDbOpen) iDb.Close(); iDbOpen=false;
		TRAP(store_exists, iStore = CPermanentFileStore::OpenL(Fs(), database_file, aFileMode));
	
		//TRAP (store_exists, iStore = ::OpenL(Fs(), database_file, EFileRead|EFileWrite));

		if (store_exists==KErrNone) { 
			TRAPD(err, iDb.OpenL(iStore, iStore->Root()));
			if (err==KErrNotFound) {
				// store not constructed properly
				Fs().Delete(database_file);
				retry=true;
			} else if(err!=KErrNone) {
				User::Leave(err);
			} else {
				iDbOpen=true;
				TInt err=iDb.Recover();
				User::LeaveIfError(err);
			}
		} else {
			// construct database
			TRAPD(err, {
				iStore = CPermanentFileStore::ReplaceL(Fs(), database_file,aFileMode);
				iStore->SetTypeL(iStore->Layout());
				TStreamId id=iDb.CreateL(iStore);
				iDbOpen=true;
				iStore->SetRootL(id);
				iStore->CommitL();
			});
			if (err!=KErrNone) {
				// roll back
				Fs().Delete(database_file);
				User::Leave(err);
			}
		}
	}
	/*
	 * named databases not supported?
	 *
	if (BaflUtils::FileExists(Fs(), database_file)) {
		TInt err=iDb.Open(Fs(), database_file);
		User::LeaveIfError(err);
	} else {
		TInt err=iDb.Create(Fs(), database_file);
		User::LeaveIfError(err);
	}
	*/
}

EXPORT_C void CDb::BeginL()
{
	User::LeaveIfError(iDb.Begin());
}

EXPORT_C void CDb::CommitL()
{
	TInt err=iDb.Commit();
	if (err!=KErrNone) {
		iDb.Rollback();
		if (iDb.IsDamaged()) iDb.Recover();
	}
	++iTransactionCount;
	if (iTransactionCount>4) iDb.Compact();
	User::LeaveIfError(err);
}

EXPORT_C void CDb::RollBack()
{
	iDb.Rollback();
	if (iDb.IsDamaged()) iDb.Recover();
	++iTransactionCount;
	if (iTransactionCount>4) iDb.Compact();
}

EXPORT_C MDBStore::MDBStore(RDbDatabase& Db) : iDb(Db)
{
}

EXPORT_C void MDBStore::CloseTable()
{
	if (iTableOpen) { TRAPD(err, iTable.Close()); }
	iTableOpen=false;
}
EXPORT_C MDBStore::~MDBStore()
{
	CloseTable();
}

EXPORT_C void MDBStore::CreateIndices(int* columns, int* idx_columns, bool unique_idx, const TDesC& name)
{
	TBuf<10> colname;
	if (idx_columns[0]!=-1) {
		CDbKey* idx_key=CDbKey::NewLC();
		TInt idx_count=0;

		for (int* idx_i=idx_columns; ; idx_i++) {
			if (*idx_i<0) {
				if (unique_idx && idx_count==0) 
					idx_key->MakeUnique();
				TBuf<30> idx_name;
				if (idx_count>0) idx_name.Format(_L("IDX%d"), idx_count);
				else idx_name=_L("IDX");
				TInt err=iDb.CreateIndex(idx_name, name, *idx_key);
				if (err)
					User::Leave(err);
				idx_key->Clear();
				++idx_count;

			} else {
				colname.Format(_L("C%d"), *idx_i);

				/*
				 * we must limit the key size, e.g. 255 characters
				 * is too long, so truncate the last text column in the idx
				 * (but we can't say 50 if the column is actually _shorter_
				 */
				if (*(idx_i+1)>=0) {
					idx_key->AddL(TDbKeyCol(colname));
				} else {
					if ( columns[*idx_i -1] == EDbColText)  {
						if (iTextLen!=0 && iTextLen<50) {
							idx_key->AddL(TDbKeyCol(colname));
						} else {
							idx_key->AddL(TDbKeyCol(colname, 50));
						}

					} else {
						idx_key->AddL(TDbKeyCol(colname));
					}
				}
			}
			if (*idx_i==-1) break;
		}

		CleanupStack::PopAndDestroy(); // idx_key
	}
}

EXPORT_C void MDBStore::DeleteIndices(const TDesC& name)
{
	CDbIndexNames* idxs=iDb.IndexNamesL(name);
	CleanupStack::PushL(idxs);
	TInt err;
	TBuf<20> idxname;
	for (int i=0; i< idxs->Count(); i++) {
		idxname=(*idxs)[i];
		err=iDb.DropIndex( idxname, name);
	}
	CleanupStack::PopAndDestroy();
}

EXPORT_C void MDBStore::ConstructL(int* columns, int* idx_columns, bool unique_idx,
			  const TDesC& name, TBool Alter, int* col_flags)
{
	if (name.Length()>KMaxTableNameLen) User::Leave(KErrTooBig);
	iTableName=name;

	iCurrentIdx=-1;

	TInt table_exists=iTable.Open(iDb, name);
	iTable.Close();

	CDbColSet* cols=CDbColSet::NewLC();

	TBuf<10> colname; int colcount=1;
	for (int* col_i=columns; *col_i!=-1; col_i++) {
		colname.Format(_L("C%d"), colcount);
		colcount++;
		TDbCol n(colname, (TDbColType)*col_i);
		if (col_flags) {
			TInt flags=col_flags[colcount-2];

			/* the Database layer automatically sets ENotNull on
			 * EAutoIncrement columns. If we don't do the same,
			 * trying later to alter the table definition will fail
			 * with KErrArgument (-6)
			 */
			if (flags & TDbCol::EAutoIncrement) flags|=TDbCol::ENotNull;
			n.iAttributes=flags;
		}
		if (n.iType==EDbColText && iTextLen!=0) n.iMaxLength=iTextLen;
		if (n.iType==EDbColBinary && iTextLen!=0) n.iMaxLength=iTextLen;
		cols->AddL(n);
	}
#ifdef DEBUG_DBCOLS
	{
		TBuf<100> msg;
		RDebug::Print(_L("Asked to create table with columns:"));
		for (int i=1; i <= cols->Count(); i++) {
			msg=_L("Column name: ");
			const TDbCol& col=(*cols)[i];
			msg.Append(col.iName);
			msg.Append(_L(" type: "));
			msg.AppendNum(col.iType);
			msg.Append(_L(" attr: "));
			msg.AppendNum(col.iAttributes);
			msg.Append(_L(" maxlength: "));
			msg.AppendNum(col.iMaxLength);
			RDebug::Print(msg);
		}
	}
#endif

	if (table_exists!=KErrNone) {
		TInt err_create=iDb.CreateTable(name, *cols);
		if (err_create==KErrAlreadyExists) 
			User::Leave(table_exists);
		User::LeaveIfError(err_create);

		CleanupStack::PopAndDestroy(); // cols

		TRAPD(err, CreateIndices(columns, idx_columns, unique_idx, name));
		if (err!=KErrNone) {
			DeleteIndices(name);
			iDb.DropTable(name);
			User::Leave(err);
		}

		User::LeaveIfError(iTable.Open(iDb, name));
		iTableOpen=true;
	} else {
		if (Alter) {
			TInt indexcount=0;
			{
				int* idx=idx_columns;
				if (*idx >= 0) {
					while (*idx != -1) {
						if (*idx == -2) indexcount++;
						idx++;
					}
					indexcount++;
				}
			}
			TInt actual_indexcount=0;
			{
				auto_ptr<CDbIndexNames> indexnames(iDb.IndexNamesL(name));
				actual_indexcount=indexnames->Count();
			}
			if (actual_indexcount != indexcount) DeleteIndices(name);
			TInt err=iDb.AlterTable(name, *cols);
			if (err!=KErrNone) {
				if (err==KErrArgument) {
					// have to recreate indexes
					DeleteIndices(name);
					TInt err=iDb.AlterTable(name, *cols);
					if (err!=KErrNone) 
						User::Leave(err);
					TRAP(err, CreateIndices(columns, idx_columns, unique_idx, name));
					if (err!=KErrNone) {
						DeleteIndices(name);
						User::Leave(err);
					}
					actual_indexcount=indexcount;

				} else {
					User::Leave(err);
				}
			}
			if (actual_indexcount != indexcount) {
				TRAP(err, CreateIndices(columns, idx_columns, unique_idx, name));
				if (err!=KErrNone) {
					DeleteIndices(name);
					User::Leave(err);
				}
			}
		}
		CleanupStack::PopAndDestroy(); // cols
		User::LeaveIfError(iTable.Open(iDb, name));
		iTableOpen=true;
	}

	if (idx_columns[0]!=-1) {
		TRAPD(err, SwitchIndexL(0));
		if (err==KErrNotFound) {
			DeleteIndices(name);
			CreateIndices(columns, idx_columns, unique_idx, name);
			SwitchIndexL(0);
		} else {
			User::LeaveIfError(err);
		}

	}
#ifdef DEBUG_DBCOLS
	{
		TBuf<100> msg;
		RDebug::Print(_L("Created with columns:"));
		auto_ptr<CDbColSet> cols(iDb.ColSetL(name));
		for (int i=1; i <= cols->Count(); i++) {
			msg=_L("Column name: ");
			const TDbCol& col=(*cols)[i];
			msg.Append(col.iName);
			msg.Append(_L(" type: "));
			msg.AppendNum(col.iType);
			msg.Append(_L(" attr: "));
			msg.AppendNum(col.iAttributes);
			msg.Append(_L(" maxlength: "));
			msg.AppendNum(col.iMaxLength);
			RDebug::Print(msg);
		}
	}
#endif
}

EXPORT_C void MDBStore::AlterL(int* columns, int* idx_columns, bool unique_idx,
	const TDesC& name, int* col_flags)
{
	if (iTableOpen) { TRAPD(err, iTable.Close()); }
	ConstructL(columns, idx_columns, unique_idx, name, ETrue, col_flags);
}

EXPORT_C void MDBStore::SwitchIndexL(TInt Idx)
{
	if (iCurrentIdx==Idx) return;

	TBuf<30> idx_name;
	if (Idx==-1) {
		RESET_IF_NECESSARY( User::LeaveIfError(iTable.SetNoIndex()) );
		iCurrentIdx=Idx;
		return;
	}
	if (Idx>0) idx_name.Format(_L("IDX%d"), Idx);
	else idx_name=_L("IDX");
	RESET_IF_NECESSARY( User::LeaveIfError(iTable.SetIndex(idx_name)) );
	iCurrentIdx=Idx;
}

EXPORT_C TInt MDBStore::GetCurrentIndex()
{
	return iCurrentIdx;
}

EXPORT_C void MDBStore::Reset(TInt aError)
{
	// FIXME!
	// there should be a better way...
	//
	if (iInReset) return;
	iTable.Cancel();
	iInReset=true;
	if (aError==KErrNotReady) {
		iTable.Reset();
	} else if (aError==KErrDisconnected) {
		if (iTableOpen) iTable.Close();
		iTableOpen=false;
		TInt err=iTable.Open(iDb, iTableName);
		if (err!=KErrNone) {
			iInReset=false;
			User::Leave(err);
		}
		iTableOpen=true;
	} else if (aError==KErrCorrupt) {
		if (iTableOpen) iTable.Close();
		iTableOpen=false;
		iDb.Recover();
		TInt err=iTable.Open(iDb, iTableName);
		if (err!=KErrNone) {
			iInReset=false;
			User::Leave(err);
		}
		iTableOpen=true;
	} else {
		iInReset=false;
		User::Leave(aError);
	}
	TInt prev_idx=iCurrentIdx;
	iCurrentIdx=-2;
	TRAPD(err2, SwitchIndexL(prev_idx));
	iInReset=false;
	User::LeaveIfError(err2);
}

EXPORT_C void MDBStore::PutL()
{
	// fix me : if there is an error here, it doesn't get trapped!!
	// at least the error code id KErrNone, and the app closes
	TRAPD(err, iTable.PutL());
	if (err!=KErrNone) {
		iTable.Cancel();
		User::Leave(err);
	}
	if (!iDb.InTransaction() && iUpdateCount>=COMPACT_BETWEEN_UPDATES) {
		iUpdateCount=0;
		iDb.Compact();
	}
	++iUpdateCount;
}

EXPORT_C void MDBStore::DeleteL()
{
	TRAPD(err, iTable.DeleteL());
	if (err!=KErrNone) {
		iTable.Cancel();
		User::Leave(err);
	}
	if (!iDb.InTransaction() && iUpdateCount>=COMPACT_BETWEEN_UPDATES) {
		iUpdateCount=0;
		iDb.Compact();
	}
	++iUpdateCount;
}

EXPORT_C void MDBStore::SetTextLen(TInt Len)
{
	if (Len>255) 
		iTextLen=255;
	else
		iTextLen=Len;
}

EXPORT_C CSingleColDbBase::CSingleColDbBase(MApp_context& Context, RDbDatabase& Db) : MDBStore(Db), MContextBase(Context)
{
}

EXPORT_C CSingleColDbBase::~CSingleColDbBase()
{
}

EXPORT_C bool CSingleColDbBase::SeekInnerL(TInt Idx, bool Updatable, bool Add)
{
	TDbSeekKey rk(Idx);
	if (iTable.SeekL(rk) ) {
		iTable.GetL();
		if (Updatable) iTable.UpdateL();
		return true;
	} else if (Add) {
		if (!Updatable) User::Leave(-1001);
		iTable.InsertL();
		iTable.SetColL(1, Idx);
		return true;
	}
	return false;
}

EXPORT_C bool CSingleColDbBase::SeekL(TInt Idx, bool Updatable, bool Add)
{
	bool ret=false;
	RESET_IF_NECESSARY(ret=SeekInnerL(Idx, Updatable, Add));
	return ret;
}

EXPORT_C void CSingleColDbBase::ConstructL(const TDesC& TableName,
	int ColType)
{
	int columns[] = { EDbColUint32, 0, -1 };
	columns[1]=ColType;
	int id_cols[]= { 1, -1 };

	// fix me : is this correct??
	// MDBStore::SetTextLen(255);
	MDBStore::ConstructL(columns, id_cols, true, TableName);
}


EXPORT_C TBool CSingleColDbBase::FirstL()
{
	return iTable.FirstL();
}

EXPORT_C TBool CSingleColDbBase::NextL()
{
	return iTable.NextL();
}
