#include "errorhandling.h"

#include "symbian_auto_ptr.h"

_LIT(KUnsetNoMemory, "<failed to allocate memory for string>");

class CErrorInfoImpl : public CErrorInfo {
private:
	CErrorInfoImpl(TErrorCode aErrorCode, TErrorType aErrorType, TSeverity aSeverity,
		HBufC* aUserMsg, HBufC* aTechMsg, HBufC* aCallStack,
		MErrorInfo* aInnerInfo);
		
	virtual ~CErrorInfoImpl();

	virtual TErrorCode ErrorCode() const;
	virtual const TDesC& StackTrace() const;
	virtual const TDesC& UserMessage() const;
	virtual const TDesC& TechMessage() const;
	virtual TErrorType ErrorType() const;
	virtual TSeverity Severity() const;
	virtual const MErrorInfo* InnerError() const;
	virtual const MErrorInfo* NextError() const;

	void AddRef() const;
	void Release() const;

	friend class CErrorInfo;
	friend class auto_ptr<CErrorInfoImpl>;

	TErrorCode	iErrorCode;
	HBufC		*iUserMessage, *iTechMessage, *iCallStack;
	MErrorInfo	*iInnerError, *iNextError;
	TErrorType	iErrorType;
	TSeverity	iSeverity;
	mutable TUint	iRefCount;
};


EXPORT_C CErrorInfo* CErrorInfo::NewL(TErrorCode aErrorCode,
	HBufC* aUserMsg, HBufC* aTechMessage, HBufC* aCallStack,
	TErrorType aErrorType, TSeverity aSeverity,
	MErrorInfo* aInnerError)
{
	auto_ptr<CErrorInfoImpl> ret(new (ELeave) CErrorInfoImpl(aErrorCode,
		aErrorType, aSeverity, aUserMsg, aTechMessage, aCallStack, aInnerError) );
	ret->AddRef();
	return ret.release();
}

CErrorInfoImpl::CErrorInfoImpl(TErrorCode aErrorCode, TErrorType aErrorType, TSeverity aSeverity,
		HBufC* aUserMsg, HBufC* aTechMsg, HBufC* aCallStack, MErrorInfo* aInnerInfo) :
	iErrorCode(aErrorCode), 
	iUserMessage(aUserMsg), iTechMessage(aTechMsg), iCallStack(aCallStack),
	iInnerError(aInnerInfo), iErrorType(aErrorType),iSeverity(aSeverity)
{
	if (iInnerError) iInnerError->AddRef();
}

void CErrorInfoImpl::AddRef() const
{
	++iRefCount;
}
void CErrorInfoImpl::Release() const
{
	--iRefCount;
	if (iRefCount<=0) delete this;
}

CErrorInfoImpl::~CErrorInfoImpl()
{
	delete iUserMessage;
	delete iTechMessage;
	if (iInnerError) iInnerError->Release();
	if (iNextError) iNextError->Release();
}

TErrorCode CErrorInfoImpl::ErrorCode() const
{
	return iErrorCode;
}

const TDesC& CErrorInfoImpl::UserMessage() const
{
	if (iUserMessage) return *iUserMessage;
	return KNullDesC;
}

const TDesC& CErrorInfoImpl::TechMessage() const
{
	if (iTechMessage) return *iTechMessage;
	return KNullDesC;
}

TErrorType CErrorInfoImpl::ErrorType() const
{
	return iErrorType;
}

TSeverity CErrorInfoImpl::Severity() const
{
	return iSeverity;
}

const MErrorInfo* CErrorInfoImpl::InnerError() const
{
	return iInnerError;
}

const MErrorInfo* CErrorInfoImpl::NextError() const
{
	return iNextError;
}

EXPORT_C TErrorCode MakeErrorCode(TUint iUid, TInt iCode)
{
	TErrorCode e={iUid, iCode};
	return e;
}

const TDesC& CErrorInfoImpl::StackTrace() const
{
	if (iCallStack) return *iCallStack;
	return KNullDesC;
}
