#include "mutexrequest.h"
#include "symbian_auto_ptr.h"

/*
 * OK: this isn't actually quite safe
 * if this thread gets killed while the other
 * is waiting on the mutex, it never gets Signal()ed
 *
 * we assume that clients will eventually close all
 * handles to the mutex, retry and so recover
 *
 */

EXPORT_C CMutexRequest* CMutexRequest::NewL(MApp_context& Context,
		const TDesC& aName, TTimeIntervalMicroSeconds32 aTimeOut, 
		TRequestStatus* aToSignal)
{
	CALLSTACKITEM2_N(_CL("CMutexRequest"), _CL("NewL"), &Context);

	auto_ptr<CMutexRequest> ret(new (ELeave) CMutexRequest(Context));
	ret->ConstructL();
	ret->OpenGlobalL(aName);
	ret->WaitL(aToSignal, aTimeOut);
	return ret.release();
}

CMutexRequest::CMutexRequest(MApp_context& Context) : 
	CActive(EPriorityNormal), MContextBase(Context) { }

void CMutexRequest::ConstructL()
{
	CALLSTACKITEM_N(_CL("CMutexRequest"), _CL("ConstructL"));

	CActiveScheduler::Add(this);
}

void CMutexRequest::OpenGlobalL(const TDesC& aName)
{
	CALLSTACKITEM_N(_CL("CMutexRequest"), _CL("OpenGlobalL"));

	if (iIsOpen) User::Leave(KErrAlreadyExists);
	TInt err=iMutex.OpenGlobal(aName);
	if (err!=KErrNone) {
		User::LeaveIfError(iMutex.CreateGlobal(aName, 1));
	}
	iIsOpen=ETrue;
}

void WaitLoopL(TAny* aArgs)
{
	CMutexRequest* mr=(CMutexRequest*)aArgs;
	mr->iMutex.Wait();
	mr->iParent.RequestComplete(mr->iRequestStatus, KErrNone);
}

TInt WaitLoop(TAny* aPtr)
{
        CTrapCleanup *cl;
        cl=CTrapCleanup::New();

        TInt err=0;
        TRAP(err,
                WaitLoopL(aPtr));

	delete cl;

	return err;
}

void CMutexRequest::WaitL(TRequestStatus* aToSignal, TTimeIntervalMicroSeconds32 aTimeOut)
{
	CALLSTACKITEM_N(_CL("CMutexRequest"), _CL("WaitL"));

	//DebugLog(_L("CMutexRequest::WaitL"));

	if (iChildIsOpen) User::Leave(KErrAlreadyExists);
	if (!iParentIsOpen) {
		User::LeaveIfError( iParent.Open(iParent.Id()) );
		iParentIsOpen=ETrue;
	}

	if (!iIsOpen) User::Leave(KErrNotReady);
	*aToSignal=KRequestPending;
	iRequestStatus=aToSignal;

	TBuf<100> name=_L("MutextWait ");
	name.AppendNum(iParent.Id());
	name.AppendNum( (TUint)this );
	User::LeaveIfError(iChild.Create(name, WaitLoop, 8192, 8192, 64*1024, this));
	iChildIsOpen=ETrue;
	iChild.Resume();
	iWaited=ETrue;

	User::LeaveIfError( iTimer.CreateLocal() );
	iTimer.After(iStatus, aTimeOut);

	SetActive();
}

EXPORT_C CMutexRequest::~CMutexRequest()
{
	CALLSTACKITEM_N(_CL("CMutexRequest"), _CL("~CMutexRequest"));

	//DebugLog(_L("~CMutexRequest"));
	Cancel();
	if (iChildIsOpen) {
		iChild.Kill(KErrNone);
		iChild.Close();
	}
	if (iParentIsOpen) {
		iParent.Close();
	}
	if (iRequestStatus && *iRequestStatus==KRequestPending) {
		User::RequestComplete(iRequestStatus, KErrDied);
	}
	if (iWaited) iMutex.Signal();
	if (iIsOpen) iMutex.Close();
	iTimer.Close();
}

void CMutexRequest::DoCancel()
{
	CALLSTACKITEM_N(_CL("CMutexRequest"), _CL("DoCancel"));

	iTimer.Cancel();
}

void CMutexRequest::RunL()
{
	CALLSTACKITEM_N(_CL("CMutexRequest"), _CL("RunL"));

	// timed out

	iChild.Kill(KErrNone);
	iChild.Close();
	iChildIsOpen=EFalse;
	
	if (iRequestStatus && *iRequestStatus==KRequestPending) {
		User::RequestComplete(iRequestStatus, KErrTimedOut);
	}
}