/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "timeout.h"
#include "symbian_auto_ptr.h"
#include "app_context.h"
#include "app_context_impl.h"
#include "callstack.h"

class CTimeOutImpl : public CTimeOut {
public:
	~CTimeOutImpl();
private:
	virtual void Wait(int seconds);
	virtual void WaitMax(int seconds); 
	virtual void Reset();

	CTimeOutImpl(MTimeOut& i_cb);
	void ConstructL();
	void RunL();

	MTimeOut& cb;
	TTime	iWaitUntil;
	MApp_context* iContext;

	friend class CTimeOut;
};

CTimeOut::CTimeOut() : CTimer(EPriorityNormal)
{
}

CTimeOut::~CTimeOut()
{
}

CTimeOutImpl::CTimeOutImpl(MTimeOut& i_cb) : cb(i_cb)
{
}

void CTimeOutImpl::ConstructL()
{
	iContext=GetContext();

	CTimer::ConstructL();
	CActiveScheduler::Add(this);
}

EXPORT_C CTimeOut* CTimeOut::NewL(MTimeOut& i_cb)
{
	CALLSTACKITEMSTATIC_N(_CL("CTimeOut"), _CL("NewL"));

	auto_ptr<CTimeOutImpl> ret (new (ELeave) CTimeOutImpl(i_cb));
	ret->ConstructL();
	return ret.release();
}

void CTimeOutImpl::RunL()
{
	// there's no guarantee that we aren't deleted
	// from the expired() callback, so we have to cache
	// the member variable iContext in a stack variable

	MApp_context* ctx=iContext;
	if (ctx) ctx->CallStackMgr().ResetCallStack();

	TTime now; now.HomeTime(); 
	
	now+=TTimeIntervalMicroSeconds(100*1000);
	if (now < iWaitUntil) {
		At(iWaitUntil);
	} else {
		cb.expired(this);
	}

	if (ctx) ctx->CallStackMgr().ResetCallStack();
}

void CTimeOutImpl::Wait(int seconds)
{
	CALLSTACKITEM2_N(_CL("CTimeOutImpl"), _CL("Wait"), iContext);
	Reset();
	TTime at;
	at.HomeTime();
	
	at+=TTimeIntervalSeconds(seconds);
	iWaitUntil=at;
	At(at);
}

void CTimeOutImpl::WaitMax(int seconds)
{
	TTime t; t.HomeTime();
	t+=TTimeIntervalSeconds(seconds);
	if (t >= iWaitUntil) return;
	Reset();
	iWaitUntil=t;
	At(t);
}

void CTimeOutImpl::Reset()
{
	CALLSTACKITEM2_N(_CL("CTimeOutImpl"), _CL("Reset"),  iContext);

	Cancel();
}

CTimeOutImpl::~CTimeOutImpl()
{
	CALLSTACKITEM2_N(_CL("CTimeOutImpl"), _CL("~CTimeOutImpl"), iContext);

	iContext=0;
	Cancel();
}

EXPORT_C void SetTime()
{
	TTime now;
	now.HomeTime();
	now+=TTimeIntervalHours(3);
	User::SetHomeTime(now);
}