/* 
Copyright (C) 2004  Mika Raento - Renaud Petit

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
   
    
     email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/

/*
 * Concepts:
 * !Accessing Phonebook!
 */

#include "phonebook.h"

#include <cpbkcontactengine.h> 
#include <cpbkcontactiter.h> 
#include <cpbkcontactitem.h> 
#include <cpbkmmsaddressselect.h>
#include <cpbkemailaddressselect.h>

#include <bamatch.h>
#include <contextcommon.mbg>
#include <contextcommon.rsg>
#include <cntdb.h>
#include <CPbkPhoneNumberSelect.h>
#include <cpbksmsaddressselect.h>
#include <e32math.h>

#include <eikenv.h>
#include <bcardeng.h>
#include "icons.h"
#include <avkon.mbg>


#pragma warning(disable: 4706)

EXPORT_C contact* contact::NewL(TContactItemId i_id, const TDesC& i_first_name, const TDesC& i_last_name) 
{
	CALLSTACKITEM_N(_CL("contact"), _CL("NewL"));

	contact* c=new (ELeave) contact;
	CleanupStack::PushL(c);
	c->id=i_id;
	
	c->first_name=HBufC::NewL(i_first_name.Length());
	*(c->first_name)=i_first_name;
	
	c->last_name=HBufC::NewL(i_last_name.Length());
	*(c->last_name)=i_last_name;

	CleanupStack::Pop();
	return c;
}

EXPORT_C void contact::Name(HBufC*& Into, bool last_name_first)
{
	contact *c=this;
	TInt len=c->first_name->Length()+c->last_name->Length()+50;
	if (len > Into->Des().MaxLength()) {
		delete Into;
		Into=HBufC::NewL(len);
	}

	if (last_name_first) 
	{
		*(Into)=*(c->last_name);
		if (c->last_name->Length()>0 && c->first_name->Length()>0) 
		{
			Into->Des().Append(_L(" "));
		}
		Into->Des().Append(*(c->first_name));
	} else {
		*(Into)=*(c->first_name);
		if (c->first_name->Length()>0 && c->last_name->Length()>0) 
		{
			Into->Des().Append(_L(" "));
		}
		Into->Des().Append(*(c->last_name));
	}
	if (Into->Des().Length() == 0)
	{
		Into->Des().Append(_L("(Unnamed)"));
	}
}

EXPORT_C void contact::set_presence(CBBPresence* data)
{
	CALLSTACKITEM_N(_CL("contact"), _CL("set_presence"));

	if (presence) presence->Release();
	presence=data;
	if (data) data->AddRef();

}


EXPORT_C contact::~contact() {
	if (presence) presence->Release();
	delete first_name;
	delete last_name;
}


class contacts_key : public TKey {
public:
	contacts_key(CArrayFixFlat<contact*> & i_arr, HBufC*& aname1, HBufC*& aname2, bool alast_name_first=true) : arr(i_arr), 
			name1(aname1), name2(aname2), last_name_first(alast_name_first) { }
	TInt Compare(TInt aLeft,TInt aRight) const {
		contact* left=arr[aLeft];
		contact* right=arr[aRight];
		if (left->has_nick && !right->has_nick) return -1;
		if (right->has_nick && !left->has_nick) return 1;
		
		left->Name(name1, last_name_first);
		right->Name(name2, last_name_first);
		return name1->Compare(*name2);
	}
private:
	CArrayFixFlat<contact*>& arr;
	HBufC*& name1; HBufC*& name2;
	bool last_name_first;
};

class contacts_swap : public TSwap {
public:
	contacts_swap(CArrayFixFlat<contact*> & i_arr) : arr(i_arr) { }
	void Swap(TInt aLeft,TInt aRight) const {
		contact* tmp=arr[aLeft];
		arr[aLeft]=arr[aRight];
		arr[aRight]=tmp;
	}
private:
	CArrayFixFlat<contact*>& arr;
};

EXPORT_C phonebook::phonebook(MApp_context& Context, CJabberData* JabberData, CPresenceHolder* PresenceHolder) : 
MContextBase(Context), iJabberData(JabberData), iPresenceHolder(PresenceHolder)
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("phonebook"));


}

EXPORT_C phonebook::~phonebook()
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("~phonebook"));

	if (obs) obs->exiting();
	
	delete current_ids;
	
	if (contacts) {
		reset_contacts();
	}
	delete contacts;
	
	delete contactitem;
	delete chnot;
	
	if (owns_engine) {
		delete eng;
	}
	
	delete iContactToIndex;
}

EXPORT_C TInt phonebook::Count()
{
	return current_ids->Count();
}

EXPORT_C TInt phonebook::GetContactId(TInt Index)
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("GetContactId"));


	if (Index<0 || Index >= current_ids->Count()) return KErrNotFound;
	return (*current_ids)[Index];
}

EXPORT_C void phonebook::ConstructL()
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("ConstructL"));


	iContactToIndex=CGenericIntMap::NewL();
	
	current_ids=new (ELeave) CArrayFixFlat<TContactItemId>(50);
	
	eng=CPbkContactEngine::Static();
	
	if (eng) {
		owns_engine=false;
	} else {
		eng=CPbkContactEngine::NewL();
		owns_engine=true;
	}
	
	read_db();
	
	chnot=eng->CreateContactChangeNotifierL(this);
}

void phonebook::read_db()
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("read_db"));


	read_data(eng->CreateContactIteratorLC(ETrue));
	CleanupStack::PopAndDestroy();

	HBufC* name1=HBufC::NewL(200);
	CleanupStack::PushL(name1);
	HBufC* name2=HBufC::NewL(200);
	CleanupStack::PushL(name2);
	User::QuickSort(contacts->Count(), contacts_key(*contacts, name1, name2), contacts_swap(*contacts));
	CleanupStack::PopAndDestroy(2);
	
	filter(previous_filter, true);
}

void phonebook::HandleDatabaseEventL(TContactDbObserverEvent /*aEvent*/)
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("HandleDatabaseEventL"));


	read_db();
}

EXPORT_C void phonebook::PresenceChangedL(TInt ContactId, CBBPresence* Info)
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("PresenceChangedL"));

	if (obs) obs->before_change();
	
	TInt idx=(TInt)iContactToIndex->GetData(ContactId);
	if (!idx) return;
	contact* c=(*contacts)[idx-1];
	c->set_presence(Info);
	
	if (obs) obs->contents_changed();
}

EXPORT_C void phonebook::Notify(const TDesC & /*aMessage*/)
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("Notify"));


	// no impl
}

EXPORT_C phonebook_observer * phonebook::get_observer()
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("get_observer"));


	return obs;
}

EXPORT_C TInt phonebook::GetIndex(TInt ContactId)
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("GetIndex"));


	TInt idx=(TInt)iContactToIndex->GetData(ContactId);
	if (!idx) return KErrNone;
	contact* c=(*contacts)[idx-1];
	TInt curr_idx=c->current_idx;
	if (!curr_idx) return KErrNone;
	return curr_idx-1;
}

void phonebook::reset_contacts()
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("reset_contacts"));


	//iContactToIndex->Reset();
	
	for (int i=0; i<contacts->Count(); i++) {
		delete (*contacts)[i];
	}
	contacts->Reset();
}

void phonebook::read_data(CPbkContactIter * iter)
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("read_data"));


	if (contacts) {
		TInt c=contacts->Count();
		reset_contacts();
		contacts->SetReserveL(c+1);
	} else {
		contacts=new (ELeave) CArrayFixFlat<contact*>(50);
	}
	
	CPbkContactItem *item=0;
	TBuf<100> first_name;
	TBuf<100> last_name;
	TBuf<100> jabber_nick;
	
	for (iter->FirstL(); (item=iter->CurrentL()); iter->NextL()) {
		
		TPbkContactItemField* f;
		
		f=item->FindField(EPbkFieldIdLastName);
		if (f) {
			last_name=f->Text();
		} else {
			last_name=_L("");
		}
		f=item->FindField(EPbkFieldIdFirstName);
		if (f) {
			first_name=f->Text();
		} else {
			first_name=_L("");
		}

		if (first_name.Length()==0 && last_name.Length()==0) {
			f=item->FindField(EPbkFieldIdCompanyName);
			if (f) last_name=f->Text();
		}

		contact* c=contact::NewL(item->Id(), first_name, last_name);
		if (iJabberData && (iJabberData->GetJabberNickL(item->Id(), jabber_nick)) && (jabber_nick.Length()>1) ) {
			c->has_nick=true;
		}
		if (iPresenceHolder) {c->set_presence(iPresenceHolder->GetPresence(item->Id()));}
		contacts->AppendL(c);
	}
}

EXPORT_C void phonebook::ReRead()
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("ReRead"));

	read_db();
}


EXPORT_C void phonebook::set_observer(phonebook_observer* i_obs)
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("set_observer"));


	obs=i_obs;
}

EXPORT_C contact * phonebook::GetContact(TInt index)
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("GetContact"));


	TInt contactId = GetContactId(index);
	TInt idx=(TInt)iContactToIndex->GetData(contactId);
	if (!idx) return KErrNone;
	contact* c=(*contacts)[idx-1];
	return c;
}

EXPORT_C bool phonebook::filter(const TDesC& /*substr*/, bool /*force*/)
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("filter"));

	if (obs) obs->before_change();
	
	current_ids->Reset();
	
	contact* ci;
	TInt idx=0;
	for (int i=0; i<contacts->Count(); i++) 
	{
		ci=(*contacts)[i];

		current_ids->AppendL( ci->id);
		iContactToIndex->AddDataL(ci->id, (void*) (i+1), true);
		ci->current_idx=idx+1;
		idx++;
	}

	if (obs) obs->contents_changed();
	return true;
}

EXPORT_C CPbkContactEngine* phonebook::get_engine()
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("get_engine"));


	return eng;
}

EXPORT_C CArrayFixFlat<contact*>* phonebook::GetContacts()
{
	CALLSTACKITEM_N(_CL("phonebook"), _CL("GetContacts"));


	return contacts;
}


// -------------------------------------------------------------

const TInt CPresenceUpdater::KTimeOut = 60;

EXPORT_C CPresenceUpdater * CPresenceUpdater::NewL(phonebook_i& book)
{
	CALLSTACKITEM_N(_CL("CPresenceUpdater"), _CL("NewL"));


	auto_ptr<CPresenceUpdater> ret(new (ELeave) CPresenceUpdater(book));
	ret->ConstructL();
	return ret.release();
}

EXPORT_C CPresenceUpdater::CPresenceUpdater(phonebook_i& book) : book(book), iWaiting(EFalse)
{
	CALLSTACKITEM_N(_CL("CPresenceUpdater"), _CL("CPresenceUpdater"));

}

EXPORT_C CPresenceUpdater::~CPresenceUpdater()
{
	CALLSTACKITEM_N(_CL("CPresenceUpdater"), _CL("~CPresenceUpdater"));


	delete iWait;
}

EXPORT_C void CPresenceUpdater::ConstructL()
{
	CALLSTACKITEM_N(_CL("CPresenceUpdater"), _CL("ConstructL"));


	iWait=CTimeOut::NewL(*this);
	Start();
}

EXPORT_C void CPresenceUpdater::Start()
{
	CALLSTACKITEM_N(_CL("CPresenceUpdater"), _CL("Start"));


	Refresh();
	if (!iWaiting)
	{
		iWait->Wait(KTimeOut);
		iWaiting= ETrue;
	}
}

EXPORT_C void CPresenceUpdater::Stop()
{
	CALLSTACKITEM_N(_CL("CPresenceUpdater"), _CL("Stop"));


	iWait->Reset();	
	iWaiting = EFalse;
}

void CPresenceUpdater::expired(CBase*)
{
	CALLSTACKITEM_N(_CL("CPresenceUpdater"), _CL("expired"));


	RDebug::Print(_L("refreshing..."));
	Refresh();
	if (iWaiting)
	{
		iWaiting = EFalse;
		Start();
	}
}

EXPORT_C void CPresenceUpdater::Refresh()
{
	CALLSTACKITEM_N(_CL("CPresenceUpdater"), _CL("Refresh"));
}
