/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "cb_presence.h"
#include "symbian_auto_ptr.h"
#include <contextCommon.h>
#include "bbxml.h"

const TInt CPresenceHolder::KDefaultBufferSize = 256;

CPresenceHolder* CPresenceHolder::NewL(CJabberData& JabberData)
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("NewL"));


	auto_ptr<CPresenceHolder> ret(new (ELeave) CPresenceHolder(JabberData));
	ret->ConstructL();
	return ret.release();
}

CPresenceHolder::~CPresenceHolder()
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("~CPresenceHolder"));


	Cancel();
	if (iSessionOpen) iSession.Close();
	delete iWait;
	delete iListeners;
	delete iPresenceData;

	delete iContact;
	delete iPresence;
	
}

void CPresenceHolder::AddListener(MPresenceListener* Listener)
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("AddListener"));


	iListeners->AppendL(Listener);
}

const CBBPresence* const CPresenceHolder::GetPresence(TInt ContactId) const
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("GetPresence"));

	return (CBBPresence*) iPresenceData->GetData(ContactId);
}

CPresenceHolder::CPresenceHolder(CJabberData& JabberData) : CCheckedActive(EPriorityStandard, _L("CPresenceHolder")), iJabberData(JabberData),
	  iC(0,0), iP(0,0)
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("CPresenceHolder"));


    	CActiveScheduler::Add(this);
}

void CPresenceHolder::Start()
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("Start"));


	if (iSession.ConnectToContextServer() == KErrNone)
	{
		iSessionOpen=true;
		RequestPresence();
	} else {
		Restart();
	}
}

void CPresenceHolder::Restart()
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("Restart"));


	Cancel();
	iWait->Cancel();
	if (iSessionOpen) {
		iSession.Close();
	}
	iSessionOpen=false;
	RDebug::Print(_L("restarting"));
	iWait->Wait(20);
}

void CPresenceHolder::expired(CBase*)
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("expired"));


	Start();
}

void CPresenceHolder::ConstructL()
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("ConstructL"));


	iPresenceData=CGenericIntMap::NewL();
	iPresenceData->SetDeletor(CBBPresenceDeletor);
	iListeners=CList<MPresenceListener*>::NewL();

	iContact = HBufC::NewL(KDefaultBufferSize/4);
	iPresence= HBufC::NewL(KDefaultBufferSize);

	iSendTimeStamp = TTime();

	iWait=CTimeOut::NewL(*this);

	Start();
}

void CPresenceHolder::NewPresence(const TDesC& Nick, const TDesC& Info, const TTime & send_timestamp)
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("NewPresence"));

	TInt contact;
	contact=iJabberData.GetContactIdL(Nick);
	if (contact==KErrNotFound) return;

	// only supports new format
	TBuf<15> pv2=_L("<presencev2");
	if (Info.Left(pv2.Length()).Compare(pv2)) return;

	TPtrC8 xml8( (TUint8*)Info.Ptr(), Info.Size());

	auto_ptr<CBBPresence> data( CBBPresence::NewL() );
	auto_ptr<CSingleParser> parser( CSingleParser::NewL(data.get(), false) );
	TRAPD(err, parser->ParseL(xml8) );
	if (err!=KErrNone) return;
	data->iSentTimeStamp()=send_timestamp;
	data->iSent=ETrue;

	iPresenceData->AddDataL(contact, data.get(), true);
	CBBPresence* dp=data.release();
	dp->AddRef();

	CList<MPresenceListener*>::Node* i=iListeners->iFirst;
	while (i) {
		TRAPD(err, i->Item->PresenceChangedL(contact, dp));
		i->Item->Notify(_L(""));
		i=i->Next;
	}
}

CBBPresence* CPresenceHolder::GetPresence(TInt ContactId)
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("GetPresence"));


	return (CBBPresence*)iPresenceData->GetData(ContactId);
}

void CPresenceHolder::CBBPresenceDeletor(void* p)
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("CBBPresenceDeletor"));

	CBBPresence* b=(CBBPresence*)p;
	b->Release();
}

void CPresenceHolder::CheckedRunL()
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("CheckedRunL"));


	if (iStatus == ECSRequestCompleted)
	{
		switch (current_state)
		{
			case EWaitingForPresence:
				{
					RDebug::Print(iC.Left(128));
					RDebug::Print(iP.Left(128));
					NewPresence(iC,iP, iSendTimeStamp);
					RequestPresenceNotification();
					break;
				}
			default:
				Restart();
				// ASSERT(0); //Unexpected error
				break;
		}
	}
	else if (iStatus == ECSBufferTooSmall)
	{
		iContact = iContact->ReAllocL(iC.MaxLength() *2);
		iC.Set(iContact->Des());

		iPresence = iPresence->ReAllocL(iP.MaxLength() *2);
		iP.Set(iPresence->Des());

		RequestPresenceNotification();
	}
	else if (iStatus == ECSServerUnreachable || iStatus == ECSIdentificationError )
	{
		CList<MPresenceListener*>::Node* i=iListeners->iFirst;
		while (i) 
		{
			i->Item->Notify(_L("Server unreachable"));
			i=i->Next;
		}
		// ok, now we now that the connection is not up
		// let's just wait for next presence notification
		RequestPresenceNotification();
	}
	else if (iStatus == EContextServerTerminated)
	{
		CList<MPresenceListener*>::Node* i=iListeners->iFirst;
		while (i) 
		{
			i->Item->Notify(_L("Server Stopped"));
			i=i->Next;
		}
		// do nothing, it's a proper stop of the context server ...
	}
	else 
	{
		Restart();
	}
}

void CPresenceHolder::RequestPresence()
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("RequestPresence"));


	if(!IsActive())
	{
		current_state = EWaitingForPresence;
		
		iC.Set(iContact->Des());
		iP.Set(iPresence->Des());

		iSession.MsgRequestPresenceInfo(iC, iP, iSendTimeStamp, iStatus);
		SetActive();
	}	
}

void CPresenceHolder::RequestPresenceNotification()
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("RequestPresenceNotification"));


	if(!IsActive())
	{
		current_state = EWaitingForPresence;

		iC.Set(iContact->Des());
		iP.Set(iPresence->Des());

		iSession.MsgRequestPresenceNotification(iC, iP, iSendTimeStamp, iStatus);
		SetActive();
	}		
}

void CPresenceHolder::CancelRequest()
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("CancelRequest"));


    	Cancel() ;
}

void CPresenceHolder::DoCancel()
{
	CALLSTACKITEM_N(_CL("CPresenceHolder"), _CL("DoCancel"));


	iSession.Cancel();
}

