#include "videoengine.h"
#include <mmferrors.h>
#include "MPlayerUIControllerListener.h"
#include "app_context.h"

CVideoFileDetails::CVideoFileDetails()
{
	CALLSTACKITEM_N(_CL("CVideoFileDetails"), _CL("CVideoFileDetails"));

}

CVideoFileDetails::~CVideoFileDetails()
{
	CALLSTACKITEM_N(_CL("CVideoFileDetails"), _CL("~CVideoFileDetails"));

	delete iTitle;
	delete iFileName;
	delete iUrl;
	delete iFormat;
	delete iCopyright;
}

CVideoEngine* CVideoEngine::NewL( )
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("NewL"));

	CVideoEngine* self = new (ELeave) CVideoEngine( );
	CleanupStack::PushL( self );
	self->ConstructL( );
	CleanupStack::Pop();
	return self;
}

CVideoEngine::CVideoEngine()
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("CVideoEngine"));

}

void CVideoEngine::ConstructL()
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("ConstructL"));

	iMediaFile = HBufC::NewL(0);
	iEngineState = EPNotInitialised; 
	iFileDetails = new (ELeave) CVideoFileDetails();
	iProgressUpdater = CHeartbeat::NewL(0);
}

    
CVideoEngine::~CVideoEngine()
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("~CVideoEngine"));

	delete iMediaFile;
	delete iFileDetails;
	delete iProgressUpdater;
}

void CVideoEngine::SetNewFileL(const TDesC &aFileName)
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("SetNewFileL"));

	HBufC* newFile = aFileName.AllocL();
	delete iMediaFile;		 
	iMediaFile = newFile;
}

void CVideoEngine::DoPlayL()
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("DoPlayL"));

	if ( !iProgressUpdater->IsActive()) {
		iProgressUpdater->Start(ETwelveOClock,this); 
        }
	iPlayer->Play();

	switch (iEngineState){
		case EPStopped: {
			iPlayPosition = 0;
			//iListener->PlaybackPositionChangedL(iPlayPosition,iFileDetails->iDurationInSeconds);
			break;
		}
		case EPPaused:
			//iListener->PlaybackPositionChangedL(iPlayPosition,iFileDetails->iDurationInSeconds);
			break;
		default:
			break;
        }

	iEngineState = EPPlaying;
}

void CVideoEngine::DoStopL()
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("DoStopL"));

	if ( iEngineState != EPPlaying && iEngineState != EPPaused ) return;
	if ( iProgressUpdater->IsActive()) iProgressUpdater->Cancel();
        
	iPlayPosition = 0;
	iPlayer->Stop();
	iPlayer->Close();
	iEngineState = EPStopped;
	MvpuoPlayComplete(KErrNone);

}

void CVideoEngine::MvpuoOpenComplete(TInt aError )
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("MvpuoOpenComplete"));

	if (aError == KErrNone) {
		iPlayer->Prepare();
	} else {
		// acts as playing is over
		MvpuoPlayComplete(aError);
	}
}

void CVideoEngine::MvpuoFrameReady(CFbsBitmap& /*aFrame*/,TInt /*aError*/)
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("MvpuoFrameReady"));

	// no impl
}
 
void CVideoEngine::MvpuoPlayComplete(TInt aError)
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("MvpuoPlayComplete"));

	if (iProgressUpdater->IsActive()) iProgressUpdater->Cancel();
        
	iPlayPosition = 0;
	iListener->PlayCompletedL(aError);
	iEngineState = EPStopped;
	iPlayer->Close();
}

void CVideoEngine::MvpuoPrepareComplete(TInt aError )
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("MvpuoPrepareComplete"));

	TInt ret = aError;
 
	iEngineState = EPStopped;
	iFileDetails->iDurationInSeconds = 0;

	if ( ret == KErrNone || ret == KErrMMPartialPlayback ) {
		TSize size( 0, 0 );
		iPlayer->VideoFrameSizeL(size);
		iFileDetails->iResolutionHeight = size.iHeight;
		iFileDetails->iResolutionWidth = size.iWidth;

		// Is there any audio track?
		iFileDetails->iAudioTrack = iPlayer->AudioEnabledL();   
		if (iFileDetails->iAudioTrack) iPlayer->SetVolumeL(5);
		// video track
		iFileDetails->iVideoTrack = iPlayer->VideoBitRateL();	            
		// duration of the video clip
		iFileDetails->iDurationInSeconds = iPlayer->DurationL().Int64() / KMPOneSecond;           
	} else {
		iListener->PlayCompletedL(aError);
		return;
	}

	if ( iListener ) {
		TRAPD( ignore,iListener->InitControllerCompletedL( ret ) );
        }
}

void CVideoEngine::MvpuoEvent(const TMMFEvent& aEvent)
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("MvpuoEvent"));

	// Higher priority application has taken over the audio device. --> Do pause.
	if (aEvent.iEventType == KMMFEventCategoryVideoPlayerGeneralError &&
        aEvent.iErrorCode == KErrHardwareNotAvailable) {
		TRAPD(ignore,iListener->PlayCompletedL(KErrAudioLost));    
        }
}
 
void CVideoEngine::InitControllerL( MPlayerUIControllerListener* aCallback,
                                    RWsSession& aWs,
                                    CWsScreenDevice& aScreenDevice,
                                    RWindowBase& aWindow,
                                    TRect& aScreenRect,
                                    TRect& aClipRect )   
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("InitControllerL"));

	iEngineState = EPInitialising;
	iListener = aCallback;
    
	delete iPlayer;
	iPlayer = NULL;
	iPlayer = CVideoPlayerUtility::NewL( *this, EMdaPriorityNormal, 
		                             EMdaPriorityPreferenceNone, aWs, 
						aScreenDevice,aWindow,aScreenRect, aClipRect );

	iPlayer->OpenFileL( iMediaFile->Des() );
}

void CVideoEngine::Beat()
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("Beat"));

	// keep backlights on if clip has video
	if ( iFileDetails->iVideoTrack ) {
		User::ResetInactivityTime();
        }
	if ( iEngineState != EPPlaying ) return;

	TInt64 ret = iPlayPosition%2;
    
	if ( ret.GetTInt() ) {
		iPlayPosition = iPlayPosition + 1;
		if (iListener) {
			//TRAPD(ignore,iListener->PlaybackPositionChangedL(iPlayPosition,iFileDetails->iDurationInSeconds));
		}
        } else {
		Synchronize();
        }
}


void CVideoEngine::Synchronize()
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("Synchronize"));

	if ( iEngineState != EPPlaying ) return;
	
	// if live stream, fake progress
	TRAPD(ignore, 
		iPlayPosition = iPlayer->PositionL().Int64() / KMPOneSecond;
		if (iListener){
			//iListener->PlaybackPositionChangedL(iPlayPosition,iFileDetails->iDurationInSeconds);
		}
        );
}

TInt CVideoEngine::GetEngineState()
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("GetEngineState"));

	return iEngineState;
}

void CVideoEngine::DoPauseL()
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("DoPauseL"));

	if ( iEngineState == EPPlaying ) {
		if (iProgressUpdater->IsActive()) {
			iProgressUpdater->Cancel();
		}
		iPlayer->PauseL();
		iEngineState = EPPaused;
	}
}

void CVideoEngine::CloseController()
{
	CALLSTACKITEM_N(_CL("CVideoEngine"), _CL("CloseController"));

	iListener=0;
	if (iPlayer) iPlayer->Close();
	delete iPlayer;
	iPlayer = 0;
}

TInt CVideoEngine::IncreaseVolume()
{
	if (!iFileDetails->iAudioTrack) return 0;
	TInt vol = iPlayer->Volume();

	if (vol < iPlayer->MaxVolume()) {
		TRAPD(ignore, iPlayer->SetVolumeL(++vol));
	}
	return vol;
}

TInt CVideoEngine::DecreaseVolume()
{
	if (!iFileDetails->iAudioTrack) return 0;
        TInt vol = iPlayer->Volume();

	if (vol > 0) {
		TRAPD(ignore, iPlayer->SetVolumeL(--vol));
	}
	return vol;
}
