#include "bb_listener.h"

#include <blackboardclientsession.h>
#include "bbdata.h"
#include "symbian_auto_ptr.h"
#include "context_uids.h"
#include "bbtypes.h"
#include "csd_event.h"
#include "bbxml.h"
#include "csd_loca.h"
#include "timeout.h"
#include <s32mem.h>

void Log(const TDesC& msg);
void Log(const TDesC8& msg);
void Log(const TDesC& msg, TInt i);

#ifndef __WINS__
const TInt KMaxAckInterval = 15*60;
const TInt KMaxAckWait = 5*60;
#else
const TInt KMaxAckInterval = 5*60;
const TInt KMaxAckWait = 2*60;
#endif

const TComponentName KListener = { { CONTEXT_UID_CONTEXTNETWORK}, 1 };

class CBBListenerImpl : public CBBListener, public MContextBase,
	public MTimeOut {
private:
	CBBListenerImpl(CBBProtocol* aProtocol, MApp_context& aContext);
	void ConstructL();
	~CBBListenerImpl();

	void CheckedRunL();
	virtual void Acked(TUint id);
	virtual void IncomingTuple(const CBBTuple* aTuple);
	virtual void Error(TInt aError, TInt aOrigError, const TDesC& aDescr);
	virtual void ReadyToWrite(TBool aReady);
	virtual void Disconnected();

	void ConnectL();
	void SetFilterL();
	void WaitForNotify();
	void WriteL();
	void DoCancel();
	void WriteAckL();
	void PutTupleL(const TTupleName& aTuple,
		const TDesC& aSubName, const MBBData* aData);

	void expired(CBase* aSource);

	CBBProtocol*	iProtocol;
	RBBClient	iBBClient;

	HBufC8*		iSerializedData;
	TPtr8		iP;
	CXmlBufExternalizer	*iCurrentBuf, *iFreeBuf;
	CXmlBufExternalizer	*iWaiting[2];

	TFullArgs	iFullArgs;
	TInt		iAsyncErrorCount;
	TBool		iReadyToWrite, iPendingWrite;
	MBBDataFactory*	iFactory;
	CBBTuple	*iTuple;
	CTimeOut	*iTimer;
	CList<TUint>	*iToAck;
	TTime		iPreviousAck;

	enum TGetState {
		EIdle,
		EGettingListener,
		EGettingLoca,
		EWaitForNotify
	};
	TGetState	iGetState;
	void GetListener();
	void GetLoca();
	void GetOrWaitL();

	friend class CBBListener;
	friend class auto_ptr<CBBListenerImpl>;

};

CBBListener::CBBListener() : CCheckedActive(EPriorityNormal, _L("CBBListener"))
{
	CALLSTACKITEM_N(_CL("CBBListener"), _CL("CBBListener"));

}

EXPORT_C CBBListener* CBBListener::NewL(CBBProtocol* aProtocol, MApp_context& aContext)
{
	CALLSTACKITEM_N(_CL("CBBListener"), _CL("NewL"));

	auto_ptr<CBBListenerImpl> ret(new (ELeave) CBBListenerImpl(aProtocol, aContext));
	ret->ConstructL();
	return ret.release();
}

CBBListenerImpl::CBBListenerImpl(CBBProtocol* aProtocol, MApp_context& aContext) : 
	MContextBase(aContext),
		iProtocol(aProtocol), iP(0, 0) { }

void CBBListenerImpl::ConstructL()
{
	CALLSTACKITEM_N(_CL("CBBListenerImpl"), _CL("ConstructL"));

	iCurrentBuf=CXmlBufExternalizer::NewL(2048);
	iFreeBuf=CXmlBufExternalizer::NewL(2048);
	iSerializedData=HBufC8::NewL(2048);

	iTimer=CTimeOut::NewL(*this);
	iTimer->Wait(KMaxAckInterval);

	iFactory=BBDataFactory();
	iTuple=new (ELeave) CBBTuple(iFactory);
	iToAck=CList<TUint>::NewL();

	ConnectL();
	SetFilterL();
	iGetState=EWaitForNotify;
	
	CActiveScheduler::Add(this);
}

CBBListenerImpl::~CBBListenerImpl()
{
	CALLSTACKITEM_N(_CL("CBBListenerImpl"), _CL("~CBBListenerImpl"));

	Cancel();
	iBBClient.Close();
	delete iCurrentBuf;
	delete iFreeBuf;
	delete iSerializedData;
	delete iTuple;
	delete iToAck;
	delete iTimer;
}

void CBBListenerImpl::ConnectL()
{
	CALLSTACKITEM_N(_CL("CBBListenerImpl"), _CL("ConnectL"));

	Log(_L("CBBListener::ConnectL()"));

	Cancel();
	TInt errorcount=0;
	TInt err=KErrNone;
	while (errorcount<5) {
		iBBClient.Close();
		err=iBBClient.Connect();
		if (err==KErrNone) return;
		errorcount++;
	}
	User::Leave(err);
}

void CBBListenerImpl::SetFilterL()
{
	CALLSTACKITEM_N(_CL("CBBListenerImpl"), _CL("SetFilterL"));

	Log(_L("CBBListener::SetFilterL()"));

	TInt errorcount=0, err=KErrNone;
	while (errorcount<5) {
		TRequestStatus s;
		iBBClient.AddNotificationL(KListener, s);
		User::WaitForRequest(s);
		err=s.Int();
		if (err==KErrNone) {
			iBBClient.AddNotificationL(KLocaMessageStatusTuple, 
				ETrue, EBBPriorityNormal,
				s);
			User::WaitForRequest(s);
			err=s.Int();
			if (err==KErrNone) return;
		}

		ConnectL();
		errorcount++;
	}
	User::Leave(err);
}

void CBBListenerImpl::WaitForNotify()
{
	CALLSTACKITEM_N(_CL("CBBListenerImpl"), _CL("WaitForNotify"));

	Log(_L("CBBListener::WaitForNotify()"));

	if (IsActive()) return;

	iGetState=EWaitForNotify;
	iSerializedData->Des().Zero();
	iP.Set(iSerializedData->Des());
	iBBClient.WaitForNotify(iFullArgs, iP, iStatus);
	SetActive();
}

void CBBListenerImpl::DoCancel()
{
	CALLSTACKITEM_N(_CL("CBBListenerImpl"), _CL("DoCancel"));

	iBBClient.CancelNotify();
}

void CBBListenerImpl::CheckedRunL()
{
	CALLSTACKITEM_N(_CL("CBBListenerImpl"), _CL("CheckedRunL"));

	Log(_L("CheckedRunL()"));

	{
		if (iStatus.Int()!=KErrNone) {
			if (iStatus.Int()==KClientBufferTooSmall) {
				iSerializedData->Des().Zero();
				iSerializedData=iSerializedData->ReAllocL(iSerializedData->Des().MaxLength()*2);
				iAsyncErrorCount=0;
				GetOrWaitL();
				return;
			} else if (iStatus.Int()==KErrNotFound) {
				if (iGetState==EWaitForNotify) {
					User::Leave(KErrNotFound);
				} else if (iGetState==EGettingListener) {
					GetLoca();
				} else if (iGetState==EGettingLoca) {
					SetFilterL();
					WaitForNotify();
				}
				return;
			}
			if (iAsyncErrorCount>5) User::Leave(iStatus.Int());
			++iAsyncErrorCount;
			ConnectL();
			SetFilterL();
			WaitForNotify();
			return;
		}
	}

	MBBData* d=0;
	{
		RDesReadStream rs(*iSerializedData);
		CleanupClosePushL(rs);	
		TTypeName read_type=TTypeName::IdFromStreamL(rs);
		{
			d=iFactory->CreateBBDataL(read_type, KEvent, iFactory);
			CleanupPushBBDataL(d);
		}
		{
			d->InternalizeL(rs);
		}
		CleanupStack::Pop();
		CleanupStack::PopAndDestroy();
	}

	{
		delete iTuple->iData(); iTuple->iData()=d;
		iTuple->iTupleMeta.iModuleUid()=iFullArgs.iTupleName.iModule.iUid;
		iTuple->iTupleMeta.iModuleId()=iFullArgs.iTupleName.iId;
		iTuple->iTupleMeta.iSubName=iFullArgs.iSubName;
		iTuple->iTupleId()=iFullArgs.iId;
		
		iFreeBuf->Zero();
		iTuple->IntoXmlL(iFreeBuf);
		iFreeBuf->Characters(_L("\n"));
	}

	
	if (iReadyToWrite) {
		iWaiting[0]=iFreeBuf;
		WriteL();
		GetOrWaitL();
	} else {
		iWaiting[1]=iFreeBuf;
		iPendingWrite=ETrue;
	}
}

void CBBListenerImpl::GetOrWaitL()
{
	if (iGetState==EWaitForNotify) {
		WaitForNotify();
	} else if (iGetState==EGettingListener) {
		GetListener();
	} else if (iGetState==EIdle || iGetState==EGettingLoca) {
		GetLoca();
	}
}

void CBBListenerImpl::GetListener()
{
	if (IsActive()) return;

	iGetState=EGettingListener;
	iSerializedData->Des().Zero();
	iP.Set(iSerializedData->Des());
	iBBClient.Get(KListener, iFullArgs, iP, iStatus);
	SetActive();

}

void CBBListenerImpl::GetLoca()
{
	if (IsActive()) return;

	iGetState=EGettingLoca;
	iSerializedData->Des().Zero();
	iP.Set(iSerializedData->Des());
	iBBClient.Get(KLocaMessageStatusTuple, KNullDesC, iFullArgs, iP, iStatus);
	SetActive();
}

void CBBListenerImpl::WriteAckL()
{
	CALLSTACKITEM_N(_CL("CBBListenerImpl"), _CL("WriteAckL"));
	TUint ack=iToAck->Top();
	TBuf<100> xml=_L("<ack><id>");
	xml.AppendNum(ack);
	xml.Append(_L("</id></ack>\n"));
	iProtocol->WriteL(xml);
	iToAck->Pop();
}

void CBBListenerImpl::WriteL()
{
	CALLSTACKITEM_N(_CL("CBBListenerImpl"), _CL("WriteL"));

	Log(_L("WriteL"));

	iReadyToWrite=EFalse;
	if (iFreeBuf==iWaiting[0]) {
		iFreeBuf=iCurrentBuf;
		iCurrentBuf=iWaiting[0];
	}
	TRAPD(err, iProtocol->WriteL( iWaiting[0]->Buf() ));
	if (err==KErrNotReady) iPendingWrite=ETrue;
	else User::LeaveIfError(err);

	iTimer->WaitMax(KMaxAckWait);
}

_LIT(KLastAck, "last_ack");

void CBBListenerImpl::Acked(TUint id)
{
	CALLSTACKITEM_N(_CL("CBBListenerImpl"), _CL("Acked"));

	iTimer->Wait(KMaxAckInterval);
	TRequestStatus s;
	iBBClient.Delete(id, s);
	User::WaitForRequest(s);
#ifdef __WINS__
	TBuf<100> msg=_L("deleted ");
	msg.AppendNum(id); msg.Append(_L(": "));
	msg.AppendNum(s.Int());
	RDebug::Print(msg);
#endif
	if (iPreviousAck + TTimeIntervalMinutes(1) < GetTime()) {
		TBBTime t(KLastAck);
		iPreviousAck=GetTime();
		t()=iPreviousAck;
		TRAPD(err, PutTupleL(KStatusTuple, KLastAck, &t));
	}
}

const TTupleName KRemoteLocaLogicTuple = { { CONTEXT_UID_CONTEXTSENSORS }, 37 };

void CBBListenerImpl::PutTupleL(const TTupleName& aTuple,
	const TDesC& aSubName, const MBBData* aData)
{
	if (!aData) return;

	auto_ptr<HBufC8> serialized(HBufC8::NewL(1024));
	TInt err=KErrOverflow;
	while (err==KErrOverflow) {
		TPtr8 bufp(serialized->Des());
		RDesWriteStream ws(bufp);

		aData->Type().ExternalizeL(ws);
		TRAP(err, aData->ExternalizeL(ws));
		if (err==KErrNone) {
			ws.CommitL();
		} else if (err==KErrOverflow) {
			serialized->Des().Zero();
			serialized.reset( serialized->ReAllocL(
				serialized->Des().MaxLength()*2) );
		}
	}
	User::LeaveIfError(err);

	TUint id;
	TRequestStatus s;

	TBool replace=ETrue;
	if (aTuple==KRemoteLocaLogicTuple) replace=EFalse;

	iBBClient.Put(aTuple, aSubName,
		KNoComponent, *serialized, EBBPriorityNormal,
		replace, id, s, ETrue, EFalse);
	User::WaitForRequest(s);
	User::LeaveIfError(s.Int());
}


void CBBListenerImpl::IncomingTuple(const CBBTuple* aTuple)
{
	iTimer->Wait(KMaxAckInterval);

	TTupleName tn= { aTuple->iTupleMeta.iModuleUid(), aTuple->iTupleMeta.iModuleId() };

	auto_ptr<HBufC8> serialized(HBufC8::NewL(1024));
	PutTupleL(tn, aTuple->iTupleMeta.iSubName(),
		aTuple->iData());

	iToAck->AppendL(aTuple->iTupleId());
}

void CBBListenerImpl::Error(TInt /*aError*/, TInt /*aOrigError*/, const TDesC& /*aDescr*/)
{
	CALLSTACKITEM_N(_CL("CBBListenerImpl"), _CL("Error"));

	iReadyToWrite=EFalse;
}

void CBBListenerImpl::ReadyToWrite(TBool aReady)
{
	CALLSTACKITEM_N(_CL("CBBListenerImpl"), _CL("ReadyToWrite"));

	Log(_L("ReadyToWrite"));

	if (aReady) {
		if (iToAck->iCount > 0) {
			WriteAckL();
			return;
		}
		iWaiting[0]=iWaiting[1]; iWaiting[1]=0;
		if (iWaiting[0]) {
			WriteL();
		} else {
			iReadyToWrite=ETrue;
		}
		GetOrWaitL();
	} else {
		iReadyToWrite=EFalse;
	}
}

void CBBListenerImpl::Disconnected()
{
	CALLSTACKITEM_N(_CL("CBBListenerImpl"), _CL("Disconnected"));

	iReadyToWrite=EFalse;
}

void CBBListenerImpl::expired(CBase* aSource)
{
	User::Leave(KContextErrTimeoutInBBProtocol);
}
