#include "cn_bt_obex.h"

_LIT(KServerTransportName,"RFCOMM");

#define KBTServiceOBEX 0x1105

#include <btmanclient.h>
#include <btextnotifiers.h>
#include <btsdp.h>
#include <obex.h>
#include "mutexrequest.h"
#include "timeout.h"

#include "symbian_auto_ptr.h"
#include "reporting.h"

//#include <bteng.h>
// reverse engineered from bteng.lib
class RBTDeviceHandler {
public:
	IMPORT_C int  AreExistingConnectionsL(int &);
	IMPORT_C int  CloseAllBasebandConnectionsL(void);
};


_LIT(KBtAccessMutex, "ContextBtMutex");

#define ACCESS_WAIT_MSECS 500

/*
 * Concepts:
 * !Sending Bluetooth messages!
 * !Bluejacking!
 * !Closing Bluetooth connection!
 */


/*
 * There's a few tricks that this class has to do:
 * 
 *  - we use a mutex (CMutexRequest) to arbitrate access
 *    to the Bluetooth stack between these objects and CDiscoverImpl
 *  - after doing the SDP search, we don't immediately close
 *    the sdp object and don't immediately try to do the Obex send, it
 *    seems that these sometimes result in a KERN-EXEC 0. Instead
 *    we wait 1 sec before sending, and don't delete the sdp object
 *    before the next request
 *  - there's no 'Cancel' on the CObexClient, so we just delete it
 *    and manually complete the iStatus if necessary. There seems to
 *    be no way of stopping the connection before the other end
 *    says Yes/No, so we pull the rug from under the obex client by
 *    actually closing all bluetooth connection via RBTDeviceHandler
 *  - if asked to (aConnectCount), we connect-wait_a_bit-disconnect-connect
 *    so that the recipient gets multiple alerts, since the single beep
 *    is so easy to miss. That may be a bit nasty.
 *  - It took about a week to get it to actually work. Don't change it
 *    unless you really know what you are doing.
 *
 *  MR
 */

class CBtObexImpl: public CBtObex, public MSdpAgentNotifier, public MSdpAttributeValueVisitor,
	public MContextBase, public MTimeOut {
private:
	CBtObexImpl(MApp_context& aContext, MObexNotifier& aNotifier);
	~CBtObexImpl();
	
	void ConstructL();

	virtual void SendMessage(const TBTDevAddr& aAddr, 
		CObexBaseObject* aObject,
		TInt	aConnectCount); // doesn't take ownership
	void CancelSend();

	virtual void DoCancel();
	virtual void CheckedRunL();

	enum TCommState { IDLE, WAITING_ON_MUTEX, WAITING_FOR_ACCESS, 
		GETTING_SERVICE, WAITING_FOR_CONNECT, CONNECTING, SENDING_FILE,
		DISCONNECTING, ABORTING_FOR_NEW, ABORTING_FOR_CANCEL, 
		DISCONNECTING_FOR_CANCEL };
	TCommState iCurrentState;

	MObexNotifier& iNotifier;

	TBTDevAddr	iAddr;
	CSdpAgent* agent;
	CSdpSearchPattern* list;
	CObexClient* obex_client;
	CMutexRequest* iMutex;
	void ReleaseMutex();

	TObexBluetoothProtocolInfo info;
	TUint port;
	bool seen_rfcomm;

	CObexBaseObject* current_object;

	RTimer	iTimer;
	RBTDeviceHandler iBtHandler;

	TInt iRetryCount;
	void RetryOrReport(TInt aError, MObexNotifier::TState aState);

	void get_service();
	void release();
	void send_next_file(); 
	void Async();

	TInt	iRemainingConnects, iBetweenConnects;
	void expired(CBase* aSource);
	CTimeOut	*iTimeOut;
	enum TWaitState { EWaitToStopConnect, EWaitToStartConnect };
	TWaitState	iWaitState;
	MObexNotifier::TState	iStateBeforeCancel;
	TBool		iInCancel;
private:
	//MSdpAgentNotifier
	virtual void AttributeRequestComplete(TSdpServRecordHandle aHandle, TInt aError);
	virtual void AttributeRequestResult(TSdpServRecordHandle aHandle, 
		TSdpAttributeID aAttrID, CSdpAttrValue* aAttrValue);
	virtual void NextRecordRequestComplete(TInt aError, 
		TSdpServRecordHandle aHandle, TInt aTotalRecordsCount);
	//MSdpAttributeValueVisitor
	virtual void VisitAttributeValueL(CSdpAttrValue &aValue, TSdpElementType aType);
	virtual void StartListL(CSdpAttrValueList &aList);
	virtual void EndListL();

	friend class CBtObex;
	friend class auto_ptr<CBtObexImpl>;
};

EXPORT_C CBtObex::CBtObex() : CCheckedActive(EPriorityNormal, _L("CBtObex")) { }

CBtObexImpl::CBtObexImpl(MApp_context& Context, MObexNotifier& aNotifier) : 
	MContextBase(Context), iNotifier(aNotifier) //, targetServiceClass(0x2345)
{
	CALLSTACKITEM_N(_CL("CBtObexImpl"), _CL("CBtObexImpl"));

}

void CBtObexImpl::ConstructL() 
{
	CALLSTACKITEM_N(_CL("CBtObexImpl"), _CL("ConstructL"));

	User::LeaveIfError(iTimer.CreateLocal());
	CActiveScheduler::Add(this); // add to scheduler
	iTimeOut=CTimeOut::NewL(*this);
	iBetweenConnects=5;
}

EXPORT_C CBtObex* CBtObex::NewL(MApp_context& aContext, MObexNotifier& aNotifier)
{
	CALLSTACKITEM_N(_CL("CBtObex"), _CL("NewL"));

	auto_ptr<CBtObexImpl> ret(new (ELeave) CBtObexImpl(aContext, aNotifier));
	ret->ConstructL();
	return ret.release();
}

void CBtObexImpl::SendMessage(const TBTDevAddr& aAddr, 
		CObexBaseObject* aObject, TInt aConnectCount)
{
	CALLSTACKITEM_N(_CL("CBtObexImpl"), _CL("SendMessage"));
	Reporting().DebugLog(_L("CBtObexImpl::SendMessage"));

	if (iCurrentState!=IDLE) iNotifier.Error(KErrNotReady, MObexNotifier::EInitializing);

	iRemainingConnects=aConnectCount;
	if (iRemainingConnects<0) iRemainingConnects=0;

	iRetryCount=0;

	release();
	current_object=aObject;
	iAddr=aAddr;
	
	TTimeIntervalMicroSeconds32 w(120*1000*1000);
	//Reporting().DebugLog(_L("obex:wait_on_mutex"));
	iMutex=CMutexRequest::NewL(AppContext(), KBtAccessMutex, w, &iStatus);
	iCurrentState=WAITING_ON_MUTEX;
	SetActive();
}

void CBtObexImpl::RetryOrReport(TInt aError, MObexNotifier::TState aState)
{
	CALLSTACKITEM_N(_CL("CBtObexImpl"), _CL("RetryOrReport"));

	++iRetryCount;
	release();
	if (iRetryCount<5) {
		iCurrentState=WAITING_FOR_ACCESS;
		TTimeIntervalMicroSeconds32 w(ACCESS_WAIT_MSECS*1000);
		iTimer.After(iStatus, w);
		SetActive();
		port=0;
	} else {
		ReleaseMutex();
		iNotifier.Error(aError, aState);
	}
}

void CBtObexImpl::release()
{
	CALLSTACKITEM_N(_CL("CBtObexImpl"), _CL("release"));

	delete agent; agent=0;
	delete list; list=0;
	
	delete obex_client; obex_client=0;
	if (iTimeOut) iTimeOut->Reset();
	iCurrentState=IDLE;
}

void CBtObexImpl::CancelSend()
{
	CALLSTACKITEM_N(_CL("CBtObexImpl"), _CL("CancelSend"));

	switch (iCurrentState) {
	case GETTING_SERVICE:
		iStateBeforeCancel=MObexNotifier::EGettingService;
		break;
	case CONNECTING:
		iStateBeforeCancel=MObexNotifier::EConnecting;
		break;
	case SENDING_FILE:
		iStateBeforeCancel=MObexNotifier::ESending;
		break;
	default:
		iStateBeforeCancel=MObexNotifier::EInitializing;
		break;
	}

	switch(iCurrentState) {
	case GETTING_SERVICE:
	case WAITING_FOR_CONNECT:
		iCurrentState=DISCONNECTING_FOR_CANCEL;
		break;
	case CONNECTING:
		{
		iBtHandler.CloseAllBasebandConnectionsL();
		/*
		TRequestStatus s;
		obex_client->Disconnect(s);
		User::WaitForRequest(s);
		*/
		iCurrentState=DISCONNECTING_FOR_CANCEL;
		}
		break;
	case WAITING_ON_MUTEX:
	case WAITING_FOR_ACCESS:
		Cancel();
		iCurrentState=IDLE;
		iNotifier.Cancelled(iStateBeforeCancel);
		break;
	case SENDING_FILE:
		obex_client->Abort();
		iCurrentState=ABORTING_FOR_CANCEL;
		break;
	default:
		User::Leave(KErrNotReady);
	};
}

CBtObexImpl::~CBtObexImpl()
{
	CALLSTACKITEM_N(_CL("CBtObexImpl"), _CL("~CBtObexImpl"));

	Cancel();
	
	iTimer.Close();

	release();
	delete iTimeOut;
	delete iMutex;
}

void CBtObexImpl::DoCancel()
{
	CALLSTACKITEM_N(_CL("CBtObexImpl"), _CL("DoCancel"));

	if (iCurrentState==SENDING_FILE) {
		obex_client->Abort();
	}
	if (iCurrentState==CONNECTING) {
		TRAPD(err, iBtHandler.CloseAllBasebandConnectionsL());
		/*iInCancel=ETrue;
		CActiveScheduler::Start();
		iInCancel=EFalse;*/
		obex_client->Error(KErrCancel);
		/*
		TRequestStatus* s=&iStatus;
		User::RequestComplete(s, KErrNone);
		*/
		/*
		TRequestStatus s;
		obex_client->Disconnect(s);
		User::WaitForRequest(s);
		*/
	}
	if (iCurrentState==WAITING_ON_MUTEX) {
		Reporting().DebugLog(_L("ReleaseMutex"));
		ReleaseMutex();
	}
	if (iCurrentState==WAITING_FOR_ACCESS || iCurrentState==WAITING_FOR_CONNECT) {
		Reporting().DebugLog(_L("timer cancel"));
		iTimer.Cancel();
	}

	if (iStatus==KRequestPending) {
		Reporting().DebugLog(_L("delete obex"));
		delete obex_client; obex_client=0;
	}
	if (iStatus==KRequestPending) {
		Reporting().DebugLog(_L("complete request"));

		TRequestStatus* s=&iStatus;
		User::RequestComplete(s, KErrNone);
	}
}

void CBtObexImpl::get_service()
{
	CALLSTACKITEM_N(_CL("CBtObexImpl"), _CL("get_service"));

	port=0;
	iCurrentState=GETTING_SERVICE;
	agent = CSdpAgent::NewL(*this, iAddr);
	list = CSdpSearchPattern::NewL();
	list->AddL(KBTServiceOBEX);
	agent->SetRecordFilterL(*list);
	agent->NextRecordRequestL();
}

void CBtObexImpl::send_next_file()
{	
	CALLSTACKITEM_N(_CL("CBtObexImpl"), _CL("send_next_file"));

	release();

	info.iTransport.Copy(KServerTransportName);
	info.iAddr.SetBTAddr(iAddr);
	info.iAddr.SetPort(port);

	obex_client=CObexClient::NewL(info);

	obex_client->Connect(iStatus);
	iCurrentState=CONNECTING;
	if (iRemainingConnects) {
		--iRemainingConnects;
		iWaitState=EWaitToStopConnect;
		iTimeOut->Wait(iBetweenConnects);
	} else {
		iTimeOut->Reset();
	}
		
	SetActive();
	
}

void CBtObexImpl::CheckedRunL()
{
	CALLSTACKITEM_N(_CL("CBtObexImpl"), _CL("CheckedRunL"));
	if (iInCancel) {
		CActiveScheduler::Stop();
	}

	if (iCurrentState==ABORTING_FOR_NEW) {
		obex_client->Connect(iStatus);
		iCurrentState=CONNECTING;
		SetActive();
		return;
	}
	if (iCurrentState==ABORTING_FOR_CANCEL) {
		obex_client->Disconnect(iStatus);
		iCurrentState=DISCONNECTING_FOR_CANCEL;
		SetActive();
		return;
	}

	if (iStatus.Int()!=KErrNone) {
		MObexNotifier::TState st;
		switch(iCurrentState) {
		case SENDING_FILE:
		case CONNECTING:
			st=MObexNotifier::ESending;
			break;
		default:
			st=MObexNotifier::EInitializing;
			break;
		};
		iCurrentState=IDLE;
		release();
		ReleaseMutex();
		iNotifier.Error(iStatus.Int(), st);
		return;
	}
	switch(iCurrentState) {
	case WAITING_ON_MUTEX:
		{
		//Reporting().DebugLog(_L("obex:got_mutex"));
		iCurrentState=WAITING_FOR_ACCESS;
		TTimeIntervalMicroSeconds32 w(ACCESS_WAIT_MSECS*1000);
		iTimer.After(iStatus, w);
		SetActive();
		}
		break;
	case WAITING_FOR_ACCESS:
		//Reporting().DebugLog(_L("obex:get_service"));
#ifndef __WINS__
		get_service();
#else
		ReleaseMutex();
		iCurrentState=IDLE;
		iNotifier.Success();
		return;
#endif
		break;
	case CONNECTING:
		iTimeOut->Reset();
		obex_client->Put(*current_object, iStatus);
		iCurrentState=SENDING_FILE;
		SetActive();
		break;
	case SENDING_FILE:
		iCurrentState=DISCONNECTING;
		obex_client->Disconnect(iStatus);
		SetActive();
		break;
	case DISCONNECTING:
		release();
		ReleaseMutex();
		iNotifier.Success();
		break;
	case WAITING_FOR_CONNECT:
	case GETTING_SERVICE:
		TRAPD(err, send_next_file());
		if (err!=KErrNone) {
			Cancel();
			release();
			ReleaseMutex();
			iNotifier.Error(err, MObexNotifier::EInitializing);
		}
		break;
	case IDLE:
		ReleaseMutex();
		iNotifier.Error(KErrGeneral, MObexNotifier::EInitializing);
		break;
	case DISCONNECTING_FOR_CANCEL:
		ReleaseMutex();
		iNotifier.Cancelled(iStateBeforeCancel);
		break;
	}
}

void CBtObexImpl::AttributeRequestComplete(TSdpServRecordHandle /*aHandle*/, TInt aError)
{
	CALLSTACKITEM_N(_CL("CBtObexImpl"), _CL("AttributeRequestComplete"));
	//Reporting().DebugLog(_L("AttributeRequestComplete"));

	if (aError!=KErrNone && aError!=KErrEof) {
		//Reporting().DebugLog(_L("AttributeRequestComplete:0:retry"));
		RetryOrReport(aError, MObexNotifier::EGettingService);
	} else if (aError==KErrNone) {
		TRAPD(err, agent->NextRecordRequestL());
		if (err!=KErrNone) {
			//Reporting().DebugLog(_L("AttributeRequestComplete:1:retry"));
			RetryOrReport(aError, MObexNotifier::EGettingService);
		}
	}
}

void CBtObexImpl::AttributeRequestResult(TSdpServRecordHandle /*aHandle*/, 
				      TSdpAttributeID /*aAttrID*/, CSdpAttrValue* aAttrValue)
{
	CALLSTACKITEM_N(_CL("CBtObexImpl"), _CL("AttributeRequestResult"));
	//Reporting().DebugLog(_L("AttributeRequestResult"));

	
	if (aAttrValue) 
		aAttrValue->AcceptVisitorL(*this);	
}

void CBtObexImpl::VisitAttributeValueL(CSdpAttrValue &aValue, TSdpElementType aType)
{
	CALLSTACKITEM_N(_CL("CBtObexImpl"), _CL("VisitAttributeValueL"));
	//Reporting().DebugLog(_L("VisitAttributeValueL"));

	if (aType==ETypeUUID) {
		if (aValue.UUID()==KRFCOMM) {
			seen_rfcomm=true;
		} else {
			seen_rfcomm=false;
		}
	} else if (aType==ETypeUint && seen_rfcomm) {
		port=aValue.Uint();
	}
}

void CBtObexImpl::StartListL(CSdpAttrValueList &/*aList*/)
{
	//Reporting().DebugLog(_L("StartListL"));

}

void CBtObexImpl::EndListL()
{
	//Reporting().DebugLog(_L("EndListL"));

}

void CBtObexImpl::ReleaseMutex()
{
	CALLSTACKITEM_N(_CL("CBtObexImpl"), _CL("ReleaseMutex"));

	if (!iMutex) return;

	delete iMutex;iMutex=0;
	//Reporting().DebugLog(_L("obex::releasemutex"));
}

void CBtObexImpl::NextRecordRequestComplete(TInt aError, 
					 TSdpServRecordHandle aHandle, TInt aTotalRecordsCount)
{
	//Reporting().DebugLog(_L("NextRecordRequestComplete"));

	TBuf<30> msg;
	if (aError==KErrEof) {
		//Reporting().DebugLog(_L("NextRecordRequestComplete::Eof"));
		if (port==0) {
			//_LIT(err, "didn't find service");
			// release();
			iCurrentState=IDLE;
			ReleaseMutex();
			iNotifier.Error(KErrNotFound, MObexNotifier::EGettingService);
		} else {
			// release();
			//_LIT(f, "found port %d");
			//TBuf<30> msg;
			//msg.Format(f, port);
			//cb->status_change(msg);
			//connect_to_service();
			/*iCurrentState=GETTING_SERVICE; */

			TTimeIntervalMicroSeconds32 w(1*1000*1000);
			iTimer.After(iStatus, w);
			SetActive();

			if (iCurrentState==GETTING_SERVICE)
				iCurrentState=WAITING_FOR_CONNECT;
			return;
		}
	} else if (aError!=KErrNone) {
		//Reporting().DebugLog(_L("NextRecordRequestComplete::Other_error"));
		//_LIT(err, "service error: %d");
		//msg.Format(err, aError);
		Reporting().DebugLog(_L("NextRecordRequestComplete:0:retry"));
		RetryOrReport(aError, MObexNotifier::EGettingService);
	} else if(aTotalRecordsCount==0) {
		//Reporting().DebugLog(_L("NextRecordRequestComplete::0records"));
		release();
		ReleaseMutex();
		iNotifier.Error(KErrNotFound, MObexNotifier::EGettingService);
	} else {
		//Reporting().DebugLog(_L("NextRecordRequestComplete::attributerequest"));
		agent->AttributeRequestL(aHandle, KSdpAttrIdProtocolDescriptorList);
		return;
	}
}

void CBtObexImpl::expired(CBase* aSource)
{
	if (iCurrentState!=CONNECTING) return;

	if (iWaitState==EWaitToStopConnect) {
		iCurrentState=ABORTING_FOR_NEW;
		//obex_client->Abort();
		TRAPD(err, iBtHandler.CloseAllBasebandConnectionsL());
	} else {
		if (!obex_client) obex_client=CObexClient::NewL(info);
		obex_client->Connect(iStatus);
		iCurrentState=CONNECTING;
		if (iRemainingConnects) {
			--iRemainingConnects;
			iWaitState=EWaitToStopConnect;
			iTimeOut->Wait(iBetweenConnects);
		} else {
			iTimeOut->Reset();
		}
		
		SetActive();
	}
}
