#include "downloader.h"
#include "symbian_auto_ptr.h"
#include "cn_http.h"
#include "settings.h"
#include "cl_settings.h"
#include "reporting.h"
#include "notifystate.h"
#include <contextnetwork.mbg>

#define MAX_ERRORS 5

#ifndef __WINS__
_LIT(KIconFile, "c:\\system\\data\\contextnetwork.mbm");
#else
_LIT(KIconFile, "z:\\system\\data\\contextnetwork.mbm");
#endif

class CDownloaderImpl : public CDownloader, public MContextBase, public MDBStore, public MHttpObserver {

	CDownloaderImpl(MApp_context& aContext, RDbDatabase& Db, 
		MDownloadObserver& aObserver);
	void ConstructL(const TDesC& aDirectory);
	~CDownloaderImpl();

	virtual void DownloadL(TInt64 aRequestId, const TDesC& aUrl, TBool aForce);

	friend class CDownloader;
	friend class auto_ptr<CDownloaderImpl>;

	virtual void NotifyHttpStatus(THttpStatus st, TInt aError);
	virtual void NotifyNewHeader(const CHttpHeader &aHeader);
	virtual void NotifyNewBody(const TDesC8 &chunk);

	void CheckedRunL();
	void DoCancel();

	void Reset();
	void StartDownload();
	void Async(TInt aSeconds=0);
	void GotAllL();
	TBool MarkError(TInt aError, const TDesC& aDescr);
	TBool MarkErrorL(TInt aError, const TDesC& aDescr);
	void CloseFile();
	void OpenFileL();
	TBool SeekToRequestId(TInt64 aId);
	virtual void RemoveRequest(TInt64 aRequestId);
	virtual void SetFixedIap(TInt aIap);
	TInt GetErrorWait();
	void GetFileExtensionL(const TDesC& aMimeType, TDes& aExtensionInto);
	TBool CheckForLocal(const TDesC& aUrl);

	enum TState { EIdle, EFetching };
	TState iCurrentState;

	enum TDownloadingStatus { EDone, EDownloading, EDownloadErrors };
	void ShowDownloading(TDownloadingStatus aStatus);

	enum TColumns {
		ERequestId = 1,
		EPriority,
		EUrl,
		EErrors,
		EErrorCode,
		EErrorDescr,
		EFileName,
		EContentType,
		EFinished,
	};
	enum TIndices {
		ERequestIdx = 0,
		EPriorityIdx
	};

	MDownloadObserver& iObserver;
	TInt		iPriority;
	TFileName	iDirectory;
	CHttp*		iHttp;
	RFile		iFile; TBool iFileIsOpen;
	TFileName	iCurrentFileName, iCurrentUrl, iCurrentContentType;
	TInt		iSize, iGot, iOffset;
	TInt64		iCurrentRequestId;
	TBool		iGotError, iGotReply, iDisabled;
	TInt		iIAP, iFixedIAP;
	TBool		iReseting;
	RTimer		iTimer; TBool iTimerIsOpen;
	CNotifyState*	iNotifyState;
};

EXPORT_C CDownloader* CDownloader::NewL(MApp_context& aContext, RDbDatabase& Db, 
	const TDesC& aDirectory, MDownloadObserver& aObserver)
{
	CALLSTACKITEM_N(_CL("CDownloader"), _CL("NewL"));
	auto_ptr<CDownloaderImpl> ret(new (ELeave) CDownloaderImpl(aContext, Db, aObserver));
	ret->ConstructL(aDirectory);
	return ret.release();
}

_LIT(KDownloader, "CDownloader");

CDownloader::CDownloader() : CCheckedActive(EPriorityIdle, KDownloader) { }

CDownloaderImpl::CDownloaderImpl(MApp_context& aContext, RDbDatabase& Db, 
				 MDownloadObserver& aObserver) : MContextBase(aContext), MDBStore(Db),
				iObserver(aObserver) { }

void CDownloaderImpl::ConstructL(const TDesC& aDirectory)
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("ConstructL"));
	iDirectory=aDirectory;

	if (iDirectory.Right(1).Compare(_L("\\"))) iDirectory.Append(_L("\\"));
	TInt err=Fs().MkDirAll(iDirectory);
	if (err!=KErrNone && err!=KErrAlreadyExists) User::Leave(err);

	TInt cols[]={ EDbColInt64, EDbColInt32, EDbColText, EDbColInt32, EDbColInt32, 
		EDbColText, EDbColText, EDbColText, EDbColInt32, -1 };
	TInt idx[]={ ERequestId, -2, EPriority, -1 };
	SetTextLen(255);
	MDBStore::ConstructL(cols, idx, EFalse, _L("downloads"));

	SwitchIndexL(EPriorityIdx);
	TBool found;
	RESET_IF_NECESSARY(found=iTable.LastL());
	if (found) {
		iTable.GetL();
		iPriority=iTable.ColInt(EPriority)+1;
	} else {
		iPriority=1;
	}
	iCurrentState=EIdle;
	User::LeaveIfError(iTimer.CreateLocal()); iTimerIsOpen=ETrue;
	CActiveScheduler::Add(this);
	Async();
}

CDownloaderImpl::~CDownloaderImpl()
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("~CDownloaderImpl"));
	Cancel();
	if (iTimerIsOpen) iTimer.Close();
	CloseFile();
	Reset();
	delete iNotifyState;
}

void CDownloaderImpl::Async(TInt aSeconds)
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("Async"));
	if (iCurrentState==EIdle && !IsActive()) {
		if (aSeconds==0) {
			TRequestStatus *s=&iStatus;
			User::RequestComplete(s, KErrNone);
		} else {
			TTimeIntervalMicroSeconds32 w(aSeconds*1000*1000);
			iTimer.After(iStatus, w);
		}
		SetActive();
	}
}

TBool CDownloaderImpl::CheckForLocal(const TDesC& aUrl)
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("CheckForLocal"));
	if (aUrl.Mid(1, 2).Compare(_L(":\\"))==0) {
		TBuf<50> aContentType;
		if (aUrl.Right(3).CompareF(_L("jpg"))==0) {
			aContentType.Append(_L("image/jpeg"));
		} else if (aUrl.Right(3).CompareF(_L("amr"))==0) {
			aContentType.Append(_L("audio/amr"));
		} else if (aUrl.Right(3).CompareF(_L("3gp"))==0) {
			aContentType.Append(_L("video/3gp"));
		} else {
			aContentType.Append(_L("text/plain"));
		}
		iTable.SetColL(EFileName, aUrl);
		iTable.SetColL(EContentType, aContentType);
		iTable.SetColL(EFinished, 1);
		return ETrue;
	}
	return EFalse;
}

void CDownloaderImpl::DownloadL(TInt64 aRequestId, const TDesC& aUrl, TBool aForce)
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("DownloadL"));
#ifdef __WINS__
	RDebug::Print(_L("h:%d"), aRequestId.High());
	RDebug::Print(_L("l:%d"), aRequestId.Low());
	RDebug::Print(aUrl);
#endif

	TBool found=SeekToRequestId(aRequestId);
	if (found) {
		iTable.GetL();
		if (iTable.ColDes(EUrl).Compare(aUrl)) 
			User::Leave(KErrArgument);
		iTable.UpdateL();
		if (aForce) {
			iTable.SetColL(EErrors, 0);
		}
		iTable.SetColL(EPriority, ++iPriority);
		PutL();
	} else {
		RESET_IF_NECESSARY(iTable.InsertL());
		iTable.SetColL(ERequestId, aRequestId);
		iTable.SetColL(EPriority, ++iPriority);
		iTable.SetColL(EUrl, aUrl);
		iTable.SetColL(EErrors, 0);
		iTable.SetColL(EErrorCode, 0);
		if (! CheckForLocal(aUrl)) {
			iTable.SetColL(EFinished, 0);
		}
		PutL();
	}
	Async();
}

void CDownloaderImpl::CheckedRunL()
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("CheckedRunL"));
#ifdef __WINS__
	RDebug::Print(_L("CheckedRunL()"));
#endif
	SwitchIndexL(EPriorityIdx);
	TBool found;
	TInt errcount=0; TInt err; TInt err_row=-1, current_row;
again:
	current_row=0;
	if (errcount > 5) User::Leave(err);
	RESET_IF_NECESSARY(found=iTable.LastL());

	TBool requests_left=EFalse;

	while (found) {
		current_row++;
		TRAP(err, iTable.GetL());
		if (err==KErrCorrupt) {
			++errcount;
			if (errcount==2 && current_row==err_row) {
				iTable.DeleteL();
			}
			err_row=current_row;
			MDBStore::Reset(err);
			goto again;
		}
		if (iTable.ColInt(EFinished)) {
			iCurrentFileName=iTable.ColDes(EFileName);
			iCurrentContentType=iTable.ColDes(EContentType);
			iCurrentRequestId=iTable.ColInt64(ERequestId);
			Async();
			GotAllL();
			return;
		}
		if (iTable.ColInt(EErrors) <= MAX_ERRORS) {
			iCurrentRequestId=iTable.ColInt64(ERequestId);
			iCurrentUrl=iTable.ColDes(EUrl);
			TRAPD(err, StartDownload());
			if (err!=KErrNone) {
				iCurrentState=EIdle;
				if (MarkError(err, _L("error starting download")))
					requests_left=ETrue;
				Reset();
			} else {
				break;
			}
		} 
		RESET_IF_NECESSARY(found=iTable.PreviousL());
	}
	if (!found) {
		Reset();
		if (requests_left) {
			ShowDownloading(EDownloading);
			Async(GetErrorWait());
		} else {
			ShowDownloading(EDone);
		}
	} else {
		ShowDownloading(EDownloading);
	}
}

TInt CDownloaderImpl::GetErrorWait()
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("GetErrorWait"));
#ifdef __WINS__
	return 1;
#else
	return 10;
#endif
}

void CDownloaderImpl::OpenFileL()
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("OpenFileL"));
	if (iFileIsOpen) return;

	SeekToRequestId(iCurrentRequestId);
	iTable.GetL();

	iCurrentFileName.Zero();
	iCurrentFileName.Append(iDirectory);
	iCurrentFileName.AppendNum(iCurrentRequestId);

	TBuf<10> extension;
	GetFileExtensionL(iCurrentContentType, extension);
	iCurrentFileName.Append(extension);

	RESET_IF_NECESSARY(iTable.UpdateL());
	iTable.SetColL(EFileName, iCurrentFileName);
	iTable.SetColL(EContentType, iCurrentContentType);
	PutL();

	User::LeaveIfError(iFile.Replace(Fs(), iCurrentFileName, EFileWrite));
	iFileIsOpen=ETrue;

}

void CDownloaderImpl::StartDownload()
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("StartDownload"));
#ifdef __WINS__
	TBuf<100> msg=_L("StartDownload()");
	RDebug::Print(msg);
#endif
	iReseting=iGotReply=iGotError=EFalse;
	CloseFile();

	iOffset=0;
	iCurrentFileName.Zero();

	if (! iTable.IsColNull(EFileName)) {
		iCurrentFileName=iTable.ColDes(EFileName);
		TEntry entry;
		TInt err=Fs().Entry(iCurrentFileName, entry);
		if (err==KErrNone) {
			err=iFile.Open(Fs(), iCurrentFileName, EFileWrite);
			if (err==KErrNone) {
				iFileIsOpen=ETrue;
				iOffset=entry.iSize;
				err=iFile.Seek(ESeekStart, iOffset);
				if (err!=KErrNone) {
					CloseFile();
				}
			}
		}
		if (err!=KErrNone) {
			Fs().Delete(iCurrentFileName);
			iCurrentFileName.Zero();
			iOffset=0;
		}
	}

	if (!iHttp) iHttp=CHttp::NewL(*this, AppContext());

	if (iFixedIAP>0) {
		iIAP=iFixedIAP;
	} else {
		Settings().GetSettingL(SETTING_IP_AP, iIAP);
	}
	iHttp->GetL(iIAP, iCurrentUrl, TTime(0), iOffset);
	iCurrentState=EFetching;
}

void CDownloaderImpl::NotifyHttpStatus(THttpStatus st, TInt aError)
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("NotifyHttpStatus"));
#ifdef __WINS__
	TBuf<100> msg=_L("NotifyHttpStatus: ");
	msg.AppendNum(st); msg.Append(_L(" err: "));
	msg.AppendNum(aError);
	RDebug::Print(msg);
#endif
	if (iReseting) return;

	TInt err;
	switch(st) {
		case EHttpConnected:
			break;
		case EHttpDisconnected:
			iCurrentState=EIdle;
			iHttp->Disconnect();
			if (iGotReply || (aError==KErrEof && iSize==-1 && !iGotError)) {
				TRAP(err, GotAllL());
				if (err!=KErrNone) {
					MarkError(err, _L("error notifying"));
					Reset();
				}
			} else if (iGotError) {
				Reset();
			} else {
				Reset();
				MarkError(aError, _L("http disconnected"));
			}
			Async();
			break;
		case EHttpError:
			iGotError=true;
			MarkError(aError, _L("got http error"));
			break;
	}
}

void CDownloaderImpl::NotifyNewHeader(const CHttpHeader &aHeader)
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("NotifyNewHeader"));
#ifdef __WINS__
	TBuf<100> msg=_L("NotifyNewHeader: ");
	msg.AppendNum(aHeader.iHttpReplyCode);
	RDebug::Print(msg);
#endif
	iSize=aHeader.iSize;
	if (aHeader.iHttpReplyCode<200 || aHeader.iHttpReplyCode>299) {
		TBuf<100> msg=_L("Server replied with ");
		msg.AppendNum(aHeader.iHttpReplyCode);
		MarkError(KErrGeneral, msg);
		iGotError=ETrue;
		CloseFile();
	} else {
		if (aHeader.iChunkStart > iOffset) {
			iGotError=ETrue;
			MarkError(KErrGeneral, _L("Server gave offset larger than what we asked for"));
			CloseFile();
		} else {
			iCurrentContentType=aHeader.iContentType;
			if (iOffset > aHeader.iChunkStart && iOffset>0) {
				// offset is non-zero only if
				// the file is already open
				iOffset=aHeader.iChunkStart;
				if (iOffset<0) iOffset=0;
				User::LeaveIfError(iFile.Seek(ESeekStart, iOffset));
			}
			OpenFileL();
		}
	}
}

void CDownloaderImpl::NotifyNewBody(const TDesC8 &chunk)
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("NotifyNewBody"));
#ifdef __WINS__1
	TBuf<100> msg=_L("NotifyNewBody");
	RDebug::Print(msg);
#endif
	if (iGotError) return;

	iFile.Write(chunk);
	iGot+=chunk.Length();
	if (iGot>=iSize) iGotReply=ETrue;
}

void CDownloaderImpl::GotAllL()
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("GotAllL"));
	TAutoBusy busy(Reporting());
#ifdef __WINS__
	TBuf<100> msg=_L("GotAllL");
	RDebug::Print(msg);
#endif
	if (SeekToRequestId(iCurrentRequestId)) {
		RESET_IF_NECESSARY(iTable.UpdateL());
		iTable.SetColL(EFinished, 1);
		PutL();
		iObserver.DownloadFinished(iCurrentRequestId, iCurrentFileName, iCurrentContentType);
		RESET_IF_NECESSARY(iTable.DeleteL());
	} else {
		iObserver.DownloadFinished(iCurrentRequestId, iCurrentFileName, iCurrentContentType);
	}
}

void CDownloaderImpl::CloseFile()
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("CloseFile"));
	if (iFileIsOpen) {
		iFile.Close();
		iFileIsOpen=EFalse;
	}
}

TBool CDownloaderImpl::SeekToRequestId(TInt64 aId)
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("SeekToRequestId"));
	SwitchIndexL(ERequestIdx);
	TDbSeekKey k(aId);
	TBool ret=EFalse;
	RESET_IF_NECESSARY(ret=iTable.SeekL(k));
	return ret;
}

void CDownloaderImpl::Reset()
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("Reset"));
	iReseting=ETrue;
	CloseFile();
	if (iHttp) iHttp->Disconnect();
	delete iHttp; iHttp=0;
	iReseting=EFalse;
	iCurrentState=EIdle;
	return;
}

TBool CDownloaderImpl::MarkError(TInt aError, const TDesC& aDescr)
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("MarkError"));
	TBool ret=ETrue;
	TRAPD(err, ret=MarkErrorL(aError, aDescr));
	if (err!=KErrNone) return ETrue;
	return ret;
}

TBool CDownloaderImpl::MarkErrorL(TInt aError, const TDesC& aDescr)
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("MarkErrorL"));
	TBool ret=ETrue;
	if (SeekToRequestId(iCurrentRequestId)) {
		iTable.GetL();
		TInt count=iTable.ColInt(EErrors)+1;
		if (count>MAX_ERRORS) {
			iObserver.DownloadError(iCurrentRequestId, aError, aDescr);
			ret=EFalse;
		} 
		iTable.UpdateL();
		iTable.SetColL(EErrors, count);
		iTable.SetColL(EErrorCode, aError);
		iTable.SetColL(EErrorDescr, aDescr);
		PutL();
	} else {
		iObserver.DownloadError(iCurrentRequestId, KErrGeneral, _L("internal error: row disappeared"));
		ret=EFalse;
	}
	return ret;
}

void CDownloaderImpl::RemoveRequest(TInt64 aRequestId)
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("RemoveRequest"));
	// TODO: implement
}

void CDownloaderImpl::SetFixedIap(TInt aIap)
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("SetFixedIap"));
	iFixedIAP=aIap;
}

void CDownloaderImpl::DoCancel()
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("DoCancel"));
	if (iStatus==KRequestPending)
		iTimer.Cancel();
}

void CDownloaderImpl::GetFileExtensionL(const TDesC& aMimeType, TDes& aExtensionInto)
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("GetFileExtensionL"));
	TBuf<50> basictype;
	TInt semicol_pos;
	semicol_pos=aMimeType.Locate(';');
	if (semicol_pos==KErrNotFound) {
		basictype=aMimeType;
	} else {
		basictype=aMimeType.Left(semicol_pos);
	}

	if (basictype.CompareF(_L("image/jpeg"))==0) {
		aExtensionInto=_L(".jpg");
	} else if (basictype.CompareF(_L("image/jpg"))==0) {
		aExtensionInto=_L(".jpg");
	} else if (basictype.CompareF(_L("audio/wav"))==0) {
		aExtensionInto=_L(".wav");
	} else if (basictype.CompareF(_L("audio/amr"))==0) {
		aExtensionInto=_L(".amr");
	} else if (basictype.Left(5).CompareF(_L("video"))==0) {
		aExtensionInto=_L(".3gp");
	} else if (basictype.CompareF(_L("text/plain"))==0) {
		aExtensionInto=_L(".txt");
	} else if (basictype.CompareF(_L("text/html"))==0) {
		aExtensionInto=_L(".html");
	} else {
		aExtensionInto=_L(".dat");
	}
}


void CDownloaderImpl::ShowDownloading(TDownloadingStatus aStatus)
{
	CALLSTACKITEM_N(_CL("CDownloaderImpl"), _CL("ShowDownloading"));
	if ( aStatus == EDone ) {
		delete iNotifyState; iNotifyState=0;
	} else {
		if (! iNotifyState ) {
			iNotifyState=CNotifyState::NewL(AppContext(), KIconFile);
		}
		if (aStatus == EDownloading) {
			iNotifyState->SetCurrentState(EMbmContextnetworkDownloading, EMbmContextnetworkDownloading);
		} else {
			iNotifyState->SetCurrentState(EMbmContextnetworkDownloading, EMbmContextnetworkDownloading);
		}
	}
}
