#include "cn_http.h"
#include <charconv.h>

const TInt CHttp::KDefaultPortNumber = 80;
_LIT8(HTTP_HEAD, "HEAD %S HTTP/1.0\r\n\r\n");
_LIT8(HTTP_GET_MAIN, "GET %S HTTP/1.0\r\n");
_LIT8(HTTP_POST_MAIN, "POST %S HTTP/1.0\r\n");
_LIT8(HTTP_POST_HOST, "Host: %S\r\n");
_LIT8(HTTP_POST_CONTENTTYPE, "Content-Type: %S\r\n");
_LIT8(HTTP_POST_CONTENTLENGTH, "Content-Length: %d\r\n");
_LIT8(HTTP_POST_CONTENTDISPOSITION, "Content-Disposition: form-data; name=\"%S\"\r\n");
_LIT8(HTTP_POST_CONTENTDISPOSITIONFILE, "Content-Disposition: form-data; name=\"%S\"; filename=\"%S\"\r\n");
_LIT8(FORMDATA_TYPE, "multipart/form-data; boundary=");

_LIT8(HTTP_GET_MODIFIED_SINCE, "If-Modified-Since: %S\r\n");
_LIT8(HTTP_GET_RANGE, "Range: bytes=%u-%u\r\n");
_LIT(KDefaultServerName, "www.cs.helsinki.fi");

#define SIMPLE_UPLOAD 0
#include "util.h"

EXPORT_C CHttp * CHttp::NewL(MHttpObserver& aObserver, MApp_context &Context)
{
	CHttp* self = CHttp::NewLC(aObserver, Context);
	CleanupStack::Pop(self);
	return self;
}

EXPORT_C CHttp * CHttp::NewLC(MHttpObserver& aObserver, MApp_context &Context)
{
	
	CHttp* self = new (ELeave) CHttp(aObserver, Context);
	CleanupStack::PushL(self);
	self->ConstructL();
	return self;
}

CHttp::CHttp(MHttpObserver& aObserver, MApp_context& Context): 
		MContextBase(Context), iObserver(aObserver), iPort(KDefaultPortNumber), iServerName(KDefaultServerName)
{

}

void CHttp::ConstructL()
{
	iHttpStatus = ENotConnected;
	iEngine = CSocketsEngine::NewL(*this, iContext);
	iHeader = CHttpHeader::NewL();
	
	iParsingState = EUndefined;
}

EXPORT_C CHttp::~CHttp()
{
	if (iDebugFileIsOpen) iDebugFile.Close();

	DeleteParts();
	delete iParseBuffer;
	delete iHeader;
	delete iEngine;
	delete iCommand;
}

void CHttp::MakeBoundary()
{
	TTime t; t=GetTime();
	TInt64 i=t.Int64();
	iBoundary=_L8("------");
	iBoundary.AppendNum(i.Low());
	iBoundary.AppendNum(i.High());
	iBoundary.Append(_L8("---------\r\n"));
}

void CHttp::ChangeStatus(THttpState aNewStatus, TInt aError)
{
	iHttpStatus = aNewStatus;
	switch (aNewStatus)
        {
        case ENotConnected:

		iObserver.NotifyHttpStatus(MHttpObserver::EHttpDisconnected, aError);
		Log(_L("Http::Not connected"));
		break;
		
        case EConnecting:
		Log(_L("Http::Connecting"));
		break;

	case EConnected:
		iObserver.NotifyHttpStatus(MHttpObserver::EHttpConnected, KErrNone);
		Log(_L("Http::Connected"));
		break;
	case EDisconnectForNew:
		Log(_L("Http::Disconnecting For New Request"));
		break;
        default:
		Log(_L("Http::Unknown status"));
		break;
        }
}

void CHttp::ConnectL()
{
	CALLSTACKITEM_N(_CL("CHttp"), _CL("ConnectL"));

	if (iHttpStatus == EConnected) {
		ChangeStatus(EDisconnectForNew, KErrNone);
		iEngine->Disconnect(EFalse);
	} else if (iHttpStatus == ENotConnected) {
		ChangeStatus(EConnecting, KErrNone);
		iEngine->ConnectL(iServerName, iPort, iAccessPoint);
	}
}

void CHttp::NotifyEngineStatus(TInt st, TInt aError)
{
#ifdef __WINS__
	TBuf<100> msg=_L("CHttp::NotifyEngineStatus: ");
	msg.AppendNum(st); msg.Append(_L(" http status before: "));
	msg.AppendNum(iHttpStatus); msg.Append(_L(" err: "));
	msg.AppendNum(aError);
	RDebug::Print(msg);
#endif
	if (st == MEngineObserver::ESocketConnected) {
		ChangeStatus(EConnected, KErrNone);
		ResetParser();
		if (iCommand) Log(*iCommand);
		if (iCommand) iEngine->WriteL(*iCommand);
		if (iDebugFileIsOpen) iDebugFile.Write(*iCommand);
		iEngine->Read();
	} else if (st == MEngineObserver::ESocketDisconnected) {
		if (iHttpStatus==EDisconnectForNew) {
			iHttpStatus = EConnecting;
			ConnectL();
		} else if (iHttpStatus!=ENotConnected) {
			ChangeStatus(ENotConnected, aError);
		}
	} else {
		ChangeStatus(ENotConnected, aError);
	}
}


void CHttp::ReportError(TInt aErrorType, TInt errorCode)
{
	switch (aErrorType)
	{
	case EServerError:
		Log(_L("Http: Server error"), errorCode);
		break;

	case EParseError:
		Log(_L("CJabber: StreamError"));
		break;
	
	default:
		break;
	}
}


EXPORT_C void CHttp::GetL(const TUint& iAP, const TDesC &url, const TTime &modified_since, const TUint &chunkStart, const TUint &chunkEnd)
{
	iMethod=EGet;
	iAccessPoint = iAP;
	ParseUrl(url);

	delete iCommand;
	iCommand=0;

	iCommand = HBufC8::NewL(url.Length() + HTTP_GET_MAIN().Length()+2);
	iCommand->Des().AppendFormat(HTTP_GET_MAIN(), &iUrl);
	if (modified_since != TTime(0)) {
		TBuf8<30> internet_time;
		TimeIntoInternetTime(internet_time, modified_since);
		iCommand=iCommand->ReAllocL(iCommand->Length()+HTTP_GET_MODIFIED_SINCE().Length() + internet_time.Length() );
		iCommand->Des().AppendFormat(HTTP_GET_MODIFIED_SINCE(), &internet_time);
	}
	if ( chunkEnd!=0 ) {
		iCommand=iCommand->ReAllocL(iCommand->Length()+HTTP_GET_RANGE().Length() + 10 );
		iCommand->Des().AppendFormat(HTTP_GET_RANGE(), chunkStart, chunkEnd);
	}
	iCommand->Des().Append(_L("\r\n"));
	Log(*iCommand);

	TRAPD(err, ConnectL());
}


void CHttp::NotifyNewData(const TDesC8 &aBuffer)
{
	if (!Parse(aBuffer)) {
		iObserver.NotifyHttpStatus(MHttpObserver::EHttpError, KErrGeneral);
		Log(_L("Parse Error"));
	}
}


void CHttp::NotifyCanWrite()
{
begin:
	if (iMethod==EGet) return;

	iCommand->Des().Zero();

	if (iPartState==ESendingHeader) {
		if (iCurrentPartNode) {
			TBuf8<256> buf8, buf8_2;
			Log(_L("send header"));
			CPostPart* p=iCurrentPartNode->Item;
			if (iCurrentPartNode != iParts->iFirst) 
				iCommand->Des()=_L8("\r\n");

			iCommand->Des().Append(iBoundary);
			if (p->FileName().Length() > 0) {
				CC()->ConvertFromUnicode(buf8, p->iName);
				CC()->ConvertFromUnicode(buf8_2, p->FileName());
				iCommand->Des().AppendFormat(HTTP_POST_CONTENTDISPOSITIONFILE, 
					&buf8, &buf8_2);
			} else {
				CC()->ConvertFromUnicode(buf8, p->iName);
				iCommand->Des().AppendFormat(HTTP_POST_CONTENTDISPOSITION, 
					&buf8);
			}
			buf8.Zero(); CC()->ConvertFromUnicode(buf8, p->iMimeType);
			iCommand->Des().AppendFormat(HTTP_POST_CONTENTTYPE, &buf8);
			iCommand->Des().Append(_L8("\r\n"));
		} else {
			if (!SIMPLE_UPLOAD || iParts->iCount > 1) {
				Log(_L("send last boundary"));
				iCommand->Des()=_L8("\r\n");
				iCommand->Des().Append(iBoundary);
			} else {
				Log(_L("sent all"));
				return;
			}
		}
	} else {
		Log(_L("send body"));
		if (iCurrentPartNode) {
			CPostPart* p=iCurrentPartNode->Item;
			TPtr8 cp=iCommand->Des();
			p->ReadChunkL(cp);
			if (iCommand->Des().Length()==0) {
				//EOF on part
				iCurrentPartNode=iCurrentPartNode->Next;
				iPartState=ESendingHeader;
				goto begin;
			}
		} else {
			return;
		}
	}
	iPartState=ESendingBody;
	
	if (iDebugFileIsOpen) iDebugFile.Write(*iCommand);
	iEngine->WriteL(*iCommand);
}

TBool CHttp::Parse(const TDesC8 &aBuffer)
{
	TInt eol=KErrNotFound;
	TInt begin_substring = 0;
	TBool ok = ETrue;
	TBool cont = ETrue;
	_LIT8(KCRLF, "\r\n");

	if ( (iParsingState == EUndefined) || (iParsingState==EHeader) ) {
		while ( cont ) {
			eol=aBuffer.Mid(begin_substring, aBuffer.Length()- begin_substring).Find(KCRLF);
			if (eol != KErrNotFound) {
				/*if (eol == 24) { 
					RDebug::Print(_L("EndOfLine")); 
				}*/
				RDebug::Print(_L("found eol at %d"), eol);
				if (iParseBuffer) {
					iParseBuffer=iParseBuffer->ReAllocL(iParseBuffer->Des().Length() + eol);
					iParseBuffer->Des().Append(aBuffer.Mid(begin_substring, eol));
					ok = ParseLine(*iParseBuffer);
					delete iParseBuffer; iParseBuffer=0;
				} else {
					ok = ParseLine(aBuffer.Mid(begin_substring, eol));
				}
				cont = (iParsingState==EHeader);
				begin_substring += eol + KCRLF().Length();
			} else {
				cont = EFalse;
				if (iParseBuffer) {
					TInt len=iParseBuffer->Des().Length();
					// case when \r\n split over the parsebuffer and
					// the incoming buffer
					if (iParseBuffer->Des()[len-1]=='\r') {
						if (aBuffer[0]=='\n') {
							ok=ParseLine(iParseBuffer->Des().Mid(0, len-1));
							delete iParseBuffer; iParseBuffer=0;
							begin_substring=1;
							if (iParsingState == EHeader) cont=ETrue;
						}
					}
				}
			}
		}
		if (!ok) return ok;

      		if ( begin_substring < aBuffer.Length() ) {
			if (iParseBuffer) {
				iParseBuffer=iParseBuffer->ReAllocL(iParseBuffer->Des().Length() + aBuffer.Length() - begin_substring );
				iParseBuffer->Des().Append( aBuffer.Mid(begin_substring, aBuffer.Length() - begin_substring) );
			} else {
				iParseBuffer = HBufC8::NewL(aBuffer.Length() - begin_substring);
				iParseBuffer->Des().Append(aBuffer.Mid(begin_substring, aBuffer.Length() - begin_substring));
			}
			if (iParsingState==EBody) {
				iObserver.NotifyNewBody(*iParseBuffer);
				delete iParseBuffer; iParseBuffer=0;
			}
		}
	} else if (iParsingState == EBody) {
		iObserver.NotifyNewBody(aBuffer);
		delete iParseBuffer; iParseBuffer=0;
	} else {
		//error!
		ReportError(EParseError, 0);
	}
	return ok;
}

TBool CHttp::ParseLine(const TDesC8 &aLine)
{
	_LIT8(HTTP_REPLY, "HTTP/*");
	_LIT8(HTTP_DATE, "Date:*");
	_LIT8(HTTP_SERVER, "Server:*");
	_LIT8(HTTP_LASTMOD, "Last-Modified:*");
	_LIT8(HTTP_ETAG, "ETag:*");
	_LIT8(HTTP_ACCEPT_RANGE, "Accept-Ranges:*");
	_LIT8(HTTP_SIZE, "Content-Length:*");
	_LIT8(HTTP_MIME, "Content-Type:*");
	_LIT8(HTTP_CONNECTION, "Connection:*");
	_LIT8(HTTP_RANGE, "Content-Range: bytes*");
	
	TLex8 lex;
	TInt err;

	Log(aLine);
	if (aLine.Length() == 0) {
		// end of header - about to receive a body
		iParsingState = EBody;
		iObserver.NotifyNewHeader(*iHeader);
		return ETrue;
	} else if (aLine.Match(HTTP_REPLY) != KErrNotFound) {
		iParsingState = EHeader;
		iHeader->iFilename.Copy(iUrl);
		iHeader->iServername.Copy(iServerName);
		lex=aLine.Mid(5, 3);
		err = lex.Val(iHeader->iHttpVersion);
		lex=aLine.Mid(9, 3);
		err = lex.Val(iHeader->iHttpReplyCode);
		return (err==KErrNone);
	} else if (aLine.Match(HTTP_DATE) != KErrNotFound) {
		iHeader->iServerTime = InternetTimeIntoTime(aLine.Mid(6, aLine.Length()-6));
		return ETrue;
	} else if (aLine.Match(HTTP_SERVER) != KErrNotFound) {
		// not supported yet
		return ETrue;
	} else if (aLine.Match(HTTP_LASTMOD) != KErrNotFound) {
		iHeader->iLastMod = InternetTimeIntoTime(aLine.Mid(15, aLine.Length()-15));
		return ETrue;
	} else if (aLine.Match(HTTP_ETAG) != KErrNotFound) {
		//not handled yet
		return ETrue;
	} else if (aLine.Match(HTTP_ACCEPT_RANGE) != KErrNotFound) {
		return ETrue;
	} else if (aLine.Match(HTTP_RANGE) != KErrNotFound ) {
			// we're receiving a chunk
			TInt dash = aLine.Mid(HTTP_RANGE().Length(), aLine.Length() - HTTP_RANGE().Length()).Find(_L8("-"));
			TInt slash = aLine.Mid(HTTP_RANGE().Length(), aLine.Length() - HTTP_RANGE().Length()).Find(_L8("/"));

			if ((dash!=KErrNotFound) && (slash!=KErrNotFound)) {
				lex=aLine.Mid(HTTP_RANGE().Length(), dash);
				err=lex.Val(iHeader->iChunkStart);

				lex=aLine.Mid(HTTP_RANGE().Length()+dash+1, slash-dash-1);
			        err=lex.Val(iHeader->iChunkEnd);

				lex=aLine.Mid(HTTP_RANGE().Length()+slash+1, aLine.Length() - (slash+1+HTTP_RANGE().Length()));
				err=lex.Val(iHeader->iSize);
				return (err==KErrNone);
			} else 	return EFalse;
	} else if (aLine.Match(HTTP_SIZE) != KErrNotFound) {
		lex=aLine.Mid(16, aLine.Length()-16);
		err=lex.Val(iHeader->iSize);
		iHeader->iChunkStart=0;
		iHeader->iChunkEnd=iHeader->iSize;
		return (err==KErrNone);
	} else if (aLine.Match(HTTP_MIME) != KErrNotFound) {
		TInt state=CCnvCharacterSetConverter::KStateDefault;
		CC()->ConvertToUnicode(iHeader->iContentType, aLine.Mid(HTTP_MIME().Length()), state);
		return ETrue;
	} else if (aLine.Match(HTTP_CONNECTION) != KErrNotFound) {
		return ETrue;
	} else {
		return EFalse;
	}
}

void CHttp::ResetParser()
{
	delete iParseBuffer; iParseBuffer=0;
	iParsingState = EUndefined;
	iHeader->Reset();
}

TBool CHttp::ParseUrl(const TDesC &url)
{
	_LIT(HTTP, "http://");

	// port not handled yet
	CC()->ConvertFromUnicode(iUrl, url);
		
	if (url.Left(HTTP().Length()).Compare(HTTP()) == 0) {
		TInt length = url.Length()-HTTP().Length();
		TInt slash = url.Mid(HTTP().Length(), length).Find(_L("/"));
		if (slash==KErrNotFound) slash=length;

		iServerName=url.Mid(HTTP().Length(), slash);
		return ETrue;
	} else {
		return EFalse;
	}
}

EXPORT_C CHttpHeader * CHttpHeader::NewL()
{
	CHttpHeader* self = CHttpHeader::NewLC();
	CleanupStack::Pop(self);
	return self;
}

EXPORT_C CHttpHeader * CHttpHeader::NewLC()
{
	CHttpHeader* self = new (ELeave) CHttpHeader();
	CleanupStack::PushL(self);
	self->ConstructL();
	return self;
}

void CHttpHeader::ConstructL()
{
	Reset();
}

EXPORT_C void CHttpHeader::Reset()
{
	iFilename.Zero();
	iHttpReplyCode = -1;
	iHttpVersion=-1;
	iLastMod=TTime(0);
	iServername.Zero();
	iSize=-1;
	iChunkStart = -1;
	iChunkEnd = -1;
	iServerTime=TTime(0);
}

EXPORT_C void CHttpHeader::Copy(const CHttpHeader &aHeader)
{
	this->iFilename.Copy(aHeader.iFilename);
	this->iHttpReplyCode = aHeader.iHttpReplyCode;
	this->iHttpVersion = aHeader.iHttpVersion;
	this->iLastMod = aHeader.iLastMod;
	this->iServername.Copy(aHeader.iServername);
	this->iServerTime = aHeader.iServerTime;
	this->iSize = aHeader.iSize;
	this->iChunkStart = aHeader.iChunkStart;
	this->iChunkEnd = aHeader.iChunkEnd;
}

TTime CHttp::InternetTimeIntoTime(const TDesC8 &aInternetTimeString)
{
	// GMT offset not supported! 
	TTime t;
	TBuf<20> date_string;
	
	TBuf<2> day;
	TBuf<3> month;
	TBuf<4> year;
	TBuf<2> hour;
	TBuf<2> min;
	TBuf<2> sec;

	day.Copy(aInternetTimeString.Mid(5,2));
	month.Copy(aInternetTimeString.Mid(8,3));
	year.Copy(aInternetTimeString.Mid(12,4));
	
	hour.Copy(aInternetTimeString.Mid(17,2));
	min.Copy(aInternetTimeString.Mid(20,2));
	sec.Copy(aInternetTimeString.Mid(23,2));

	date_string.Append(day);
	date_string.Append(_L("-"));
	if (month.Compare(_L("Jan")) == 0) {
		date_string.Append(_L("01"));
	} else if (month.Compare(_L("Feb")) == 0) {
		date_string.Append(_L("02"));
	} else if (month.Compare(_L("Apr")) == 0) {
		date_string.Append(_L("03"));
	} else if (month.Compare(_L("May")) == 0) {
		date_string.Append(_L("05"));
	} else if (month.Compare(_L("Jun")) == 0) {
		date_string.Append(_L("06"));
	} else if (month.Compare(_L("Jul")) == 0) {
		date_string.Append(_L("07"));
	} else if (month.Compare(_L("Aug")) == 0) {
		date_string.Append(_L("08"));
	} else if (month.Compare(_L("Sep")) == 0) {
		date_string.Append(_L("09"));
	} else if (month.Compare(_L("Oct")) == 0) {
		date_string.Append(_L("10"));
	} else if (month.Compare(_L("Nov")) == 0) {
		date_string.Append(_L("11"));
	} else if (month.Compare(_L("Dec")) == 0) {
		date_string.Append(_L("12"));
	}

        date_string.Append(_L("-"));
	date_string.Append(year);
	date_string.Append(_L(" "));
	date_string.Append(hour);
	date_string.Append(_L(":"));
	date_string.Append(min);
	date_string.Append(_L(":"));
	date_string.Append(sec);

	t.Parse(date_string);
	return t;
}

void CHttp::TimeIntoInternetTime(TDes8 &ret, const TTime &time)
{
	switch(time.DayNoInWeek()) {
		case EMonday:
			ret.Append(_L8("Mon"));
			break;
		case ETuesday:
			ret.Append(_L8("Tue"));
			break;
		case EWednesday:
			ret.Append(_L8("Wed"));
			break;
		case EThursday:
			ret.Append(_L8("Thu"));
			break;
		case EFriday:
			ret.Append(_L8("Fri"));
			break;
		case ESaturday:
			ret.Append(_L8("Sat"));
			break;
		case ESunday:
			ret.Append(_L8("Sun"));
			break;
	}
	ret.Append(_L8(", "));
	ret.AppendFormat(_L8("%02d"), time.DayNoInMonth()+1);
	ret.Append(_L8(" "));
	switch(time.DateTime().Month()) {
		case EJanuary:
			ret.Append(_L8("Jan"));
			break;
		case EFebruary:
			ret.Append(_L8("Feb"));
			break;
		case EMarch:
			ret.Append(_L8("Mar"));
			break;
		case EApril:
			ret.Append(_L8("Apr"));
			break;
		case EMay:
			ret.Append(_L8("May"));
			break;
		case EJune:
			ret.Append(_L8("Jun"));
			break;
		case EJuly:
			ret.Append(_L8("Jul"));
			break;
		case EAugust:
			ret.Append(_L8("Aug"));
			break;
		case ESeptember:
			ret.Append(_L8("Sep"));
			break;
		case EOctober:
			ret.Append(_L8("Oct"));
			break;
		case ENovember:
			ret.Append(_L8("Nov"));
			break;
		case EDecember:
			ret.Append(_L8("Dec"));
			break;
	}
	ret.Append(_L8(" "));
	ret.AppendFormat(_L8("%04d"), time.DateTime().Year());
	ret.AppendFormat(_L8(" %02d:%02d:%02d GMT"), time.DateTime().Hour(), time.DateTime().Minute(), time.DateTime().Second());
	Log(ret);
}


EXPORT_C CPostPart::CPostPart(const TDesC& aName, const TDesC& aMimeType)
{
	iName=aName;
	iMimeType=aMimeType;
}

EXPORT_C CFilePart* CFilePart::NewL(RFs& aFs, const TDesC& aFileName, 
	const TDesC& aName, const TDesC& aMimeType)
{
	auto_ptr<CFilePart> ret(new (ELeave) CFilePart(aName, aMimeType));
	ret->ConstructL(aFs, aFileName);

	return ret.release();
}

CFilePart::CFilePart(const TDesC& aName, const TDesC& aMimeType) : CPostPart(aName, aMimeType) { }

void CFilePart::ConstructL(RFs& aFs, const TDesC& aFileName)
{
	iFileName=aFileName;
	User::LeaveIfError(iFile.Open(aFs, aFileName, EFileShareAny | EFileRead));
	iFileIsOpen=ETrue;
	User::LeaveIfError(iFile.Size(iSize));
}

TInt	CFilePart::Size()
{
	return iSize;
}

void CFilePart::ReadChunkL(TDes8& aIntoBuffer)
{
	aIntoBuffer.Zero();
	TInt toread;
	if (iSize-iRead > aIntoBuffer.MaxLength()) toread=aIntoBuffer.MaxLength();
	else toread=iSize-iRead;
	User::LeaveIfError(iFile.Read(aIntoBuffer, toread));
	iRead+=aIntoBuffer.Length();
}

EXPORT_C CFilePart::~CFilePart()
{
	if (iFileIsOpen) iFile.Close();
}

EXPORT_C CBufferPart* CBufferPart::NewL(HBufC8* aBuffer, TBool aTakeOwnership,
	const TDesC& aName, const TDesC& aMimeType)
{
	auto_ptr<CBufferPart> ret(new (ELeave) CBufferPart(aName, aMimeType));
	ret->ConstructL(aBuffer, aTakeOwnership);

	return ret.release();
}

EXPORT_C CBufferPart* CBufferPart::NewL(const TDesC8& aBuffer,
	const TDesC& aName, const TDesC& aMimeType)
{
	auto_ptr<CBufferPart> ret(new (ELeave) CBufferPart(aName, aMimeType));
	ret->ConstructL(aBuffer);

	return ret.release();
}

CBufferPart::CBufferPart(const TDesC& aName, const TDesC& aMimeType) : CPostPart(aName, aMimeType) { }

EXPORT_C CBufferPart::~CBufferPart()
{
	if (iOwnsBuffer) delete iBuffer;
}

void CBufferPart::ConstructL(const TDesC8& aBuffer)
{
	iBuffer=aBuffer.AllocL();
	iOwnsBuffer=ETrue;
}

void CBufferPart::ConstructL(HBufC8* aBuffer, TBool aTakeOwnership)
{
	iOwnsBuffer=aTakeOwnership;
	iBuffer=aBuffer;
}

TInt CBufferPart::Size()
{
	return iBuffer->Des().Length();
}

void CBufferPart::ReadChunkL(TDes8& aIntoBuffer)
{
	TInt toread;
	toread=Size()-iPos;
	if (toread > aIntoBuffer.MaxLength()) toread=aIntoBuffer.MaxLength();
	aIntoBuffer=iBuffer->Des().Mid(iPos, toread);
	iPos+=toread;
}

void CHttp::DeleteParts()
{
	delete iParts; iParts=0;
}

EXPORT_C void CHttp::ReleaseParts()
{
	DeleteParts();
}


EXPORT_C void CHttp::PostL(const TUint& iAP, const TDesC &url, CPtrList<CPostPart>* aBodyParts)
{
	DeleteParts();
	iParts=aBodyParts;

#ifdef __WINS__
	if (iDebugFileIsOpen) {
		iDebugFile.Close();
		iDebugFileIsOpen=EFalse;
	}
	if (iDebugFile.Replace(Fs(), _L("c:\\post.txt"), EFileWrite)==KErrNone) {
		iDebugFileIsOpen=ETrue;
	}
#endif

	if (!aBodyParts || aBodyParts->iCount==0) 
		User::Leave(KErrArgument);
	if (! ParseUrl(url)) 
		User::Leave(KErrArgument);

	MakeBoundary();
	iMethod=EPost;
	iAccessPoint = iAP;

	TInt len=url.Length() + HTTP_POST_MAIN().Length()+2;
	len+=HTTP_POST_HOST().Length()+iServerName.Length();
	len+=HTTP_POST_CONTENTTYPE().Length()+128;
	len+=HTTP_POST_CONTENTLENGTH().Length()+20;
	len+=HTTP_POST_CONTENTDISPOSITIONFILE().Length()+256+128;
	len+=iBoundary.Length();

	if (!iCommand || iCommand->Des().MaxLength() < len) {
		delete iCommand; iCommand=0;
		iCommand=HBufC8::NewL(len);
	} else {
		iCommand->Des().Zero();
	}

	iCommand->Des().AppendFormat(HTTP_POST_MAIN(), &iUrl);

	TBuf8<256> buf8, buf8_2;
	CC()->ConvertFromUnicode(buf8, iServerName);
	iCommand->Des().AppendFormat(HTTP_POST_HOST(), &buf8);

	if (SIMPLE_UPLOAD && aBodyParts->iCount == 1) {
		CPostPart* p=aBodyParts->Top();
		buf8.Zero(); CC()->ConvertFromUnicode(buf8, p->iMimeType);
		iCommand->Des().AppendFormat(HTTP_POST_CONTENTTYPE, &buf8);
		iCommand->Des().AppendFormat(HTTP_POST_CONTENTLENGTH, p->Size());
		
		if (p->FileName().Length() > 0) {
			buf8.Zero(); CC()->ConvertFromUnicode(buf8, p->iName);
			buf8_2.Zero(); CC()->ConvertFromUnicode(buf8_2, p->FileName());
			iCommand->Des().AppendFormat(HTTP_POST_CONTENTDISPOSITIONFILE, &buf8, &buf8_2);
		}
	} else {
		TBuf8<158> type=FORMDATA_TYPE();
		type.Append(iBoundary.Mid(2, iBoundary.Length()-4));
		iCommand->Des().AppendFormat(HTTP_POST_CONTENTTYPE, &type);
		TInt len=0;
		for (CPtrList<CPostPart>::Node* n=aBodyParts->iFirst; n; n=n->Next) {
			len+=iBoundary.Length();
			TInt filen_len=n->Item->FileName().Length();
			TInt name_len=n->Item->iName.Length();
			if ( filen_len > 0 ) {
				len+=HTTP_POST_CONTENTDISPOSITIONFILE().Length()-4;
			} else {
				len+=HTTP_POST_CONTENTDISPOSITION().Length()-2;
			}
			len+=filen_len; len+=name_len;
			len+=HTTP_POST_CONTENTTYPE().Length()-2;
			len+=n->Item->iMimeType.Length();
			len+=2; //linebreak after header and before content
			len+=2; //linebreak after content and before boundary
			len+=n->Item->Size();
		}
		len+=iBoundary.Length();
		iCommand->Des().AppendFormat(HTTP_POST_CONTENTLENGTH, len);
	}

	iCommand->Des().Append(_L8("\r\n"));
	Log(*iCommand);

	if (!SIMPLE_UPLOAD || iParts->iCount > 1) iPartState=ESendingHeader;
	else iPartState=ESendingBody;

	iCurrentPartNode=iParts->iFirst;

	ConnectL();
}

EXPORT_C const TDesC& CPostPart::FileName()
{
	return iFileName;
}


EXPORT_C void CPostPart::SetFileName(const TDesC& aFileName)
{
	iFileName=aFileName;
}
EXPORT_C void CHttp::AppendUrlEncoded(TDes& into, const TDesC& str)
{
	int i;
	for (i=0; i<str.Length(); i++) {
		TChar c(str[i]);
		TUint cval=(TUint)c & 0xff;
		if (c.IsAlphaDigit() && cval<0x7f) into.Append(c);
		else {
			TBuf<5> b;
			b.Format(_L("%%%02x"), cval);
			into.Append(b);
		}
	}
}

EXPORT_C void CHttp::Disconnect()
{
	iEngine->Disconnect(ETrue);
}
