/* 
Copyright (C) 2004  Mika Raento - Renaud Petit

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
   
    
     email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/

#include "jabber.h"
#include "sha1.h"
#include <flogger.h>
#include "concretedata.h"

const TInt CJabber::KDefaultPortNumber = 5222;
const TInt CJabber::KCloseSessionTimeOut = 2;
_LIT(KDefaultJabberRessource, "Context");
_LIT(KDefaultServerName, "jabber.org");


_LIT(KStream,"stream:stream");
_LIT(KStreamError, "stream:error");
_LIT(KIq, "iq");
_LIT(KPresence, "presence");
_LIT(KMessage, "message");
_LIT(KStatus, "status");
_LIT(KBody, "body");
_LIT(KSubject, "subject");

_LIT(KId, "id");
_LIT(KType, "type");
_LIT(KResult, "result");
_LIT(KFrom, "from");
_LIT(KError, "error");
_LIT(KCode, "code");
_LIT(KUnavailable, "unavailable");

#include "util.h"

EXPORT_C CJabber * CJabber::NewL(MJabberObserver& aObserver, MApp_context &Context)
{
	CJabber* self = CJabber::NewLC(aObserver, Context);
	CleanupStack::Pop(self);
	return self;
}

EXPORT_C CJabber * CJabber::NewLC(MJabberObserver& aObserver, MApp_context &Context)
{
	CJabber* self = new (ELeave) CJabber(aObserver, Context);
	CleanupStack::PushL(self);
	self->ConstructL();
	return self;
}

CJabber::CJabber(MJabberObserver& aObserver, MApp_context& Context): 
		MContextBase(Context), iObserver(aObserver), iPort(KDefaultPortNumber), 
		iServerName(KDefaultServerName),
		iResource(KDefaultJabberRessource), restart(ETrue)
{
#ifdef __WINS__
	iPresenceInterval=60;
	iFreshnessInterval=4*60;
#else
	iPresenceInterval=60;
	iFreshnessInterval=4*60;
#endif
}

void CJabber::ConstructL()
{	
	iStack = CList<TInt>::NewL();
	iParser = XML_ParserCreate(NULL);
	iXmlBuf=CXmlBuf16::NewL(256);

	iErrorValue = HBufC::NewL(64);
	
	iFrom = HBufC::NewL(64);
	iPresenceInfo=HBufC::NewL(512);
	iMessage=HBufC::NewL(512);
	iSubject=HBufC::NewL(128);
	iUserPresenceInfo=HBufC::NewL(512);

	iSendTimer = CTimeOut::NewL(*this);
	iCloseSessionTimer = CTimeOut::NewL(*this);
	ChangeStatus(ENotConnected, KErrNone);
}

void CJabber::ChangeStatus(TJabberState aNewStatus, TInt aError)
{
	switch (aNewStatus)
        {
        case ENotConnected:
		Log(_L("Jabber::Not connected"), aError);
		break;
		
        case EConnecting:
		Log(_L("Jabber::Connecting"), aError);
		break;

	case EConnected:
		Log(_L("Jabber::Connected"), aError);
		break;
		
	case EInitiating:
		Log(_L("Jabber::Initiating"), aError);
		break;
		
        case EIdentifying:
		Log(_L("Jabber::Identifying"), aError);
		break;
		
	case EDisconnecting:
		Log(_L("Jabber::Disconnecting"), aError);
		break;
		
	case EWaitingForRetry:
		Log(_L("Jabber::Waiting For Retry"), aError);
		break;
		
        default:
		Log(_L("Jabber::Unknown status"), aError);
		break;
        }
	iJabberStatus = aNewStatus;
}

EXPORT_C CJabber::~CJabber()
{
	delete iFrom;
	delete iPresenceInfo;
	delete iMessage;
	delete iSubject;
	delete iUserPresenceInfo;
	delete iErrorValue;
	delete iXmlBuf;	
	XML_ParserFree(iParser);
	delete iStack;
	delete iEngine;
	delete iSendTimer;
	delete iCloseSessionTimer;
}

EXPORT_C void CJabber::ConnectL(const TDesC16 &u, const TDesC16 &p, const TDesC16 &s, TUint32 iAP)
{
	CALLSTACKITEM_N(_CL("CJabber"), _CL("ConnectL"));

	if (iJabberStatus == EConnected) {
		iObserver.NotifyJabberStatus(MJabberObserver::EJabberConnected);
		// start presence timer, and send presence??
	} else if (iJabberStatus == ENotConnected) {
		delete iEngine; iEngine=0;
		TBool compress=EFalse;
		if (s.CompareF(_L("armi.hiit.fi"))==0) compress=ETrue;
		iEngine = CSocketsEngine::NewL(*this, iContext, compress);

		iCloseConnection = EFalse;
		iUsername.Copy(u);
		iFullNick.Zero();
		iFullNick.Append(u);
		iFullNick.Append(_L16("@"));
		iFullNick.Append(s);
		iFullNick.Append(_L("/"));
		iFullNick.Append(iResource);
		iPassword.Copy(p);
		iServerName.Copy(s);
		iAccessPoint = iAP;
		iJabberSessionId.Zero();

		XML_ParserReset(iParser, NULL);
		XML_SetUserData(iParser,this);
		XML_SetElementHandler(iParser, CJabber::startElement, CJabber::endElement);
		XML_SetCharacterDataHandler(iParser, CJabber::charData);
		iXmlBuf->Zero();
		iStack->reset();

		ChangeStatus(EConnecting, KErrNone);
		iEngine->ConnectL(iServerName, iPort, iAccessPoint);
	}
}

EXPORT_C void CJabber::Disconnect(TBool close_connection)
{
	if (iJabberStatus != ENotConnected) {
		restart = EFalse;
		DoDisconnect(close_connection);
	}
}

void CJabber::DoDisconnect(TBool close_connection)
{
	TJabberState prev_state=iJabberStatus;
	iCloseConnection = close_connection;
	iSendTimer->Reset();
	ChangeStatus(EDisconnecting, KErrNone);
	iEngine->StopRead();
	if (prev_state==EConnected) {
		TRAPD(err, SendDisconnectionL());
		Log(_L("Error in SendDisconnectionL %d"), err);
	}
	iCloseSessionTimer->Wait(KCloseSessionTimeOut);
}

void CJabber::NotifyEngineStatus(TInt st, TInt aError)
{
	if (st == MEngineObserver::ESocketConnected) {
		switch (iJabberStatus)
		{
		case EConnecting:
			ChangeStatus(EInitiating, KErrNone);
			SendXMLStreamHeaderL();
			iEngine->Read();
			break;
		default:
			break;
		}
	} else if (st == MEngineObserver::ESocketDisconnected) {
		switch (iJabberStatus)
		{
		case EDisconnecting:
			// should arrive here through a nice manual close
			ChangeStatus(ENotConnected, aError);
			iObserver.NotifyJabberStatus(MJabberObserver::EJabberDisconnected);
			// should we wait a bit??
			if (restart) ConnectL(iUsername, iPassword, iServerName, iAccessPoint);
			break;
                default:
			// unexpected error from Socket engine - not critical
			// no need to tell Jabber observer, just reconnect
			ChangeStatus(ENotConnected, aError);
			TRAPD(err, ConnectL(iUsername, iPassword, iServerName, iAccessPoint));			
			break;
		}
	} else { 
		ReportError(st);
	}
}

void CJabber::SendXMLStreamHeaderL()
{
	CALLSTACKITEM_N(_CL("CJabber"), _CL("SendXMLStreamHeaderL"));

	if ( (iJabberStatus == EInitiating) ) {
		TBuf16<200> header;
		header.Append(_L16("<?xml version='1.0'?><stream:stream"));
		header.Append(_L16(" xmlns='jabber:client' "));
		header.Append(_L16("xmlns:stream='http://etherx.jabber.org/streams'  to='"));
		header.Append(iServerName);
		header.Append(_L16("'>"));
		
		iEngine->WriteL(header);
	}
}



void CJabber::SendDisconnectionL()
{
	CALLSTACKITEM_N(_CL("CJabber"), _CL("SendDisconnectionL"));

	if ( (iJabberStatus == EDisconnecting) ) {
		TBuf16<200> msg;
		msg.Append(_L16("</stream:stream>"));
		iEngine->WriteL(msg);
	}
}



void CJabber::IdentifyL()
{
	CALLSTACKITEM_N(_CL("CJabber"), _CL("IdentifyL"));

	if (iJabberStatus == EIdentifying ) {
		TBuf8<100> iDigest;
		iDigest.Copy(iJabberSessionId);
		iDigest.Append(iPassword);
		
		SHA1 * sha1 = new SHA1;
		sha1->Input((char*)(iDigest.Ptr()),iDigest.Size());
		unsigned int message_digest_array[5];
		
		sha1->Result(message_digest_array);
		
		TBuf<40> digest;
		for (int i=0;i<5;i++)
		{
			TBuf<8> h;
			h.Format(_L("%08x"),message_digest_array[i]);
			digest.Append(h);
		}
		delete sha1;
		
		TBuf16<400> ident;
		
		ident.Append(_L16("<iq type='set'><query xmlns='jabber:iq:auth'><username>"));
		ident.Append(iUsername);
		
		ident.Append(_L16("</username><resource>"));
		ident.Append(iResource);
		ident.Append(_L16("</resource><digest>"));
		ident.Append(digest);	
		ident.Append(_L16("</digest></query></iq>"));
		
		iEngine->WriteL(ident);
	}
}

EXPORT_C TBool CJabber::IsConnected()
{
	return (iJabberStatus == EConnected);
}


void CJabber::DoSendPresenceInfoL()
{
	CALLSTACKITEM_N(_CL("CJabber"), _CL("DoSendPresenceInfoL"));

	if (iJabberStatus == EConnected && iUserPresenceInfo && iUserPresenceInfo->Des().Length()>0)
	{
		Log(_L("Jabber::SendingPresence"));
		iXmlBuf->Zero();
		auto_ptr<CPtrCArray> attr(new (ELeave) CPtrC16Array(1));
		attr->AppendL(_L16("from"));
		attr->AppendL(iFullNick.Mid(0));
		attr->AppendL(_L16("type"));
		attr->AppendL(_L16("available"));
		iXmlBuf->BeginElement(_L16("presence"), attr.get());
		
		iXmlBuf->Leaf(_L16("show"), _L16("xa"));
		iXmlBuf->BeginElement(_L16("status"));
		
		TBuf<15> timestamp;
		
		TTime time; time=GetTime();
		
		TDateTime dt;
		dt=time.DateTime();
		_LIT(KFormatTxt,"%04d%02d%02dT%02d%02d%02d");
		timestamp.Format(KFormatTxt, dt.Year(), (TInt)dt.Month()+1, (TInt)dt.Day()+1,
			dt.Hour(), dt.Minute(), dt.Second());
		
		iXmlBuf->Characters(timestamp);
		iXmlBuf->Characters(*iUserPresenceInfo);

		iXmlBuf->EndElement(_L16("status"));
		iXmlBuf->EndElement(_L16("presence"));
		
		iEngine->WriteL(iXmlBuf->Buf());
		
		iLastUpdateSent=GetTime();
		
		iSendTimer->Wait(iFreshnessInterval);
	}
}

EXPORT_C void CJabber::SendPresenceInfoL(const TDesC &presenceInfo)
{
	CALLSTACKITEM_N(_CL("CJabber"), _CL("SendPresenceInfoL"));

	Log(_L("Jabber::QueueingPresence To send"));

	//local copy of presenceInfo, in case of disconnection...
	while ( presenceInfo.Length()>iUserPresenceInfo->Des().MaxLength() ) {
		iUserPresenceInfo=iUserPresenceInfo->ReAllocL(iUserPresenceInfo->Des().MaxLength()*2);
	}
	*iUserPresenceInfo=presenceInfo;
	
	TTime now; now=GetTime();
	
	if (iJabberStatus == EConnected && now-iPresenceInterval > iLastUpdateSent)
	{
		iSendTimer->Reset();
		DoSendPresenceInfoL();
		
	} else if (iJabberStatus == EConnected) {
		//TBuf<50> msg;
		TTime next_send=iLastUpdateSent+iPresenceInterval;
		TInt w= ((next_send.Int64() - now.Int64())/(1000*1000)).Low();
		if (w<1) w=1;
		//msg.Format(_L("Queuing presence sending after %d secs"), w);
		iSendTimer->Wait(w);
	}
}

void CJabber::NotifyNewData(const TDesC8& aBuffer)
{
	if (!XML_Parse(iParser,(char*)(aBuffer.Ptr()),aBuffer.Size(), EFalse)) {
		XML_Error code=XML_GetErrorCode(iParser);
		TPtrC descr((TUint16*)XML_ErrorString(code));
		Log(_L("XML parse error"));
		Log(descr);
		ReportError(EXmlParseError/*, descr*/);
	}
}

//------------------------------------------------------------------------
// expat handlers

void  CJabber::startElement(void *userData, const XML_Char *el, const XML_Char **atts)
{
	CJabber* jab=(CJabber*)userData;
	jab->startElement(el, atts);
}

void  CJabber::endElement(void *userData, const XML_Char *name)
{
	CJabber* jab=(CJabber*)userData;
	jab->endElement(name);
}

void CJabber::charData(void *userData, const XML_Char *s, int len)
{
	CJabber* jab=(CJabber*)userData;
	jab->charData(s, len);
}

void  CJabber::startElement(const XML_Char *el, const XML_Char **atts)
{
	CALLSTACKITEM_N(_CL("CJabber"), _CL("startElement"));

	TInt currentState = EStackUndefined;
	
	if ( iStack->iCurrent != NULL )	{
		currentState = iStack->iCurrent->Item;
	}
	
	// if we're already ignoring parent tag
	if ( currentState == EStackIgnoreElement || currentState == EStackError)
	{
		// then let's ignore all children
		iStack->AppendL(EStackIgnoreElement);
		return;
	}
	
	TPtrC element = TPtrC( (TUint16*)el );
	
	if (element.Compare(KStream) == 0) {
		// Case: <stream:stream>
		iStack->AppendL(EStackSession);
		
		for (int i = 0; atts[i]; i += 2) {
			TPtrC attribute = TPtrC((TUint16*)atts[i]);
			if ( attribute.Compare(KId) == 0) {
				(iJabberSessionId).Copy(TPtrC((TUint16*)atts[i+1]));
			}
		}
		ChangeStatus(EIdentifying, KErrNone);
		IdentifyL();
	} else if ( element.Compare(KStreamError) == 0) {
		// Case: <stream:error>
		iStack->AppendL(EStackStreamError);
	} else if ( (element.Compare(KIq) == 0) && ( currentState == EStackSession ) ) {
		// Case: <iq>
		TBool found = EFalse;
		for (int i = 0; atts[i]; i += 2) {
			TPtrC attribute = TPtrC((TUint16*)atts[i]);
			TPtrC value = TPtrC((TUint16*)atts[i+1]);
			
			if ( attribute.Compare(KType) == 0 ) {
				if (value.Compare(KResult) == 0 ) {
					iStack->AppendL(EStackConnected);
					ChangeStatus(EConnected, KErrNone);
					iObserver.NotifyJabberStatus(MJabberObserver::EJabberConnected);
				} else if (value.Compare(KError) == 0) {
					iStack->AppendL(EStackIdentFailure);
				}
				found = ETrue;
			}
		}
		if (!found)
		{
			iStack->AppendL(EStackIgnoreElement);				
		}
	} else if ( element.Compare(KError) == 0 ) {
		// Case: <error>
		iErrorCode.Zero();
		
		delete iErrorValue;
		iErrorValue = iErrorValue = HBufC::NewL(0);
		
		for (int i=0; atts[i]; i+=2)
		{
			TPtrC attribute = TPtrC((TUint16*)atts[i]);
			TPtrC value = TPtrC((TUint16*)atts[i+1]);
			if ( attribute.Compare(KCode) == 0 )
			{
				iErrorCode.Copy(value);			
			}
		}
		iStack->AppendL(EStackError);	
	} else if (element.Compare(KPresence) == 0) {
		// Case: <presence>
		TBool contextBuddyFound = EFalse;
		TBool offlinePresenceInfo = EFalse;
		
		for (int i = 0; atts[i]; i += 2)
		{
			TPtrC attribute = TPtrC((TUint16*)atts[i]);
			TPtrC value = TPtrC((TUint16*)atts[i+1]);
			
			if ( (attribute.Compare(KFrom) == 0) && ( value.Find(KDefaultJabberRessource) != KErrNotFound) )
			{
				contextBuddyFound = ETrue;
				
				TInt len=value.Length() - KDefaultJabberRessource().Length() -1;
				if (iFrom->Des().MaxLength() < len) {
					iFrom = iFrom->ReAllocL(len);
				}
				*(iFrom) = value.Left(value.Length() - KDefaultJabberRessource().Length()-1);
			}
			else if ( (attribute.Compare(KType) == 0) && (value.Compare(KUnavailable)==0) )
			{
				offlinePresenceInfo = ETrue;
			}
		}
		if (contextBuddyFound)
		{
			if (!offlinePresenceInfo) 
			{ 
				iStack->AppendL(EStackPresenceInfo); 
			}
			else
			{
				iStack->AppendL(EStackOfflinePresenceInfo);	
			}
		}
		else
		{
			iStack->AppendL(EStackIgnoreElement);	
		}
	} else if (element.Compare(KMessage) == 0) {
		// Case: <message>
		for (int i = 0; atts[i]; i += 2)
		{
			TPtrC attribute = TPtrC((TUint16*)atts[i]);
			TPtrC value = TPtrC((TUint16*)atts[i+1]);
			
			if (attribute.Compare(KFrom) == 0)
			{
				TInt len=value.Length();
				if (iFrom->Des().MaxLength() < len) {
					iFrom = iFrom->ReAllocL(len);
				}
				*iFrom=value;
			}
		}
		iStack->AppendL(EStackMessage);
	} else if (element.Compare(KSubject)==0 && currentState == EStackMessage) {
		iStack->AppendL(EStackSubject);
	} else if (element.Compare(KBody)==0 && currentState == EStackMessage) {
		iStack->AppendL(EStackBody);
	} else if (element.Compare(KStatus) == 0) {
		// Case: <status>
		iStack->AppendL(EStackStatus);	
	} else {
		// Case: unhandled tag
		iStack->AppendL(EStackIgnoreElement);
	}
}



void  CJabber::endElement(const XML_Char *name)
{
	CALLSTACKITEM_N(_CL("CJabber"), _CL("endElement"));

	RDebug::Print(_L("-%s"),name);
	switch (iStack->iCurrent->Item)
	{
	case EStackIgnoreElement:
		// just pop the last item
		iStack->DeleteLast();
		break;
		
	case EStackIdentFailure:
		iStack->DeleteLast();
		ReportError(EIdentificationFailed);
		break;
		
	case EStackStatus:
		iStack->DeleteLast();
		break;
		
	case EStackPresenceInfo:
		{
			TInt timestamp_len=15;
			if (iPresenceInfo->Des().Length() > 0)
			{
				_LIT(KTime, "t");
				RDebug::Print(iPresenceInfo->Des().Left(timestamp_len));
				TBBTime stamp(KTime);
				TRAPD(err, stamp.FromStringL(iPresenceInfo->Des().Left(timestamp_len)));
				
				if (stamp() == TTime(0))
				{
					iObserver.NotifyNewPresenceInfo(*iFrom, 
					*iPresenceInfo, stamp());
				}
				else
				{
					iObserver.NotifyNewPresenceInfo(*iFrom, 
					iPresenceInfo->Mid(timestamp_len), stamp());
				}
			}
			
			// reset values
			
			iFrom->Des().Zero();
			
			iPresenceInfo->Des().Zero();
			iStack->DeleteLast();
			break;
		}
	case EStackMessage:
		{
			if (iMessage->Des().Length() > 0 || iSubject->Des().Length()>0) {
				iObserver.NotifyNewMessage(*iFrom, *iSubject, *iMessage);
			}
			iMessage->Des().Zero();
			iFrom->Des().Zero();
			iSubject->Des().Zero();
			iStack->DeleteLast();
		}
		break;
	case EStackBody:
	case EStackSubject:
		iStack->DeleteLast();
		break;
	case EStackOfflinePresenceInfo:
		//send something to context server to notify the offline status of contact
		
		//reset values
		
		delete iFrom;
		iFrom = iFrom = HBufC::NewL(0);
		
		iPresenceInfo->Des().Zero();
		
		iStack->DeleteLast();
		break;
		
	case EStackConnected:
		iStack->DeleteLast();
		if (iUserPresenceInfo->Length() >0) {
			SendPresenceInfoL(*iUserPresenceInfo);	
		}
		break;
		
	case EStackSession:
		// disconnect
		iStack->DeleteLast();
		iCloseSessionTimer->Reset();
		iEngine->Disconnect(ETrue);
		break;
		
	case EStackError:
		iStack->DeleteLast();
		break;
		
	case EStackStreamError:
		iStack->DeleteLast();
		ReportError(EStreamError);
		break;
	}
}

void CJabber::charData(const XML_Char *s, int len)
{
	CALLSTACKITEM_N(_CL("CJabber"), _CL("charData"));

	if (!iStack->iCurrent) return;
	
	if (iStack->iCurrent->Item == EStackStatus) {
		TPtrC t = TPtrC((TUint16*)s,len);
		while ( iPresenceInfo->Length()+len > iPresenceInfo->Des().MaxLength()) {
			iPresenceInfo=iPresenceInfo->ReAllocL(iPresenceInfo->Des().MaxLength()*2);
		}
		iPresenceInfo->Des().Append(t);
		
		//Log(*iPresenceInfo);
	} else if (iStack->iCurrent->Item == EStackError)	{
		iErrorValue = iErrorValue->ReAllocL(iErrorValue->Length()+len);
		iErrorValue->Des().Append((TUint16*)s,len);
	} else if (iStack->iCurrent->Item == EStackBody) {
		TPtrC t = TPtrC((TUint16*)s,len);
		while ( iMessage->Length()+len > iMessage->Des().MaxLength()) {
			iMessage=iMessage->ReAllocL(iMessage->Des().MaxLength()*2);
		}
		iMessage->Des().Append(t);
	} else if (iStack->iCurrent->Item == EStackSubject) {
		TPtrC t = TPtrC((TUint16*)s,len);
		while ( iSubject->Length()+len > iSubject->Des().MaxLength()) {
			iSubject=iSubject->ReAllocL(iSubject->Des().MaxLength()*2);
		}
		iSubject->Des().Append(t);
	}
}

void CJabber::expired(CBase* Source)
{
	if (Source==iSendTimer) {
		DoSendPresenceInfoL();
	} else if (Source== iCloseSessionTimer) { 
		iEngine->Disconnect(iCloseConnection);
	}
}

void CJabber::ReportError(TInt aErrorType)
{
	DoDisconnect(EFalse);

	switch (aErrorType)
	{
	case EIdentificationFailed:
		Log(_L("CJabber: Identification Failed") );
		Log(_L("ErrorDetail:"));
		Log(iErrorCode);
		Log(*iErrorValue);
		restart = EFalse;
		iObserver.NotifyJabberStatus(MJabberObserver::EIdentificationFailed);
		break;

	case EStreamError:
		Log(_L("CJabber: StreamError"));
		restart = ETrue;
		break;
			
	case EXmlParseError:
		Log(_L("CJabber: XML Parse Error"));
		restart = ETrue;
		break;
	default:
		// critical errors from Sockets Engine
		restart = EFalse;
		iObserver.NotifyJabberStatus(aErrorType);
		break;
	}
}

void CJabber::NotifyCanWrite()
{
}
