#include "transfer2.h"

#include "db.h"
#include "symbian_auto_ptr.h"
#include "cn_http.h"
#include "settings.h"
#include "cl_settings.h"
#include <bautils.h>
#include "file_output_base.h"
#include <plpvariant.h>
#include "bbutil.h"
#include "raii_f32file.h"
#include "raii_d32dbms.h"
#include "xmlbuf.h"
#include "expect.h"
#include "bbxml.h"
#include <apgcli.h>
#include "notifystate.h"
#include <contextnetwork.mbg>

_LIT(KUploadDir, "ToBeUploaded\\");
_LIT(KNoUploadDir, "NotShared\\");
_LIT(KUploadedDir, "Shared\\");
_LIT(KOldDir, "Old\\");

#ifndef __WINS__
_LIT(KIconFile, "c:\\system\\data\\contextnetwork.mbm");
#else
_LIT(KIconFile, "z:\\system\\data\\contextnetwork.mbm");
#endif



class CHttpTransfer2Impl : public CHttpTransfer2, public MContextBase, public MDBStore, public MSocketObserver,
	MHttpObserver {
private:
	// CHttpTransfer2
	virtual TFileName AddFileToQueueL(const TDesC& aFileName, TInt UrlSetting,
			TInt ScriptSetting, TBool DeleteAfter, const TDesC& aMeta,
			MBBData* aPacket, TInt aWaitTime=0, TBool aDontMove=EFalse);
	virtual TInt DeleteFile(const TDesC& aFileName);
	virtual TFileOpStatus DoNotTransfer(const TDesC& aFileName);
	virtual TFileOpStatus MoveToOld(const TDesC& aFileName);
	virtual TBool FileIsQueued(const TDesC& aFileName);
	virtual void Trigger(TBool Force=EFalse);

	CHttpTransfer2Impl(MApp_context& Context,
		RDbDatabase& Db, MSocketObserver* aStatusNotif);
	void ConstructL();

	friend class CHttpTransfer2;

	// CheckedActive
	void CheckedRunL();
	TInt CheckedRunError(TInt aError);
	void DoCancel();

	// MSocketObserver
	virtual void success(CBase* source);
	virtual void error(CBase* source, TInt code, const TDesC& reason);
	virtual void info(CBase* source, const TDesC& msg);

	// MHttpObserver
	virtual void NotifyHttpStatus(THttpStatus st, TInt aError);
	virtual void NotifyNewHeader(const CHttpHeader &aHeader);
	virtual void NotifyNewBody(const TDesC8 &chunk);

	// internal
	void StartFileTransferL();
	void ConnectL();
	void HandleSent();
	TBool MoveToCurrent();
	void Close();
	CPtrList<CPostPart>* GetFileInfoL(const TDesC& aFileName, const TDesC& PacketFileName);
	void Reset();
	void MakeFileName();
	TFileName AddFileToQueueL(const TDesC& aFileName, TInt UrlSetting,
			TInt ScriptSetting, TBool DeleteAfter, const TDesC& aMeta,
			MBBData* aPacket, const TDesC& PacketFileName, TInt aWaitTime, TBool aDontMove);

	enum state { IDLE, CONNECTING, MKDING, CWDING, SENDING_FILE, CLOSING };
	state current_state;
	enum TOp { UPLOAD, DELETE, DONTSHARE, MOVETOOLD };

	void QueueOpL(const TDesC& aSource, TOp Op, const TDesC& aDestination=KNullDesC);

	void DoNotTransferL(const TDesC& aSource, const TDesC& aDestination);
	void MoveToOldL(const TDesC& aSource, const TDesC& aDestination);

	TFileName GetDoNotTransferFileName(const TDesC& aFileName);
	TFileName GetMoveToOldFileName(const TDesC& aFileName);

	enum TUploadingStatus { EDone, EUploading, EErrors };
	void ShowUploading(TUploadingStatus aStatus);

	CHttp*		iHttp;
	TFileName	iTempFileName, iTempFileName2, iPacketFileName;
	TDbBookmark	iCurrentFileMark;
	MSocketObserver* iStatusNotif;
	RTimer		iTimer;
	TInt		iWaitTime;

	TInt		iRetries;
	TBuf<20>	iImsi;
	TBuf<20>	iDT, iPrevDT;
	TBuf<30>	iFileNameBuf;
	TInt		iDTCount;
	TInt		iFileNameIndexLen;
	bool		iNoTransfer;
	bool		iGotReply, iGotError;

	TInt		iIap, iProxyPort;
	TBuf<256>	iUrl, iProxy;
	CExpect*	iExpect;

	CNotifyState*	iNotifyState;
public:
	~CHttpTransfer2Impl();
};


EXPORT_C CHttpTransfer2* CHttpTransfer2::NewL(MApp_context& Context,
	RDbDatabase& Db, MSocketObserver* aStatusNotif)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2"), _CL("NewL"));


	auto_ptr<CHttpTransfer2Impl> ret(new (ELeave) CHttpTransfer2Impl(Context, Db, aStatusNotif));
	ret->ConstructL();
	return ret.release();
}

CHttpTransfer2::CHttpTransfer2() : CCheckedActive(EPriorityIdle, _L("CHttpTransfer2"))
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2"), _CL("CHttpTransfer2"));

}

EXPORT_C CHttpTransfer2::~CHttpTransfer2()
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2"), _CL("~CHttpTransfer2"));

}


void CHttpTransfer2Impl::MakeFileName()
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("MakeFileName"));

	TTime now; now=GetTime();
	format_datetime(iDT, now);
	iDT.Trim();

	iFileNameBuf=iDT;
	if (! (iDT.Compare(iPrevDT)) ) {
		++iDTCount;
		iFileNameBuf.Append(_L("-"));
		iFileNameBuf.AppendNum(iDTCount);
	} else {
		iDTCount=0;
	}
	iPrevDT=iDT;
}

TFileName CHttpTransfer2Impl::AddFileToQueueL(const TDesC& aFileName, TInt UrlSetting,
	TInt ScriptSetting, TBool DeleteAfter, const TDesC& aMeta,
	MBBData* aPacket, const TDesC& PacketFileName, TInt aWaitTime, TBool aDontMove) 
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("AddFileToQueueL"));

	iTable.Cancel();

	TParse p; p.Set(aFileName, 0, 0);

	iTable.InsertL();
	iTable.SetColL(1, aFileName);
	iTable.SetColL(2, UrlSetting);
	iTable.SetColL(3, ScriptSetting);
	iTable.SetColL(4, DeleteAfter);
	iTable.SetColL(5, aMeta.Left(KMaxFileName));
	iTable.SetColL(6, PacketFileName);
	iTable.SetColL(7, UPLOAD);
	iTable.SetColL(8, aDontMove);
	if (aPacket) {
		RADbColWriteStream w; w.OpenLA(iTable, 9);
		aPacket->Type().ExternalizeL(w);
		aPacket->ExternalizeL(w);
		w.CommitL();
	}
	PutL();

	if (!aDontMove) {
		iTempFileName = p.DriveAndPath();
		if (! iTempFileName.Right( KOldDir().Length() ).Compare( KOldDir )) {
			iTempFileName = p.DriveAndPath().Left(p.DriveAndPath().Length()-KOldDir().Length());
		}
		iTempFileName.Append(KUploadDir);
		BaflUtils::EnsurePathExistsL(Fs(), iTempFileName);
		iTempFileName.Append(iFileNameBuf);
		iTempFileName.Append(p.Ext());

		if (Fs().Rename(aFileName, iTempFileName)==KErrNone) {
			iTable.UpdateL();
			iTable.SetColL(1, iTempFileName);
			PutL();
		}
	}

	iWaitTime=aWaitTime;
	
	if (!aDontMove) return iTempFileName;
	else return aFileName;
}

TFileName CHttpTransfer2Impl::AddFileToQueueL(const TDesC& aFileName, TInt UrlSetting,
	TInt ScriptSetting, TBool DeleteAfter, const TDesC& aMeta,
	MBBData* aPacket, TInt aWaitTime, TBool aDontMove)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("AddFileToQueueL"));


	TParse p; p.Set(aFileName, 0, 0);

	MakeFileName();

	RDebug::Print(aFileName);
	iPacketFileName = p.DriveAndPath();
	if (! iPacketFileName.Right( KOldDir().Length() ).Compare( KOldDir )) {
		iPacketFileName = p.DriveAndPath().Left(p.DriveAndPath().Length()-KOldDir().Length());
	}
	iPacketFileName.Append(KUploadDir);
	iPacketFileName.Append(iFileNameBuf);
	iPacketFileName.Append(_L(".xml"));

	AddFileToQueueL(aFileName, UrlSetting, ScriptSetting, DeleteAfter, aMeta, 
		aPacket, iPacketFileName, aWaitTime, aDontMove);

	Trigger();
	return iPacketFileName;
}

void CHttpTransfer2Impl::QueueOpL(const TDesC& aFileName, TOp Op, const TDesC& aFileName2)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("QueueOpL"));

	iTable.Cancel();

	iTable.InsertL();
	iTable.SetColL(1, aFileName);
	iTable.SetColL(2, 0);
	iTable.SetColL(3, 0);
	iTable.SetColL(4, 0);
	iTable.SetColL(5, aFileName2);
	iTable.SetColL(6, _L(""));
	iTable.SetColL(7, Op);
	iTable.SetColL(8, EFalse);
	PutL();

	iWaitTime=5;
	Trigger();
}

TInt CHttpTransfer2Impl::DeleteFile(const TDesC& aFileName)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("DeleteFile"));


	if(Fs().Delete(aFileName)==KErrNone) return KErrNone;

	TRAPD(err, QueueOpL(aFileName, DELETE));
	return err;
}

void CHttpTransfer2Impl::DoNotTransferL(const TDesC& aSource, const TDesC& aDestination)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("DoNotTransferL"));

	User::LeaveIfError(Fs().Rename(aSource, aDestination));
	RDebug::Print(_L("moved %S to %S"), &aSource, &aDestination);
}

TFileOpStatus CHttpTransfer2Impl::DoNotTransfer(const TDesC& aFileName)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("DoNotTransfer"));

	TFileOpStatus ret;
	TFileName f = GetDoNotTransferFileName(aFileName);
	ret.fn=f;

	TRAP(ret.err, DoNotTransferL(aFileName, f));
	if (ret.err==KErrNone) return ret;
	TRAP(ret.err, QueueOpL(aFileName, DONTSHARE, f));
	return ret;
}

void CHttpTransfer2Impl::MoveToOldL(const TDesC& aSource, const TDesC& aDestination)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("MoveToOldL"));

	TFileName fn;
	TBool del=EFalse;
	ToDel(aSource, fn);
	if (BaflUtils::FileExists(Fs(), fn)) {
		del=ETrue;
	}
	ToPacket(aSource, fn);
	if (BaflUtils::FileExists(Fs(), fn)) {
		// compatibility with old upload
		MakeFileName();
		TRAPD(err,
			AddFileToQueueL(aSource, SETTING_PUBLISH_URLBASE, 
			SETTING_PUBLISH_SCRIPT, del, _L("unknown"), 0, fn, 0, EFalse));
		if (err==KErrNone) {
			ToDel(aSource, fn);
			Fs().Delete(fn);
		}
		User::LeaveIfError(err);
		return;
	}

	User::LeaveIfError(Fs().Rename(aSource, aDestination));
	RDebug::Print(_L("moved %S to %S"), &aSource, &aDestination);
}

TFileOpStatus CHttpTransfer2Impl::MoveToOld(const TDesC& aFileName)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("MoveToOld"));

	TFileOpStatus ret;
	TFileName f = GetMoveToOldFileName(aFileName);
	ret.fn=f;

	TRAP(ret.err, MoveToOldL(aFileName, f));
	if (ret.err==KErrNone) return ret;
	TRAP(ret.err, QueueOpL(aFileName, MOVETOOLD, f));
	return ret;
}

CHttpTransfer2Impl::CHttpTransfer2Impl(MApp_context& Context,
				       RDbDatabase& Db, MSocketObserver* aStatusNotif) : 
	MContextBase(Context), MDBStore(Db), iStatusNotif(aStatusNotif)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("CHttpTransfer2Impl"));


}

void CHttpTransfer2Impl::ConstructL()
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("ConstructL"));

	iFileNameIndexLen=50;

	TInt cols[] = { EDbColText, EDbColInt32, EDbColInt32, EDbColBit, EDbColText, 
		EDbColText, EDbColInt32, EDbColBit, EDbColLongBinary, -1 };
	TInt idx[] = { 1, -1 };

#ifndef __WINS__
	TPlpVariantMachineId machineId;
	PlpVariant::GetMachineIdL(machineId);
	iImsi=machineId;
#else
	// Return a fake IMEI when working on emulator
	_LIT(KEmulatorImsi, "244050000000000");
	iImsi.Copy(KEmulatorImsi);
#endif
	iTimer.CreateLocal();

	iExpect=CExpect::NewL();

	SetTextLen(KMaxFileName);

	MDBStore::ConstructL(cols, idx, false, _L("TRANSFER2"), ETrue);

	CActiveScheduler::Add(this);

	Trigger();
}

void CHttpTransfer2Impl::CheckedRunL()
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("CheckedRunL"));

	iTable.Cancel();

	if (current_state==IDLE) {
		if (iTable.FirstL()) {
			iCurrentFileMark=iTable.Bookmark();
			ConnectL();
			iWaitTime=0;
		} else {
			ShowUploading(EDone);
			return;
		}
		ShowUploading(EUploading);
	} else if (current_state==SENDING_FILE) {
		ShowUploading(EUploading);
		StartFileTransferL();
	}
}

TInt CHttpTransfer2Impl::CheckedRunError(TInt aError)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("CheckedRunError"));

	if (iStatusNotif) {
		if (iLatestErrorStack)
			iStatusNotif->error(this, aError, *iLatestErrorStack);
		else
			iStatusNotif->error(this, aError, _L("http2runl"));
	}
	if (iWaitTime==0) {
		iWaitTime=10;
	} else {
		iWaitTime=(TInt)(iWaitTime*1.5);
	}
	Reset();
	Trigger();
	return KErrNone;
}

void CHttpTransfer2Impl::DoCancel()
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("DoCancel"));

	iTimer.Cancel();
}

void CHttpTransfer2Impl::success(CBase* /*source*/)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("success"));

	iTable.Cancel();

	switch(current_state) {
	case CONNECTING:
		StartFileTransferL();
		break;
	case SENDING_FILE:
		{
		if (iStatusNotif) iStatusNotif->success(this);
		if (iStatusNotif) iStatusNotif->info(this, _L("transfered file"));

		HandleSent();
		if (iTable.NextL() || iTable.FirstL()) {
			iCurrentFileMark=iTable.Bookmark();
			Trigger(ETrue);
		} else {
			if (iStatusNotif) iStatusNotif->info(this, _L("all files transfered"));
			Close();
			ShowUploading(EDone);
		}
		}
		break;
	case CLOSING:
		Reset();
		iWaitTime=1;
		Trigger();
		break;
	}
}

TBool CHttpTransfer2Impl::MoveToCurrent()
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("MoveToCurrent"));


	TRAPD(err, iTable.GotoL(iCurrentFileMark));
	if (err!=KErrNone) return EFalse;
	return ETrue;
}

void CHttpTransfer2Impl::error(CBase* /*source*/, TInt code, const TDesC& reason)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("error"));

	iTable.Cancel();

	if (iStatusNotif) iStatusNotif->error(this, code, reason);

	if (current_state==SENDING_FILE) {
		if (MoveToCurrent() && iTable.NextL()) {
			iCurrentFileMark=iTable.Bookmark();
			iWaitTime=1;	
		} else if (iTable.FirstL()) {
			iCurrentFileMark=iTable.Bookmark();
			++iRetries;
			iWaitTime=300;
		} else {
			Reset();
			ShowUploading(EErrors);
			return;
		}
	} else {
		Reset();
		++iRetries;
		iWaitTime=300;
	}
	Trigger(ETrue);
	ShowUploading(EUploading);
}

void CHttpTransfer2Impl::info(CBase* /*source*/, const TDesC& msg)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("info"));


	if (iStatusNotif) iStatusNotif->info(this, msg);
}

void CHttpTransfer2Impl::Trigger(TBool Force)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("Trigger"));


	if (!Force && current_state!=IDLE) return;

	Cancel();
	ShowUploading(EUploading);

	if (iWaitTime==0) {
		TRequestStatus *s=&iStatus;
		User::RequestComplete(s, KErrNone);
	} else {
		TTimeIntervalMicroSeconds32 w(iWaitTime*1000*1000);
		iTimer.After(iStatus, w);
	}
	SetActive();
}

void CHttpTransfer2Impl::ConnectL()
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("ConnectL"));

	iTable.Cancel();
	TInt& Iap=iIap;
	Settings().GetSettingL(SETTING_IP_AP, Iap);

	if (Iap==-1) {
		// 'No connection'
		current_state=SENDING_FILE;
		iNoTransfer=true;
		Trigger(ETrue);
	} else {
		iNoTransfer=false;

		if (!iHttp) {
			iHttp=CHttp::NewL(*this, AppContext());
		}

		if (!MoveToCurrent()) {
			if (iStatusNotif) iStatusNotif->error(this, -1031, _L("current record disappeared"));
			Reset();
			return;
		}
		iTable.GetL();

		TInt url_setting, script_setting;

		url_setting=iTable.ColInt32(2);
		script_setting=iTable.ColInt32(3);

		TDes& url=iUrl;
		TBuf<50> script;
		Settings().GetSettingL(url_setting, url);
		Settings().GetSettingL(script_setting, script);
		url.Append(script);

		TDes& Proxy=iProxy;
		Settings().GetSettingL(SETTING_PROXY, Proxy);
		TInt&	ProxyPort=iProxyPort;
		Settings().GetSettingL(SETTING_PROXY_PORT, ProxyPort);

		current_state=CONNECTING;

		StartFileTransferL();
	}
}

void CHttpTransfer2Impl::Close()
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("Close"));

	current_state=CLOSING;
	success(this);
}

CHttpTransfer2Impl::~CHttpTransfer2Impl()
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("~CHttpTransfer2Impl"));

	Cancel();
	iTimer.Close();

	delete iHttp;
	delete iExpect;
	delete iNotifyState;
}

_LIT(KPacket, "packet");

CPtrList<CPostPart>*  CHttpTransfer2Impl::GetFileInfoL(const TDesC& aFileName,
				      const TDesC& PacketFileName)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("GetFileInfoL"));

	auto_ptr< CPtrList<CPostPart> > parts(CPtrList<CPostPart>::NewL());

	//TBuf<40> mime=_L("application/octect-stream");
	//if (aFileName.Right(3).CompareF(_L("jpg"))==0) mime=_L("image/jpeg");
	//if (aFileName.Right(3).CompareF(_L("txt"))==0) mime=_L("text/plain");
	//if (aFileName.Right(3).CompareF(_L("3gp"))==0) mime=_L("video/mpeg");
	//if (aFileName.Right(3).CompareF(_L("amr"))==0) mime=_L("audio/mpeg"); //FIXME
	
	TBuf<40> mime;
	GetMimeTypeL(aFileName, mime);

	auto_ptr<CFilePart> file(CFilePart::NewL(Fs(), aFileName, _L("photo"), mime));
	{
		TFileName aPostFileName=iImsi;
		aPostFileName.Append(_L("/"));
		TParse p; p.Set(aFileName, 0, 0);
		aPostFileName.Append(p.NameAndExt());
		file->SetFileName(aPostFileName);
	}

	if (BaflUtils::FileExists(Fs(), PacketFileName)) {
		TEntry fe;
		User::LeaveIfError(Fs().Entry(PacketFileName, fe));
		if (fe.iSize!=0) {
			auto_ptr<CFilePart> packet(CFilePart::NewL(Fs(), PacketFileName, _L("packet"), _L("text/xml")));
			packet->SetFileName(PacketFileName);
			parts->AppendL(packet.get());
			packet.release();
		}
	}

	if (! iTable.IsColNull(9) ) {
		RADbColReadStream s; s.OpenLA(iTable, 9);
		TTypeName tn=TTypeName::IdFromStreamL(s);
		auto_ptr<CBBDataFactory> f(CBBDataFactory::NewL());
		bb_auto_ptr<MBBData> d(f->CreateBBDataL(tn, KPacket, f.get()));
		d->InternalizeL(s);

		auto_ptr<CXmlBufExternalizer> xmlb(CXmlBufExternalizer::NewL(2048));
		xmlb->Declaration(_L("iso-8859-1"));
		d->IntoXmlL(xmlb.get());

		auto_ptr<HBufC8> b8(HBufC8::NewL( xmlb->Buf().Length()+20 ) );
		TPtr8 p=b8->Des();
		CC()->ConvertFromUnicode(p, xmlb->Buf() );

		auto_ptr<CBufferPart> bp(CBufferPart::NewL(b8.get(), ETrue, _L("packet"), _L("text/xml")));
		b8.release();

		TBuf<128> aPostFileName=iImsi;
		aPostFileName.Append(_L("/"));
		TParse fp; fp.Set(aFileName, 0, 0);
		aPostFileName.Append(fp.Name());
		aPostFileName.Append(_L("_"));
		aPostFileName.Append(fp.Ext().Mid(1));
		aPostFileName.Append(_L(".xml"));
		
		bp->SetFileName(aPostFileName);

		parts->AppendL(bp.get());
		bp.release();
	}
	parts->AppendL(file.get());
	file.release();
	return parts.release();
}

void CHttpTransfer2Impl::StartFileTransferL()
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("StartFileTransferL"));

	iGotReply=iGotError=false;
	iExpect->Reset();
	iExpect->SetPatternL(_L8("^OK"));

	iTable.Cancel();

	if (! MoveToCurrent()) {
		if (iStatusNotif) iStatusNotif->error(this, -1031, _L("current record disappeared"));
		Reset();
		return;
	}

	bool done=false;
	while (!done) {
		// skip files in use
		iTable.GetL();
		iTempFileName=iTable.ColDes16(1);
		iTempFileName2=iTable.ColDes16(5);
		RFile f;
		TInt err;
		err=f.Open(Fs(), iTempFileName, EFileRead|EFileShareAny);
		TOp Op=(TOp)iTable.ColInt32(7);
		TBool DontMove;
		if (iTable.IsColNull(8)) {
			DontMove=EFalse;
		} else {
			DontMove=iTable.ColInt32(8);
		}
		bool goto_next=false;
		if (err==KErrNone) {
			f.Close();
			switch (Op) {
			case UPLOAD:
				if (!DontMove) {
					if (iTempFileName.Find(KUploadDir)==KErrNotFound) {
						iPacketFileName=iTable.ColDes16(6);
						if (iPacketFileName.Length()>0) {
							_LIT(Kcontext, "context");
							TFileName filen; TParse p;
							if (iPacketFileName.FindC(Kcontext)!=KErrNotFound) {
								p.Set(iTempFileName, 0, 0);
								MakeFileName();
								filen=p.DriveAndPath();
								filen.Append(KUploadDir);
								filen.Append(iFileNameBuf);
								filen.Append(p.Ext());
							} else {
								p.Set(iTempFileName, 0, 0);
								filen=iPacketFileName;
								filen.Replace( filen.Length()-4, 4, p.Ext());
							}
							iTable.UpdateL();
							if ( Fs().Rename(iTempFileName, filen) == KErrNone ) {
								iTempFileName=filen;
								iTable.SetColL(1, iTempFileName);
							}
							PutL();
						}
					}
				}
				break;
			case DELETE:
				User::LeaveIfError(Fs().Delete(iTempFileName));
				goto_next=true;
				break;
			case DONTSHARE:
				DoNotTransferL(iTempFileName, iTempFileName2);
				goto_next=true;
				break;
			case MOVETOOLD:
				MoveToOldL(iTempFileName, iTempFileName2);
				goto_next=true;
				break;
			}
		}
		if (goto_next) {
			iTable.DeleteL();
			if (iTable.NextL()) {
				current_state=SENDING_FILE;
				iCurrentFileMark=iTable.Bookmark();
				iWaitTime=0;
				Trigger(ETrue);
			} else {
				Reset();
				iWaitTime=15;
				Trigger();
			}
			return;
		}
		if (err==KErrAccessDenied || err==KErrInUse) {
		} else if (err==KErrNotFound) {
			iPacketFileName=iTable.ColDes16(6);
			Fs().Delete(iPacketFileName);
			iTable.DeleteL();
		} else {
			iCurrentFileMark=iTable.Bookmark();
			done=true;
		}
		if (!done) {
			if (! iTable.NextL()) {
				Reset();
				iWaitTime=15;
				Trigger();
				return;
			}
		}
	}

	if (iNoTransfer) {
		if (iTable.NextL()) {
			iCurrentFileMark=iTable.Bookmark();
			Trigger();
		} else {
			Reset();
		}
	} else {
		if (!iHttp) {
			ConnectL();
		} else {

			iPacketFileName=iTable.ColDes16(6);

			CPtrList<CPostPart>* parts=GetFileInfoL(iTempFileName, iPacketFileName);

			current_state=SENDING_FILE;
			iHttp->PostL(iIap, iUrl, parts);
		}
	}
}

void CHttpTransfer2Impl::Reset()
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("Reset"));

	delete iHttp; iHttp=0;
	current_state=IDLE;
}

void CHttpTransfer2Impl::HandleSent()
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("HandleSent"));

	iTable.Cancel();

	if (! MoveToCurrent() ) {
		if (iStatusNotif) iStatusNotif->error(this, -1031, _L("current record disappeared"));
		Reset();
		return;
	}

	iTable.GetL();

	iTempFileName=iTable.ColDes16(1);
	iPacketFileName=iTable.ColDes16(6);

	Fs().Delete(iPacketFileName);

	TBool del=iTable.ColInt(4);
	if(del) {
		Fs().Delete(iTempFileName);
	} else {
		TFileName moveto=iTempFileName;
		TParse p;
		if (moveto.Find(KUploadDir)) {
			moveto.Replace(moveto.Find(KUploadDir), KUploadDir().Length(), KUploadedDir);
		} else {
			p.Set(iTempFileName, 0, 0);
			moveto=p.DriveAndPath();
			moveto.Append(KUploadedDir);
			MakeFileName();
			moveto.Append(iFileNameBuf);
			moveto.Append(p.Ext());
		}
		p.Set(moveto, 0, 0);
		BaflUtils::EnsurePathExistsL(Fs(), p.DriveAndPath());
		Fs().Rename(iTempFileName, moveto);
	}
	iTable.DeleteL();

}

TBool CHttpTransfer2Impl::FileIsQueued(const TDesC& aFileName)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("FileIsQueued"));
	iTable.Cancel();

	TDbSeekKey k(aFileName);
	TBool ret=iTable.SeekL(k);
	while(ret) {
		iTable.GetL();
		if ( aFileName.Left(iFileNameIndexLen).Compare(iTable.ColDes(1).Left(iFileNameIndexLen)) ) return EFalse;
		if ( aFileName.Compare(iTable.ColDes(1)) ) return ETrue;
		ret=iTable.NextL();
	}
	return EFalse;
}

void CHttpTransfer2Impl::NotifyHttpStatus(THttpStatus st, TInt aError)
{
	switch(st) {
		case EHttpConnected:
			if (iStatusNotif) iStatusNotif->info(this, _L("connected"));
			break;
		case EHttpDisconnected:
			if (iGotReply) {
				iHttp->ReleaseParts();
				success(iHttp);
			} else if (iGotError) {
				//ignore
			} else {
				error(iHttp, aError, _L("http disconnected"));
			}
			break;
		case EHttpError:
			iGotError=true;
			error(iHttp, aError, _L("http error"));
			break;
	};

}

void CHttpTransfer2Impl::NotifyNewHeader(const CHttpHeader &aHeader)
{
	if (aHeader.iHttpReplyCode < 200 || aHeader.iHttpReplyCode>299) {
		iGotError=true;
		TBuf<40> err=_L("server replied with ");
		err.AppendNum(aHeader.iHttpReplyCode);
		error(iHttp, KErrGeneral, err);
	}
}

void CHttpTransfer2Impl::NotifyNewBody(const TDesC8 &chunk)
{
	if (iGotReply || iGotError) return;
	TInt i;
	for (i=0; i< chunk.Length(); i++) {
		if (iExpect->handle_input( chunk[i] )) {
			iGotReply=true;
			return;
		}
	}
}

EXPORT_C void GetMimeTypeL(const TDesC& FileName, TDes& aBuf)
{
	CALLSTACKITEM_N(_CL("CHttpTransfer2Impl"), _CL("NotifyNewBody"));

	aBuf.Zero();
	aBuf.Append(_L("application/octect-stream"));

	TUid appUid(KNullUid);
	RApaLsSession ls;
	User::LeaveIfError(ls.Connect());
	TDataType dataType;
	TInt err = ls.AppForDocument(FileName, appUid, dataType);
	if ((err==KErrNone) && (dataType.Des().Length()!=0)){
		aBuf.Zero();	
		aBuf.Append(dataType.Des());
	}
	ls.Close();
	RDebug::Print(_L("file: %S"), &FileName);
	RDebug::Print(_L("mime: %S"), &aBuf);
}

EXPORT_C void ToPacket(const TDesC& filen, TDes& packet)
{
	CALLSTACKITEM_N(_CL("RDebug"), _CL("Print"));

	TInt dirpos=filen.LocateReverse('\\');
	packet=filen.Left(dirpos);
	packet.Append(_L("\\context"));
	packet.Append(filen.Mid(dirpos));
	TInt extpos=packet.LocateReverse('.');
	if (extpos!=KErrNotFound) {
		packet[extpos]='_';
	}
	packet.Append(_L(".xml"));
}

EXPORT_C void ToDel(const TDesC& filen, TDes& del)
{
	CALLSTACKITEM_N(_CL("TDummyPrompt"), _CL("TDummyPrompt"));

	TInt dirpos=filen.LocateReverse('\\');
	del=filen.Left(dirpos);
	del.Append(_L("\\context"));
	del.Append(filen.Mid(dirpos));

	TInt extpos=del.LocateReverse('.');
	if (extpos!=KErrNotFound) {
		del[extpos]='_';
	}
	del.Append(_L(".del"));
}

TFileName CHttpTransfer2Impl::GetDoNotTransferFileName(const TDesC& aFileName)
{
	TFileName filen;
	TParse p; p.Set(aFileName, 0, 0);
	filen = p.DriveAndPath();
	if (! filen.Right( KOldDir().Length() ).Compare( KOldDir )) {
		filen = p.DriveAndPath().Left(p.DriveAndPath().Length()-KOldDir().Length());
	}
	filen.Append(KNoUploadDir);
	BaflUtils::EnsurePathExistsL(Fs(), filen);

	MakeFileName();
	filen.Append(iFileNameBuf);
	filen.Append(p.Ext());

	return filen;
}


TFileName CHttpTransfer2Impl::GetMoveToOldFileName(const TDesC& aFileName)
{
	TFileName fn;
	
	TParse p; p.Set(aFileName, 0, 0);
	fn = p.DriveAndPath();
	if (! fn.Right( KOldDir().Length() ).Compare( KOldDir ) ) {
		return aFileName;
	}
	fn.Append(KOldDir);
	BaflUtils::EnsurePathExistsL(Fs(), fn);
	MakeFileName();
	fn.Append(iFileNameBuf);
	fn.Append(p.Ext());

	return fn;
}

void CHttpTransfer2Impl::ShowUploading(TUploadingStatus aStatus)
{
	if ( aStatus == EDone ) {
		delete iNotifyState; iNotifyState=0;
	} else {
		if (! iNotifyState ) {
			iNotifyState=CNotifyState::NewL(AppContext(), KIconFile);
		}
		if (aStatus == EUploading) {
			iNotifyState->SetCurrentState(EMbmContextnetworkUploading, EMbmContextnetworkUploading);
		} else {
			iNotifyState->SetCurrentState(EMbmContextnetworkUploading, EMbmContextnetworkUploading_err);
		}
	}
}
