#include "transferdir2.h"
#include "transfer2.h"
#include "list.h"
#include "symbian_auto_ptr.h"
#include "db.h"
#include "cl_settings.h"
#include <bautils.h>
#include "reporting.h"

class CTransferDirImpl : public CTransferDir, public MContextBase, public MUploadCallBack {
private:
	TInt ProcessDir(const TDesC& aDirName, TInt UrlSetting,
		TInt ScriptSetting, const TDesC& aMeta,  MUploadPrompt* Prompt,
		TTime TimeLimit, TBool aMoveToMMC, TBool aDontMoveToSubDirs);
	CTransferDirImpl(MApp_context& Context, MSocketObserver& aObserver);
	void ConstructL(const TDesC& aDbName);
	TFileOpStatus Back(bool Upload, bool DeleteFromPhone, MBBData* Packet);
	virtual CHttpTransfer2* Transferer();

	struct TDirItem {
		TFileName	iDirName;
		TInt		iUrlSetting;
		TInt		iScriptSetting;
		TFileName	iMeta;
		MUploadPrompt*  iPrompt;
		TTime		iTimeLimit;
		TBool		iMoveToMMC;
		TBool		iDontMoveToSubDirs;

		TDirItem(const TDesC& aDirName, TInt UrlSetting,
			TInt ScriptSetting, const TDesC& aMeta, MUploadPrompt* Prompt, TTime TimeLimit,
			TBool aMoveToMMC, TBool aDontMoveToSubDirs) : 
				iDirName(aDirName),
				iUrlSetting(UrlSetting), iScriptSetting(ScriptSetting), iPrompt(Prompt),
				iTimeLimit(TimeLimit), iMoveToMMC(aMoveToMMC), iDontMoveToSubDirs(aDontMoveToSubDirs)
		{
			iMeta=aMeta.Left(KMaxFileName);
		}
	};

	CList<TDirItem*>	*iList;
	TDirItem		*iCurrentDir;
	TInt			iCurrentItem;
	CDir			*iDir;
	CDb			*iDb;
	CHttpTransfer2		*iTransfer;
	TFileName		iCurrentDirName;
	MSocketObserver&	iObserver;
	enum TState { EIdle, EProcessing };
	TState			iCurrentState;

	void CheckedRunL();
	TInt CheckedRunError(TInt aError);
	void DoCancel();

	void FileStep();
	void Async();
	TFileName DoBackL(bool Upload, bool DeleteFromPhone, MBBData* Packet);

	friend class CTransferDir;
	bool has_memory_card;
public:
	~CTransferDirImpl();
};


EXPORT_C CTransferDir* CTransferDir::NewL(MApp_context& Context, MSocketObserver& aObserver, const TDesC& aDbName)
{
	CALLSTACKITEM_N(_CL("CTransferDir"), _CL("NewL"));


	auto_ptr<CTransferDirImpl> ret(new (ELeave) CTransferDirImpl(Context, aObserver));
	ret->ConstructL(aDbName);
	return ret.release();
}

CTransferDir::CTransferDir() : CCheckedActive(EPriorityIdle, _L("CTransferDir"))
{
	CALLSTACKITEM_N(_CL("CTransferDir"), _CL("CTransferDir"));


}

EXPORT_C CTransferDir::~CTransferDir() { }

TInt CTransferDirImpl::ProcessDir(const TDesC& aDirName, TInt UrlSetting,
	TInt ScriptSetting, const TDesC& aMeta, MUploadPrompt* Prompt, TTime TimeLimit, 
	TBool aMoveToMMC, TBool aDontMoveToSubDirs)
{
	CALLSTACKITEM_N(_CL("CTransferDirImpl"), _CL("ProcessDir"));

	TDirItem* i=new (ELeave) TDirItem(aDirName, UrlSetting, ScriptSetting, aMeta, 
		Prompt, TimeLimit, aMoveToMMC, aDontMoveToSubDirs);
	TRAPD(err, iList->AppendL(i));
	if (err!=KErrNone) delete i;
	else {
		if (iCurrentState!=EIdle) return KErrNone;
		Async();
	}
	return err;
}

CTransferDirImpl::CTransferDirImpl(MApp_context& Context, MSocketObserver& aObserver) : 
		MContextBase(Context), iObserver(aObserver)
{
	CALLSTACKITEM_N(_CL("CTransferDirImpl"), _CL("CTransferDirImpl"));

}

void CTransferDirImpl::ConstructL(const TDesC& aDbName)
{
	CALLSTACKITEM_N(_CL("CTransferDirImpl"), _CL("ConstructL"));

	TBool use_mmc=ETrue;
	Settings().GetSettingL(SETTING_USE_MMC, use_mmc);
	has_memory_card=false;
	TDriveInfo i;
	if (use_mmc && Fs().Drive(i, EDriveE)==KErrNone) {
		// memory card
		has_memory_card=true;
	}

	iList=CList<TDirItem*>::NewL();
	iDb=CDb::NewL(AppContext(), aDbName, EFileRead|EFileWrite|EFileShareAny);

	CActiveScheduler::Add(this);

	iTransfer=CHttpTransfer2::NewL(AppContext(), iDb->Db(), &iObserver);
}

void CTransferDirImpl::CheckedRunL()
{
	CALLSTACKITEM_N(_CL("CTransferDirImpl"), _CL("CheckedRunL"));


	if (iCurrentState==EIdle) {
		delete iCurrentDir; iCurrentDir=0;
		iCurrentDir=iList->Pop();
		if (!iCurrentDir) return;

		delete iDir; iDir=0;
		TInt err=Fs().GetDir(iCurrentDir->iDirName, KEntryAttNormal, ESortByName, iDir);
		{
			TParse p; p.Set(iCurrentDir->iDirName, 0, 0);
			iCurrentDirName = p.DriveAndPath();
			RDebug::Print(iCurrentDirName);
		}
		if (err!=KErrNone) {
			Async();
			return;
		}
		iCurrentItem=0;
		FileStep();
	} else {
		FileStep();
	}
}

TInt CTransferDirImpl::CheckedRunError(TInt aError)
{
	CALLSTACKITEM_N(_CL("CTransferDirImpl"), _CL("CheckedRunError"));


	return aError;
}

void CTransferDirImpl::DoCancel()
{
	CALLSTACKITEM_N(_CL("CTransferDirImpl"), _CL("DoCancel"));
}

TFileName CTransferDirImpl::DoBackL(bool Upload, bool DeleteFromPhone, MBBData* Packet)
{
	CALLSTACKITEM_N(_CL("CTransferDirImpl"), _CL("DoBackL"));

	TFileOpStatus f;

	TEntry e=(*iDir)[iCurrentItem-1];
	TFileName filen=iCurrentDirName; filen.Append(e.iName);
	if (Upload) {
		Reporting().DebugLog(_L("CTransferDirImpl::DoBackL queuing"));
		Reporting().DebugLog(filen);
		TRAPD(err, iTransfer->AddFileToQueueL(filen, iCurrentDir->iUrlSetting,
			iCurrentDir->iScriptSetting, DeleteFromPhone,
			iCurrentDir->iMeta, Packet, iCurrentDir->iDontMoveToSubDirs));
		if (err!=KErrNone) {
			User::Leave(err);
		}
	} else {
		if (DeleteFromPhone) {
			iTransfer->DeleteFile(filen);
		} else {
			f=iTransfer->DoNotTransfer(filen);
		}
	}
	return f.fn;
}

TFileOpStatus CTransferDirImpl::Back(bool Upload, bool DeleteFromPhone, MBBData* Packet)
{
	CALLSTACKITEM_N(_CL("CTransferDirImpl"), _CL("Back"));

	TFileOpStatus ret;
	TFileName fn;
	TRAP(ret.err, ret.fn = DoBackL(Upload, DeleteFromPhone, Packet));
	if (ret.err!=KErrNone) {
		TAknGlobalNoteType notetype=EAknGlobalErrorNote;
		TBuf<100> msg=_L("Failed to queue file for upload: ");
		msg.AppendNum(ret.err);
		Reporting().ShowGlobalNote(notetype, msg);
	}
	Async();
	return ret;
}

void CTransferDirImpl::FileStep()
{
	CALLSTACKITEM_N(_CL("CTransferDirImpl"), _CL("FileStep"));

	if (iCurrentItem < iDir->Count()) {
		iCurrentState=EProcessing;
		TEntry e=(*iDir)[iCurrentItem++];
		TFileName filen=iCurrentDirName; filen.Append(e.iName);

		TBool use_mmc=EFalse;
		if (iCurrentDir->iMoveToMMC && has_memory_card && filen.Left(1).CompareF(_L("c"))==0 && 
				Settings().GetSettingL(SETTING_USE_MMC, use_mmc) && use_mmc &&
				e.iName.Left(12).Compare(_L("cellid_names")) ) {
			TFileName filen2;
			filen2=filen;
			filen2.Replace(0, 1, _L("e"));
			if (BaflUtils::CopyFile(Fs(), filen, filen2) == KErrNone) {
				BaflUtils::DeleteFile(Fs(), filen);
				filen=filen2;
			}
		}

		if (iTransfer->FileIsQueued(filen)) {
			Async();
		} else {
			if (e.iModified > iCurrentDir->iTimeLimit) {
				iCurrentDir->iPrompt->Prompt( filen, this );
			} else {
				iTransfer->MoveToOld( filen );
				Async();
			}
		}
	} else {
		iCurrentState=EIdle;
		Async();
	}
}

void CTransferDirImpl::Async()
{
	CALLSTACKITEM_N(_CL("CTransferDirImpl"), _CL("Async"));


	if (IsActive()) return;

	TRequestStatus *s=&iStatus;
	User::RequestComplete(s, KErrNone);
	SetActive();
}

CTransferDirImpl::~CTransferDirImpl()
{
	CALLSTACKITEM_N(_CL("CTransferDirImpl"), _CL("~CTransferDirImpl"));

	Cancel();
	delete iTransfer;
	delete iDb;

	if (iList) {
		TDirItem* i;
		while (i=iList->Pop()) delete i;
	}
	delete iCurrentDir;
	delete iList;
	delete iDir;
}

CHttpTransfer2* CTransferDirImpl::Transferer()
{
	return iTransfer;
}