/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


//CContextNotify.cpp
#include "ContextNotify.h"
#include "ContextNotifySession.h"
#include "NotifyCommon.h"
#include <E32SVR.H>
#include <basched.h>

#include <flogger.h>

void Log(const TDesC& msg);

bool ClientAlive(TInt ThreadId);

#pragma warning(disable:4706)

CContextNotify::CContextNotify(TInt aPriority) : CServer(aPriority)
{
	
}

CContextNotify::~CContextNotify()
{
	delete iForeground;
	delete iDrawer;
	if (ws_is_open) iWsSession.Close();
}

CContextNotify* CContextNotify::NewL()
{
	CContextNotify* ContextNotifyer = CContextNotify::NewLC();
    	CleanupStack::Pop(ContextNotifyer);
    	return ContextNotifyer;
}

CContextNotify* CContextNotify::NewLC()
{
	CContextNotify* ContextNotifyer = new (ELeave) CContextNotify(EPriorityNormal);
    	CleanupStack::PushL(ContextNotifyer);
    	ContextNotifyer->ConstructL();
    	return ContextNotifyer;
}

void CContextNotify::ConstructL()
{
	User::LeaveIfError(iWsSession.Connect());
	ws_is_open=true;
	iDrawer=CDrawer::NewL(iWsSession);
	iForeground=CForeground::NewL(iWsSession);
	iForeground->AddObserver(iDrawer);

	StartL(KContextNotifyName);
}

CSharableSession* CContextNotify::NewSessionL(const TVersion& aVersion) const
{
	// check version
	if (!User::QueryVersionSupported(TVersion(KContextNotifyMajorVersionNumber,
                                                KContextNotifyMinorVersionNumber,
                                                KContextNotifyBuildVersionNumber),
                                                aVersion))
	{
		User::Leave(KErrNotSupported);
	}
	
	// create new session
	RThread client = Message().Client();
	return CContextNotifySession::NewL(client, *const_cast<CContextNotify*> (this));
}


void CContextNotify::IncrementSessions()
{
    	iSessionCount++;
}

void CContextNotify::DecrementSessions()
{
    	iSessionCount--;
    	if (iSessionCount <= 0)
	{
      		iSessionCount =0;
		CActiveScheduler::Stop();
	}    
}

TInt CContextNotify::CheckedRunError(TInt aError)
{
	if (aError == KErrBadDescriptor)
	{
        	// A bad descriptor error implies a badly programmed client, so panic it;
        	// otherwise report the error to the client
        	PanicClient(Message(), EBadDescriptor);
	}
	else
	{
		Message().Complete(aError);
	}

	//
	// The leave will result in an early return from CServer::RunL(), skipping
	// the call to request another message. So do that now in order to keep the
	// server running.
	ReStart();

	return KErrNone;	// handled the error fully
}

void CContextNotify::PanicClient(const RMessage& aMessage, TContextNotifyPanic aPanic)
{
	aMessage.Panic(KContextNotify, aPanic);
}

void CContextNotify::PanicServer(TContextNotifyPanic aPanic)
{
    	User::Panic(KContextNotify, aPanic);
}

void CContextNotify::RunL()
{
	TRAPD(err, CServer::RunL());
	if (err!=KErrNone) {
		TBuf<20> msg;
		msg.Format(_L("Error in RunL: %d"), err);
		Log(msg);
		CheckedRunError(err);
	}
}

void CContextNotify::ThreadFunctionL()
{
	#if defined (__WINS__)
	UserSvr::ServerStarted();
	#endif

	__UHEAP_MARK;

	{
    	CActiveScheduler* activeScheduler = new (ELeave) CBaActiveScheduler;

    	CleanupStack::PushL(activeScheduler) ;

    	// Install active scheduler
    	// We don't need to check whether an active scheduler is already installed
    	// as this is a new thread, so there won't be one
    	CActiveScheduler::Install(activeScheduler);

    	// Construct our server
	CContextNotify::NewLC();    

	RSemaphore semaphore;
	if (semaphore.CreateGlobal(KContextNotifySemaphoreName, 0)!=KErrNone) {
		User::LeaveIfError(semaphore.OpenGlobal(KContextNotifySemaphoreName));
	}
	
	// Semaphore opened ok
	semaphore.Signal();
	semaphore.Close();

	// Start handling requests
	CActiveScheduler::Start();
	CleanupStack::PopAndDestroy(2); // CServer, activeScheduler, 
	}
		 
	__UHEAP_MARKEND;

}

TInt CContextNotify::ThreadFunction(TAny* /*aNone*/)
{
    	CTrapCleanup* cleanupStack = CTrapCleanup::New();
	if (cleanupStack == NULL)
	{
      	PanicServer(ECreateTrapCleanup);
	}

	TRAPD(err, ThreadFunctionL());
    	if (err != KErrNone)
	{
        	PanicServer(ESrvCreateServer);
	}
	
    	delete cleanupStack;
    	cleanupStack = NULL;

	return KErrNone;
}


//---------------------------------------------------------------------------

void CContextNotify::TerminateContextNotify()
{
	NotifySessions(ETerminated);
	//Log();
	CActiveScheduler::Stop();
}

void CContextNotify::NotifySessions(TEvent aEvent)
{
	CContextNotifySession* session=0;

	iSessionIter.SetToFirst();
	while( (session = reinterpret_cast<CContextNotifySession*>(iSessionIter++)) ) {
		session->NotifyEvent(aEvent);
	}
}

void CContextNotify::ReportError(TContextNotifyRqstComplete aErrorType, TDesC & aErrorCode, TDesC & aErrorValue)
{
	CContextNotifySession* session=0;

	iSessionIter.SetToFirst();
	while( (session = reinterpret_cast<CContextNotifySession*>(iSessionIter++)) ) {
		session->ReportError(aErrorType, aErrorCode, aErrorValue);
	}
}

void CContextNotify::CancelRequest(const RMessage &aMessage)
{
	// there's nothing to cancel (at least yet)
	switch(aMessage.Function()) {
	default:
		break;
	}
}

TInt CContextNotify::AddIconL(TInt aIconHandle, TInt aMaskHandle)
{
	CFbsBitmap* bm=new (ELeave) CFbsBitmap;
	CleanupStack::PushL(bm);
	User::LeaveIfError(bm->Duplicate(aIconHandle));
	CFbsBitmap* mask=new (ELeave) CFbsBitmap;
	CleanupStack::PushL(mask);
	User::LeaveIfError(mask->Duplicate(aMaskHandle));
	TInt id=iDrawer->AddIconL(bm, mask);
	CleanupStack::Pop(2);
	return id;
}

void CContextNotify::ChangeIconL(TInt aId, TInt aIconHandle, TInt aMaskHandle)
{
	CFbsBitmap* bm=new (ELeave) CFbsBitmap;
	CleanupStack::PushL(bm);
	User::LeaveIfError(bm->Duplicate(aIconHandle));
	CFbsBitmap* mask=new (ELeave) CFbsBitmap;
	CleanupStack::PushL(mask);
	User::LeaveIfError(mask->Duplicate(aMaskHandle));
	iDrawer->ChangeIconL(bm, mask, aId);
	CleanupStack::Pop(2);
}

void CContextNotify::RemoveIcon(TInt aId)
{
	iDrawer->RemoveIcon(aId);
}
