/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


//CContextNotifySession.cpp

#include "ContextNotifySession.h"
#include "NotifyCommon.h"
#include <e32svr.h>
#include <flogger.h>
#include "symbian_auto_ptr.h"

bool ClientAlive(TInt ThreadId)
{
	if (ThreadId==0) return false;

	RThread c; bool ret = false;
	if (c.Open(ThreadId) != KErrNone) return false;
	if (c.ExitType() == EExitPending) ret=true;
	c.Close();
	return ret;
}


CContextNotifySession::CContextNotifySession(RThread& aClient, CContextNotify& aServer) : CSession(aClient), iServer(aServer)
{
    
}

CContextNotifySession::~CContextNotifySession()
{
	if (iIds) {
		CList<TInt>::Node* n;
		for (n=iIds->iFirst; n; n=n->Next) {
			iServer.RemoveIcon(n->Item);
		}
	}
	delete iIds;
    	iServer.DecrementSessions();
}

void CContextNotifySession::CompleteMessage(TInt Code)
{
	if (ClientAlive(iMessageThreadId)) {
		iMessage.Complete(Code);
	}
	SetMessage(RMessage());
}

void CContextNotifySession::ServiceL(const RMessage& aMessage)
{
	if (aMessage.Function()!=ECancel) SetMessage(aMessage);

	switch (aMessage.Function())
	{
		case ETerminateContextNotify:
			CompleteMessage(ERequestCompleted);
			iServer.TerminateContextNotify();
			break;

		case EAddIcon:
			AddIcon();
			break;
		case ERemoveIcon:
			RemoveIcon();
			break;
		case EChangeIcon:
			ChangeIcon();
			break;
		case ECancel:
			{
			RMessage prev=iMessage;
			CompleteMessage(KErrCancel);
			iServer.CancelRequest(prev);
			aMessage.Complete(0);
			break;
			}

		default :
            	PanicClient(EBadRequest);
	}

}

void CContextNotifySession::AddIcon()
{
	TInt handle, mask, id=-1;
	handle=(TInt)iMessage.Ptr0();
	mask=(TInt)iMessage.Ptr1();
	TInt err;
	TRAP(err, id=iServer.AddIconL(handle, mask));
	if (err==KErrNone) {
		TPckgC<TInt> idp(id);
		TRAP(err, iMessage.WriteL(iMessage.Ptr2(), idp, 0));
	}
	if (err==KErrNone) {
		TRAP(err, iIds->AppendL(id));
	}
	if (err!=KErrNone) {
		iServer.RemoveIcon(id);
	}
	CompleteMessage(err);
}

void CContextNotifySession::RemoveIcon()
{
	TInt id;
	id=(TInt)iMessage.Ptr0();
	iServer.RemoveIcon(id);
	CompleteMessage(KErrNone);
}

void CContextNotifySession::ChangeIcon()
{
	TInt handle, mask, id;
	id=(TInt)iMessage.Ptr2();
	handle=(TInt)iMessage.Ptr0();
	mask=(TInt)iMessage.Ptr1();
	TRAPD(err, iServer.ChangeIconL(id, handle, mask));
	CompleteMessage(err);
}

void CContextNotifySession::PanicClient(TInt aPanic) const
{
	Panic(KContextNotify, aPanic) ; // Note: this panics the client thread, not server
}

CContextNotifySession* CContextNotifySession::NewL(RThread& aClient, CContextNotify& aServer)
{
	CContextNotifySession* self = CContextNotifySession::NewLC(aClient, aServer) ;
    	CleanupStack::Pop(self) ;
    	return self ;
}

CContextNotifySession* CContextNotifySession::NewLC(RThread& aClient, CContextNotify& aServer)
{
	CContextNotifySession* self = new (ELeave) CContextNotifySession(aClient, aServer);
    	CleanupStack::PushL(self) ;
    	self->ConstructL() ;
    	return self ;
}

void CContextNotifySession::ConstructL()
{
	iIds=CList<TInt>::NewL();
	iServer.IncrementSessions();
}

//------------------------------------------------------------------------

void CContextNotifySession::TerminateContextNotify(const RMessage& /*aMessage*/)
{
	CompleteMessage(ERequestCompleted);
	iServer.TerminateContextNotify();
}

void CContextNotifySession::SetMessage(const RMessage& aMessage) 
{
	if (aMessage==RMessage()) {
		iMessageThreadId=0;
	} else {
		iMessageThreadId=aMessage.Client().Id();
	}
	iMessage=aMessage;
}

void CContextNotifySession::NotifyEvent(CContextNotify::TEvent aEvent)
{
	if (!ClientAlive(iMessageThreadId)) return;

	if (aEvent==CContextNotify::ETerminated) {
		CompleteMessage(EContextNotifyTerminated);
	}
}

void CContextNotifySession::ReportError(TContextNotifyRqstComplete aErrorType, 
					TDesC & /*aErrorCode*/, TDesC & /*aErrorValue*/)
{
	CompleteMessage(aErrorType);
}
