#include "drawer.h"
#include "symbian_auto_ptr.h"
#include <e32std.h>
#include <w32std.h>
#include <coedef.h>
#include <apgwgnam.h>

/*
 * Concepts:
 * !Drawing on top of Phone screen!
 */

const TUid KUidPhone = { 0x100058b3 };
const TUid KUidMenu = { 0x101f4cd2 };

const TInt KIconSep = 1;

class CDrawerImpl : public CDrawer {
public:
	virtual TInt	AddIconL(CFbsBitmap* data, CFbsBitmap* mask);
	virtual void	RemoveIcon(TInt Id);
	virtual void	ChangeIconL(CFbsBitmap* data, CFbsBitmap* mask, TInt Id);
	CDrawerImpl(RWsSession& aWsSession);
	virtual ~CDrawerImpl();

	virtual void ForegroundChanged(TUid AppUid);
        
	void ConstructL();
	void StartL();
	void Listen();
	void Release();
	void Draw(const TRect& aRect);

	void RunL();
	TInt RunError(TInt aError);
	void DoCancel();

	TBool reception_bar_on_screen();

	RWsSession& iWsSession; 
        CWindowGc* gc;
	CWsScreenDevice* screen;
	RWindowGroup wg; bool wg_is_open;
	RWindow	w; bool w_is_open;

	RPointerArray<CFbsBitmap> iIcons;
	RPointerArray<CFbsBitmap> iMasks;
	bool	full_redraw;
};

CDrawer::CDrawer(CActive::TPriority aPriority) : CActive(aPriority)
{
}

CDrawer::~CDrawer()
{
}

CDrawer* CDrawer::NewL(RWsSession& aWsSession)
{
	auto_ptr<CDrawerImpl> ret(new (ELeave) CDrawerImpl(aWsSession));
	ret->ConstructL();
	return ret.release();
}

TInt	CDrawerImpl::AddIconL(CFbsBitmap* aIcon, CFbsBitmap* aMask)
{
	User::LeaveIfError(iIcons.Append(aIcon));
	TInt err=iMasks.Append(aMask);
	if (err!=KErrNone) {
		iIcons.Remove(iIcons.Count()-1);
		User::Leave(err);
	}
	TSize ws=w.Size(), is=aIcon->SizeInPixels();
	ws.iWidth+=is.iWidth+KIconSep;
	if (is.iHeight > ws.iHeight) ws.iHeight=is.iHeight;
	w.SetSize(ws);

	full_redraw=true; Draw(TRect());
	return iIcons.Count()-1;
}

void	CDrawerImpl::RemoveIcon(TInt Id)
{
	if (Id<0 || Id >= iIcons.Count()) return;

	TSize ws=w.Size(), is=iIcons[Id]->SizeInPixels();

	delete iIcons[Id];
	iIcons[Id]=0;
	delete iMasks[Id];
	iMasks[Id]=0;

	ws.iWidth-=(is.iWidth+KIconSep);
	w.SetSize(ws);
	full_redraw=true; Draw(TRect());
}

void	CDrawerImpl::ChangeIconL(CFbsBitmap* data, CFbsBitmap* aMask, TInt Id)
{
	if (Id<0 || Id >= iIcons.Count()) User::Leave(KErrNotFound);

	delete iIcons[Id];
	iIcons[Id]=data;
	delete iMasks[Id];
	iMasks[Id]=aMask;

	full_redraw=true; Draw(TRect());
}

CDrawerImpl::CDrawerImpl(RWsSession& aWsSession) : CDrawer(EPriorityHigh), iWsSession(aWsSession)
{
}

void CDrawerImpl::ConstructL()
{
        screen=new (ELeave) CWsScreenDevice(iWsSession);
        User::LeaveIfError(screen->Construct());

        User::LeaveIfError(screen->CreateContext(gc));

        wg=RWindowGroup(iWsSession);
        User::LeaveIfError(wg.Construct((TUint32)&wg, EFalse));
	wg_is_open=true;
        wg.SetOrdinalPosition(1, ECoeWinPriorityAlwaysAtFront+1);
        wg.EnableReceiptOfFocus(EFalse);

	CApaWindowGroupName* wn=CApaWindowGroupName::NewLC(iWsSession);
	wn->SetHidden(ETrue);
	wn->SetWindowGroupName(wg);
	CleanupStack::PopAndDestroy();

        w=RWindow(iWsSession);
        User::LeaveIfError(w.Construct(wg, (TUint32)&w));
	w_is_open=true;

        w.Activate();
//<<<<<<< Drawer.cpp
        //w.SetExtent( TPoint(1, 45), TSize(0, 0) );
//	w.SetExtent( TPoint(156, 1), TSize(0,0));
//=======
        w.SetExtent( TPoint(1, 46), TSize(0, 0) );
//>>>>>>> 1.5
        w.SetBackgroundColor( KRgbWhite );
        w.SetOrdinalPosition(1, ECoeWinPriorityAlwaysAtFront+1);
        w.SetNonFading(ETrue);

	TInt wgid=iWsSession.GetFocusWindowGroup();
	CApaWindowGroupName* gn;
	gn=CApaWindowGroupName::NewLC(iWsSession, wgid);

        ForegroundChanged(gn->AppUid());
	CleanupStack::PopAndDestroy();

	CActiveScheduler::Add(this);

	Listen();
}

CDrawerImpl::~CDrawerImpl()
{
	Cancel();

	int i;
	for (i=0; i<iIcons.Count(); i++) {
		delete iIcons[i];
		delete iMasks[i];
	}
	iIcons.Close(); iMasks.Close();

	delete gc;
	delete screen;
	if (w_is_open) w.Close();
	if (wg_is_open) wg.Close();
}

void CDrawerImpl::StartL()
{
}

void CDrawerImpl::Listen()
{
	iStatus=KRequestPending;
	iWsSession.RedrawReady(&iStatus);
	SetActive();
}

void CDrawerImpl::Release()
{
}

void CDrawerImpl::RunL()
{
	TWsRedrawEvent e;
	iWsSession.GetRedraw(e);

	Draw(e.Rect());

	Listen();
}

void CDrawerImpl::Draw(const TRect& serverRect)
{
	TRect aRect=serverRect;

	gc->Activate(w);

	if (full_redraw) {
		aRect=TRect(TPoint(0,0), w.Size());
	}
	full_redraw=false;
	w.Invalidate(aRect);
	w.BeginRedraw();

	gc->SetPenStyle(CGraphicsContext::ENullPen);
	gc->SetBrushColor(KRgbWhite);
	gc->SetBrushStyle(CGraphicsContext::ESolidBrush);
	gc->DrawRect(aRect);

	int xpos=0, i;
	for (i=0; i<iIcons.Count(); i++) {
		if (iIcons[i]!=0 && iMasks[i]!=0) {
			xpos+=KIconSep;
			TRect r( TPoint(0, 0), iIcons[i]->SizeInPixels());
			gc->BitBltMasked(TPoint(xpos, 0), iIcons[i], r, iMasks[i], ETrue);
			xpos+=r.Width();
		}
	}

	w.EndRedraw();
	gc->Deactivate();
	iWsSession.Flush();
}

TInt CDrawerImpl::RunError(TInt aError)
{
	return aError;
}

void CDrawerImpl::DoCancel()
{
	iWsSession.RedrawReadyCancel();
}

void CDrawerImpl::ForegroundChanged(TUid AppUid)
{

#ifdef __WINS__
	if (AppUid==KUidMenu) {
#else
	if (AppUid==KUidPhone) {
#endif
		w.SetVisible(ETrue);
	} else {
		w.SetVisible(EFalse);
	}
}