/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


//CContextNotifyClientSession.cpp

#include "NotifyCommon.h"
#include "ContextNotifyClientSession.h"
#include <e32math.h>
#include <eikenv.h>
#include "list.h"
#include "app_context.h"
#include <eikspane.h>
#include <eikimage.h>
#include <avkon.hrh>
#include <avkon.rsg>
#include <bautils.h>

static const TUint KDefaultMessageSlots = 2;
static const TUid KServerUid3 = {0x10204BAA}; // matches UID in server/group/ContextNotifyer.mbm file

_LIT(KContextNotifyFilename, "ContextNotify");

#ifdef __WINS__
static const TUint KServerMinHeapSize =  0x1000;  //  4K
static const TUint KServerMaxHeapSize = 0x10000;  // 64K
#endif

static TInt StartServer();
static TInt CreateServerProcess();


/*
 * Concepts:
 * !Customizing the status pane!
 */

class CLocalNotifyWindowImpl : public CLocalNotifyWindow, 
		public MContextBase {
	CLocalNotifyWindowImpl() { }

	~CLocalNotifyWindowImpl() {
		CALLSTACKITEM_N(_CL("CLocalNotifyWindowImpl"), _CL("CLocalNotifyWindowImpl"));

		if (iResource) iEikonEnv->DeleteResourceFile(iResource);

		CList<TSlot>::Node	*i;
		for (i=iSlots->iFirst; i; i=i->Next) {
			delete i->Item.iImage;
		}
		delete iSlots;
		delete iToBeNotified;
	}

	void SetPositions() {
		iPrevIndex=0; iPrevNode=iSlots->iFirst;

		TInt xpos=Rect().Width()-9, ypos=0;

		CList<TSlot>::Node	*i;
		for (i=iSlots->iFirst; i; i=i->Next) {
			ypos += 1;
			i->Item.iImage->SetPosition(TPoint(xpos, ypos));
			ypos += i->Item.iImage->Size().iHeight;
		}
	}

	void DrawOnGc(CWindowGc& gc, TPoint& aFromPos) {
		TInt xpos=aFromPos.iX, ypos=aFromPos.iY;

		CList<TSlot>::Node	*i;
		for (i=iSlots->iFirst; i; i=i->Next) {
			ypos += 1;
			TRect r( TPoint(0, 0), i->Item.iImage->Bitmap()->SizeInPixels());
			gc.BitBltMasked(TPoint(xpos, ypos), 
				i->Item.iImage->Bitmap(), r, i->Item.iImage->Mask(), ETrue);

			ypos += i->Item.iImage->Size().iHeight;
		}
	}

	void ConstructL() {
		CALLSTACKITEM_N(_CL("CLocalNotifyWindowImpl"), _CL("ConstructL"));

		TFileName resfile=_L("z:\\System\\data\\avkon.rsc");
		BaflUtils::NearestLanguageFile(iEikonEnv->FsSession(), resfile); //for localization
		TRAPD(err, iResource=iEikonEnv->AddResourceFileL(resfile));
		User::LeaveIfError(err);
		{
			HBufC8* res = 
				iEikonEnv->AllocReadResourceAsDes8LC(R_AVKON_STATUS_PANE_TITLE_DEFAULT);
			TResourceReader theReader;
			theReader.SetBuffer(res);
			CAknTitlePane::ConstructFromResourceL(theReader);
			CleanupStack::PopAndDestroy();
		}
		iSp=iEikonEnv->AppUiFactory()->StatusPane();
		iSlots=CList<TSlot>::NewL();
		iToBeNotified=CList<CCoeControl*>::NewL();

	}

	void Redraw() {
		iSp->DrawNow();
		CList<CCoeControl*>::Node *i;
		for (i=iToBeNotified->iFirst; i; i=i->Next) {
			i->Item->DrawNow();
		}
	}
	virtual TInt AddIconL(CFbsBitmap* aIcon, CFbsBitmap* aMask) {
		CALLSTACKITEM_N(_CL("CLocalNotifyWindowImpl"), _CL("AddIconL"));

		CFbsBitmap* bm=new (ELeave) CFbsBitmap;
		CleanupStack::PushL(bm);
		User::LeaveIfError(bm->Duplicate(aIcon->Handle()));
		CFbsBitmap* mask=new (ELeave) CFbsBitmap;
		CleanupStack::PushL(mask);
		User::LeaveIfError(mask->Duplicate(aMask->Handle()));
		TInt id=iLastId++;

		CEikImage* im=new (ELeave) CEikImage();
		im->SetPicture(bm, mask);
		im->SetPictureOwnedExternally(EFalse);
		CleanupStack::Pop(2);
		CleanupStack::PushL(im);
		im->SetSize(bm->SizeInPixels());

		iSlots->AppendL(TSlot(id, im));
		CleanupStack::Pop(1);

		SetPositions();
		Redraw();
		return id;

	}
	virtual void ChangeIconL(CFbsBitmap* aIcon, CFbsBitmap* aMask, TInt aId)
	{
		CALLSTACKITEM_N(_CL("CLocalNotifyWindowImpl"), _CL("ChangeIconL"));

		CList<TSlot>::Node	*i;
		for (i=iSlots->iFirst; i; i=i->Next) {
			if (i->Item.iId==aId) break;
		}
		if (!i) User::Leave(KErrNotFound);
		CFbsBitmap* bm=new (ELeave) CFbsBitmap;
		CleanupStack::PushL(bm);
		User::LeaveIfError(bm->Duplicate(aIcon->Handle()));
		CFbsBitmap* mask=new (ELeave) CFbsBitmap;
		CleanupStack::PushL(mask);
		User::LeaveIfError(mask->Duplicate(aMask->Handle()));

		CEikImage* im=new (ELeave) CEikImage();
		im->SetPicture(bm, mask);
		im->SetPictureOwnedExternally(EFalse);
		im->SetSize(bm->SizeInPixels());
		CleanupStack::Pop(2);
		CleanupStack::PushL(im);

		delete i->Item.iImage;
		i->Item.iImage=im;
		CleanupStack::Pop(1);

		SetPositions();
		Redraw();
	}
	virtual void RemoveIcon(TInt aId) {
		CALLSTACKITEM_N(_CL("CLocalNotifyWindowImpl"), _CL("RemoveIcon"));

		CList<TSlot>::Node	*i;
		for (i=iSlots->iFirst; i; i=i->Next) {
			if (i->Item.iId==aId) {
				delete i->Item.iImage;
				iSlots->DeleteNode(i, ETrue);
				break;
			}
		}

		SetPositions();
		Redraw();
	}


	TInt CountComponentControls() const {
		TInt ret=CAknTitlePane::CountComponentControls();
		return ret+iSlots->iCount;
	}
	CCoeControl* ComponentControl(TInt aIndex) const {
		if (aIndex < CAknTitlePane::CountComponentControls()) {
			return CAknTitlePane::ComponentControl(aIndex);
		} else {
			aIndex-=CAknTitlePane::CountComponentControls();
			if (aIndex==0) {
				iPrevNode=iSlots->iFirst;
				iPrevIndex=0;
			} else {
				while (iPrevIndex < aIndex) {
					iPrevNode=iPrevNode->Next;
					iPrevIndex++;
				}
			}
			return iPrevNode->Item.iImage;
		}
	}

	virtual void AddNotifiedControl(CCoeControl* aControl) {
		iToBeNotified->AppendL(aControl);
	}
	virtual void RemoveNotifiedControl(CCoeControl* aControl) {
		CList<CCoeControl*>::Node *i;
		for (i=iToBeNotified->iFirst; i; i=i->Next) {
			if (i->Item == aControl) {
				iToBeNotified->DeleteNode(i, ETrue);
				return;
			}
		}
	}

	TInt	iLastId;
	mutable TInt	iRefCount;

	struct TSlot {
		TInt		iId;
		CEikImage*	iImage;

		TSlot() : iId(-1), iImage(0) { }
		TSlot(TInt aId, CEikImage* aImage) :
		iId(aId), iImage(aImage) { }
	};
	CList<TSlot>*	iSlots;
	CList<CCoeControl*> *iToBeNotified;
	CEikStatusPane* iSp;
	CAknTitlePane*	iPrevious;

	mutable CList<TSlot>::Node *iPrevNode; 
	mutable TInt iPrevIndex;
	TInt iResource;

	friend class CLocalNotifyWindow;
};

EXPORT_C void CNotifyWindowControl::Draw(const TRect& ) const
{
	CWindowGc& gc = SystemGc();
	TPoint pos=Position();
	iLocal->DrawOnGc(gc, pos);
}

EXPORT_C void CNotifyWindowControl::ConstructL(CCoeControl *aTopLevel)
{
	iTopLevel=aTopLevel;
	iLocal=CLocalNotifyWindow::Global();
	if (iLocal) iLocal->AddNotifiedControl(iTopLevel);
}

EXPORT_C CNotifyWindowControl::~CNotifyWindowControl()
{
	if (iLocal)  {
		iLocal->RemoveNotifiedControl(iTopLevel);
	}
}

CLocalNotifyWindow::~CLocalNotifyWindow() { }

EXPORT_C CLocalNotifyWindow* CLocalNotifyWindow::Global()
{
	if (! CEikonEnv::Static()) return 0;
	CLocalNotifyWindowImpl* w=(CLocalNotifyWindowImpl*)Dll::Tls();
	return w;
}

EXPORT_C void CLocalNotifyWindow::Destroy()
{
	if (! CEikonEnv::Static()) return;
	CLocalNotifyWindowImpl* w=(CLocalNotifyWindowImpl*)Dll::Tls();

	if (!w) return;
	CEikStatusPane* sp=CEikonEnv::Static()->AppUiFactory()->StatusPane();
	sp->SwapControlL( TUid::Uid(EEikStatusPaneUidTitle), w->iPrevious);
	User::LeaveIfError(Dll::SetTls(0));
	delete w;
}

EXPORT_C void CLocalNotifyWindow::CreateAndActivateL()
{
	if (! CEikonEnv::Static()) return;
	void* existing=Dll::Tls();
	CLocalNotifyWindowImpl* w=0;
	if (existing) {
		User::Leave(KErrAlreadyExists);
	} else {
		auto_ptr<CLocalNotifyWindowImpl> wp(new (ELeave) CLocalNotifyWindowImpl);
		w=wp.get();

		CEikStatusPane* sp=CEikonEnv::Static()->AppUiFactory()->StatusPane();

		// FIXME: if an error occurs, a partially constructed object
		// is left as the title pane
		//
		CAknTitlePane* tp=w;
		CAknTitlePane* oldtitle=(CAknTitlePane*)
			sp->SwapControlL( TUid::Uid(EEikStatusPaneUidTitle), tp);
#ifndef __WINS__
// crashes on 2.0 emu on redraw?
		w->SetMopParent(oldtitle);
#endif
		w->ConstructL();
		w->iPrevious=oldtitle;
		wp.release();
		User::LeaveIfError( Dll::SetTls(w) );

		w->SetTextL(*oldtitle->Text());
		w->SetRect( oldtitle->Rect() );
	}
}

EXPORT_C RContextNotifyClientSession::RContextNotifyClientSession() : RSessionBase(), iIdPckg(0)
{
	
}

EXPORT_C TInt RContextNotifyClientSession::Connect()
{
	TInt result;
	
	result = ::StartServer();
	if (result == KErrNone)
	{
		result = CreateSession(KContextNotifyName,
			Version(),
			KDefaultMessageSlots);	
	}
	return result;
}

EXPORT_C void RContextNotifyClientSession::Close()
{
	RSessionBase::Close();
	delete iIdPckg; iIdPckg=0;
}

EXPORT_C TVersion RContextNotifyClientSession::Version() const
{
	return(TVersion(KContextNotifyMajorVersionNumber,
		KContextNotifyMinorVersionNumber,
		KContextNotifyBuildVersionNumber));
}

//----------------------------------------------------------------------------------------

EXPORT_C void RContextNotifyClientSession::Cancel() const
{
	SendReceive(ECancel, NULL);
}

EXPORT_C void RContextNotifyClientSession::TerminateContextNotify(TRequestStatus& aStatus)
{
	SendReceive(ETerminateContextNotify, NULL, aStatus);
}

EXPORT_C void RContextNotifyClientSession::AddIcon(CFbsBitmap* aIcon, CFbsBitmap* aMask, TInt& aId, TRequestStatus& aStatus)
{
	delete iIdPckg; iIdPckg=0; iIdPckg=new (ELeave) TPckg<TInt>(aId);

	TAny* messageParameters[KMaxMessageArguments];
	messageParameters[0] = (void*)aIcon->Handle();
	messageParameters[1] = (void*)aMask->Handle();
	messageParameters[2] = (void*)iIdPckg;

	SendReceive(EAddIcon, &messageParameters[0], aStatus);
}

EXPORT_C void RContextNotifyClientSession::RemoveIcon(TInt aId, TRequestStatus& aStatus)
{
	TAny* messageParameters[KMaxMessageArguments];
	messageParameters[0] = (void*)aId;
	SendReceive(ERemoveIcon, &messageParameters[0], aStatus);
}

EXPORT_C void RContextNotifyClientSession::ChangeIcon(CFbsBitmap* aIcon, CFbsBitmap* aMask, TInt aId, TRequestStatus& aStatus)
{
	TAny* messageParameters[KMaxMessageArguments];
	messageParameters[0] = (void*)aIcon->Handle();
	messageParameters[1] = (void*)aMask->Handle();
	messageParameters[2] = (void*)aId;

	SendReceive(EChangeIcon, &messageParameters[0], aStatus);
}

static TInt StartServer()
{
	TInt result;
	
	TFindServer findContextNotify(KContextNotifyName);
	TFullName name;
	
	result = findContextNotify.Next(name);
	if (result == KErrNone)
        {
		// Server already running
		return KErrNone;
        }
	
	RSemaphore semaphore;        
	result = semaphore.CreateGlobal(KContextNotifySemaphoreName, 0);
	if (result != KErrNone)
        {
		result=semaphore.OpenGlobal(KContextNotifySemaphoreName);
        }
	if (result != KErrNone)
        {
		return  result;
	}

	result = CreateServerProcess();
	if (result != KErrNone)
        {
		return  result;
        }
	
	semaphore.Wait();
	semaphore.Close();       
	
	return  KErrNone;
}

static TInt CreateServerProcess()
{
	TInt result;
	
	const TUidType serverUid(KNullUid, KNullUid, KServerUid3);
	
#ifdef __WINS__
	
	RLibrary lib;
	result = lib.Load(KContextNotifyFilename, _L("z:\\system\\programs"), serverUid);
	if (result != KErrNone)
        {
		return  result;
        }
	
	//  Get the WinsMain function
	TLibraryFunction functionWinsMain = lib.Lookup(1);
	
	//  Call it and cast the result to a thread function
	TThreadFunction serverThreadFunction = reinterpret_cast<TThreadFunction>(functionWinsMain());
	
	TName threadName(KContextNotifyName);
	
	// Append a random number to make it unique
	threadName.AppendNum(Math::Random(), EHex);
	
	RThread server;
	
	result = server.Create(threadName,   // create new server thread
		serverThreadFunction, // thread's main function
		KDefaultStackSize,
		NULL,
		&lib,
		NULL,
		KServerMinHeapSize,
		KServerMaxHeapSize,
		EOwnerProcess);
	
	lib.Close();    // if successful, server thread has handle to library now
	
	if (result != KErrNone)
        {
		return  result;
        }
	
	server.SetPriority(EPriorityMore);
	
	
#else
	
	RProcess server;
	
	result = server.Create(KContextNotifyFilename, _L(""), serverUid);
	if (result != KErrNone)
        {
		return  result;
        }
	
#endif
	
	server.Resume();
	server.Close();
	
	return KErrNone;
}
